/*
 * V Kernel - Copyright (c) 1984 by David Cheriton, Tim Mann
 *
 * Header file describing Sun-2 memory mapping and onboard device
 *   addresses.  See the Sun-2 Architecture Manual (if you can get one)
 *   for details.  Some of the definitions may only be valid for "machine
 *   type 1", the Multibus-based Sun-2 implementation.
 *
 * This file uses current SMI terminology, which differs from older V
 *   terminology.  In particular, PMEG = "page map entry group" is
 *   equivalent to the old term "physical segment".
 */

#ifndef MEMORY
#define MEMORY

/* Location of the interrupt vector page */
#define VECTOR_BASE 0

typedef unsigned long	Bit32;
typedef unsigned short	Bit16;
typedef unsigned char	Bit8;
typedef long		sBit32;
typedef short		sBit16;
typedef char		sBit8;

/* Routines to read and write function-code-3 space, exported by sun2mem.c */
extern Bit8 Fc3ReadBit8(/*addr*/);		/* read an 8-bit quantity */
extern Bit16 Fc3ReadBit16(/*addr*/);		/* read a 16-bit quantity */
extern Bit32 Fc3ReadBit32(/*addr*/);		/* read a 32-bit quantity */
extern void Fc3WriteBit8(/*addr, value*/);	/* write an 8-bit quantity */
extern void Fc3WriteBit16(/*addr, value*/);	/* write a 16-bit quantity */
extern void Fc3WriteBit32(/*addr, value*/);	/* write a 32-bit quantity */

/* CPU Layer (Architecure manual, chapter 4) */
/* Addresses and formats for various things in "function code 3" space.
 *  These are accessed using the MOVS instruction with the
 *  value 3 in the function code registers */

/* - Identification PROM (read only)*/
#define IdProm(k) (0x8+(k<<11))		/* address of PROM byte k */
#define IDP_FORMAT		0	/* format byte */
#define IDP_MACHINE_TYPE	1	/* machine type byte */
#define IDP_ENET_ADDRESS	2	/* start of Enet address (6 bytes) */
#define IDP_DATE		8	/* start of manuf. date (4 bytes) */
#define IDP_SERIAL_NUMBER	12	/* start of serial number (3 bytes) */
#define IDP_CHECKSUM		15	/* checksum (1 byte) */
#define IDP_RESERVED		16	/* reserved field (16 bytes) */

/* - Diagnostic LED register (1 byte, write only) */
#define DIAGNOSTIC_REGISTER	0xA

/* - Bus error register (16 bits, read only) */
#define BUS_ERROR_REGISTER	0xC
typedef union
  {
    struct
      {
	Bit16 reserved1:8;
	Bit16 pageValid:1;	/* 1=Error on valid page, 0=on invalid page */
	Bit16 reserved2:3;
	Bit16 protection:1;	/* 1=Protection violation */
	Bit16 timeout:1;	/* 1=Timeout */
	Bit16 parityUpper:1;	/* 1=Parity error, upper byte */
        Bit16 parityLower:1;	/* 1=Parity error, lower byte */
      }
    f;
    Bit16 u;	/* Whole thing as an unsigned quantity */
  }
BusErrorRegister;

/* - System Enable register (16 bits, read/write) */
#define SYSTEM_ENABLE_REGISTER	0xE
typedef struct
  {
    Bit16 reserved:8;
    Bit16 notBootState:1;	/* 0=Boot, 1=Normal */
    Bit16 enableInt:1;		/* Enable all interrupts */
    Bit16 enableDvma:1;		/* Enable Direct Virtual Memory Access */
    Bit16 enableParityCheck:1;	/* Enable parity checking */
    Bit16 causeInt3:1;		/* Cause a level 3 interrupt */
    Bit16 causeInt2:1;		/* Cause a level 2 interrupt */
    Bit16 causeInt1:1;		/* Cause a level 1 interrupt */
    Bit16 enableParityGen:1;	/* Enable parity generation */
  }
SystemEnableRegister;

/* - MMU entries for virtual address v in the current address space */
#define PageMapAddr(v)		(0x0+(((Bit32)(v))&0xfff800))
#define SegMapAddr(v)		(0x5+(((Bit32)(v))&0xff8000))  /* as a byte */
#define CONTEXT_REGISTER	0x6


/* MMU hardware characteristics (manual, chapter 5) */

/* Size of a page, a segment, and a context (address space) */
#define PAGE_SIZE	0x800		/* 2 Kbytes */
#define SEG_SIZE	0x8000		/* 32 Kbytes */
#define CONTEXT_SIZE	0x1000000	/* 16 Mbytes */

#define NUM_CONTEXTS	8		/* Total number of contexts */
#define	NUM_PMEGS	256		/* Number of PMEGs available */

#define SEGS_PER_CTX	(CONTEXT_SIZE/SEG_SIZE)	/* 512 */
#define PGS_PER_SEG	(SEG_SIZE/PAGE_SIZE)	/* 16 */

/* - Structure of the context register */
typedef union
  {
    /* Fields */
    struct
      {
        Bit8 supervisor;	/* Supervisor state context number */
        Bit8 user;		/* User state context number */
      }
    f;
    Bit16 u;	/* Whole thing as an unsigned quantity */
  }
ContextRegister;

/* For LRU segment map swapping */
#define MAP_NOT_RESIDENT 0xffff	/* an invalid context register value */
#define GetAddressableTeam() AddressableTeam
extern	struct _TD *AddressableTeam;

/* - Segment map entry */
typedef Bit8 SegMapEntry;

/* - Page map entry */
typedef union
  {
    /* Fields */
    struct
      {
        Bit32 valid:1;		/* 1 = Valid page */
	Bit32 protection:6;	/* Supervisor rwx, user rwx */
	Bit32 type:3;		/* Page type; see below */
	Bit32 accessed:1;	/* Page has been accessed */
	Bit32 modified:1;	/* Page has been modified */
	Bit32 pagenum:20;	/* Page number (physical address) */
      }
    f;
    Bit32 u;	/* Whole thing as an unsigned integer */
  }
PageMapEntry;
	
/* Protection bits */
#define SR		040	/* Supervisor read */
#define SW		020	/* Supervisor write */
#define SX		010	/* Supervisor execute */
#define UR		004	/* User read */
#define UW		002	/* User write */
#define UX		001	/* User execute */

/* Page types */
#define ONBOARD_MEM	0	/* "Onboard" (P2 bus) */
#define ONBOARD_IO	1	/* Onboard devices */
#define MULTIBUS_MEM	2	/* Multibus memory address space */
#define MULTIBUS_IO	3	/* Multibus I/O address space */

/* - Structure of a virtual address */
typedef union
  {
    /* Fields */
    struct
      {
	Bit32	unused:8;	/* High 8 bits unused in 68010 */
	Bit32	segment:9;	/* Segment number */
	Bit32	page:4;		/* Page number within segment */
	Bit32	byte:11;	/* Byte number */
      }
    f;
    char *p;		/* Whole thing as a character pointer */
    Bit32 u;		/* Whole thing as an unsigned 32-bit quantity */
  }
VirtualAddress;


/* Extract the page number (for the pagenum field of a page table entry)
 *   from a physical address */
#define ExtractPageNumber(p) ((p)>>11)	/* effectively same as p%PAGE_SIZE */
/* Extract the segment number from an address */
#define ExtractSegNumber(p) ((p)>>15)	/* effectively same as p%SEG_SIZE */

/*
 * Physical addresses for things in onboard memory space (machine type 1).
 *   These are addressed through page map entries of type ONBOARD_MEM.
 */
#define P_ONBOARD_RAM		0
#define P_ONBOARD_RAM_LIMIT	0x400000	/* No more than 4 Mbytes */
#define P_BW_FRAMEBUFFER	0x700000	/* "new" frame buffer */
#define P_KB_MOUSE_UART		0x780000
#define P_SOUND_GENERATOR	0x780800
#define P_VIDEO_CTRL_REG	0x781800

/* Addresses for things in onboard mappable device space (machine type 1).
 *   These are addressed through page map entries of type ONBOARD_IO.
 */
#define P_EPROM			0		/* Weird: see manual */
#define P_ENCRYPTION_CHIP	0x001000
#define P_PARALLEL_PORT		0x001800
#define P_SERIAL_PORT		0x002000
#define P_TIMER_CHIP		0x002800
#define P_RASTEROP_CHIP		0x003000
#define P_REAL_TIME_CLOCK	0x003800

/* Addresses in Multibus memory space.  These are addressed through
 *   page map entries of type MULTIBUS_MEM, or directly by Multibus
 *   devices.
 */
#define P_DVMA_START		0
#define P_DVMA_LIMIT		0x40000
#define P_SUN1_FRAMEBUFFER	0x0C0000   /* conventional Sun-1 FB address */
#define SUN1_FRAMEBUFFER_SIZE	0x020000
#define P_3COM_ETHERNET         0x0E0000   
#define THREECOM_ETHERNET_SIZE  0x002000   /* 8k */
#define P_CMC_ETHERNET          0x0A0000   /* CMC board should be jumpered to
                                            *   respond to this address */
#define CMC_ETHERNET_SIZE       0x020000   /* 128k */

/* Conventional virtual addresses.
 */
#define KERNEL_START	0x2000		/* Kernel is loaded at this address */
#define KERNEL_LIMIT	(4*SEG_SIZE)	/* end of kernel */
#define TEAM_START 	KERNEL_LIMIT	/* Start of team space */
#define TEAM_LIMIT	0xE68000	/* Team size limit */
#define	STACK		0xE68000
#define ENET_DATA_SEG	0xE70000	/* 64k for ethernet buffer/data space*/
#define TEAM_SEG	0xE80000	/* 64k for team descriptors */
#define PD_SEG		0xE90000	/* 64k for process descriptors */
#define PROFILESEG	0xEA0000	/* A segment for profiling data */
#define TESTPAGE	PROFILESEG
#define XFERSEG		0xEA8000	/* Used by kernel for moving things */
#define V_MBIO_WINDOW	0xEB0000	/* Window on Multibus I/O space */ 
#define V_MBIO_LIMIT	0xEC0000
#define P_MBIO_WINDOW	0x000000	/* ...starting at address 0 */
#define P_MBIO_LIMIT	0x010000
#define V_SUN2_FRAMEBUFFER	0xEC0000
#define SUN2_FRAMEBUFFER_SIZE	0x020000
#define V_TIMER_CHIP		0xEE0000
#define V_RASTEROP_CHIP		0xEE0800
#define V_REAL_TIME_CLOCK	0xEE1000
#define V_ENCRYPTION_CHIP	0xEE1800
#define V_PARALLEL_PORT		0xEE2000
#define V_VIDEO_CTRL_REG	0xEE3800
#define V_SOUND_GENERATOR	0xEE4000
#define V_KBD_MOUSE_UART	0xEEC000
#define V_SERIAL_PORT		0xEEC800
#define V_EPROM_LIMIT   0xF00000
#define V_DVMA_START	0xF00000	/* Virtual address of DVMA space */
#define V_DVMA_LIMIT	0xF40000
#define V_MBMEM_WINDOW	0xF40000	/* Window on Multibus mem space */
#define V_MBMEM_LIMIT  0x1000000
#define P_MBMEM_WINDOW  0x040000	/* ...starting past DVMA space */
#define P_MBMEM_LIMIT	0x100000
#define MBMEM_OFFSET	(V_MBMEM_WINDOW-P_MBMEM_WINDOW)
#define V_SUN1_FRAMEBUFFER (MBMEM_OFFSET+P_SUN1_FRAMEBUFFER)
#define V_3COM_ETHERNET    (MBMEM_OFFSET+P_3COM_ETHERNET)
#define V_CMC_ETHERNET     (MBMEM_OFFSET+P_CMC_ETHERNET)

/* end conventional addresses */


/* Page map entry used to indicate an unmapped page.  Note that the 
 * valid bit is not set!
 */
#define NULL_PAGE 0

/* Segment map entry used to indicate an unused segment.  This pmeg
 *   is filled with NULL_PAGE page entries.
 */
#define NULL_PMEG (NUM_PMEGS-1)

/* Round an address up to the nearest page boundary */
#define uptopage(addr)\
  ( ((((Bit32) (addr)) - 1) & (~(PAGE_SIZE-1))) + PAGE_SIZE )
/* Round an address up to the nearest segment boundary */
#define uptoseg(addr)\
  ( ((((Bit32) (addr)) - 1) & (~(SEG_SIZE-1))) + SEG_SIZE )
/* Round an address down to the nearest page boundary */
#define downtopage(addr)\
  ( ((Bit32) (addr)) & ~(PAGE_SIZE-1) )
/* Round an address down to the nearest segment boundary */
#define downtoseg(addr)\
  ( ((Bit32) (addr)) & ~(SEG_SIZE-1) )

/* Read and write MMU registers */
#define SetPageMap(addr, value)\
  Fc3WriteBit32(PageMapAddr(addr), (Bit32)(value))
#define SetSegMap(addr, value)\
  Fc3WriteBit8(SegMapAddr(addr), (Bit8)(value))
#define GetPageMap(addr)\
  Fc3ReadBit32(PageMapAddr(addr))
#define GetSegMap(addr)\
  Fc3ReadBit8(SegMapAddr(addr))
#define SetContextNum(num)\
  Fc3WriteBit16(CONTEXT_REGISTER, (Bit16)(num))
#define SetContextReg(cxt)\
  Fc3WriteBit16(CONTEXT_REGISTER, (Bit16)(cxt).u)
#define GetContextReg(cxt)\
  (cxt)->u = Fc3ReadBit16(CONTEXT_REGISTER)
#define GetBusErrorReg()\
  Fc3ReadBit16(BUS_ERROR_REGISTER)

/* This kernel normally sets both contexts the same */
#define SetBothContexts(ctx)\
  Fc3WriteBit16( CONTEXT_REGISTER, (((Bit8)(ctx))<<8) | ((Bit8)(ctx)) )
/* ...except in a few cases during initialization */
#define SetSuperContext(ctx)\
  Fc3WriteBit8(CONTEXT_REGISTER, (Bit8)(ctx))
#define SetUserContext(ctx)\
  Fc3WriteBit8(CONTEXT_REGISTER+1, (Bit8)(ctx))

/* Modes for Probe() function -- see memory.c */
#define PROBE_READ 1
#define PROBE_INIT 2
#define PROBE_PRESERVE 3

#endif
