/*
 * V Kernel - Copyright (c) 1982 by David Cheriton, Tim Mann
 *
 * Sun-2 memory mapping routines
 */

#include "process.h"
#include "sun2mem.h"
#include "sunromvec.h"
#include "interrupt.h"
#include "Vexceptions.h"
#include "Vquerykernel.h"
#include <b.out.h>

/* Imports */
extern MachineConfigurationReply MachineConfig;
extern Team *TeamDescriptors;
extern unsigned long MaxTeams;
extern TimeRecord Time;
extern FindConfiguration();

/* Exports */
void Init_memory_mapping();
SystemCode AllocateMemory();
SystemCode SetTeamSize();
char *KSetTeamSize();
void SetAddressableTeam();
void ReclaimMemory();
void InterTeamCopy();
int Probe();
int DiagnoseBusError();
void GetMemoryStats();
void AllocateDvmaSpace();
void MapDvma();

void FreePage();
SystemCode AllocatePage();
void FreePmeg();
SystemCode AllocatePmeg();
int GetMemorySize();
void SaveSegMap();
void RestoreSegMap();

/* Globals */
PageMapEntry FreePageList;	/* list head */
SegMapEntry FreePmegList;

int FreePages = 0;
VirtualAddress FreeKernelSpace;	/* free pages in kernel area */
VirtualAddress FreeDvmaSpace;

Team *AddressableTeam;

struct bhdr KernelBheader = {0},
	    TeamBheader = {0};		/* Force to data segment */

void Init_memory_mapping()
  /*
   * Setup the memory mapping as required.
   *
   * We assume the EPROM monitor has initialized the memory map for
   * context 0 in its standard way.  The kernel and first team are 
   * assumed to have been put at their normal addresses by the 
   * bootloader, and the pages and segments they are loaded into are 
   * assumed to be mapped 1-to-1, that is, the virtual and physical 
   * addresses are the same, and the pmeg and segment numbers are the same.
   *
   * We assign the kernel virtual addresses between 0 and TEAM_START in
   * all contexts.  Only pages that are actually used are mapped to
   * physical memory.
   *
   * Addresses from TEAM_START to TEAM_LIMIT are available to teams.  The 
   * first team is initially given just enough mapped memory in this area
   * to hold its code and data.  All other memory is added to the free
   * list.
   *
   * The area above TEAM_LIMIT is mapped in a complex way by the EPROM
   * monitor (described elsewhere), and we leave it alone (may change
   * protection on somethings later), both to keep the EPROM monitor 
   * happy when we have to call it, and because it is a reasonable 
   * arrangement for our use.
   * 
   * We don't support using Multibus memory as ordinary physical memory
   * on the Sun-2, since such pages would not work in the normal way
   * with the DVMA feature -- references from P1 space, through the
   * memory map, and back to P1 space, are prohibited (see the Sun-2 
   * Architecture Manual).
   */
  {
    int pages;
    char *cptr;
    int ctx, i;
    PageMapEntry freepg, oldpg, pme;
    SegMapEntry sme, xsme, topsme;
    VirtualAddress addr, ttop, ktop;
    Team *td;
    extern struct sunromvec *RomVecPtr;

    /* Set the rom vector ptr to the correct Sun-2 location. It is initialized
     * to the Sun-1 location in the rawio library. This works ok up to this
     * point because at startup the prom is mapped into both locations.
     */
    RomVecPtr = (struct sunromvec *) SUN2_ROM_VEC_PTR;

    /* Copy the b.out headers into kernel data space */
    TeamBheader = *((struct bhdr *) (TEAM_START - sizeof(struct bhdr)));
    KernelBheader = *((struct bhdr *) (TEAM_START - 2*sizeof(struct bhdr)));

    ttop.u = TEAM_START + TeamBheader.tsize 
		+ TeamBheader.dsize + TeamBheader.bsize + INIT_STACK;
    ktop.u = KERNEL_START + KernelBheader.tsize
		+ KernelBheader.dsize + KernelBheader.bsize;

    /* Zero the bss area in case the loader didn't do it */
    cptr = (char *) TEAM_START + TeamBheader.tsize + TeamBheader.dsize;
    for (i = TeamBheader.bsize; i; i--) *cptr++ = '\0';

    cptr = (char *) KERNEL_START + KernelBheader.tsize + KernelBheader.dsize;
    for (i = KernelBheader.bsize; i; i-- ) *cptr++ = '\0';

    SetUserContext(0);	/* just to be sure */
    FreeKernelSpace.u = uptopage(ktop.u);
    FreeDvmaSpace.u = V_DVMA_START;
    FindConfiguration();
    pages = GetMemorySize();
    MachineConfig.fastMemory = pages * PAGE_SIZE;
    MachineConfig.slowMemory = 0;
    MachineConfig.memory = MachineConfig.fastMemory;

    printx("Memory size: 0x%x bytes.\r\n", MachineConfig.memory);

    printx("Size in bytes: Kernel( %d ) First team( %d )\n",
	ktop.u - KERNEL_START, ttop.u - TEAM_START );
    if (ktop.u > TEAM_START) Kabort("Kernel too big");

    /* Initialize free lists */
    FreePageList.u = NULL_PAGE;
    FreePmegList = NULL_PMEG;

    /* Set page protection on kernel memory appropriately */
    for (addr.u = 0; addr.u < ktop.u; addr.u += PAGE_SIZE)
      {
	pme.u = GetPageMap(addr.u);
	pme.f.protection = SR|SW|SX;
	SetPageMap(addr.u, pme.u);
      }

    /* Set page protection on team memory appropriately */
    for (addr.u = TEAM_START; addr.u < ttop.u; addr.u += PAGE_SIZE)
      {
	pme.u = GetPageMap(addr.u);
	pme.f.protection = SR|SW|UR|UW|UX;
	SetPageMap(addr.u, pme.u);
      }

    /* Unmap segments in context 0 above the top of the first team,
     *   up to TEAM_LIMIT.
     */
    for (addr.u = uptoseg(ttop.u); addr.u < TEAM_LIMIT;
		addr.u += SEG_SIZE)
      {
	SetSegMap(addr.u, NULL_PMEG);
      }

    /* Initialize the segment maps for non-zero contexts.  We copy in the
     *   context 0 mapping for the kernel and the area above TEAM_LIMIT,
     *   and set the rest to NULL_PMEG.
     */
    for (addr.u = 0; addr.u < TEAM_START; addr.u += SEG_SIZE)
      {
	sme = GetSegMap(addr.u);
	for (ctx = 1; ctx < NUM_CONTEXTS; ctx++)
	  {
	    SetUserContext(ctx);
	    SetSegMap(addr.u, sme);
	  }
	SetUserContext(0);
      }
    for ( ; addr.u < TEAM_LIMIT; addr.u += SEG_SIZE)
      {
	for (ctx = 1; ctx < NUM_CONTEXTS; ctx++)
	  {
	    SetUserContext(ctx);
	    SetSegMap(addr.u, NULL_PMEG);
	  }
      }
    SetUserContext(0);
    for ( ; addr.u < CONTEXT_SIZE; addr.u += SEG_SIZE)
      {
	sme = GetSegMap(addr.u);
	for (ctx = 1; ctx < NUM_CONTEXTS; ctx++)
	  {
	    SetUserContext(ctx);
	    SetSegMap(addr.u, sme);
	  }
	SetUserContext(0);
      }

    /* Construct free pmeg list.  We put all the pmegs numbered
     *   above the ones used by the kernel and first team and 
     *   below the ones used above TEAM_LIMIT on the list.  This
     *   technique frees the right ones because the EPROM allocates
     *   pmeg numbers monotonically within this area.
     */
    xsme = GetSegMap(XFERSEG);
    topsme = GetSegMap(TEAM_LIMIT);
    for (sme = ExtractSegNumber(uptoseg(ttop.u)); sme < topsme; sme++)
      {
	SetSegMap(XFERSEG, sme);
	FreePmeg(XFERSEG);
      }
    SetSegMap(XFERSEG, xsme);

    /* Construct free page list */

    /* First, free unused pages from kernel area */
    for (addr.u = uptopage(ktop.u); addr.u < TEAM_START; addr.u += PAGE_SIZE)
	FreePage(addr.u);

    /* Next, unmap unused pages in team's last pmeg (they are added to
     *   the free list in the next step).
     */
    for (addr.u = uptopage(ttop.u); 
	 addr.u < uptoseg(ttop.u); addr.u += PAGE_SIZE)
      {
	SetPageMap(addr.u, NULL_PAGE);
      }

    /* Now free other unused pages */
    pme.f.valid = 1;
    pme.f.protection = SR|SW;
    pme.f.type = ONBOARD_MEM;
    for ( i = ExtractPageNumber(uptopage(ttop.u)); i < pages; i++ )
      {
	pme.f.pagenum = i;
	SetPageMap(TESTPAGE, pme.u);
	FreePage(TESTPAGE);
      }

    /* Unmap onboard memory pages from DVMA region */
    for (addr.u = V_DVMA_START; addr.u < V_DVMA_LIMIT; addr.u += PAGE_SIZE)
      {
	SetPageMap(addr.u, NULL_PAGE);
      }

    /* Set page protection on high memory */
    for (addr.u = TEAM_LIMIT;
	 addr.u < CONTEXT_SIZE;
	 addr.u += PAGE_SIZE)
      {
	pme.u = GetPageMap(addr.u);
	if (pme.u == NULL_PAGE) continue;
	pme.f.protection &= ~(UR|UW|UX);
	SetPageMap(addr.u, pme.u);
      }

    /* Unprotect the Sun-1 frame buffer for now */
    for (addr.u = V_SUN1_FRAMEBUFFER;
	 addr.u < V_SUN1_FRAMEBUFFER+SUN1_FRAMEBUFFER_SIZE;
	 addr.u += PAGE_SIZE)
      {
	pme.u = GetPageMap(addr.u);
	pme.f.protection = SR|SW|UR|UW;
	SetPageMap(addr.u, pme.u);
      }

    /* And the Sun-2 frame buffer */
    for (addr.u = V_SUN2_FRAMEBUFFER;
         addr.u < V_SUN2_FRAMEBUFFER+SUN2_FRAMEBUFFER_SIZE;
	 addr.u += PAGE_SIZE)
      {
	pme.u = GetPageMap(addr.u);
	pme.f.protection = SR|SW|UR|UW;
	SetPageMap(addr.u, pme.u);
      }
    /* and its control register */
    pme.u = GetPageMap(V_VIDEO_CTRL_REG);
    pme.f.protection = SR|SW|UR|UW;
    SetPageMap(V_VIDEO_CTRL_REG, pme.u);
  }

SystemCode SetTeamSize( active )
    Process *active;
  /*
   * Set the memory size for a team, mapping or unmapping pages as needed.
   */
  {
    KernelRequest *req = (KernelRequest *) &(active->msg);
    register Process *pd;

    if( !(pd = MapPid(req->pid)) ) return( NONEXISTENT_PROCESS );

    req->unspecified[0] = (unsigned)KSetTeamSize(pd->team,req->unspecified[0]);
    return( OK );
  }

char *KSetTeamSize(td, size)
    register Team *td;
    char *size;
  /*
   * Called by SetTeamSize to do the actual work.
   */
  {
    register VirtualAddress firstUnusedPage, newFirstUnusedPage;
    register int fail;
    Team *oldteam;

    /* Range check on size */
    if (size < (char *) TEAM_START) size = (char *) TEAM_START;
    if (size >= (char *) TEAM_LIMIT) size = (char *) TEAM_LIMIT;

    /* Temporarily switch contexts */    
    oldteam = GetAddressableTeam();
    SetAddressableTeam(td);

    firstUnusedPage.u = uptopage(td->team_space.size);
    newFirstUnusedPage.u = uptopage(size);

    fail = 0;
    if (firstUnusedPage.u <= newFirstUnusedPage.u)
      {
	/* More memory needed */
	for ( ; firstUnusedPage.u < newFirstUnusedPage.u; 
		firstUnusedPage.u += PAGE_SIZE )
	  {
	    if (fail = AllocatePage(firstUnusedPage.u)) break;
	  }
      }
    else
      {
	/* Less memory needed */
	for (firstUnusedPage.u -= PAGE_SIZE;
	     firstUnusedPage.u >= newFirstUnusedPage.u;
	     firstUnusedPage.u -= PAGE_SIZE)
	  {
	    FreePage(firstUnusedPage.u);
	    /* We are assuming the simple memory model in which
	     *   address spaces do not have holes, so if we free the
	     *   first page in a pmeg here, we can free the pmeg.
	     */
	    if (firstUnusedPage.u == uptoseg(firstUnusedPage.u))
	      {
		FreePmeg(firstUnusedPage.u);
	      }
	  }
      }

    SetAddressableTeam(oldteam);

    if (fail)
      {
	return (td->team_space.size = firstUnusedPage.p);
      }
    else
      {
	return (td->team_space.size = size);
      }

  }



void FreePage(pgaddr)
    VirtualAddress pgaddr;
  /*
   * Add a page to the free list, unmapping it from the current context
   */
  {
    PageMapEntry freepg;

    freepg.u = GetPageMap(pgaddr.u); /* Get map entry pointing to this page */

    /* Point page to old list top */
    *(PageMapEntry *) downtopage(pgaddr.u) = FreePageList;
    FreePageList = freepg;
    FreePages++;

    SetPageMap(pgaddr.u, NULL_PAGE);	 /* Unmap page */
  }


SystemCode AllocatePage(pgaddr)
    VirtualAddress pgaddr;
  /*
   * Map a page into the current context at the given address, if one
   *  can be found.  Return OK for success.
   */
  {
    PageMapEntry newpg;

    /* Find a page, if available */
    if ( (newpg.u = FreePageList.u) == NULL_PAGE )
      {
	return (NO_MEMORY);	/* fail */
      }

    /* Make sure there is a pmeg mapped to this segment */
    if (GetSegMap(pgaddr.u) == NULL_PMEG)
      {
	if (AllocatePmeg(pgaddr.u)) return (NO_MEMORY); /* fail */
      }

    /* Map in the page */
    newpg.f.valid = 1;
    newpg.f.protection = SR|SW|UR|UW|UX;
    SetPageMap(pgaddr.u, newpg.u);

    /* Remove it from the free list */
    FreePageList = *(PageMapEntry *) downtopage(pgaddr.u);
    FreePages--;

    return (OK);
  }

/* Allocate physical memory. This routine is meant to be called by the kernel
 * to allocate memory for such uses as process decriptor space, ethernet
 * buffer space, and profiling data space. Is does not check to see if the
 * memory has already been allocated.
 */
SystemCode AllocateMemory( start, length )
char *start;
unsigned length;
  {
    register char *ptr, *end;
    
    end = start + length;
    
    for ( ptr = start; ptr < end; ptr += PAGE_SIZE )
	if ( AllocatePage( ptr ) != OK )
	  {
	    for ( end = start; end < ptr; end += PAGE_SIZE )
		FreePage( end );
	    return( NO_MEMORY );
	  }
    return( OK );    
  }

/* Very rudimentary. All of the memory BETTER BE ALLOCATED. */
FreeMemory( start, length )
char *start;
unsigned length;
  {
    register char *ptr, *end;
    
    end = start + length;
    
    for ( ptr = start; ptr < end; ptr += PAGE_SIZE ) FreePage( ptr );
    return( OK );    
  }

void FreePmeg(addr)
    VirtualAddress addr;
  /*
   * Free the pmeg currently mapped at address addr, clearing out all
   *   the pme's except the first (which is used as a link field).
   */
  {
    VirtualAddress start;

    start.u = downtoseg(addr.u);

    /* Link new pmeg onto start of chain */
    SetPageMap(start.u, FreePmegList);
    FreePmegList = GetSegMap(start.u);

    /* Clear the rest of the pmeg */
    for (addr.u = start.u + PAGE_SIZE; 
	 addr.u < start.u + SEG_SIZE; addr.u += PAGE_SIZE)
      {
	SetPageMap(addr.u, NULL_PAGE);
      }

    /* Unmap the pmeg */
    SetSegMap(start.u, NULL_PMEG);
  }


SystemCode AllocatePmeg(addr)
    VirtualAddress addr;
  /*
   * Map an empty pmeg into the current context at the given address, if one
   *  can be found.  Return OK for success.
   */
  {
    SegMapEntry newpm;

    /* Find a pmeg */
    newpm = FreePmegList;
    if (newpm == NULL_PMEG)
      {
	return (NO_MEMORY);  /* fail */
      }

    /* Map it in */
    SetSegMap(addr.u, newpm);

    /* Remove it from the free list and clear its link field */
    FreePmegList = (SegMapEntry) GetPageMap(downtoseg(addr.u));
    SetPageMap(downtoseg(addr.u), NULL_PAGE);
  }


void ReclaimMemory(td)
    Team *td;
    /* Reclaim the memory allocated to a defunct team */
  {
    KSetTeamSize(td, NULL);
  }


void InterTeamCopy(dest, dteam, src, steam, bytes)
    Team *dteam, *steam;
    char *dest, *src;
    long unsigned bytes;
  /*
   * This routine is used by MoveFrom and MoveTo to copy bytes between
   *   different teams.
   */
  {
    Team *oldteam;
    SegMapEntry oldxfs, xfs;
    int sbytes;

    oldteam = GetAddressableTeam();
    SetAddressableTeam(dteam);
    oldxfs = GetSegMap(XFERSEG);

    while (bytes > 0)
      {
	/* Set up to move all the desired bytes from one source segment */
	sbytes = ((char *) uptoseg(src+1)) - src;
	sbytes = min(sbytes, bytes);

	/* Map segment containing source bytes into xfer segment */
	/* Could speed things up a trifle by assuming neither steam
	 *   nor dteam will be LRU after the first time, and thereafter
	 *   calling SetContextReg instead of SetAddressableTeam */
	SetAddressableTeam(steam);
	xfs = GetSegMap(src);
	SetAddressableTeam(dteam);
	SetSegMap(XFERSEG, xfs);

	/* Move the bytes */
	Copy(dest, XFERSEG + ((unsigned) src) % SEG_SIZE, sbytes);
	bytes -= sbytes;
	src += sbytes;
	dest += sbytes;

      }

    SetSegMap(XFERSEG, oldxfs);
    SetAddressableTeam(oldteam);

  }



void AllocateDvmaSpace(size, multibusAddress, virtualAddress)
    unsigned size;
    char **multibusAddress, **virtualAddress;
  {
    /*
     * Allocate 'size' bytes of DVMA space to the caller, rounded up
     *   to a page boundary.  Returns both Multibus and virtual addresses
     *   of the area allocated.  There is no provision for freeing DVMA
     *   space.
     */

    *virtualAddress = (char *) FreeDvmaSpace.u;
    *multibusAddress = (char *) FreeDvmaSpace.u - V_DVMA_START + P_DVMA_START;

    FreeDvmaSpace.u += uptopage(size);

    if (FreeDvmaSpace.u > V_DVMA_LIMIT) 
	Kabort("Out of DVMA space");
  }



void MapDvma( dest, source, length )
  register char      *dest, *source;
  register unsigned  length;

  {
    /* Map starting at 'source' address into Dmva space starting at
     *   'dest' address.
     */

    PageMapEntry  pme;
    register char *p, *q;  

    q = (char *) downtopage( dest );
    for( p = (char *) downtopage( source );
         p < (char *) uptopage( source+length );
         p += PAGE_SIZE )
      {
        pme.u = GetPageMap( p );
        SetPageMap( q, pme.u );
        q += PAGE_SIZE;
      }
  }



	
int GetMemorySize()
  /*
   * Probe to determine size of on-board memory on a Sun-2. 
   * Returns the amount in pages.
   */
  {
    PageMapEntry oldPage, pme;
    register int pages, *p;

    /* Save old contents of page map entry we are using */
    oldPage.u = GetPageMap(TESTPAGE);

    /* Test onboard pages until one doesn't work */
    pages = TEAM_START/PAGE_SIZE;  /* Don't test kernel memory */
    pme.f.valid = 1;
    pme.f.protection = SR|SW;
    pme.f.type = ONBOARD_MEM;
    while (pages < ExtractPageNumber(P_ONBOARD_RAM_LIMIT))
      {
	pme.f.pagenum = pages;
	SetPageMap(TESTPAGE, pme.u);
	if (Probe(TESTPAGE, PROBE_PRESERVE))
	    pages++;	/* more memory here */
	else
	    break;	/* no more */
      }
	
    SetPageMap(TESTPAGE, oldPage.u);
    return (pages);
  }



int Probe(address, how)
    short *address;
    int how;
  /*
   * Look for memory at address.
   * If how == PROBE_INIT,
   *   write before reading to initialize, then look for writeable memory.
   * If how == PROBE_PRESERVE,
   *   look for writeable memory, but preserve contents of location.
   * If how == PROBE_READ,
   *   just try reading and see whether it causes a bus error.
   */
  {
    register short *Zero = (short *) 0;
    register short z, x;
    int BusErrorOnProbe(), (*oldBusErrorVector)();

    /* Set bus error vector */
    oldBusErrorVector = *( (int (**)()) BUSERROR );
    *( (int (**)()) BUSERROR ) = BusErrorOnProbe;

    z = *Zero;  /* Save old contents of address 0 */
		
    if (how == PROBE_INIT) 
	*address = 0x5c5c;	/* Initialize the test location */

    x = ~*address;		/* Read from it */

    if (how == PROBE_READ)
      {
	/* Success if we get here with no bus error */
	*( (int (**)()) BUSERROR ) = oldBusErrorVector;
	return (1);
      }
    else
      {
	*address = x;		/* Write back the complement */
	if (*address == x && *Zero == z) 
	  {
	    /* Success */
	    *address = ~x;
	    *( (int (**)()) BUSERROR ) = oldBusErrorVector;
	    return (1);
          }
    	else
	  {
	    /* Failure */
	    asm("ProbeFailed:");
	    *( (int (**)()) BUSERROR ) = oldBusErrorVector;
	    *Zero = z;
            return (0);
          }
      }

    asm("	.globl BusErrorOnProbe");
    asm("BusErrorOnProbe:");
#ifdef MC68010
    asm("	addl	#58, sp");
#else !MC68010
    asm("	addl	#14, sp");
#endif MC68010
    asm("	jmp	ProbeFailed");

  }

/*
 * Another routine to tell you whether you can read a given location in
 *   memory.  This one, however, is intended specifically for virtual
 *   addresses.  It doesn't actually try to read from the location, just
 *   look at the memory mapping hardware, which had better be set
 *   up properly.
 *
 * May need major fiddling if we introduce virtual memory.
 */

KernelCanRead(address, length)
    unsigned    address, length;
  {
    register unsigned	testadr;
    register PageMapEntry pme;	/* Do we really gain anything by putting */
				/*   bit-field variables in registers?   */

    /* Test one byte in each page from address to address+length-1 */
    /*   If we can read one byte in the page, we can read anything */
    for (testadr = address; (unsigned)(testadr-address) < length;
			     /* ^ Does this work for all cases? */
	 testadr = (testadr+PAGE_SIZE) & (-PAGE_SIZE) /*gag*/)
      {
	pme.u = GetPageMap(testadr);
	if ( !(pme.f.protection & SR) || !(pme.f.valid) )
	    return 0;
      }
    return 1;
  }


int DiagnoseBusError(req)
    ExceptionRequest *req;
  /*
   * Diagnose the cause of a bus error.  Used by exception handler.
   */
  {
    BusErrorRegister cause;

    if (req->accaddr >= CONTEXT_SIZE)
      {
	/* Garbage address -- these processors have 
	 * only 24 bit addressing */
	return (OUT_OF_RANGE);
      }

    cause.u = GetBusErrorReg();

    if (!cause.f.pageValid)
	return (PAGE_INVALID);

    if (cause.f.protection)
        return (PROTECTION);

    if (cause.f.timeout)
        return (MB_TIMEOUT);

    if (cause.f.parityUpper || cause.f.parityLower)
	return (PARITY);
  }


/*
 * Get memory statistics for QueryKernel operation
 */
void GetMemoryStats(reply)
    register MemoryStatisticsReply *reply;
  {
    reply->unusedFastMemory = FreePages * PAGE_SIZE;
    reply->unusedSlowMemory = 0;
  }


/*
 * Routines to set/get the currently addressable team space.
 * The Sun-2 only has 8 sets of segment mapping registers, so if
 *   we have more than 8 teams, we have to do LRU swapping of the
 *   segment maps into and out of registers.
 */

/* GetAddressableTeam() is now a macro */

void SetAddressableTeam(newteam)
    register Team *newteam;
  {
    register Team *td, *oldest;
    
    /* Bring in segment map if not resident */
    if (newteam->team_space.context.u == MAP_NOT_RESIDENT)
      {
	/* Find the LRU context... */
	oldest = NULL;
	for (td = TeamDescriptors; td < (TeamDescriptors + MaxTeams); td++)
	  {
	    if ( MapPid(td->team_root) == NULL )
		td->team_space.timestamp = 0;		/* a dead team */

	    if ( td->team_space.context.u != MAP_NOT_RESIDENT &&
		    (oldest == NULL ||
		     td->team_space.timestamp < oldest->team_space.timestamp) )
		oldest = td;
	  }

	/* ...and swap out its segment map */
	newteam->team_space.context = oldest->team_space.context;
	oldest->team_space.context.u = MAP_NOT_RESIDENT;
        SetContextReg(newteam->team_space.context);
	SaveSegMap(oldest->team_space.segmap);
	RestoreSegMap(newteam->team_space.segmap);
      }
    else
      {
        SetContextReg(newteam->team_space.context);
      }

    AddressableTeam->team_space.timestamp = Time.seconds;
				/* %%% should be in clicks, really */
    AddressableTeam = newteam;
  }
  
SystemCode ClearModifiedPages( req, segPtr, segSize )
KernelRequest *req;
register long **segPtr;
unsigned long segSize;
  {
    PageMapEntry pme;
    register VirtualAddress addr, firstUnused;
    register int returnModifiedPages, count;
    int maxCount;
    Process *cpd;
    Team *oldteam;
    
    if ( !Local( req->pid ) ) return( MODE_NOT_SUPPORTED );
    
    if ( !MAP_TO_RPD( cpd, req->pid ) ) return( NONEXISTENT_PROCESS );
    
    if ( returnModifiedPages = ( req->opcode == RETURN_MODIFIED_PAGES ) )
	if( ( count = maxCount = segSize / sizeof( long * ) ) == 0 )
	    return( BAD_BUFFER );

    oldteam = GetAddressableTeam();
    SetAddressableTeam( cpd->team );
    
    firstUnused.u = uptopage( cpd->team->team_space.size );
    for ( addr.u = TEAM_START; addr.u < firstUnused.u; addr.u += PAGE_SIZE )
      {
	pme.u = GetPageMap( addr.u );
	if ( pme.f.modified )
	  {
	    pme.f.modified = 0;
	    SetPageMap( addr.u, pme.u );
	    if ( returnModifiedPages )
	      {
		*segPtr++ = (long *) addr.u;
		if ( --count == 0  ) break;
	      }
	  }
      }
    
    SetAddressableTeam( oldteam );
    
    if ( ( req->unspecified[0] = maxCount - count ) != maxCount )
      {
        *segPtr = NULL;
	return( OK );
      }
    else
	return( RETRY );
  }  


/* Routines to read and write function-code-3 space */

/* All routines assume that both function code registers contain "3"
 *   already.  The EPROM monitor is known to initialize them to that
 *   value, and the kernel never changes them.  */
asm("	.text			");
asm("	.globl Fc3ReadBit8	");
asm("Fc3ReadBit8:		");
asm("   movl sp@(4),a0		");	/* get arg into a0 */
asm("	movsb a0@, d0		");	/* move from outer space */
asm("	rts			");

asm("	.text			");
asm("	.globl Fc3ReadBit16	");
asm("Fc3ReadBit16:		");
asm("   movl sp@(4),a0		");	/* get arg into a0 */
asm("	movsw a0@, d0		");	/* move from outer space */
asm("	rts			");

asm("	.text			");
asm("	.globl Fc3ReadBit32	");
asm("Fc3ReadBit32:		");
asm("   movl sp@(4),a0		");	/* get arg into a0 */
asm("	movsl a0@,d0		");	/* move from outer space */
asm("	rts			");

asm("	.text			");
asm("	.globl Fc3WriteBit8	");
asm("Fc3WriteBit8:		");
asm("	movl sp@(4),a0		");	/* get address into a0 */
asm("	movl sp@(8),d0		");	/* get value into d0 */
asm("	movsb d0,a0@		");	/* move to outer space */
asm("	rts			");

asm("	.text			");
asm("	.globl Fc3WriteBit16	");
asm("Fc3WriteBit16:		");
asm("	movl sp@(4),a0		");	/* get address into a0 */
asm("	movl sp@(8),d0		");	/* get value into d0 */
asm("	movsw d0,a0@		");	/* move to outer space */
asm("	rts			");

asm("	.text			");
asm("	.globl Fc3WriteBit32	");
asm("Fc3WriteBit32:		");
asm("	movl sp@(4),a0		");	/* get address into a0 */
asm("	movl sp@(8),d0		");	/* get value into d0 */
asm("	movsl d0,a0@		");	/* move to outer space */
asm("	rts			");


/* Copy segment tables between mapping registers and memory */

asm("	.text			");
asm("	.globl SaveSegMap	");
asm("SaveSegMap:		");
asm("	movl #5,a0		");	/* base of seg table */
asm("	movl sp@(4),a1		");	/* destination */
asm("	movl #511,d0		");	/* size of seg table, minus 1 */
asm("loop1:			");
asm("	movsb a0@,d1		");	/* fetch an entry */
asm("	movb d1,a1@+		");	/* save it */
asm("	addl #/8000,a0		");	/* step to next segment */
asm("	dbra d0,loop1		");
asm("	rts			");

asm("	.text			");
asm("	.globl RestoreSegMap	");
asm("RestoreSegMap:		");
asm("	movl #5,a0		");	/* base of seg table */
asm("	movl sp@(4),a1		");	/* destination */
asm("	movl #511,d0		");	/* size of seg table, minus 1 */
asm("loop2:			");
asm("	movb a1@+,d1		");	/* fetch an entry */
asm("	movsb d1,a0@,		");	/* restore it */
asm("	addl #/8000,a0		");	/* step to next segment */
asm("	dbra d0,loop2		");
asm("	rts			");
