/*
 * Mouse handling routines for the SMI optical mice.  (Both parallel and
 *  serial versions.)
 *
 * The mouse on the SMI model 100 is connected to the parallel input
 *  port and has to be polled.  It sends 3 bytes every 7.5 ms, but the
 *  documentation does not guarantee that they are evenly spaced at
 *  2.5 ms intervals, so we poll every 2 ms (??).  The timer is set
 *  to interrupt every 2 ms, and the interrupt handler calls 
 *  SmiMouseInterrupt() which polls the mouse.
 *
 * On the model 120, the mouse is connected to a serial chip, and
 *  the serial interrupt routine calls SmiMouseSqueak().  The serial
 *  mouse uses a somewhat different protocol from the parallel version.
 *
 * Marvin Theimer, 2/83
 * Tim Mann, 7/84
 */


#include "memory.h"

extern short MouseEvent, MouseX, MouseY;
extern short unsigned MouseButtons;

#ifndef SUN3
#define Port (*(short *)(V_PARALLEL_PORT))
#endif SUN3

static short cur = 0, prev = 0;
static short unsigned oldButtons1 = 0, oldButtons2 = 0;

#ifndef SUN3
/*
 * Parallel mouse handler
 */
SmiMouseInterrupt()
  {
    register short temp;
    register char dx, dy;
    register short unsigned newButtons;

    /* Be sure we get two consecutive readings that agree, to reduce
     *  the chance of glitches due to reading while the values are
     *  changing (TPM) */
    cur = Port & 0xff;
    do
      {
	temp = cur;
        cur = Port & 0xff;
       }
    while (cur != temp);

    /* If we read the same thing on two consecutive interrupts, we are
     *  ahead of the mouse, so ignore it.
     */
    if (cur == prev) return;

    /* Check for sync byte */
    if ((cur & 0xf0) == 0x80)
      {
	/* Current input was the sync byte, previous was dy */
        dy = prev;
	dy >>= 1;
	if (dy < -40) dy = 0;
	MouseY -= dy;			/* NOTE the minus sign */
	if (dy)
	    MouseEvent = 1;

	/* Make sure we get two readings that agree before returning
	 *  the mouse buttons, to eliminate glitches. (TPM)
	 *  (Not sure this is still needed, due to the antiglitch 
         *  code above.) */
	newButtons = (short unsigned) ((cur>>1) & 7);
        newButtons = ((newButtons & 4) >> 2) | ((newButtons & 3) << 1);
	if (newButtons == oldButtons2 && oldButtons2 != oldButtons1)
	  {
	    MouseButtons = newButtons;
	    MouseEvent = 1;
	  }
	oldButtons1 = oldButtons2;
	oldButtons2 = newButtons;
      }
    else if ((prev & 0xf0) == 0x80)
      {
	/* Previous reading was the sync byte, so current is dx */
        dx = cur;
	dx >>= 1;
	if (dx < -40) dx = 0;
	MouseX += dx;
	if (dx) 
	    MouseEvent = 1;
      }
    prev = cur;
  }
#endif SUN3

/*
 * Serial mouse handler
 */
SmiMouseSqueak(squeak)
    char squeak;
  {
    register short temp;
    register char dx, dy;
    register short unsigned newButtons;
    static state = 0;

    /* Check for sync byte */
    if ((squeak & 0xf8) == 0x80)
      {
	/* Synchronize */
	state = 0;
      }

    switch (state++)
      {
       case 0:
        /* Sync byte + buttons */
        MouseButtons = ((squeak & 4) >> 2) | ((squeak & 3) << 1);
	break;

       case 1:
       case 3:
	/* Current reading is dx */
	MouseX += squeak;
	break;

       case 2:        
	/* Current reading is dy */
	MouseY -= squeak;
	break;

       case 4:
	/* Current reading is dy, last byte of sequence */
	MouseY -= squeak;
	MouseEvent = 1;
	break;

       default:
#ifdef undef /* It was decided that this was too annoying to print out */
	printx("Bogus mouse output!\n");
#endif
	break;
      }
  }
