 
/* Ethernet header for Communication Machinery Corporation's (CMC)
 *   Ethernet Node Processor (ENP).  Used by kernel driver and on-board driver.
 */

#include "Venviron.h"
#include "Vethernet.h"
#include "Vikc.h"
#include "process.h"
#include "sun2mem.h"


#define ENET_INT_LEVEL  INT3  /* multibus interrupt level */
/* Memory Map */

/* Note:  the ENP slave window is 128k as seen by the host.  The window is
 *        jumper selectable on the ENP board.
 */
#ifdef ENP10  /* VME-bus based */
#define ENP_RAM_ORIGIN 	0x00f00000
#define ADDRESS_MASK	0x0001ffff
#define BASE_ENP_RAM    0x00f01000  /* ENP RAM space -> 123.5k */
#define TOP_ENP_RAM     0x00f1fdff
#define ENP_STACK_BASE  0x00f1d000  /* stack has upper 11.5k of RAM */
#define BASE_SYS_MEM	0x00001000  /* VME address space */
#define	TOP_SYS_MEM	0x00eeffff
#define BASE_ENP_SLAVE  0x00de1000 /* 124k window for host to access ENP mem. */
#define TOP_ENP_SLAVE   0x00dfffff
#define BOTTOM_ENP_SLAVE 0x00de0000 /* window starts 4k above the bottom */   
#endif ENP10

/* Note: the Enp-30 board should be jumpered to respond to addresses starting
 *   at P_CMC_ETHERNET */
#ifdef ENP30  /* multibus based */
#define ENP_RAM_ORIGIN 	0x00F00000
#define ADDRESS_MASK	0x0001FFFF
#define BASE_ENP_RAM    0x00F01000  /* ENP RAM space -> 116k */
#define TOP_ENP_RAM     0x00F1E000
#define ENP_STACK_BASE  0x00f1B200  /* stack has upper 11.5k of RAM */
#define BASE_SYS_MEM	0x00001000  /* Multibus address space */
#define	TOP_SYS_MEM	0x00EF0000
#define BOTTOM_ENP_SLAVE V_CMC_ETHERNET /* 120k window for host to access ENP */
#define TOP_ENP_SLAVE   (V_CMC_ETHERNET+0x1E000)     
#define BASE_ENP_SLAVE  (V_CMC_ETHERNET+0x01000) /* bottom 4k is inaccessible */
/* 8-bit register to clear after an interrupt.  Located at 0x027 in the top 
 *   4k of the slave window. */
#define SYS_INTERRUPT_REG  	(BOTTOM_ENP_SLAVE+0x1F000+0x027) /* Sys view */
#define INTERRUPT_REG 		0xFFF027  /* Enp view */
#define MPU_CSR                 0xFFF001  /* Enp view of control, status reg */
#define ENABLE_BIT   		0x10      /* bit in CSR that enables ints. */
#endif ENP30

/* WARNING:  All mailboxes and flags should eventually be located in host
 *           memory to avoid possible problems of host timing out while
 *           trying to access ENP memory (LANCE chip hogs bus for 10us bursts).
 */

/* Enp code origin from enp's point of view. */
#define ENP_CODE_START  	(0x2000 + ENP_RAM_ORIGIN)
/* Enp code origin from system's point of view. */
#define ENP_LOAD_ORIGIN         (((unsigned) ENP_CODE_START & ADDRESS_MASK)  \
                                                  | BOTTOM_ENP_SLAVE )
#define JUMP_COMMAND         	0x80800000

#define JUMP_COMMAND_LOC        (0x1000 + ENP_RAM_ORIGIN)  /* enp's view */
#define SYS_JUMP_COMMAND_LOC    (((unsigned) JUMP_COMMAND_LOC & ADDRESS_MASK) \
                                      | BOTTOM_ENP_SLAVE ) /* system's view */
#define K1_STARTUP_MBOX		(0x1004 + ENP_RAM_ORIGIN)
#define ENP_STARTUP_MBOX	(0x1008 + ENP_RAM_ORIGIN)
#define ENP_STARTUP_INIT_BLOCK 	(0x101c + ENP_RAM_ORIGIN)  /* 32-byte block */
#define ERROR_LOC               (0x104c + ENP_RAM_ORIGIN)

/* location used to allow the Enp to print characters by passing them to the
 *   host and having it do it. */
#define PUTCHAR_LOC		(0x1050 + ENP_RAM_ORIGIN)  /* Enp view */
#define SYS_PUTCHAR_LOC         (((unsigned) PUTCHAR_LOC & ADDRESS_MASK)  \
                                      | BOTTOM_ENP_SLAVE ) /* system's view */
/* debugging locations */
/* TEST_LOC1 can be observed with the prom monitor by examining
 *    V_CMC_ETHERNET+1054 */
#define TEST_LOC1               (0x1054 + ENP_RAM_ORIGIN)
#define TEST_LOC2               (0x1058 + ENP_RAM_ORIGIN)
#define TEST_LOC3               (0x105c + ENP_RAM_ORIGIN)
#define TEST_LOC4               (0x1060 + ENP_RAM_ORIGIN)
#define TEST_LOC5               (0x1064 + ENP_RAM_ORIGIN)

/* K1 Kernel commands and return status codes to the host */
#define NET_INIT	0xd0	/* Initialize board */
#define NET_STATS 	0xb0	/* get stats and/or reset the board */
#define NET_RCV 	0xa0	/* read a packet */
#define NET_XMIT	0xc0	/* Transmit a packet */
#define NET_MODIFY      0xe0    /* Reset board using new mode */
#define TDATA_NOT_COPIED 0xc1   /* Data of a xmit block could not be copied */
#define TDATA_COPIED    0xc2    /* Data of a xmit block was copied */
#define HOST_PUTCHAR    0xf0    /* Requests host to print the char present
                                 *   in PUTCHAR_LOC */

/* Buffer Specifications for the enp */
#define BUFFER_LENGTH 	1536    /* standard max. ethernet packet size */
#define NUM_XMIT_BUF       8
#define NUM_RCV_BUF        8

typedef struct
  {
    char buf[BUFFER_LENGTH];
  } Buffer;

/* Buffer Specifications for the host's received segments */
#define NUM_SEG_BUFFERS    4

typedef struct  
  {
    char buf[MAX_APPENDED_SEGMENT];
  } HostSegBuffer;	


#define  ENET_PLUS_KERNEL_HEADER_SIZE   (sizeof(Enet10Header)+sizeof(kPacket))

#define SHORTS_PER_ENET_ADDR    3
#define SHORTS_PER_ADDR_FILTER  4  
	
/* Init control modes */
#define PROM	0x8000		/* promiscuous */
#define	FADR 	0x4000		/* write 1 to force kernel to use Enet Station 
                                 *   addr supplied by init */
#define	INTL	0x0040		/* internal loopback, works only if loop set */
#define	DRTY	0x0020		/* disable retries upon collision */
#define	COLL	0x0010		/* when set, a collsion is forced upon 
                                 *   next transmission */
#define	DTCR	0x0008		/* disbale generation of CRC */
#define	LOOP	0x0004		/* places lance in loopback mode */
#define	DTX	0x0002		/* inhibits access to transmitter ring */
#define	DRX	0x0001		/* inhibits access to receiver ring */


/* Some of the Receive status bits */
#define ERR     0x4000          /* error occurred */
#define MISS    0x0001          /* a missed packed was reported */

/* Some of the Transmit status bits */
/* ERR is same as above */
#define STP     0x0200          /* start of packet */
#define ENP     0x0100          /* end of packet */


/* Status command's function codes */
#define RESET	0x8000		/* causes ENP and lANCE to be reset and 
                                 *   execute startup code */
#define SETB	0x4000		/* request ENP to interrupt the host */
#define SRESET	0x0800		/* reset statistics */ 
#define SREQ    0x0400          /* copy statistics into stats block */
#define RDCSR	0x0010		/* read CSR */
#define CMODE	0x0008		/* change mode using word passed in CSR0 */
#define STOP 	0x0004		/* sets stop bit CSR0 */
#define STRT	0x0002		/* causes LANCE to start operation, 
                                 *   assumning it has been initialized */
#define	INIT	0x0001		/* causes LANCE to read the init block 
                                 *   and set up for operation */


typedef unsigned char uschar;

/* Type definitions of control blocks for K1 Kernel */

typedef struct
  {
	short	mode;		    /* mode in which ENP is to operate */
	short	RcvBufNum;	    /* number of buffers in receive ring */
   	short   XmitBufNum;	    /* number of buffers in transmit ring */
	short	reserved;
/* mask used to accept incoming	multicast addresses from Ethernet */
	unsigned short	LogAddrFilter[SHORTS_PER_ADDR_FILTER];   
	int 	(*RcvInterrupt)();  /* address of receive interrupt routine */
	int 	(*XmitInterrupt)(); /* address of transmit interrupt routine */
	int	(*BusInterrupt)();  /* address of bus interrupt routine */
/* ethernet station address used only if FADR is 0 in mode */
	unsigned short  EthernetAddr[SHORTS_PER_ENET_ADDR]; 
  } InitControl;


/* Response block from K1 kernel after initializing. */
typedef struct 
  {
        /* ethernet address of the Node Processor */
	unsigned short 	EthernetAddr[SHORTS_PER_ENET_ADDR]; 
	short 	reserved;
	int 	(*StatusRtnAddr)();/* memory address of K1 Status routine */
	int	(*RcvRtnAddr)();   /* memory address of K1 Receive routine */
	int     (*XmitRtnAddr)();  /* memory address of K1 Transmit routine */
	int	(*TimerRtnAddr)(); /* memory address of K1 Timer routine */
  } InitResponse;



struct _RcvBufCtrl
  {
	struct  _RcvBufCtrl *LinkAddr; /* used to chain receive buffers */
	short 	status;		/* status word */
	short	BufLength;	/* length of the receive buffer */	
	Buffer	*BufAddr;	/* receive buffer address */
	short 	MsgLength;	/* length of message in receive buffer(s) */
	short	reserved;
  };
typedef struct _RcvBufCtrl RcvBufCtrl;



struct _XmitBufCtrl
  {
	struct  _XmitBufCtrl *LinkAddr;	/* used to chain transmit buffers */
	short	status;		/* status word */
	short 	BufLength;	/* length of the transmit buffer */
	Buffer	*BufAddr;	/* transmit buffer address */
	short	TDR;		/* time domain reflectometry, lower 10 bits */
	short	reserved;
  };
typedef struct _XmitBufCtrl XmitBufCtrl;



typedef struct
  {
	long	EnetTransmitted; /* ethernet msgs successfully transmitted */
	long	MultRetries;	 /* multiple retries reported */
	long	SingleRetries; 	 /* single retries reported */
	long	RetryFailures;	 /* retry failures reported */
	long	Deferrals;	 /* deferrals reported */
	long	XmitErrors;	 /* transmit BUF errors */
	long	SILOUnderruns;	 /* SILO underruns */
	long	LateCollisions;	 /* late collisions */
	long	CarrierLosses;	 /* carrier losses */
	long	BabblingXmitErrors; /* babbling transmitter errors */
	long	CollisionErrors; /* collision errors */		
	long	MemErrorsXmit;   /* memory erorrs on transmit */
	long	EnetReceived;	 /* ethernet messages received */
	long	MissedPackets;	 /* missed packets reported */
	long	CRCErrors;	 /* CRC errors reported */
	long	FramingErrors;	 /* framing errors*/
	long 	RcvBufErrors;	 /* receive BUF errors */
	long	SILOOverruns;	 /* SILO overuns */
	long	MemErrorsRcv;  	 /* memory errors on receive */
  } StatsBlock;



typedef struct 
  {
	short 	   FunctionCode;    /* actions to be performed by status */
	short	   CSR0Return;	    /* holds current value of CSR0 if RDCSR 
                                     *   was requested */
	StatsBlock *StatsBlockAddr; /* address of block holding statistics */
  } StatusCtrl;



struct _TimerCtrl
  {
	struct _TimerCtrl *LinkAddr;
 	short  request_code;
 	short  time;
        int    (*HandlerAddr)();
        short  event_code;
        short  reserved;
  };
typedef struct _TimerCtrl TimerCtrl;
	    

    

/* Type definitions for Host <=> ENP interaction */

typedef struct  /* used during query and modify */
  {
        Process    *current_pd; 
        StatsBlock *StatsArray;  /* place to put the stats during query */ 
    	uschar     type;         /* either NET_STATS or NET_MODIFY */
  } QueryRecord;


typedef struct
  {
        Enet10Header   EnetHeader;
        Process        *pd;        /* pd to copy the kernel packet into */
        Process        *pd_pa;     /* phys. addr. of above */
        char           *data;      /* location to copy the segment, if any */
        char           *data_pa;   /* phys. addr. of above */
        unsigned short segment_length;  /* length  of segment */
  } RcvRecord;


typedef struct
  {
	int   length;
	char  *address;
  } XmitPiece;


typedef struct
  {
        Process        *current_pd; 
        unsigned short xmit_again; 
	Enet10Header   EnetHeader;   /* must pass the header to the interface */
        int            numpieces;
	XmitPiece      piece[5];
  } XmitRecord;




typedef struct
  {
	short          mode;
	XmitRecord     *XmitRec;         /* pointer to transmit record */
        RcvRecord      *RcvRec;          /* pointer to receive record */
        QueryRecord    *QueryRec;        /* pointer to query record */
        uschar         *interrupt_loc_ptr; /* pointer to interrupt-type loc. */
	unsigned short LogAddrFilter[SHORTS_PER_ADDR_FILTER];
  } HostInitRequest;



typedef struct
  {
        unsigned short  *XmitFlagPtr;  /* ptr to state of the xmit record */
        unsigned short  *RcvFlagPtr;   /* ptr to state of the rcv record */
        unsigned short  *QueryFlagPtr; /* ptr to state of the query record */
	unsigned short 	EthernetAddr[SHORTS_PER_ENET_ADDR];
  } EnpInitResponse;




#define Max( a, b ) (a < b ? b : a)
#define Min( a, b ) (a < b ? a : b)

/* The following macro is not used because all addresses given to the Enp
 *   by the host are physical multibus addresses. */
#define SystoENPAddr(addr)			                        \
        ( ( (unsigned) addr >= BASE_ENP_SLAVE &&                        \
            (unsigned) addr < TOP_ENP_SLAVE ) ?                         \
             ((unsigned) addr & ADDRESS_MASK) | ENP_RAM_ORIGIN :        \
              ( ( (unsigned) addr < BASE_SYS_MEM ||                     \
                  (unsigned) addr >= TOP_SYS_MEM ) ?                    \
                    ENPerror( 2 ) :                                     \
                    (unsigned) addr ) )

#define ENPtoSysAddr(addr)						\
	( ((unsigned) addr & ADDRESS_MASK) | BOTTOM_ENP_SLAVE )
            


/* note that ENPerror() will not work properly if executed 
 *   by the host kernel. */
#define ENPerror(num)					\
	(*((unsigned *) ERROR_LOC) = num)



#ifdef ENPDEBUG
#define Incr_Test_Loc1()				\
  	{						\
	  *((unsigned *) TEST_LOC1) = (*((unsigned *) TEST_LOC1) + 1);	\
	}
  
#define Incr_Test_Loc2()				\
  	{						\
	  *((unsigned *) TEST_LOC2) = (*((unsigned *) TEST_LOC2) + 1);	\
	}
#define Incr_Test_Loc3()				\
  	{						\
	  *((unsigned *) TEST_LOC3) = (*((unsigned *) TEST_LOC3) + 1);	\
	}
#define Incr_Test_Loc4()				\
  	{						\
	  *((unsigned *) TEST_LOC4) = (*((unsigned *) TEST_LOC4) + 1);	\
	}
#define Incr_Test_Loc5()				\
  	{						\
	  *((unsigned *) TEST_LOC5) = (*((unsigned *) TEST_LOC5) + 1);	\
	}
#endif ENPDEBUG
  


#define Interrupt_Host( code )					\
	{							\
     	  /* don't want a receive interrupt */ 			\
    	  Disable();						\
   	  /* wait if host is processing another interrupt 	\
   	  while( *((char *)INTERRUPT_REG) != 0 )		\
     	    ;							\
	  */							\
	  *(uschar *)interrupt_loc_ptr = code;			\
	  Interrupt();						\
          Enable();						\
	}


/* cause the actual interrupt to the host */
#define Interrupt()						\
        { InterruptReq.FunctionCode = SETB;  			\
          (*(int(*)()) K1Status)( &InterruptReq ); }

#define Disable()						\
	{							\
	  *(char *)MPU_CSR &= ~ENABLE_BIT;			\
	}

#define Enable()						\
	{							\
	  *(char *)MPU_CSR |= ENABLE_BIT;			\
	}	




/* scatter the data to the host from the enp */
#define CopyToHost( RcvBlock )						\
	{								\
          /* put enet header right in the record */     		\
	  RcvRec->EnetHeader = *((Enet10Header *)RcvBlock->BufAddr);	\
          								\
          /* copy the kernel packet (header) */				\
          Copy( SystoENPAddr(&(RcvRec->pd_pa->packetType)),		\
                  ((char *)RcvBlock->BufAddr + ENET10_HEADER_SIZE),	\
                    sizeof(kPacket) );  				\
        								\
          if( RcvBlock->MsgLength > ENET_PLUS_KERNEL_HEADER_SIZE )	\
            {								\
              RcvRec->segment_length = RcvBlock->MsgLength - 		\
                                ENET_PLUS_KERNEL_HEADER_SIZE; 		\
              Copy( SystoENPAddr(RcvRec->data_pa),			\
            ((char *)RcvBlock->BufAddr + ENET_PLUS_KERNEL_HEADER_SIZE),	\
               RcvRec->segment_length );				\
            }								\
          else								\
            RcvRec->segment_length = 0;					\
	}

              
