/*
 * Distributed V Kernel 
 * Copyright (c) 1982 by Stanford University.
 *
 * Kernel Ethernet driver
 * Sun 3 Mbit version ("en")
 */

#include "Venviron.h"
#include "Vethernet.h"
#include "Vikc.h"
#include "Vquerykernel.h"
#include "interrupt.h"
#include "ether3meg.h"
#include "dm.h"
#include "ikc.h"

typedef Process_id (*PFPID)();

typedef struct { char *ptr; unsigned long bytes; } BufferList;

/* Imports */
extern unsigned char *LastPeripheral;

/* Variables imported from enet.c */
extern DeviceInstance	*EthernetInstance;
extern ProcessId	EnetWriter;	/* Pid of writer; NULL if free */
extern NetAddress	MulticastDefault;
extern char		UserPacketBuffer[];
extern int		UserPacketLength;
extern SyncQueue	IkpOutq;

/* Exports */
extern SystemCode EnetModify();
extern SystemCode EnetQuery();
extern SystemCode EnetPowerup();
extern NetworkWrite();

/* Used internally */
SystemCode EnetReadPacket();

NetAddress	EnetHostNumber = 0; 	/* physical ethernet address */
short		EnetStatus;		/* Current status settings */
int		EnetReceiveMask = ENET_DEFAULT_MASK;
				/* addresses to listen for */
int		EnetBadCounts = 0;	/* Number of bad count values */
unsigned short	EnetCollisions = 0;	/* Number of collision errors */
unsigned short	EnetOverflows = 0;	/* Queue overflow errors */
unsigned short	EnetCRCerrors = 0;	/* Packets with bad CRC's */
unsigned short	EnetSyncErrors = 0;	/* Receiver out of sync */
unsigned short	EnetTimeouts = 0;	/* Transmitter timeouts */
int		EnetValidPackets = 0;
unsigned short  EnetPromiscuous = 0;	/* Are we in promiscuous mode. */

char		KernelPacketBuffer[ENET3_MAX_PACKET*sizeof(short)];
int		NetAlarm = 0;		/* Not used in this driver */


/* Macro expansion to interrupt-invoked C call to Ethernetinterrupt */
Call_inthandler(EnetInterrupt)


EnetFlushReceiver()

  /* Flush receiver buffer just in case of problems.
   */
  {
    register unsigned count;
    register short status;

    for( count = 0; (count < 0x4000) && (EIStatus & EnetRxInt); ++count )
      status = EIData0;
    if( count == 0x4000 ) Kabort( "Ethernet interface failure" );
  }

SystemCode EnetPowerup()

  /* Powerup and initialize the Ethernet Interface Board.
   *  Done just once, from kernel Powerup() routine. 
   */
  {
    extern int Asm_EnetInterrupt();
    register int (**intvec)() = (int(**)()) ENET_INT_LEVEL;
    
    /* Indicate we have a 3 meg board */
    *LastPeripheral++ = PRF_ENET_3MBIT;
    IkpOutq.tail = (Process *) &(IkpOutq.head);
    IkpOutq.type = IKP_OUT_QUEUE;
    EnetReceiveMask = (ENET_SELF+ENET_BROADCAST);
    
    /* Plug interrupt location */
    *intvec = Asm_EnetInterrupt;

    /* Get hardwired address of this host */
    EnetHostNumber = ( char ) EIAddress;

    EnetReset( ENET_DEFAULT_MASK );
  }

/* NOTE: This version of the driver ignores the receive mask */
EnetReset( receiveMask )
    int receiveMask;
  /*
   * Initialize or re-initialize the Ethernet board.
   */
  {
    register int count;

    EnetStatus = EnetInit + EnetIntDisable + EnetIntLevel1 +
		(EnetReceiveMask&ENET_PROMISCUOUS ? 0 : EnetNoFilterData);

    EIStatus = EnetStatus; /* Write to interface status register */
    for( count = 0; count < 256; count++ ) EIAddress = count;

    /*  Setup for self and broadcast. ( and multicast ) */

    EnetStatus &= ~EnetNoFilterData;
    EIStatus = EnetStatus;

    if( EnetReceiveMask & ENET_BROADCAST ) EIAddress = 0;

    if( EnetReceiveMask & ENET_SELF ) EIAddress = EnetHostNumber;

    /* Turn off initialization and enable interrupts */
    EnetStatus &= ~(EnetInit + EnetIntDisable);
    EIStatus = EnetStatus;
    
    EnetFlushReceiver();

    return( OK );
 }

EnetInterrupt()

   /* Handle an interrupt from the ethernet interface.
    */
  {
    extern ProcessId EnetWriter;
    register Process *pd;
    register short transmitterack;
    if( EIStatus & EnetTxInt ) /* Transmitter interrupt */
      {
	transmitterack = EIAddress;
	/* If IkpOutq is non-empty, transmit next interkernel packet. */
	if( (pd=IkpOutq.head) != NULL )
	  {
	    Lockq( &IkpOutq );
	    if( (IkpOutq.head = pd->link) == NULL )
		 IkpOutq.tail = (Process *) &(IkpOutq.head);
	    pd->queuePtr = NULL;
	    Unlockq( &IkpOutq );
	    WriteNextKPacket( pd );
	  }
	else EnetWriter = 0; /* No transmission in progress now */
	if( EIStatus & EnetTimeout )++EnetTimeouts;
      }
    else /* receiver interrupt */
        if( EIStatus&EnetRxInt && (EnetStatus&EnetRxIntDisable) == 0 )
          {
	    EnetStatus |= EnetRxIntDisable;
	    EIStatus = EnetStatus;

	    EnetReadPacket( 0 );
          }
  }

SystemCode EnetReadPacket()
   /*
    *  Physically transfer a packet from the Ethernet
    *  Interface Board into the reading process's buffer.
    */
  {
    extern PFPID RemoteTrapTable[];
    Process *pd;
    register short *save, *p;
    register unsigned words, readwords;
    DeviceInstance *inst;
    IoRequest *req;
    Team *oldteam;
    SystemCode ret;
    short status;

#define enet ((Enet3Header *) &KernelPacketBuffer[0])
#define kp ((kPacket *) &KernelPacketBuffer[sizeof(Enet3Header)])

    /* Check if anything is in the receiver FIFO queue */
    if( (EIStatus & EnetRxInt) == 0 ) return( END_OF_FILE );

    status = EIData0;
    if (status & EnetQueueEmpty)
      {
	/* This was in reality the lookahead word.
	 *   The next word is the real status word. */
	status = EIData0;
      }
    if (status & EnetQueueEmpty)
      {
	/* Still empty, although we received an interrupt.
	 *   The interface has probably been wedged by receiving
	 *   a packet of >= 2046 words, so we reinitialize it.
	 */
	EnetReset();
	EnetOverflows++;
	return (DEVICE_ERROR);
      }

    /* Find a valid error-free packet */
    while( !(status & EnetQueueEmpty) )
      {
        words = (status & EnetCountMask) - 1;
	if(((status & (EnetCollision|EnetOverflow|EnetCRCerror)) == 0) && 
	   (words <= (ENET3_MAX_PACKET>>1)) &&
	   (words > 8))
	  {
	    /* A valid packet */

	    /* Get the first eight words of the packet and figure out
	       if it is a non-alias process kernel packet */
	    save = (short *) &KernelPacketBuffer[0];
	    for (readwords = 0; readwords < 8; readwords++) *save++ = EIData0;
	    
	    /* First check if we are hearing our own packets. Discard if so. */
	    if (enet->SrcHost == EnetHostNumber)
	      {
		words -= 8;
		while (words--) status = EIData0;
		ret = NO_REPLY;
		goto finishpacket;
	      }

#ifdef undef
/* We have to swap the dstPid first */
	    if((enet->EtherType != KERNEL_PACKET) ||
	       (kp->dstPid & REMOTE_ALIAS_PROCESS))
#endif
 	    if (enet->EtherType != KERNEL_PACKET)
	      {
		ret = NO_REPLY;
		/* First case: it is not a kernel packet */
		/* and not to a remote alias process */
		/* Ethernet device not open */
		if( (inst = EthernetInstance) == NULL )
		  {
		    /* Discard packet */
		    words -= 8;
		    while( words-- ) status = EIData0;
		    goto finishpacket;
		  }

		if( !MAP_TO_RPD(pd, inst->reader) )
		  /* No reader process */
		  {
		    if( !MAP_TO_RPD(pd, inst->owner) )
		      {
			/* Ethernet device not open */
			inst->owner = 0; /* Free the instance */
			EthernetInstance = NULL;
			inst = NULL;
			EnetReset( ENET_DEFAULT_MASK );
			return( ret );
		      }

		    /* The read request for this packet hasn't been
		     * issued yet. Instead of trashing the packet,
		     * store the packet in the UserPacketBuffer. If
		     * another packet comes in while waiting overwrite
		     * the buffer.
		     */
		    pd = NULL;
		    UserPacketLength = words * sizeof(short);
		    /* Stick the first eight words where they belong */
		    Copy(UserPacketBuffer,KernelPacketBuffer,8*sizeof(short));
		    p = (short *) &UserPacketBuffer[8*sizeof(short)];
		    words -= 7;
		    while (--words) *p++ = EIData0;
		  }
		else
		  {
		    /* There is a reader waiting for this packet. Transfer the
		     * packet from the read buffer to the waiting process
		     * buffer.
		     */
		    req = (IoRequest *) &pd->msg;
		    /* First, fill in the bytecount in the reply */
	    	    if ( ( words << 1 ) < req->bytecount )
			req->bytecount = words << 1;

		    /* Now decide where to put the data */
		    if( req->bytecount <= IO_MSG_BUFFER ) 
			p = (short *) req->shortbuffer;
		     else
			p = (short *) req->bufferptr;

		    /* Copy data to correct location */
		    oldteam = GetAddressableTeam();
		    SetAddressableTeam(pd->team);

		    /* Stick the first eight words where they belong */
		    Copy(p, KernelPacketBuffer, 8*sizeof(short));
		    p += 8;
		    readwords = (req->bytecount >> 1) + (req->bytecount & 1);
		    if( words <= readwords )
		      {
			words -= 7;
			while( --words ) *p++ = EIData0;
		      }
		    else /* A larger packet than requested */
		      {
		        words -= readwords;
			readwords  -= 7;
			while( --readwords ) *p++ = EIData0;
			++words;
			/* Discard extra words */
			while( --words ) status = EIData0;
		      }

		    ret = OK;

		    SetAddressableTeam(oldteam);
		    ret = OK;
		    inst->reader = 0;
		    req->requestcode = OK;
		    Addready( pd );
		  }
	      }
	    else
	      {
		/* Second case : a kernel packet */
		words -= 8;
		/* Copies the rest of the packet into the KernelPacketBuffer */
		while(words--) *save++ = EIData0;
		if( (enet->EtherType == KERNEL_PACKET) &&
		    DifferentIKCByteOrder(kp) )
	  	  {
		    SwapIKPacket(kp);
		    kp->packetType &= ~IKC_LITTLE_ENDIAN; /* Just to be tidy */
	          }
		if (kp->packetType != remoteForward)
		    HostCacheEnter(kp->srcPid>>16, enet->SrcHost);
		(*RemoteTrapTable[(kp->packetType)&0xf])(kp);
		ret = NO_REPLY;
	      }
finishpacket:
	    /* Check that we have not hit the end of the receiver buffer */
	    if( !(EIStatus & EnetRxInt) )
	      {
		/* We have read past the end of the receiver buffer */
		++EnetSyncErrors;
		ret = DEVICE_ERROR;
		goto enableRxInt;
	      }
	    ++EnetValidPackets;
	    status = EIData0;	/* Read and discard CRC checksum */
	    goto enableRxInt;
	  }
	/* Otherwise we discard this packet */

	if( status & EnetCollision ) ++EnetCollisions;
	if( status & EnetOverflow ) ++EnetOverflows;
	if( status & EnetCRCerror ) ++EnetCRCerrors;
	words += 2;
	while( --words ) status = EIData0;

	status = EIData0;	/* Read the status word for the next packet.
				   */
      }
    
    ret = END_OF_FILE;

enableRxInt:
    EnetStatus &= ~EnetRxIntDisable;
    EIStatus = EnetStatus;
    return( ret );
#undef kp
#undef enet
  }


NetworkWrite(bufferptr)
register BufferList *bufferptr;
  {
    register short *p;
    register unsigned long bytes;
    register BufferList *bl;
    register unsigned extra;
    register unsigned words;

    if (bufferptr->bytes < sizeof(Enet3Header)) Kabort("Bad network header");
  
    for (bytes = 0, bl = bufferptr; bl->ptr != 0; bl++) bytes += bl->bytes;

    ((Enet3Header *) bufferptr->ptr)->SrcHost = EnetHostNumber;

    EIData0 = (bytes + (bytes & 1)) / 2;/* Packet length in shorts */
    
    extra = 0;					/* Extra byte on prev buffer */
    while (bufferptr->ptr)
      {
	if (bufferptr->bytes)
	  {
	    if (extra)
		EIData0 = (short) (*bufferptr->ptr |
			(*((bufferptr-1)->ptr+(bufferptr-1)->bytes - 1) << 8));
	    p = (short *) (bufferptr->ptr + extra);
	    /* add one to use predecrementing for speed */
	    words = (bufferptr->bytes - extra)/2 + 1;
	    if ( ((unsigned) p & 1) == 0 ) while ( --words ) EIData0 = *p++;
	    else
	      {
		while (--words)
	          {
#define bp ((char *) p)
		    EIData0 = (*bp) << 8 | *(bp+1);
#undef bp
		    p++;
		  }
	      }
	    extra = (bufferptr->bytes - extra) & 1;
	  }
	bufferptr++;
      }
    if (extra) EIData0 = *((bufferptr-1)->ptr + (bufferptr-1)->bytes - 1) << 8;
  }


/* NOTE: EnetModify not implemented in this driver */
SystemCode EnetModify( pd, inst, dirIndex )
    register Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
   return( OK );
  }    

SystemCode EnetQuery( pd, inst, dirIndex ) 
    Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    register QueryEnetReply *reply = (QueryEnetReply *) &pd->msg;

    reply->NetworkType = ENET_TYPE_3MBIT;
    reply->NumCollisions = EnetCollisions;
    reply->NumOverflows = EnetOverflows;
    reply->NumCRCErrors = EnetCRCerrors;
    reply->NumSyncErrors = EnetSyncErrors;
    reply->NumTimeOuts = EnetTimeouts;
    reply->ReceiveMask = EnetReceiveMask;
    reply->NumValidPackets = EnetValidPackets;
    reply->HostAddress.e3 = EnetHostNumber;

    return( OK );
  } 

int NetCheck() {}	  

SystemCode AddLogicalHostGroup(lhg) GroupId lhg; { return( OK ); };

SystemCode DeleteLogicalHostGroup(lhg) GroupId lhg; { return( OK ); };
