/*
 * Distributed V Kernel 
 * Copyright (c) 1985 by Stanford University.
 *
 * Kernel Ethernet driver
 * 
 */

/*
V-system kernel ethernet driver for the Sun Microsystems workstations
with integrated Intel 82586 LAN controller chips (Sun-2/50 and Sun-3).

All constants used in this driver were derived from the 1984 Intel LAN
Components User Manual, which describes in gory detail all aspects of the
Intel 82586 ethernet controller chip. Additional information was gleaned
from twenty pages of errata blackmailed out of Intel (Ve must haf ze secret
dokumints !) and from the Sun-2 Architecture manual. The Intel book contains
some fairly well written descriptions of both the chip and of how they
expect device drivers to interface to it. Leafing through the book would be
helpful before working on the main driver code.

In places a choice was made to have a simple driver rather than use some of
the more complicated chip features. Chief among these were the decisions to
never use transmit or receive buffer chaining (scatter/gather) by allocating
buffers that are large enough to contain packets maximum length packets, to
never use command chaining (only makes sense when transmiting multiple
packet messages), and to allocate a single transmit buffer. At some point in
the future it might be nice to add a second transmit buffer to allow one to
be filled while the second is in use by the chip. This could double the
maximum packet output rate.

NetAlarm: This chip has a nasty tendancy to wedge. The most common case
seems to occur when the chip runs out of receive buffers. A less frequent
bug occurs during packet transmission. The receive and transmit sides of the
chip are totally independant when it comes to wedging, and the only way to
get rolling again is to give the chip a hardware reset. A deadman timer
(NetAlarm) is used to detect chip deadlock, and has two timeout periods. The
first resets the chip if a transmit doesn't complete in a fairly short
amount of time; the second causes a reset if no packets are received during
a much longer interval.

Buffer allocation: The amount of buffer space allocated is determined at
powerup time. 16k is allocated on systems with 1meg or less of main memory.
Systems with more memory allocate 32k. One buffer is dedicated to packet
transmission with the rest going for receive buffer space. A 2meg Sun thus
allocates 20 receive buffers, which should be enough to keep it from missing
anything exciting even in the busiest of times.

Packet reception: Upon reception of a receive interrupt 'NextFrame' is used
to determine the 'receive ring buffer descriptor array entry' where the
incoming packet has been stored. Kernel packets are handled immediately by
calling the apporpriate routine through the RemoteTrapTable. User packets
are either dropped on the floor (if the raw device isn't open), are given
directly to the user process, or are buffered in the following way: One
internal buffer is used to store user packets received when the raw device
is open but a read request is not currently issued.  Future incoming packets
overwrite this buffer - the next read will get the last packet received, not
the first. This is meant to provide enough slop for the current crop of
applications which open the raw device (such as the internet server) and
isn't meant as a reliable buffering mechanism.

Multicast reception: The driver maintains a list of multicast addresses
being listened to. This list is modified by {Add,Delete}LogicalHostGroup
requests. A fixed maximum number of multicast addresses may be listened to
at a given instant. The chip's filter mechanism isn't perfect, so it is
possible to receive packets destined for multicast addresses that were not
included in the last MCsetup command.

Packet transmission: 'EnetWriter' is used to prevent concurrent use of the
single transmit buffer descriptor (and associated data buffer). Interkernel
transmit requests are queued on the IkpOutq. User programs have to rely on
busy waiting.
*/

/* Must be before i82586.h */
#define MAX_MULTICAST_ADDRESSES 50

#include "Vethernet.h"
#include "Vikc.h"
#include "Vquerykernel.h"
#include "Venviron.h"
#include "interrupt.h"
#include "dm.h"
#include "ikc.h"
#include "i82586.h"
#include "memory.h"

#ifndef SUN3
#define V_INTEL_ETHERNET 0xEE3000 /* Put this into sun2mem.h ! */
#endif

/* Ethernet control register address */
#define ENETREG V_INTEL_ETHERNET
#define MAX_RECEIVE_BUFFERS 32

typedef Process_id (*PFPID)();

typedef struct { char *ptr; unsigned long bytes; } BufferList;

#define ENET_READER_TIMEOUT (CLICKS_PER_SEC*5)
#define ENET_WRITER_TIMEOUT (CLICKS_PER_SEC/20)

/* Imports */
extern unsigned char *LastPeripheral;
extern MachineConfigurationReply MachineConfig;
extern SystemCode AllocateMemory();

/* Variables imported from enet.c */
extern DeviceInstance	*EthernetInstance;
extern ProcessId	EnetWriter;	/* Pid of writer; NULL if free */
extern NetAddress	MulticastDefault;
extern char		UserPacketBuffer[];
extern int		UserPacketLength;
extern SyncQueue	IkpOutq;

/* Exports */
extern SystemCode EnetModify();
extern SystemCode EnetQuery();
extern SystemCode EnetPowerup();
extern NetworkWrite();

/* Used internally */
SystemCode EnetReadPacket();
unsigned short swap1();	

NetAddress	EnetHostNumber;		/* physical ethernet address */
int		EnetReceiveMask = ENET_DEFAULT_MASK;
unsigned short	EnetCollisions = 0;	/* Number of collision errors */
unsigned short	EnetOverflows = 0;	/* Queue overflow errors */
unsigned short	EnetNetChecks = 0;
int		EnetValidPackets = 0;

/* Driver state */
unsigned short  EnetPromiscuous = 0;	/* Are we in promiscuous mode. */
int		NextFrame;		/* Next receive frame to look at */
unsigned long	ReceiveBuffers;		/* # of receive buffers */
int		LastRNR = 0;		/* EnetValidPackets last Read reset */
int		NetAlarm = 0;		/* Used for ethernet watchdog timer */
int		LogicalHostCount = 0;

struct SystemConfigurationPointer *scp;

/* All control blocks have to be in a single 64k segment, so they are
 * declared in a single structure. The iscp was thrown in since it needs
 * to also be in memory addressable by the chip (anywhere on the Sun-2,
 * upper 16 meg on the Sun-3).
 */
typedef struct
  {
    struct ISystemConfigurationPointer iscp;
    struct SystemControlBlock scb;
    struct IAsetupblock IAsetupBlock;
    struct Transmitblock xblock;
    struct TransmitBufferDescriptor tbd;
    struct ReceiveFrameDescriptor rfa[MAX_RECEIVE_BUFFERS];
    struct ReceiveBufferDescriptor rbd[MAX_RECEIVE_BUFFERS];
    struct MCsetupblock MCBlock;
  }  ControlBlockSegment;

/* We'll put the ethernet control block at the front of the enet data
 * segment, so the pointer to the control block is a constant.
 */
#define cbsp ((ControlBlockSegment *) ENET_DATA_SEG)

#define Offset(addr) ((unsigned short)(((long)(addr))-((long)&(cbsp->scb))))
/*#define Address(offset) ((char *) (((long)offset) + (long)&(cbsp->scb)))*/

/* Receive and transmit data buffers */
char *rbuf, *xbuf;

/* In order to preserve the error counts in the scb only
 * the data part of the block gets swapped.
 */
int CONTROL_SWAP_LEN;

/* Macro expansion to interrupt-invoked C call to Ethernetinterrupt */
Call_inthandler(EnetInterrupt)

 
SystemCode EnetPowerup()

  /* Powerup and initialize the Ethernet Interface Board.
   *  Done just once, from kernel Powerup() routine. 
   */
  {
    extern int Asm_EnetInterrupt();
    register int (**intvec)() = (int(**)()) (VECTOR_BASE + INT3);
    register char *p;
    register unsigned long i;
    
    /* We want to maintain the scb error counts, so we don't swap those
     * fields when byte swapping the scb. CONTROL_SWAP_LEN is the number
     * of bytes to swap.
     */
    CONTROL_SWAP_LEN = 2 * (&(cbsp->scb.CRCerrs) - &(cbsp->scb.status));
    
    /* Indicate we have a Sun ethernet interface */
    *LastPeripheral++ = PRF_ENET_SUN50;
    IkpOutq.tail = (Process *) &(IkpOutq.head);
    IkpOutq.type = IKP_OUT_QUEUE;
    EnetReceiveMask = ( ENET_SELF+ENET_BROADCAST );
    
    /* Allocate receive and transmit buffers */
    /* NOTE: Be very careful before allocating 64k of enet data space. The
     * last page in memory (0xfffe000) is already allocated by the prom. It
     * is also contains the SCPADDR, so it is used whenever the chip is
     * reset (this shouldn't be a problem since the chip is reset only after
     * all pending buffers are read). See sun3mem.h and sun3stdmap.doc for
     * details on address assignments and prom-defaulted memory mapping.
     */
    if ( MachineConfig.memory <= 0x100000 ) i = 0x4000;		/* 16k */
    else if ( MachineConfig.memory <= 0x200000 ) i = 0x8000;	/* 32k */
    else i = 0xB000;						/* 48k */
    if ( AllocateMemory( ENET_DATA_SEG, i ) != OK )
	Kabort( "Couldn't allocate ethernet data buffer space !" );
    xbuf = ((char *) ENET_DATA_SEG) + sizeof(ControlBlockSegment);
    rbuf = xbuf + ENET_MAX_DATA;
    ReceiveBuffers = min( MAX_RECEIVE_BUFFERS,
	( (ENET_DATA_SEG + i) - (unsigned long) rbuf) / ENET_MAX_DATA  );

printx("Allocating %d ethernet receive buffers.\n", ReceiveBuffers );

    /* Read the ethernet address from the SMI prom */
    p = (char *) &EnetHostNumber;
#ifdef SUN3
    for (i = 2; i < 8; i++) *p++ = Fc3ReadBit8(i);
#else
    for (i = 0x1008; i < 0x4008; i += 0x800) *p++ = Fc3ReadBit8(i);
#endif
    
    *intvec = Asm_EnetInterrupt;  /* Plug interrupt location */

    cbsp->MCBlock.addresses[0] = MulticastDefault;
    LogicalHostCount = 1;

    EnetReset( ENET_DEFAULT_MASK );
  }

EnetReset( receiveMask )
    int receiveMask;
  {
    register long i;
    register ControlBlockSegment *cbsr = cbsp;
    register struct IAsetupblock *IAblock = &(cbsp->IAsetupBlock);
    register struct SystemControlBlock *c = &cbsp->scb;
    extern AllocatePage(), FreePage();
    
#ifndef SUN3
    /* The page containing the configuration pointer isn't
     * mapped in on the Sun-2 under normal circumstances,
     * so we map it in just for the duration of the powerup
     * sequence.
     */
     AllocatePage(SCPADDR);
#endif

    /* power up the ethernet chip by:
     *     Writing the System Configuration Pointer and the
     *     Intermediate SCP.
     *     Reseting the chip.
     *     Setting up the chip's individual address and multicast buffers.
     */
    scp = (struct SystemConfigurationPointer *) SCPADDR;
    scp->sysbus = 0;
    scp->ISCPlow = LOW(&cbsp->iscp);
    scp->ISCPhigh = HIGH(&cbsp->iscp);
    SwabInPlace(scp, sizeof(struct SystemConfigurationPointer));

    cbsp->iscp.busy = 1;
    cbsp->iscp.SCBoffset = 0;
    cbsp->iscp.SCBlow = LOW(c);
    cbsp->iscp.SCBhigh = HIGH(c);
    SwabInPlace(&cbsp->iscp, sizeof(struct ISystemConfigurationPointer));

    c->RSCerrs = c->CRCerrs = c->ALNerrs = c->OVRNerrs = 0;
    c->command = 0;
    *(unsigned char *)ENETREG = 0;
    Attention();
    
    c->command = CUC_START;
    c->CBLoffset = Offset(IAblock);
    SwabInPlace(c, CONTROL_SWAP_LEN);

    IAblock->command = sIACOMMAND;
    IAblock->link = 0xffff;
    IAblock->address = EnetHostNumber; /* don't swap the host # */
    Attention();
 
#ifndef SUN3
    FreePage(SCPADDR);
#endif

    InitEnetRead();
    MulticastReset();
  }  

InitEnetRead()
  {
    register int i;
    register ControlBlockSegment *c = cbsp;
    register struct ReceiveFrameDescriptor *f = c->rfa;
    register struct ReceiveBufferDescriptor *r = c->rbd;

#ifdef undef
printx("%x ", EnetValidPackets - LastRNR);
LastRNR = EnetValidPackets;
#endif

    /* Start (or restart) the Receive Unit on the chip. First
     * relink the receive frame and buffer queues in case they've
     * been corrupted.
     */
    NextFrame = 0;
    c->scb.command = RUC_START;
    c->scb.RFAoffset = Offset(f);
    c->scb.CBLoffset = 0xffff;
    SwabInPlace(&(c->scb), CONTROL_SWAP_LEN);

    for(i = 0; i < ReceiveBuffers; i++, f++, r++)
      {
	f->command = 0;
	r->size = ENET_MAX_DATA;
	r->count = 0;
	r->bufferlow = LOW(rbuf + (i * ENET_MAX_DATA) );
	r->bufferhigh = HIGH(rbuf + (i * ENET_MAX_DATA) );
	/* special code for the first buffers */
	if (i == 0)
	  {
	    f->buffer = Offset(r); /* Only the first rfd points to a rbd */
	    (f+(ReceiveBuffers-1))->link = Offset(f);/* circular lists - */
	    (r+(ReceiveBuffers-1))->next = Offset(r);/* last points to first*/
	  }
	else
	    f->buffer = 0xffff; /* Not the first rfa */

	/* special code for the last buffers */
	if (i == ReceiveBuffers-1)
	  {
	    f->command = 0x8000; /* end of list */
	    r->size |= 0x8000;
	  }
        else
	  {
	    f->link = Offset(f+1);
	    r->next = Offset(r+1);
	  }
	SwabInPlace(f, sizeof(struct ReceiveFrameDescriptor));
	SwabInPlace(r, sizeof(struct ReceiveBufferDescriptor));
      }
    Attention();
  }

EnetInterrupt()

   /* Handle an interrupt from the ethernet interface.
    */
  {
    extern ProcessId EnetWriter;
    register Process *pd;
    int PrevFrame;
    int LastStatus;
    register struct SystemControlBlock *s = &(cbsp->scb);
    register struct ReceiveFrameDescriptor *FramePtr;
     
    /*
     * Determine type of interrupt.
     */
    LastStatus = swap1(cbsp->scb.status);
    s->command = s->status & sACK_MASK; /* Acknowledge */
    s->CBLoffset = 0xFFFF; /* palindrome - no need to swap this */
    Attention();
    if (LastStatus & CX)
      {
        EnetCollisions += (swap1(cbsp->xblock.status) & XMIT_COLLISION_MASK);
	/* If IkpOutq is non-empty, transmit next interkernel packet. */
	Lockq( &IkpOutq );
	if( (pd=IkpOutq.head) != NULL )
	  {
	    if( (IkpOutq.head = pd->link) == NULL )
		IkpOutq.tail = (Process *) &(IkpOutq.head);
	    pd->queuePtr = NULL;
	    Unlockq( &IkpOutq );
	    WriteNextKPacket( pd );
	  }
	else
	  {
	    Unlockq( &IkpOutq );
	    EnetWriter = 0; /* No transmission in progress now */
	  }
      }
    
    if (LastStatus & FR)
	while ((FramePtr = &cbsp->rfa[NextFrame])->status & 0x0080)
      {
	if (NextFrame == 0)
	    PrevFrame = ReceiveBuffers - 1;
	else
	    PrevFrame = NextFrame - 1;
	EnetReadPacket(NextFrame);
	FramePtr->status = 0;
	FramePtr->buffer = 0xffff;
	/* Make this buffer the last valid buffer by setting the end-of-list
	 * bits in its frame & buffer descriptors. Clear those bits in 'prev'
	 * which was the old last valid buffer.
	 */
	FramePtr->command = 0x0080;
	cbsp->rbd[NextFrame].size |= 0x0080;
	cbsp->rfa[PrevFrame].command = 0;    /* Clear the end-of-list bits */
	cbsp->rbd[PrevFrame].size &= 0xff7f;
	if (++NextFrame == ReceiveBuffers) NextFrame = 0;
      }
    /* If the chip has left the ready state restart the receive unit
     * (after processing all of the outstanding buffers). Reinitialize
     * the links just to be sure.
     */ 
    if (LastStatus & RNR)
      {
	EnetOverflows++;
	InitEnetRead();
      }
    if (EnetWriter == 0) NetAlarm = ENET_READER_TIMEOUT;
  }


SystemCode EnetReadPacket( FrameNum )
int FrameNum;
  {
    extern PFPID RemoteTrapTable[];
    Process *pd;
    DeviceInstance *inst;
    register kPacket *kp;
    register IoRequest *req;
    Team *oldteam;
    register unsigned	bytes;
    SystemCode		ret;
    register char *p;
    register struct ReceiveFrameDescriptor *Frame = &(cbsp->rfa[FrameNum]);
    /* SendGroupMembers() may find a remote alias process group member. */
    extern int DeliverToEnetReader;

    /* Look at the first few words of the packet to determine
       if it is a kernel packet and not addressing an alias process */
    kp = (kPacket *) ( rbuf + (FrameNum * ENET_MAX_DATA) );
    bytes = (swap1(cbsp->rbd[FrameNum].count) & 0x3fff) + sizeof(Enet10Header);
    DeliverToEnetReader = 0;

    if( Frame->type == KERNEL_PACKET && DifferentIKCByteOrder(kp) )
      {
	SwapIKPacket(kp);
	kp->packetType &= ~IKC_LITTLE_ENDIAN; /* Just to be tidy */
      }
    if( Frame->type != KERNEL_PACKET ||	(kp->dstPid & REMOTE_ALIAS_PROCESS) )
      {
nonKernelPacket:
	/* First case: it is not a kernel packet */
	/* and not to a remote alias process */

	if( (inst = EthernetInstance) == NULL )/* Ethernet device not open */
	    goto finishpacket;
	if( !MAP_TO_RPD(pd, inst->reader) )
	  {
	    /* No reader process */
	    if( !MAP_TO_RPD(pd, inst->owner) )
	      {
		inst->owner = 0; /* Free the instance */
		EthernetInstance = NULL;
		inst = NULL;
		EnetReset( ENET_DEFAULT_MASK );
		/* Ethernet device not open */
		goto finishpacket;
	      }
	    /* If in promiscuous mode, dont save packet if
	     * it is a kernel packet - too risky.*/
	    if( EnetPromiscuous &&
		(Frame->type == KERNEL_PACKET &&
		(kp->dstPid & REMOTE_ALIAS_PROCESS)) )
		goto finishpacket;

	    /* There is no reader; copy the packet into the user save area */
	    UserPacketLength = bytes;
	    Copy(UserPacketBuffer, &(Frame->destination), sizeof(Enet10Header));
	    Copy(UserPacketBuffer + sizeof(Enet10Header), kp,
						bytes - sizeof(Enet10Header));
	    goto finishpacket;
	  }
	/* First, fill in the bytecount in the reply */
	req = (IoRequest *) &(pd->msg);
    	if ( bytes < req->bytecount )
	  req->bytecount = bytes;
    	/* Now decide where to put the data */
    	if( req->bytecount <= IO_MSG_BUFFER ) 
	  p = req->shortbuffer;
    	else
	  p = req->bufferptr;

	/* Copy data to correct location */
	oldteam = GetAddressableTeam();
	SetAddressableTeam(pd->team);
	Copy(p, &(Frame->destination), sizeof(Enet10Header));
	Copy(p + sizeof(Enet10Header), kp, req->bytecount-sizeof(Enet10Header));
	SetAddressableTeam(oldteam);
	ret = OK;
	inst->reader = 0;
	req->requestcode = OK;
	Addready( pd );
      }
    else
      {
	/* Second case : a kernel packet */
	/* Don't cache remoteForward pids because multiple forward exist
	 * and determining which field was the source is too confusing.
	 * The protocol shold be fixed.
	 */
	if (kp->packetType != remoteForward)
	    HostCacheEnter(kp->srcPid>>16, Frame->source);

	/* Ensure in correct packetType range. */
	kp->packetType = kp->packetType & 0xF;
#ifdef undef
printx("%c",(kp->packetType<10?'0':('A'-10)) + kp->packetType);
#endif undef
	(*RemoteTrapTable[kp->packetType])(kp);
	ret = NO_REPLY;
	/* If in promiscuous mode, let the enet reader get too. */
	if( EnetPromiscuous || DeliverToEnetReader ) goto nonKernelPacket;

      }
finishpacket:
    ++EnetValidPackets;
  }

NetworkWrite(bufferptr)
register BufferList *bufferptr;
  {
    register ControlBlockSegment *c = cbsp;
    register char *xbufptr;
    register unsigned long bytes;

    if (bufferptr->bytes < sizeof(Enet10Header)) Kabort("Bad network header");

    c->scb.command = CUC_START;
    c->scb.CBLoffset = Offset(&(c->xblock));
    SwabInPlace(&(c->scb), CONTROL_SWAP_LEN);
    c->tbd.bufferlow = LOW(xbuf);
    c->tbd.bufferhigh = HIGH(xbuf);
    c->xblock.TBDoffset = swap1(Offset(&c->tbd));
    c->xblock.command = sTRANSMIT;
    c->xblock.status = 0;
    c->xblock.destination = ((Enet10Header *) (bufferptr->ptr))->DestHost;
    c->xblock.type = ((Enet10Header *) (bufferptr->ptr))->EtherType;
    bufferptr->bytes -= sizeof(Enet10Header);
    bufferptr->ptr += sizeof(Enet10Header);

    for (xbufptr = xbuf; bufferptr->ptr != NULL; bufferptr++)
        if ((bytes = bufferptr->bytes) != 0)
	  {
	    Copy(xbufptr, bufferptr->ptr, bytes);
	    xbufptr += bytes; 
	  }

    c->tbd.count = 0x8000 | (xbufptr - xbuf);
    SwabInPlace(&(c->tbd), sizeof(struct TransmitBufferDescriptor));
    Attention();
    NetAlarm = ENET_WRITER_TIMEOUT;
  }


unsigned short swap1(y)
register int y;
  {
/* Byte swap the argument (assumed to be an unsigned short).
 * Slightly machine dependent.
 */
    register int z;
    asm("andl #65535,d7");
    asm("movl d7,d0");
    asm("asrl #8,d0");
    asm("asll #8,d7");
    asm("addl d7,d0");
    asm("andl #65535,d0");
  }

SwabInPlace(ptr, len)
register unsigned short *ptr;
register int len;
  {
    for ((len >>= 1); len--; ptr++) *ptr = swap1(*ptr);
  }

Attention()
  {
    register unsigned short *command = &cbsp->scb.command;

    *(unsigned char *)ENETREG = (unsigned char) I_CA+I_LOOPB+I_RESET+I_INTEN;
    asm("nop");		/* Give chip time to hear interrupt */
    asm("nop");
    *(unsigned char *)ENETREG = (unsigned char) I_LOOPB+I_RESET+I_INTEN;
    while (*command != 0); /* Wait for command to be accepted */
  }


SystemCode EnetModify( pd, inst, dirIndex )
    register Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    register QueryEnetReply *reply = (QueryEnetReply *) &(pd->msg);

    if( reply->NumOverflows != -1 )
	EnetOverflows = reply->NumOverflows;
    if( reply->NumValidPackets != -1 )
	EnetValidPackets = reply->NumValidPackets;
    if( reply->ReceiveMask != EnetReceiveMask &&
        reply->ReceiveMask != -1 ) EnetReset( reply->ReceiveMask );

   return( OK );
 }    

SystemCode EnetQuery( pd, inst, dirIndex ) 
    Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    register QueryEnetReply *reply = (QueryEnetReply *) &(pd->msg);
    register ControlBlockSegment *c = cbsp;

/* The query request was aimed at the 3com board, so we
 * just fill it in as best we can.
 */
    reply->NetworkType = ENET_TYPE_10MBIT;
    reply->NumCollisions = EnetCollisions;
    reply->NumOverflows = EnetOverflows;	/* count of RNR resets */
    reply->NumCRCErrors = swap1(c->scb.CRCerrs);/* */
    reply->NumSyncErrors =
			swap1(c->scb.OVRNerrs) + /* Lost - couldn't get bus */
			swap1(c->scb.ALNerrs);   /*    alignment error */
    reply->NumTimeOuts = swap1(c->scb.RSCerrs);	/*     due to no descriptors*/
    reply->ReceiveMask = EnetReceiveMask;
    reply->NumValidPackets = EnetValidPackets;
    reply->HostAddress.e10 = EnetHostNumber;
    return( OK );
  } 


NetCheck()
  {
    register Process *pd;
    
    EnetReset( ENET_DEFAULT_MASK );
    /* If IkpOutq is non-empty, transmit next interkernel packet. */
    Lockq( &IkpOutq );
    if( (pd = IkpOutq.head) != NULL )
      {
        if( (IkpOutq.head = pd->link) == NULL )
	    IkpOutq.tail = (Process *) &(IkpOutq.head);
	pd->queuePtr = NULL;
	if (EnetWriter == 0)	/* Not really sure this is right, but it */
	    EnetWriter = 1;	/*   seems more right than nothing at all*/
	Unlockq( &IkpOutq );
	WriteNextKPacket( pd );
      }
    else
      {
        EnetWriter = 0; /* No transmission in progress now */
        Unlockq( &IkpOutq );
	NetAlarm = ENET_READER_TIMEOUT;
      }
    EnetNetChecks++;
  }

SystemCode AddLogicalHostGroup(lhg)
register GroupId lhg;
  {
    register Enet10Address *ptr, *endptr;
    Enet10Address addr;

    if (lhg & LOCAL_GROUP_BIT) return( OK );
    
    if (LogicalHostCount == MAX_MULTICAST_ADDRESSES)
      {
	printx("All %d multicast addresses are allocated\n", LogicalHostCount);
      }

    if (LogicalHostCount == MAX_MULTICAST_ADDRESSES) return( NO_GROUP_DESC );
    addr = MulticastDefault;
    addr.addrlow = lhg >> 17;
#ifdef undef /* allow duplicates until higher level software if smarter */
    ptr = &cbsp->MCBlock.addresses[0];
    endptr = &cbsp->MCBlock.addresses[LogicalHostCount];
    while (ptr < endptr)
      {
	if ( *ptr == addr) return( OK );
	ptr++;
      }
#endif
    cbsp->MCBlock.addresses[LogicalHostCount++] = addr;
    MulticastReset();
    
    return( OK );
  }

SystemCode DeleteLogicalHostGroup(lhg)
register GroupId lhg;
  {
    register Enet10Address *ptr, *endptr;
    Enet10Address addr;
    
    if (lhg & LOCAL_GROUP_BIT) return( OK );

    addr = MulticastDefault;
    addr.addrlow = lhg >> 17;

    ptr = &cbsp->MCBlock.addresses[0];
    endptr = &cbsp->MCBlock.addresses[LogicalHostCount];
    while (ptr < endptr)
      {
	if ( ptr->addrlow == addr.addrlow  &&
	     ptr->addrmid == addr.addrmid &&
	     ptr->addrhigh == addr.addrhigh )
	  {
	    while (ptr < endptr)
	      {
	        *ptr = *(ptr+1);
		ptr++;
	      }
	    LogicalHostCount--;
	    MulticastReset();
	    
	    return( OK );
	  }
	ptr++;
      }
    
    return( NOT_FOUND );
  }

MulticastReset()
  {
    register struct SystemControlBlock *c = &cbsp->scb;
    register ControlBlockSegment *cbsr = cbsp;

    c->command = CUC_START;
    c->CBLoffset = Offset(&(cbsr->MCBlock));
    SwabInPlace(c, CONTROL_SWAP_LEN);

    cbsr->MCBlock.command = MC_SETUP + COM_EL;
    cbsr->MCBlock.link = 0xffff;
    cbsr->MCBlock.count = LogicalHostCount * sizeof(Enet10Address);
    SwabInPlace(&(cbsr->MCBlock), 4 * sizeof(unsigned short));/* header only */

    Attention();
  }
