@Part( style, root "manual" )
@Appendix(C Programming Style)@Label(style)@index(style)

There has been an effort to use a consistent style in V for writing
C programs.
The style and the uniformity it encourages are motivated by the
desire for readability and maintainability of software.
Although style is to a large extent a matter of individual taste,
the following describes some general practices with which most of us agree.

@appendixsection(General Format)

Recognizing that software is written to be read by other programmers
and only incidentally by compilers,
the general format follows principles established in formatting
general English documents.
Take a few more seconds to make things more readable; it is time well spent.

First, software is written to be printed on standard size (8 by 11)
paper.
This means avoiding lines longer than about 80 columns.
In general, there is one statement or declaration per line.

As with other documents,
judicious use of white space with short lines and
blank lines is encouraged.@index(blank lines)
In particular,
@begin(enumerate)
At least 2 blank lines between individual procedures.

Blank lines surround ``large'' comments.

Blank lines around any group of statements.

Blank lines around cases of a switch statement.
@end(enumerate)
@appendixsection(Names)@index(names)

Names are chosen when possible to indicate their semantics
and to read well in use, for example:
@begin(bigProgramExample)
if (GetDevice(EtherInstance) == NULL) return NOT@uf()FOUND;
@end(bigProgramExample)
Words should be spelled out, not shortened.
A good test is to read your code aloud.
You should be able to communicate it over a telephone easily, without
resorting to spelling out abbreviations.

In addition,
character case conventions are used to improve readability
and suggest the scope and type of the name.
Global variables, procedures, structs, unions, typedefs,
and macros all begin with a capital letter, and are logically
capitalized thereafter (e.g. @t(MainHashTable)).
A global variable is one defined
outside a procedure, even though it may not be exported from the file,
or an external variable.
The motivation for treating macros
in this way is that they may then be changed to procedure calls
without renaming.

Manifest constants
either follow the above convention (since they
are essentially macros with no parameters) or else are
fully capitalized with use of the underscore to separate
components of the name. E.g. WRITE@us()INSTANCE.

Local variables begin with a lower-case letter, but are 
either logically capitalized thereafter
(e.g. @t(bltWidth), @t(power), @t(maxSumOfSquares))
or else totally lower case.
Fields within structures or unions are treated in this manner also.

Local variables of limited scope are often declared as register,
if they are used very often inside inner loops.
It is not only more efficient, but usually more readable, to put
a pointer to an array of complicated structures (a common occurrence
in object-oriented programming) into a register variable
with a short name.
For example, 

@begin(BigProgramExample)
   register struct Descriptor *p = DescriptorTable+objectIndex;
   p->count = 0;
   Initialize(p->start);
   p->usage = p->default;
   p->length = p->end - p->start;
@end(BigProgramExample)

instead of the inefficient and cluttered:

@begin(BigProgramExample)
   DescriptorTable[objectIndex].count = 0;
   Initialize(DescriptorTable[objectIndex].start);
   DescriptorTable[objectIndex].usage = DescriptorTable[objectIndex].default;
   DescriptorTable[objectIndex].length = DescriptorTable[objectIndex].end 
	- DescriptorTable[objectIndex].start;
@end(BigProgramExample)

@appendixsection(Comments)

There are generally two types of comments: block-style comments, and
on-the-line comments or @i[remarks].
Multi-line, block-style comments
have the /* and */ appearing on lines by themselves, and the body
of the comment starting with a properly aligned *.
The comment should usually be surrounded by blank lines as well.
Thus it is easy to add/delete first and last lines, and it is
easier to detect the common error of omitting the */ and thus including all
code up to and including the next */ in a comment.

@begin(Bigprogramexample)
/*
 * this is the first line of a multi-line comment,
 * this is another line 
 * the last line of text
 */
@end(Bigprogramexample)
On-line comments or remarks are used to detail declarations, to explain
single lines of code, and for brief (i.e. one line) block-style
descriptive comments.

Procedures are preceded by block-style comments, explaining their (abstract)
function in terms of their parameters, results, and side effects.
Note that the parameter declarations @i(are) indented, not flushed left.
@begin(Bigprogramexample)
SystemCode EnetCheckRequest(req)
   register IoRequest *req;
  {
	/*
	 * Check that the read or write request has a legitimate buffer, etc.
	 */
    register unsigned count;
    register SystemCode r;
@hinge
        /*  Check length */
    count = req->bytecount;
    if (count <= IO@uf()MSG@uf()BUFFER) return OK;
@hinge
    req->bytecount = 0; 	/* To be left zero if a check fails */
    if (count > ENET@uf()MAX@uf()PACKET)
      {
	r = BAD@uf()BYTE@uf()COUNT;
      }
    else  
      {   
          /*
           * Make sure data pointer is valid.
	   *  Check that on a word boundary and not in the kernel area.
	   */
@hinge
        if ((!CheckUserPointer(req->bufferPointer)) ||
            (Active->team->teamSpace.size < (req->bufferPointer + count)) ||
            ((int) req->bufferPointer) & 1)
	  {
            r = BAD@uf()BUFFER;
	  }
        else
          {
	    req->bytecount = count;
	    r = OK;
	  }
      }
    return r;
  }
@end(Bigprogramexample)

@appendixsection(Indenting)
The above example shows many of the indenting rules. 
Braces ( ``{'' and ``}'' ) appear alone on a line, and
are indented two spaces from the statement they are to contain.
The body is indented two more spaces from the braces
(for a total of four spaces). 
@t(else)'s and @t(else if)'s line up
with their dominating @t(if) statement (to avoid marching off to the
right, and to reflect the semantics of the statement).
@begin(Bigprogramexample)
if ((x = y) == 0)
  {
    flag = 1;
    printf(" the value was zero ");
  }
else if (y == 1)
  {
    switch (today)
      {
        case Thursday:
            flag = 2;
            ThursdayAction();
            break;
@hinge
        case Friday:
            flag = 3;
            FridayAction();
            break;
@hinge
        default:
            OtherDayAction();
      }
  }
else
    printf(" y had the wrong value ");
@end(Bigprogramexample)

@appendixsection(File Contents)
File contents are arranged as follows.
@begin(enumerate)
Initial descriptive comment (see example below), containing a
brief descriptive abstract of the contents.
Some programmers also add
a list of all defined procedures in their defined order, or alphabetically.

Included files (avoid the use of absolute path names)

External definitions (imports and exports)

External and forward function declarations

Constant declarations

Macro definitions

Type definitions

global variable declarations
(use static declarations whenever possible, and group variables
with the functions that use them)

procedure and function definitions
@end(enumerate)

Here is the beginning of a file as an example.
@begin(Bigprogramexample)
/*
 * Distributed V Kernel - Copyright (c) 1982 by David Cheriton, Willy Zwaenepoel
 *
 * Kernel Ethernet driver
 */

#include "../../libc/include/Vethernet.h"
#include "interrupt.h"
#include "ethernet.h"
#include "ikc.h"
#include "../mi/dm.h"
@hinge
	/* Imports */
extern Process *Map@uf()pid();
extern SystemCode NotSupported();
extern DeviceInstance *GetDevice();
@hinge
	/* Exports */
extern SystemCode EnetCreate();
extern SystemCode EnetRead();
extern SystemCode EnetWrite();
extern SystemCode EnetQuery();
extern SystemCode EnetCheckRequest();
extern SystemCode EnetReadPacket();
extern SystemCode EnetPowerup();
@hinge
unsigned char	EnetHostNumber;		/* physical ethernet address */
InstanceId	EthernetInstance;	/* Instance id for Ethernet */
int		EnetReceiveMask;	/* addresses to listen for */
short		EnetStatus;		/* Current status settings */
int		EnetFIFOempty;		/* FIFO was emptied by last read */
int		EnetCollisions = 0;	/* Number of collision errors */
int		EnetOverflows = 0;	/* Queue overflow errors */
int		EnetCRCerrors = 0;	/* Packets with bad CRC's */
int		EnetSyncErrors = 0;	/* Receiver out of sync */
int		EnetTimeouts = 0;	/* Transmitter timeouts */
int		EnetValidPackets = 0;
char		kPacketArea[WORDS@uf()PER@uf()PACKET*BYTES@uf()PER@uf()WORD+20];
					/* Save area for kernel packets */
kPacket		*kPacketSave = (kPacket *) kPacketArea;
					/* Pointer to kernel packet area */
@hinge

	/* Macro expansion to interrupt-invoked C call to Ethernetinterrupt */
CallHandler(EnetInterrupt)
@end(Bigprogramexample)

@appendixsection[Parentheses]

For function calls,
the parentheses ``belong to'' the call, so there is no space between function
name and open parentheses. (There may be some inside the parentheses to make the
argument list look nice.)
When parentheses enclose the expression for a statement (@t(if), @t(for),
etc.),
the parentheses may be treated as belonging to the 
expression, so there is a space between
the keyword and the parenthesized expression.
This also clearly distinguishes the statement from a function call.
@begin(Bigprogramexample)
if (FuncA())
  {
    FuncB((a = b) == 0);
    return Nil;
  }
else
  {
    FuncC(a, b, c);
    return ToSender;
  }
@end(Bigprogramexample)
Alternatively,
parentheses may be treated
as belonging to the 
statement
(since they are syntactically required by the statement)
so there is no space between the keyword and the expression.
@begin(BigProgramExample)
if( (bytes = req->bytecount) <= IO@uf()MSG@uf()BUFFER )
    buffer = (char *) req->shortbuffer;
else
    return req->bufferPointer;
@end(BigProgramExample)
Note that parentheses are not syntactically required around the expression of 
a @t[return] statement.  Nevertheless, such parentheses may still be included,
if so desired.

Note that spaces are used to separate operators from operands
for clarity and may be selectively
omitted to suggest precedence in evaluation.

@appendixsection(Messages)

Although V is a message-based system,
most services are available by calling standard routines, so
programming at the ``message level'' is rarely necessary or desirable.
However, the programming of new servers and the non-standard use
of services or the use of messages within a program require message-level
programming.
The following conventions have been followed in V.

Space to send or receive a message is declared of type @b[Message]
(an array) or @b[MsgStruct] (a structure with appropriate fields),
as defined in <Venviron.h>.
Standard message formats, as defined in the V header files,
declare each message format to be a new data type.
Each message format contains enough padding to fill it out
to the fixed message size used by the kernel.
Where the same space is used for messages of multiple formats
(for example, both request and reply messages),
access to the space for the message can be made by casting a pointer
to the space to be of the type of the message format requires.
The following illustrates this style.
@begin(BigProgramExample)
Read(fad, buffer, bytes)
  File *fad;
  char *buffer; 
  int bytes;
   /*
    * Read the specified number of bytes into the buffer from the
    * file instance specified by fad. The number of bytes read is
    * returned.
    */
  {
    Message  msg;
    register IoRequest  *request = (IoRequest *) msg;
    register IoReply    *reply = (IoReply *) msg;
    register unsigned r, count;
    register char *buf;
@hinge
    for(;;)
      {
        request->requestcode = READ@uf()INSTANCE;
        request->fileid = fad->fileid;
        request->bufferPointer = buffer;
        request->bytecount = bytes;
        request->blocknumber = fad->block;
@hinge
        if (Send(request, fad->fileserver) == 0)
          {
            fad->lastexception = NONEXISTENT@uf()PROCESS;
            return 0;
          }
        if ((r = reply->replycode) != RETRY) break;
      }    
@hinge
    fad->lastexception = r;
    count = reply->bytecount;
@hinge
    if (count <= IO@uf()MSG@uf()BUFFER)
      {
        buf = (char *) request->shortbuffer;
        for (r = 0; r < count; ++r) *buffer++ = *buf++;
      }
    return count;
  }
@end(BigProgramExample)
