@Part( lnaming, root "manual" )
@Chapter(Naming)@Label(naming)

The @i[naming] section of the library includes a number of functions
that provide a convenient interface to V-System naming protocol messages,
plus other naming-related services.
See chapter @ref(NamingProtocol) for an explanation
of the naming protocol.

@section(Current Context)
Each process has a @i[current context] in which object names that do not
begin with the root escape character (`[') are interpreted,
similar to the @i[current working directory] of UNIX and other systems.
The following functions are provided to query or reset the current
context.

@function{SystemCode ChangeDirectory(name)
    char *name;}@index(ChangeDirectory)@index(Change Current Context)
Change the current context (working directory)
for the calling process to be the
context specified by @t[name], and return a standard system reply code
indicating OK if successful, else the reason for failure.
@t[name] is interpreted in the (previous) current context.

@function{int chdir(name)
    char *name;}@index(chdir)
This function is identical to @t[ChangeDirectory()],
except that it returns
0 to indicate success or -1 to indicate failure.
(This interface is provided for UNIX compatibility.)

@function{char *getwd(pathname)
  char pathname[];}@index(getwd)
Copies the absolute name of the current context (working directory)
into the given character array and returns its address.
(This interface is provided for UNIX compatibility.)

@section(Descriptor Manipulation)
V-System servers generally maintain a @i[descriptor] for each
of the objects they implement.
Each descriptor contains a @i[type] field, 
the associated object's
@i[name] (relative to a particular context), and additional
type dependent information such as @i[size], @i[timestamp], @i[owner], etc.
The standard header file <Vdirectory.h> defines the
descriptor types currently known to the system.

One can read (and in some cases modify) the descriptors
of all objects defined in a given naming context
by opening the associated @i[context directory] as a file.
A context directory appears as a file of descriptors.
A context directory can be opened using the standard system
@t[Open()] routine with the additional bit FDIRECTORY specified
as a part of the requested file mode.
Context directories are ordinarily opened in FBLOCK mode and
read using the standard @t[Read()] routine.

One caveat is necessary here: an attempt to open a multi-manager
context directory in this way will currently
fail with the error code MORE@us()REPLIES, since such context directories
are modelled as multiple files, one per manager.
See section @ref(NamingProtocol) for a description of the protocol
used to reliably open all partitions of a multi-manager context,
or the @t[listdir] program for a sample implementation.

One can also read or modify an individual object descriptor
using the following functions:

@function{SystemCode NReadDescriptor(name, desc)
    char *name;
    ArbitraryDescriptor *desc;}@index(NReadDescriptor)
Read an object's descriptor, specifying the object by name.

@function{SystemCode ReadDescriptor(serverpid, instanceid, desc)
    ProcessId serverpid;
    InstanceId instanceid;
    ArbitraryDescriptor *desc;}@index(ReadDescriptor)
Read the descriptor of the object from which the specified instance
was created.

@function{SystemCode NWriteDescriptor(name, desc)
    char *name;
    ArbitraryDescriptor *desc;}@index(NWriteDescriptor)
Write an object's descriptor, specifying the object by name.

@function{SystemCode WriteDescriptor(serverpid, instanceid, desc)
    ProcessId serverpid;
    InstanceId instanceid;
    ArbitraryDescriptor *desc;}@index(WriteDescriptor)
Write the descriptor of the object from which the specified instance
was created.

@section(Local Names or Aliases)

@function{SystemCode DefineLocalName(localname, truename)
    char *localname, *truename;}@index(DefineLocalName)
Defines a local alias ``[@t(localname)]''
for ``@t(truename)'', which must be the name of a context.
If @t[truename] does not begin
with a square bracket, it is first mapped in the current context to get
an absolute name before the alias is defined.
The alias is local to the team defining it, and is inherited by
teams it creates.

@function{SystemCode UndefineLocalName(name)
    char *name;}@index(UndefineLocalName)
Undefines a local alias.

@function{char *ResolveLocalName(name)
    char *name;}@index(ResolveLocalName)
Returns the stored absolute name for the given local alias.  The
returned string should not be modified, and will be freed if the name
is later redefined, so beware.

@function{ClearLocalNames()}@index(ClearLocalNames)
Undefines all local aliases for this team.
It may be useful to call @t[PrimeCache()] after
calling this routine, to reinsert definitions for
the system standard aliases.

@function{SystemCode DefineTempArea()}@index(DefineTempArea)
If the local name [tmp] is not already defined, this
function selects an appropriate place to store temporary files
and defines [tmp] to point to it.
The function returns OK if successful, else a standard system
code describing the problem.


@section(Naming Protocol Routines)

@function{ProcessId NameSend(req)
    NameRequest *req;}@index(NameSend)
Sends off the given request message, with the destination determined by
the name given in the message.
The given name must be a null-terminated string; @t[NameSend()]
neither examines nor sets @t[req->namelength].
Like @t[Send()], @t[NameSend()] returns the pid of the replier, 
and modifies its argument to hold the reply message.
@t[GetReply()] can be used if additional replies are anticipated.

@function{SystemCode GetAbsoluteName(namebuf, namelength, context)
    char namebuf[];
    unsigned namelength;
    ContextPair *context;}@index(GetAbsoluteName)
Accepts a null-terminated name in namebuf, possibly a relative name or
local alias, and modifies it to return the absolute name.  The size of
namebuf is passed in as namelength.  If the name specified an existing
context, its context identifier is returned, as with GetContextId(),
otherwise context->pid is set to 0.  The given name need not correspond
to any existing object, as long as it is unambiguous what server would
implement such an object if it did exist, and what its absolute name
would be.

@function{SystemCode GetFileName(namebuf, namelength, serverpid, instanceid)
    char namebuf[];
    unsigned namelength;    
    ProcessId serverpid;
    InstanceId instanceid;}@index(GetFileName)
Returns the absolute name for the specified file instance in @t[namebuf].
The maximum name length is passed in @t[namelength].  
@t[GetContextName()] returns OK if the mapping was successful, or
a standard system error code if a failure occurred.


@function{SystemCode GetContextId(name, context)
    char *name;
    ContextPair *context;}@index(GetContextId)
Interprets
the given name in the current context, and returns a
corresponding <process-id, context-id> pair, suitable for caching.
The function returns OK if successful, or a standard system error
code if an error is detected,
such as the given name specifying an object that is not a context.

Callers should recognize that the ContextPair may become
invalid at any time, usually due to the server that issued it crashing
and restarting with a different pid.

@function{SystemCode GetContextName(namebuf, namelength, context)
    char namebuf[];
    unsigned namelength;    
    ContextPair context;}@index(GetContextName)
The inverse of @t[GetContextId()].  Returns the absolute name for the given
context in @t[namebuf].  The maximum name length is passed in @t[namelength].
@t[GetContextName()] returns OK if the mapping was successful, or
a standard system error code if a failure occurred.


@function{int IgnoreRetry(req, pid, segbuf, segsize, serverpid)
  register MsgStruct *req;
  ProcessId pid, serverpid;
  register char *segbuf;
  register unsigned *segsize;}
This routine is intended for use only by servers that implement
the naming protocol, not for clients that use it.
It determines whether the caller
is one of the servers that should ignore the given
request @t[req]
(probably a CREATE@us()INSTANCE@us()RETRY), returning 1 (true) if so, 0 if not.
It assumes there is a 0-terminated list of pids beginning at @t[req->segPtr]
in the client's address space, and returns true if the given
@t[serverpid] is on the list.
If an appended segment was received, @t[segbuf] and @t[*segsize] should
indicate its location and size.  This routine may read in more
of the segment; if so, it alters the @t[segsize] parameter to
reflect what it read.  The routine assumes @t[segbuf] points to an area of at
least MAX@us()APPENDED@us()SEGMENT bytes.


@section(Direct Name Cache Manipulation)

The following routines are used internally by @t[NameSend()] and
the local alias manipulation functions.  They are not ordinarily
called directly in user programs.

@function{NameCacheEntry *NameCacheAdd(prefix, length, from, to, truename, flags)
    char *prefix, *truename;
    ContextPair from, to;
    unsigned short length, flags;}@index(NameCacheAdd)
Adds a new entry to the name prefix cache and return a pointer to the
new cache record.
The cache record format is defined in the 
standard header file <Vnamecache.h>.
If a cache entry already exists for the given prefix, it is deleted.
Returns NULL if no memory is available to allocate
a cache record.

The @t[prefix] argument gives the name prefix to be added, and
@t[length] is its length (not counting the terminating null byte, if any).
The prefix is interpreted relative to the @t[from] context and must
name the @t[to] context.

The @t[flags] may be any combination of the following bits:
@begin(itemize)
DONT@us()FLUSH@\The cache entry will never be flushed, even if the specified
context-pair becomes invalid.

ALIAS@\The cache entry specifies a local alias.  In this case, @t[prefix]
is the alias, while @t[truename] is the absolute name to which
the alias maps.

LOGICAL@us()PID@\The process-id portion of the specified @t[to] context is a
logical pid.  @t[NameSend()] will perform a @t[GetPid()] with scope
ANY@us()PID each time it attempts to use this cache entry.
@p[NOTE:] @i[This feature
is provided for backward compatibility with servers that implement 
the naming protocol of V version 5.1 and earlier.
It will be removed in a future V release.]
@end(itemize)

@function{NameCacheEntry *NameCacheLookup(name, context)
    char *name;
    ContextPair context;}@index(NameCacheLookup)
Checks whether any prefix of the given name matches a cache entry.
A pointer to the cache record containing the
longest matching prefix is returned.
If there is no match, NULL is returned.
A prefix match is defined as
all the characters in the prefix matching the corresponding
characters in the given name, plus the given name containing
a delimiter immediately following the match.


@function{SystemCode NameCacheDelete(cacheEntry)
    NameCacheEntry *cacheEntry;}@index(NameCacheDelete)
Deletes the specified name cache entry.
NOT@us()FOUND is returned if @t[cacheEntry] does not point to
a record currently contained in the cache; otherwise OK is returned.


@function{PrimeCache()}@index(PrimeCache)
Adds a standard set of well-known context names and aliases to
the name cache.
Normally called only once by the first team, but also useful
after a call to @t[ClearLocalNames()].

@section(Environment Variables)

The V-System implements character-string environment variables, much
like those in UNIX.
In V, a process may set variables in its own environment
as well as reading environment variables inherited from its creator.

By default, environment variables are global to a team.
The root process of a team begins with an environment variable list
inherited from its creator (through its team environment block).
A newly created process initially shares
the environment variable list of its creator.
A process may separate its environment variable
list from that of its parent by allocating a new list head of type
(EnvironmentVariable *), setting PerProcess->env to its address,
and assigning it a value_typically either NULL,
indicating an empty list, or the result of @t[copyenv(oldlist)] (see below).

@function{char *getenv(var)
    char *var;}@index(getenv)
Returns the value of the given environment variable, or NULL if it is
undefined.  The returned string should not be modified.

@function{setenv(var, value)
    char *var, *value;}@index(setenv)
Sets the given environment variable to the given value, or if value is
NULL, sets the variable to be undefined.

@function{EnvironmentVariable *copyenv(oldlist)
    EnvironmentVariable *oldlist;}@index(copyenv)
Makes a fresh copy of the environment variable chain beginning with
@t[oldlist], and returns a pointer to the first entry.  Useful if a process
wants to separate its environment from its parent's.
The @t[oldlist] argument should be the value (not the address) of the
parent's environment list head.

@function{clearenv()}@index(clearenv)
Removes the definitions of all environment variables.
