@Part( io, root "manual" )
@Chapter(Input and Output)@Label(io)
@tag(IOchapter)@index(I/O)

The input and output routines can be divided into three
categories:
@begin(enumerate)
Basic I/O routines like @t[getchar()] that are supported
but differ in their implementation from the standard Unix versions.

I/O support routines like @t[printf()]@index(printf)
that are identical with the
standard Unix version.

V-specific I/O routines like @t[Read()] and @t[Write()] that
are used in several cases to implement the standard C routines
in the V message-based world.
@end(enumerate)
@Section(Standard C I/O Routines)

The following standard C I/O routines are available:
@begin(verbatim)
clearerr()	closedir()	fclose()	feof()
ferror()	fflush()	fgetc()		fgets()
fopen()		fprintf()	fputc()		fputs()
fread()		freopen()	fscanf()	fseek()
ftell()		fwrite()	getc()		getchar()
gets()		getw()		mktemp()	opendir()
printf()	putc()		putchar()	puts()
putw()		readdir()	rewind()	rewinddir()
scanf()		sprintf()	seekdir()	setbuf()
sscanf()	telldir()	ungetc()
@end(verbatim)

However, @t[fopen()] returns a pointer value of type *File,
where File is defined in <Vio.h> and is a totally different record
structure from that used by, for instance, the Unix standard I/O.
Also, @t[setbuf()] is a no-op under V.

@Section(V I/O Conventions)

Program input and output are provided on files, which may include
disk files, pipes, mail-boxes, terminals, program memory, printers,
and other devices.

To operate on a file, it is first ``opened'' using @t[Open()]
if the file is specified by a pathname, otherwise by 
@t[OpenFile()] if the file is specified by a
server and instance identifier.
The mode is one of the following:@index(File Modes)@index(Modes)
@begin(description)
FREAD@\No write operations are allowed. File remains unchanged.@index(FRead)

FCREATE@\Any data previously associated with the described file 
is to be ignored and a new file is to be created.  Both
read and write operations may be allowed, depending on
the file type described below.@index(FCreate)

FAPPEND@\Data previously associated with the described file is to
remain unchanged.  Write operations are required only
to append data to the existing data.@INDEX(FAppend)

FMODIFY@\Existing data is to be modified and possibly appended to.
Both read and write operations are allowed.@index(FModify)
@end(description)

Both open functions return a pointer to an open file descriptor that
is used to specify the file for subsequent operations.
@t[Close()] removes access to the file.
@t[Seek()] provides random access to the byte positions in the file.  
Note:  the value returned from a byte position that has not been
written is not defined.

Each program is executed with standard input, output and error output
files, referred to as @t[stdin], @t[stdout],
and @t[stderr] respectively.

The file type indicates the operations that may be performed on
the open file as well as the semantics of these operations.
The file type is specified as some combination of the following
attributes.@index(Type)@index(File Types)
@begin(description)
READABLE@\The file can be read.@index(Readable)

WRITEABLE@\The file can be written.@index(Writeable)

APPEND@us()ONLY@*Only bytes after the last byte of the
data previously associated with the file can be written.@index(Append Only)

STREAM@\All reading or writing is strictly sequential.@index(Stream)
No seeking is allowed.
A file instance without the STREAM attribute
must store its associated data for non-sequential access.

FIXED@us()LENGTH@\The file instance is fixed in length.
Otherwise the file instance grows to accommodate the data
written, or the length of the file instance is not
known as in the case of terminal input.@index(Fixed Length)

@Multiple{VARIABLE@us()BLOCK@\Blocks shorter than the full block size may be
returned in response to read operations other
than due to end-of-file or other exception conditions.
For example, input frames from a
communication line may differ in length under normal
conditions.@index(Variable Block)

With a file instance that is VARIABLE@us()BLOCK,
WRITEABLE, and not STREAM, blocks that are
written with less than a full block size number
of bytes return exactly the amount written when read subsequently.}

MULTI@us()BLOCK@\Read and write operations are allowed that specify
a number of bytes larger than the block size.@index(Multi Block)

INTERACTIVE@\The open file is a text-oriented stream.
It also has the connotation of supplying
interactively (human) generated input.@index(Interactive)
@end(description)
Not all of the possible combinations of attributes yield a useful file
type.

Files may also be used in a block-oriented mode by specifying 
FBLOCK@us()MODE as part of the mode when opening the file.
No byte-oriented operations are allowed on a file opened
in block mode.

See chapter @ref(IOprotocol) for more details on the semantics
of the various possible file types and modes.

@section(V I/O Routines)
@subsection(Opening Files)
@function{File *Open(pathname, mode, error)
    char *pathname;  unsigned short mode;  SystemCode *error;}@index(Open)
Open the file specified by @t[pathname] with the
specified mode and return a file pointer for use with subsequent
file operations.  

@t[mode] must be one of FREAD, FCREATE, FAPPEND, or FMODIFY,
with FBLOCK@us()MODE if block mode is required.  If @t[Open()] fails
to open the file, it returns @c[NULL] and the location pointed to by
@t[error] contains a standard system reply code indicating the reason.
If an error occurs and @t[error] is @c[NULL], @t[Open()] calls @t[abort()].

@function{File *OpenDuplex(file, mode, error)
	File *file; unsigned short mode; SystemCode *error;}@index(OpenDuplex)
Open the ``other side'' of a duplex file, such as a network connection or
terminal.  @t[Mode] and @t[error] are as in @t[Open()].

@function{File *OpenFile(server, instanceidentifier, mode, error)
    ProcessId server;  InstanceId instanceidentifier;
    unsigned short mode;  SystemCode *error;}@index(OpenFile)
Open the file instance specified by the @t[server] and
@t[instanceidentifier] arguments and return a file pointer to be used with
subsequent file operations.

@t[mode] must be one of FREAD, FCREATE, FAPPEND,
or FMODIFY, with FBLOCK@us()MODE
if block mode is required.
If the instance is to be released when @t[Close()] is called on this
file pointer, FRELEASE@US()ON@US()CLOSE must also be specified as part of
the mode.
If @t[OpenFile()]
fails to open the file, it returns @c[NULL] and the location pointed
to by @t[error] contains a standard system reply code indicating the
reason.  If an error occurs and
@t[error] is @c[NULL], @t[OpenFile()] calls @t[abort()].

@function{File *@uf()Open(req, mode, server, error)
    CreateInstanceRequest *req;  unsigned short mode;
    ProcessId server;  SystemCode *error;}@index(@us()Open)
Open a file by sending the specified I/O protocol
request message @t[req] to the server specified by @t[server]
and return a file pointer to be used with
subsequent file operations.  This function is only used when
additional server-dependent information must be passed in the request
message, or the file is to be opened on a server that cannot
be specified by a character string pathname as in @t[Open()].

The request @t[req] may be either a
CreateInstanceRequest or a QueryInstanceRequest.
@t[mode] must be one of FREAD, FCREATE, FAPPEND,
or FMODIFY, with FBLOCK@us()MODE
if block mode is required.
If @t[@uf()Open()]
fails to open the file, it returns @c[NULL] and the location pointed
to by @t[error] contains a standard system reply code indicating the
reason.  If an error occurs and
@t[error] is @c[NULL], @t[@uf()Open()] calls @t[abort()].

@function{ProcessId CreateInstance(pathname, mode, req)
    char *pathname;  unsigned short mode;  CreateInstanceRequest *req;}
@index(CreateInstance)Open
the file specified by @t[pathname] in the given mode
using the specified CreateInstanceRequest, but do not set up
a File structure for it.  A CreateInstanceReply is returned at
the location pointed to by @t[req].  The function returns the
process id of the first process that replied.
If the create instance request was sent to a group, additional replies can
be obtained using @t[GetReply()].

@function{SystemCode CreateDuplexInstance(server, id, mode, req)
	ProcessId server; InstanceId id; unsigned short mode; 
	CreateDuplexInstanceRequest req;}@index(CreateDuplexInstance)
Open the "other side" of the file specified by the @t[server] and @t[id], but
do not set up a File structure for it.  A CreateDuplexInstanceReply is
returned at the location pointed to by @t[req].  Return a standard system
reply code.

@subsection(Closing Files)
@function{Close(file)
    File *file;}@index(Close)
Remove access to the specified file, and free 
the storage allocated for the File structure and associated buffers.
If the file is WRITEABLE and not in FBLOCK@us()MODE, the
output buffer is flushed.

@function{SpecialClose(file, releasemode)
    File *file;  unsigned releasemode;}@index(SpecialClose)
Close the specified file, as in @t[Close()].
If @t[SpecialClose()] releases the file instance associated with
the specified File structure, the release mode will be set to
@t[releasemode].  @t[Close()] sets the release mode to zero.
See chapter @ref(IOprotocol)
for a explanation of release modes.

@function{ReleaseInstance(fileserver, fileid, releasemode)
    ProcessId fileserver;  InstanceId fileid;  unsigned releasemode;}
@index(ReleaseInstance)Close the file instance specified by
@t[fileserver] and @t[fileid], using the specified release mode.
This function is used only when there is no File structure for the
given file.

@subsection(Byte Mode Operations)
The purpose of the byte-mode I/O library is to maintain an abstract view
of a file instance as an array of bytes with a known (but extensible)
length, and the ability to read, write, and (in the case of non-STREAMs) seek,
at the byte level.  A layer of buffering is imposed between the client and
server maintaining the actual file instance, to reduce the amount of
actual reading and writing done.  The actual file instance is guaranteed
to be identical with the local view when the file is first opened, and after
a Flush, barring I/O errors.  (Note that Close calls Flush before releasing
the instance.) At most one block of the local view of the file instance
may differ from the actual instance.

The I/O library can be used on any file type, though somewhat
confusing results may be obtained with VARIABLE_BLOCK, non-STREAM files,
particularly if one attempts to Seek in other than ABS_BLOCK mode.

The standard Unix functions mentioned above may be used on files opened
in byte mode (i.e., not opened in FBLOCK@us()MODE).  Several other
functions are also available on such files, as described below.

@function{int Seek(file, offset, origin)
    File *file;  int offset, origin;}@index(Seek)
Set the current byte position of the specified open
file to that specified by @t[offset] and @t[origin]
and return TRUE (nonzero) if successful.

If @t[origin] is ABS@us()BLK or ABS@us()BYTE, the byte position is set to
the @t[offset]-th
block or byte in the file starting from 0.  If @t[origin] is
REL@us()BYTE, @t[offset] specifies a signed offset relative to the
current byte position.  If origin is FILE@us()END, @t[offset] is the
signed byte offset from the end of file.

The end of file position is one beyond the
last byte written.  The value of bytes in the file previous to
the end of file that have not been explicitly written is undefined.

@t[Seek()] may not be used on files opened in block mode.
@t[SeekBlock()] should be used on such files. @t[Seek()] is identical
to @t[fseek()].

@function{unsigned BytePosition(file)
    File *file;}
Return the current byte position in the specified
file.  The value returned is correct only if the current byte 
position is less than MAX@us()UNSIGNED.  This function is identical
to @t[ftell()].

@function{Flush(file)
    File *file;}@index(Flush)
Flush any buffered data associated with the 
file, providing it is WRITEABLE.  Flushing a file causes local
buffered changes to the file data to be communicated to the
real file.  If the file is in block mode or not WRITEABLE,
no action is performed.  This function is identical to @t[fflush()].

@function{Resynch(file)
    File *file;}@index(Resynch)
Identical to @t[ClearEof()].

@function{SystemCode Eof(file)
    File *file;}@index(Eof)
Any of the byte mode read or write operations
may return EOF (Exception On File) as a special value
indicating an inability to read or write further in the file.
@t[Eof()] returns a standard system reply code indicating the nature of the
exception.  This may be a true end-of-file, i.e., the current byte
position exceeds the last byte position of the file, or some type
of error.

@function{ClearEof(file) 
    File *file;}@index(ClearEof)
Clear the local record of
the last exception on the given file, and resynchronize the
local view of the associated file instance with that of the 
server, including the size of the file and (for STREAMs) the next 
block to read.
This function only clears the local record of the exception; it does
not affect the circumstances that caused the exception to occur.
See @t[Eof()].

If the file is not of type STREAM, the contents of the local buffer are
discarded even if the buffer was modified and not yet rewritten.
If the file is of type STREAM, and the current file position violates
the stream condition (always read @t[nextblock] or write
@t[lastblock]+1), reposition.  The contents of the local buffer
are discarded only if repositioning is necessary.

@function{int BufferEmpty(file)
    File *file;}@index(BufferEmpty)
Test whether or not a file's local buffer is empty.  If this function
returns TRUE (nonzero), the next @t[getc()] will cause an actual read.
If it returns FALSE (zero), the next @t[getc()] will return immediately
with a byte from the buffer.

@subsection(Block Mode Operations)
The following functions are most useful on files opened in block mode.
Unless otherwise noted, they may also be used on files opened
in byte mode.

@function{unsigned Read(file, buffer, bytes)
    File *file;  char *buffer;  unsigned bytes;}@index(Read)
Read the specified number of bytes from the file
starting at the beginning of the current block location of the
file and store contiguously into the byte array starting at @t[buffer],
returning the actual number of bytes read.

The number of bytes returned may be less than the number requested
if (1) the file has the type attribute VARIABLE@us()BLOCK and
a short block was being read, (2) end of file was encountered while reading,
(3) an error occurred while reading (in this case 0 bytes are returned),
or (4) more than one block was requested and either the file does not
have the type attribute MULTI@us()BLOCK, or the server could not
return as many blocks as were requested.
If the read request cannot be satisfied,
the reason is indicated by the standard reply code
returned by @t[FileException()].
If the end of file is encountered while reading, a partial block
is returned with the reply code END@us()OF@us()FILE.
@t[Read()] is intended
for use on files opened in block mode only.
@b[Note:] @t[Read()] does @i[not] increment the current block number
stored in the File structure for the given file.

@function{unsigned Write(file, buffer, bytes)
    File *file;  char *buffer;  unsigned bytes;}@index(Write)
Write the specified number of contiguous bytes
from the buffer to the file starting at the beginning of the
current block location of the file, and return the actual
number of bytes written.

The number of bytes to be written must be less than or equal
to the block size (as returned by @t[BlockSize()]) unless the
file has the type attribute MULTI@us()BLOCK.  If the number
of bytes written is less than the number of bytes requested,
the reason is indicated by the standard reply code returned
by @t[FileException()].

@t[Write()] should be used only on files opened in block mode.
@b[Note:] @t[Write()] does @i[not] increment the current block number
stored in the File structure for the given file.

@function{unsigned BlksInFile(file)
    File *file;}@index(BlksInFile)
Return the number of blocks in the specified file.
If the number of blocks is unknown, MAXUNSIGNED is returned.

@function{unsigned BlockPosition(file)
    File *file;}@index(BlockPosition)
Return the current block position in the specified file.

@function{SeekBlock(file, offset, origin)
    File *file;  int offset;  int origin;}@index(SeekBlock)
Set the current block position of the specified open
file to that specified by @t[origin] and @t[offset].
The new block position is the block offset from the specified
block origin.  @t[origin] is one of FILE@us()BEGINNING, FILE@us()END
or FILE@us()CURRENT@us()POS.

@function{unsigned BlockSize(file) 
    File *file;}@index(BlockSize)
Return the block size in bytes of the specified
file.

@function{unsigned FileException(file)
    File *file;}@index(FileException)
Return the standard reply code indicating 
the last exception incurred on the specified file.
This is used primarily on files opened in FBLOCK@us()MODE.
@t[Eof()] is used on byte-oriented files.

@subsection(Server-Specific Operations)
This section describes routines in the I/O library which are
specific to particular servers.

@function{SystemCode CreatePipeInstance(readOwner, writeOwner, buffers, reply)
    ProcessId readOwner, writeOwner;  int buffers;
    CreateInstanceReply *reply;}@index(CreatePipeInstance)
Interact with the pipe server to create a pipe,
with the specified owners for the reading and writing ends of
the pipe, and the specified number of buffers.  @t[buffers] should
be between 2 and 10 inclusive.  The reply to the create instance request
is returned at the location pointed to by @t[reply]; it contains
the file instance id of the writeable end of the pipe.  The id of the
readable end is equal to this value plus 1.  @t[OpenFile()] may be
used to set up File structures for either or both ends of the pipe.
@t[CreatePipeInstance()] returns a
standard system reply code, which will be OK if the operation
was successful.    

@function{File *OpenTcp(localPort, foreignPort, foreignHost, active,
	precedence, security, error)
    unsigned short localPort, foreignPort;  unsigned long foreignHost;
    int active, precedence, security;  SystemCode *error;}@index(OpenTcp)
Interact with the Internet server to create a TCP network instance,
and return a pointer to a
File structure opened in byte mode
that can be used to send data on the 
corresponding TCP connection.  

To obtain a second File structure that
can be used to read from the connection, use the call
@begin(programexample)
f2 = OpenFile(FileServer(f1), FileId(f1) + 1, 
	 FREAD + FRELEASE@uf()ON@uf()CLOSE, &error)
@end(programexample)
where @t[f1] is the value returned by @t[OpenTcp()].
Note that it is necessary to release both the readable and writeable
instances to cause the connection to be deallocated.
Releasing the writeable instance closes the caller's end of the
connection.  Data can still be read from the readable instance until
it is released, or the other end closes (resulting in an END@us()OF@us()FILE
indication).

The parameters @t[localPort], @t[foreignPort], and @t[foreignHost]
specify the sockets on which the TCP connection is to be opened.
@t[active] specifies whether the connection should be active (i.e.,
send a connection ``syn'' packet), or passive (i.e., listen for an
incoming ``syn'' packet).  @t[precedence] and @t[security] specify
the precedence and security values to be used for the connection.
Specifying zero for these parameters will cause appropriate default
values to be used.

If the open is unsuccessful, @t[OpenTcp()] returns @c[null], and
a standard system reply code indicating the reason for failure
is returned in the location pointed to by @t[error]; else OK is returned
in this location.

@function{File *OpenIp(protocol, error)
    char protocol;  SystemCode *error;}@index(OpenIp)
Interact with the Internet server to create an IP network instance,
and return a pointer to a
File structure opened in block mode
that can be used to write IP packets to the network.

To obtain a second File structure that
can be used to read IP packets, use the call
@begin(programexample)
f2 = OpenFile(FileServer(f1), FileId(f1) + 1, 
	 FREAD + FBLOCK@uf()MODE + FRELEASE@uf()ON@uf()CLOSE, &error)
@end(programexample)
where @t[f1] is the value returned by @t[OpenIp()].
Note that it is necessary to release both the readable and writeable
instances even if only one of them is used.

The @t[protocol] specifies which value of the @i[protocol] field
in the IP packet headers is of interest.  The readable instance
will only return packets with the requested protocol value, and
the client program should only write packets with the specified protocol
field to the writeable instance, though this is not currently checked
by the server.  If @t[protocol] is zero, it specifies ``promiscuous''
mode, in which all IP packets are returned which are not of protocol types
that have been requested by another client, and packets of any
protocol type may be written.

If the open is unsuccessful, @t[OpenIp()] returns @c[null], and
a standard system reply code indicating the reason for failure
is returned in the location pointed to by @t[error]; else OK is returned
in this location.

@subsection(Miscellaneous I/O Functions)
@function{InstanceId FileId(file)
    File *file;}@index(FileId)
Return the file instance identifier associated
with the open file.  This was either generated as part of
@t[Open()] or specified as an argument to the @t[OpenFile()]
operation that opened the file.

@function{ProcessId FileServer(file)
    File *file;}@index(FileServer)
Return the file server identifier associated 
with the open file.  This was either generated as part of
@t[Open()] or specified as an argument to the @t[OpenFile()]
operation that opened the file.

@function{unsigned FileType(file)
    File *file;}@index(FileType)
Return the file type, which indicates 
the operations that may be performed on the open file as 
well as the semantics of these operations.  

@function{unsigned Interactive(file) 
    File *file;}@index(Interactive)
Return TRUE (nonzero) if the file has the type attribute
INTERACTIVE, else FALSE (zero).

@function{File *OpenStr(str, size, error)
    unsigned char *str;  unsigned size;  SystemCode *error;}@index(OpenStr)
Make the specified string look like a file.  The file is FIXED@us()LENGTH,
with one block of size @t[size], and the end of file set to the end of
this block.  @t[str] must point to an area at least @t[size] bytes in
length.  A file opened by @t[OpenStr()] is identified as such by its
file server (as returned by @t[FileServer()]) being equal to 0.

@function{SystemCode RemoveFile(pathname)
    char *pathname;}@index(RemoveFile)
Remove (delete) the file specified by @t[pathname].

@function{int unlink(pathname)
    char *pathname;}@index(unlink)
Remove (delete) the file specified by @t[pathname].
Returns 0 on success, -1 on failure.  This interface is provided for UNIX
compatibility.

@function{SystemCode SetBreakProcess(file, breakprocess)
    File *file;  ProcessId breakprocess;}@index(SetBreakProcess)
Sets the break process associated with the specified file (which
must be INTERACTIVE) to @t[breakprocess].  If a break occurs on the
file after a break process has been set,
the IO@us()BREAK reply will be returned to any outstanding read
requests, and the specified break process will be destroyed.

@function{SystemCode SetInstanceOwner(fileserver, fileid, owner)
    ProcessId fileserver, owner;  InstanceId fileid;}@index(SetInstanceOwner)
Set the owner of the specified file instance to be @t[owner].

@function{PrintFile(name, file)
    char *name;  File *file;}@index(PrintFile)
Print the value of each field in the given File structure on
the standard output, identifying
the file by the name @t[name].  Useful in debugging servers and I/O
routines.

@begin(comment)These don't exist at present:

@function{unsigned SetPrompt(file, promptstr)
    File *file; char *promptstr;}@index(SetPrompt)

Set the prompt string to be used when reading from
the specified file to be @t[promptstr].  Subsequently, the prompt
string is output before each line is read from the specified file.
The file must have the type attribute INTERACTIVE for a prompt to be
set.  A standard reply code is returned indicating OK if the prompt
was set, else the reason for failure.

@function{unsigned Stream(file)
    File *file;}@index(Stream)
Return TRUE (nonzero) if the file has the type attribute
STREAM else FALSE (zero).
@end(comment)

@section(Portable binary integer I/O)
The following routines can be used to write and read integers
to a file in a standard binary representation.
This representation stores integers in 1, 2, 3, or 4 bytes, signed or
unsigned. The integers are written as binary numbers, 8 bits to a byte,
in big-endian order. For the signed routines, twos-complement is used.
(This convention is followed by a number of systems, including
files read and written by TeX and METAFONT.)

The subroutines are declared as follows, for @i[N] = 1, 2, 3, and 4:

@function{PutSigned@i[N](i, f)
    long i; File *f;}@index(PutSigned)

@function{PutUnsigned@i[N](i, f)
    long i; File *f;}@index(PutUnsigned)

@function{long GetSigned@i[N](f)
    File *f;}@index(GetSigned)

@function{long GetUnsigned@i[N](f)
    File *f;}@index(GetUnsigned)
