@part(timeipc, root "manual")
@chapter(timeipc:  A V Performance Measurement Tool)
@label(timeipc)

@index(timeipc)
@index(timeipcserver)
@index(InterProcess Communication)
@index(Performance Measurement)

The @i(timeipc) program performs timing tests of the V interprocess
communication primitives
(@t{Send, Receive[WithSegment], Reply[WithSegment], MoveTo} and
@t{MoveFrom}).

To run the program, simply enter the command @t(timeipc).
For some tests, @i(timeipc) invokes a second program named @i(timeipcserver)
to serve as a target for IPC messages; @i(timeipcserver) need never be invoked
directly by users.

@section(Types of Tests)

@i(Timeipc) allows you to conduct a number of @i(tests).
A test consists of a number of @i(trials).
A trial consists of a number of @i(message transactions).

Each test measures one of the following types of message transaction:
@begin(display)
@t(Send-Receive-Reply            ) without segments
@t(Send-Receive-ReplyWithSegment ) with short segments
@t(Send-ReceiveWithSegment-Reply ) with short segments
@t(Send-Receive-MoveTo-Reply     ) with long segments
@t(Send-Receive-MoveFrom-Reply   ) with long segments
@end(display)
Short segments are up to MAX@us()APPENDED@us()SEGMENT bytes long
(1024 in the current version of the kernel).
Long segments are longer than MAX@us()APPENDED@us()SEGMENT bytes.

For each trial, the Sender process executes the following code:
@begin(bigprogramexample)
msgCounter = msgsPerTrial;
<record start time>
do
  {
    msg->timingCode = typeOfTest;
    Send( msg, receiverPid );
    if( msg->timingCode != OK ) <abort trial>
  }
while( msgCounter-- );
<record stop time>

@end(bigprogramexample)

The Receiver process executes different code depending on the type of
message transaction being tested.
For @t(Send-Receive-Reply) tests, the Receiver executes:
@begin(bigprogramexample)
msgCounter = msgsPerTrial;
do
  {
    senderPid = Receive( msg );
    if( msg->timingCode != typeOfTest ) <abort trial>
    msg->timingCode = OK;
    Reply( msg, senderPid );
  }
while( msgCounter-- );

@end(bigprogramexample)
@newpage

For @t(Send-Receive-ReplyWithSegment) tests, the Receiver executes:
@begin(bigprogramexample)
msgCounter = msgsPerTrial;
do
  {
    senderPid = Receive( msg );
    if( msg->timingCode != typeOfTest ) <abort trial>
    msg->timingCode = OK;
    ReplyWithSegment( msg, senderPid, localSegPtr, msg->segPtr, msg->segSize );
    /* NOTE: lost reply segments are not detected! */
  }
while( msgCounter-- );

@end(bigprogramexample)

For @t(Send-ReceiveWithSegment-Reply) tests, the Receiver executes:
@begin(bigprogramexample)
msgCounter = msgsPerTrial;
do
  {
    senderPid = ReceiveWithSegment( msg, localSegPtr, &localSegSize );
    if( msg->timingCode != typeOfTest ||
        msg->segSize    != localSegSize ) <abort trial>
    msg->timingCode = OK;
    Reply( msg, senderPid );
  }
while( msgCounter-- );

@end(bigprogramexample)

For @t(Send-Receive-MoveTo-Reply) tests, the Receiver executes:
@begin(bigprogramexample)
msgCounter = msgsPerTrial;
do
  {
    senderPid = Receive( msg );
    if( msg->timingCode != typeOfTest ) <abort trial>
    MoveTo( senderPid, msg->segPtr, localSegPtr, msg->segSize );
    msg->timingCode = OK;
    Reply( msg, senderPid );
  }
while( msgCounter-- );

@end(bigprogramexample)

For @t(Send-Receive-MoveFrom-Reply) test, the Receiver executes:
@begin(bigprogramexample)
msgCounter = msgsPerTrial;
do
  {
    senderPid = Receive( msg );
    if( msg->timingCode != typeOfTest ) <abort trial>
    MoveFrom( senderPid, localSegPtr, msg->segPtr, msg->segSize );
    msg->timingCode = OK;
    Reply( msg, senderPid );
  }
while( msgCounter-- );
@end(bigprogramexample)
@newpage

@section(Process Configurations)

Each type of test can be performed in any of the following three process
configurations:

(1) IPC between two processes on the same team.
@begin(programexample)

    timeipc team

   ---->Root-----
  |     (4)      |        ====> indicates Sending of test messages
  |              |        ----> indicates Sending of control messages
  |              v        (n)   indicates process priority
Sender======>Receiver 
 (1)         (0/1/2)

@end(programexample)

(2) IPC between two processes on different teams on the same host.
@begin(programexample)

    timeipc team       :       timeipcserver team
                       :
   ---->Root--------------------------------
  |     (4)            :                    |
  |                    :                    |
  |                    :                    v
Sender=================================>Receiver
 (1)                   :                (0/1/2)

@end(programexample)

(3) IPC between two processes on different teams on different hosts.
@begin(programexample)

    timeipc team       :       timeipcserver team
                       :
   ---->Root--------------------------------
  |     (4)            :                    |
  |                    :                    |
  |                    :                    v
Sender=================================>Receiver
 (1)                   :                (0/1/2)
                       :
Looper                 :                 Looper
(254)                  :                 (254)

@end(programexample)

Both @i(timeipc) and @i(timeipcserver) execute at REAL@us()TIME1 team priority,
giving their processes precedence over all other processes except the
Kernel Process (or other teams executing at REAL@us()TIME1).
For tests within a single host,
the Sender and Receiver consume all processor cycles outside of the kernel,
for the duration of a trial.  For tests between hosts, each testing team
runs an additional Looper process which loops forever, consuming and measuring
all cycles not used by the Sender or Receiver.  Thus, during a trial, the
testing workstation(s) will appear to freeze -- the cursor will not track
mouse movement and keyboard input will be queued in the kernel.  It is also
possible that concurrently-running, time-dependent applications may suffer
(e.g. TCP connections via the internet server may time out during a trial).
@newpage

@section(Input to @i(timeipc))

Before each test, you must answer a series of prompts to specify the
type of test and the process configuration for the test.
For each prompt, you may enter a null reply to request the
default value specified in brackets, or enter @t(^z) to terminate the program.
@t(^c) is ignored during prompting.
@begin(example)

receiver host name, 'local', 'sameteam', or 'quit'? [sameteam]
@end(example)
Reply with the name of a workstation on which to execute the Receiver team,
or enter one of the three keywords.  (The Sender team always executes on the
workstation where the @i(timeipc) program was executed.)
@t(local) requests that
the test be performed between two teams on the local workstation.
@t(sameteam) requests that only one team be used.
@t(quit) terminates the @i(timeipc) program.
Any of the keywords may be abbreviated to a single letter.
@begin(example)

receiver at higher, same, or lower priority than sender? [lower]
@end(example)
This prompt occurs if the test is to be performed within a single host.
Reply @t(higher) or @t(h) to run the Receiver at process priority 0,
@t(same) or @t(s) for priority 1, or @t(lower) or @t(l) for priority 2.
@begin(example)

segment size in bytes, K bytes, or M bytes? [0]
@end(example)
Specify the size of segment to be moved in the test message transaction.
Don't leave any space between the number and the optional @t(K) or @t(M)
suffix.  A size of zero requests a simple @t(Send-Receive-Reply) test.
@begin(example)

read or write? [read]
@end(example)
This prompt occurs if a non-zero segment size was requested.
@t(read) or @t(r) requests a @t(Send-Receive-ReplyWithSegment) test
(if segment size <= MAX@us()APPENDED@us()SEGMENT)
or a @t(Send-Receive-MoveTo-Reply) test
(if segment size > MAX@us()APPENDED@us()SEGMENT).
@t(write) or @t(w) requests a @t(Send-ReceiveWithSegment-Reply) test
(if segment size <= MAX@us()APPENDED@us()SEGMENT)
or a @t(Send-Receive-MoveFrom-Reply) test
(if segment size > MAX@us()APPENDED@us()SEGMENT).
@begin(example)

number of messages per trial? [10000]
@end(example)
Enter the number of times the message transaction is to be repeated within
a single trial.  Note that @t(MoveTo)/@t(MoveFrom) operations are @i(not)
counted as separate transactions.
@begin(example)

number of trials? [10]
@end(example)
Enter the number of trials to be performed.  Enter zero to start the prompting
all over again.
@newpage

@section(Output from @i(timeipc))

The results of the tests are written to @t(stdout).
All prompts and error messages are written to @t(stderr).
Here is an example of test output:
@begin(bigprogramexample)

   Send-Receive-MoveTo-Reply test with 4096 byte segments
   between sender host 'nanaimo' and receiver host 'lubbock'
   500 messages per trial, 5 trials   Wed Nov  6 16:50:15 1985

                             sender   receiver   sender   receiver
 trial   elapsed  elapsed   overhead  overhead  idle CPU  idle CPU  seg rate
 number  seconds  usec/msg  usec/msg  usec/msg  usec/msg  usec/msg  bits/sec

    1     13.030     26060         5         5     15022     14859   1257406
    2     13.320     26640         5         5     15618     15253   1230030
    3     13.490     26980         5         5     15889     15764   1214529
    4     14.800     29600         5         5     18387     18255   1107027
    5     13.020     26040         5         5     14925     14686   1258372

 avg.     13.532     27064         5         5     15968     15763   1213473

@end(bigprogramexample)
Not all columns are printed for all tests, and the "avg." row is only
printed when there is more than one trial per test.  The meanings of the
various statistics are described here:
@begin(example)

elapsed seconds
@end(example)
is the total elapsed time taken for the specified number of message
transactions.  It is determined by calling the kernel's @t(GetTime) function
before and after the sequence of messages, and thus is only as accurate as
@t(GetTime).
Although it is printed to three decimal places, the current version
of the kernel only keeps time to hundredths of a second, so the low-order
digit should always be zero.
@begin(example)

elapsed usec/msg
@end(example)
is the elapsed seconds divided by the number of message transactions,
printed in microseconds.
@begin(example)

sender   overhead usec/msg
receiver overhead usec/msg
@end(example)
are the number of microseconds of "overhead" instructions expended
for each message transaction, in the Sender process and the Receiver process.
They are computed by calling @t(GetTime) before and after 50,000 iterations
of the transaction loop with the IPC primitives removed, and dividing by
50,000.  These overhead values should be subtracted from the
@t(elapsed usec/msg) to obtain the bare message transaction time.
@begin(example)

sender   idle CPU usec/msg
receiver idle CPU usec/msg
@end(example)
are only printed for tests between two hosts, and indicate for each
host how many microseconds of the @t(elapsed usec/msg) are spent waiting for
the other host or the network.  They are measured by having a lower priority
process on each machine that loops continuously, incrementing a counter.
At the start of a trial, the counter is set to zero.  At the end of a trial,
the counter value is saved.  Then, the looper is allowed to run alone for 1
second to determine how many times per second it can increment the counter.
That rate is divided into the saved count to arrive at the printed value.
@begin(example)

seg rate bits/sec
@end(example)
is only printed for tests with a non-zero segment size.  It is the
number of bits of segment data moved during the trial divided by the
elapsed time of the trial.
@newpage

@section(Warnings and Precautions)

@begin(itemize)

Despite having exclusive access to process-level cycles on the testing
workstation(s), the progam can yield different results for repetitions of
the same test due to varying kernel overhead and network load.  For this
reason, a test should consist of more than one trial, in order to observe
the variance in the results.  A number of steps can be taken to minimize
these perturbations, depending on your need for accuracy:
@begin(itemize)
@begin(multiple)
Run the test at a time of low network load, even if you are only
testing within one workstation -- the kernel expends cycles receiving
and handling arriving network packets.  The @i(mon) program is very
helpful for discovering the current network load (but not while a
test is running!).

Alternatively, isolate your workstation(s) from the network during the
tests.  For tests within one workstation, you can unplug the transceiver
cable after you have started up the @i(timeipc) program.  For tests between
two workstations, you can connect them both to a DELNI which can be
isolated from the network at the flick of a switch.  (This is also
a polite thing to do to avoid loading the network with your test
messages.)  Note that you will not be able to redirect your test output
to a file if you have disconnected from the network.
@end(multiple)

Don't touch your keyboard or your mouse while a test is running --
they cause interrupts that the kernel must handle.

Kill off any extraneous process that are running on your workstation(s),
such as @i(mon), @i(telnet), @i(internetserver), etc.  I haven't figured out
why their presence effects the results of the tests, but it does.

Be sure that you are logged-in in order to prevent others from remotely
executing programs on your workstation(s).

Make sure no other REAL@us()TIME1 teams are running on your workstation(s),
such as other instances of @i(timeipc) or @i(timeipcserver).

@begin(multiple)
It is possible to run @i(timeipc) on top of a bare kernel, i.e. without
any other teams present.  Only tests within a single team can be
performed because the services of the team server are not available
to set up a separate receiver team.  An example boot command to load
@i(timeipc) onto a Sun2 workstation with 3Com Ethernet interface is:
@begin(example)
b V /usr/V/bin/timeipc.m68k Vkernel/sun2+ec
@end(example)
When running in this mode, answers to prompts must be entered using
LINEFEED rather than RETURN.  You may safely ignore warnings about
inability to set the team priority.

(Currently, a simple @t(Send-Receive-Reply) transaction on a Sun2 workstation
is 6 microseconds
faster when performed on top of a bare kernel, compared to running on top
of a freshly-booted standard first team and VGTS.  Surprisingly, using
the STS instead of the VGTS makes the simple message exchange @i(slower)
by 2 microseconds.)
@end(multiple)
@end(itemize)

Be aware that your workstation will appear to freeze up during a trial.
You may enter @t(^c) to abort the test and return to the first prompt, but
the interrupt will not take effect until the end of the current trial.
If you don't know how long a particular trial will take, try it first
with a small number of messages.  However, test results for trials running
less than a second or two should not be considered accurate.

For @t(Send-Receive-ReplyWithSegment) tests, lost reply segments are @i(not)
detected.  Be wary of inter-host results from such a test.

For @t(Send-ReceiveWithSegment-Reply) tests, a trial will be aborted if the
receiver does not receive the sent segment.  This occurs if the receiver
is not ready when the segment arrives, for example if the receiver is
running at lower priority than a sender on the same host.

Watch out for VGTS page mode -- you may think you are waiting for a trial
to finish when in fact the program is blocked trying to write to your
virtual terminal.

Be aware that inter-host tests consume considerable Ethernet bandwidth
(up to 3 megabits/second or more) and you are in danger of becoming
unpopular with other users of the network.

@i(Timeipc) does not currently measure the performance of @t(Forward) or
any group IPC operations.
@end(itemize)
