/* vgts.c - Virtual Graphics Terminal Server
 *
 * Marvin Theimer July 1982
 * Updated by Bill Nowicki August 1982
 * Split out other processes November 1982
 *
 * This file contains the routines which define the process structure
 * of the virtual graphics terminal service (vgts) and its applications
 * (in this case that is the Yale vlsi layout editor).  It also contains
 * all references to the Vkernel for the vgts; i.e. no other files 
 * contain references to things that are specific to the Vkernel proper.
 *
 * The process structure of the vgts consists of 4 processes: 
 *	the vgts process,
 * 	a mouse device helper process,
 *	a keyboard helper process.
 *	a timer helper process
 * The helper processes block on input from resp. the mouse
 * and the keyboard kernel calls and then send the input to the
 * vgts process.  The vgts process receives messages from both the
 * helper processes and from applications processes who desire 
 * virtual graphics terminal services.
 *
 * The vgts includes a window manager which is invoked by the user
 * by typing the window escape character.  Typing this character
 * causes the vgts process to call a window manager routine which
 * interpretes all further user input until he types the window
 * escape character again.
 *
 * Currently the vgts process operates in a mode where the user's
 * input is either directed to the window manager or is directed
 * to the (well-known) Yale application process.  Eventually this
 * scheme will have to be replaced with one which understands that
 * there are more than one applications processes running.
 * This will require policies and mechanisms for determining who
 * should currently be receiving input and also how applications
 * register themselves with the vgts so that it knows that they
 * are to receive input.
 */


#include <Vio.h>
#include <Vioprotocol.h>
#include "aledefs.h"

#define MaxInputClients 2	/* Max clients that the vgts can handle. */


struct InputClientRec
	/* Descriptor for an applications client.  Contains queues
	   (currently of length 1) for input characters and mouse
	   clicks and flags for indicating the state of a client's
	   requests. */
  {
    ProcessId pid;
		/* Process id of the applications client. */
    BOOLEAN requestFlag;
		/* Flag signalling an outstanding getInput request. */
    char c;	/* Input queue for characters. */
    int mouseX, mouseY, mouseButtons;	/* TRD mouseButton --> mouseButtons */
		/* Input queue for mouse clicks. */
    BOOLEAN charAvail;
		/* Flag signalling that a character is available. */
    BOOLEAN mouseAvail;
		/* Flag signalling that a mouse click is available. */
    short vgt;
		/* vgt number (pad) for input to go to */
  };


char BrainEscapeCharacter = 3;	/* control-c */


extern short MouseX, MouseY, MouseButtons;

ProcessId VgtsPid;		/* Well-known process id for the vgts process. */
ProcessId YalePid;		/* Well-known process id for the Yale  process. */
ProcessId ExitPid;		/* sending to this process exits the program */
ProcessId KbHelperPid, MousePid, TimerHelperPid;
		/* Process id's for the input helper processes. */

File *FileId;	/* File descriptor handle for the mouse device. */

/*
 * Process stack sizes.
 */
# define MainSize 9000
# define VgtsSize 4000
# define HelperSize 1000

static struct InputClientRec InputClient[MaxInputClients];
		/* Input handler state descriptors. */


/*******************************************************************/


/*
 * main:
 * Main routine for the vgts package.
 * This routine sets up the vgts process and any applications
 * processes; in this case, Yale.
 */
short Debug;

main()
  {
    ProcessId pid;
    int Yale(), Vgts();
    Message msg;

    InitExceptionServer();
    ExitPid = GetPid(0);
    VgtsPid = Create(9, Vgts, VgtsSize);
    YalePid = Create(10, Yale, MainSize);
    Reply(msg, VgtsPid);
    Reply(msg, YalePid);

    Receive(msg);
  }


/*******************************************************************/


/*
 * InitHelperProcesses:
 * Starts up keyboard and mouse helper processes for Vgts.
 */

InitHelperProcesses()
  {
    Message msg;
    ProcessId pid;
    int MouseProcess(), KeyboardProcess(), TimerProcess();

    MousePid = Create(8, MouseProcess, HelperSize);
    Reply(msg, MousePid);
    KbHelperPid = Create(8, KeyboardProcess, HelperSize );
    Reply(msg, KbHelperPid);
    TimerHelperPid = Create(8, TimerProcess, HelperSize);
    Reply(msg, TimerHelperPid);
  }


/*******************************************************************/


/*
 * InitVgts:
 * Initializes the Vgts.
 */

InitVgts()
  {
    int i;

    for (i = 0; i < MaxInputClients; i++)
      {
	InputClient[i].pid = 0;
	InputClient[i].requestFlag = FALSE;
	InputClient[i].charAvail = FALSE;
	InputClient[i].mouseAvail = FALSE;
      }
    InputClient[0].pid = YalePid;
    InputClient[0].vgt = 1;		/*** TEMPORARY ***/

  }


/*******************************************************************/


/*
 * Vgts:
 * Virtual graphics terminal service process.
 *
 * Note: Currently the reply to messages is scattered all over the
 * place.  This should be replaced with the proper, clean reply
 * structure as is used in most other Vkernel-based server processes.
 */

Vgts()
  {
    IoRequest msg;

    InitHelperProcesses();
    InitVgts();
	
    while (1) 
      GetMessage(0,0,&msg);
  }

GetMessage(mouseFlag, keyboardFlag, msg)
  int mouseFlag;		/* true if we return on mouse input */
  int keyboardFlag;		/* true if we return on keyboard input */
  register IoRequest *msg;	/* message that we return */
   {
    ProcessId pid;
    int index;
    char c;

    if (Debug) printf("Get Message called in process %x\n", GetPid(0) );
    while (1)
      {
	pid = Receive(msg);
	switch (msg->requestcode)
	  {
	    case GetInputRequest:
	        if (Debug) printf("Input request\n");
		index = LookUpClient(pid);
		if (InputClient[index].charAvail)
		  {
		    InputClient[index].charAvail = FALSE;
		    msg->requestcode = (int)CharInput;
		    msg->shortbuffer[0] = InputClient[index].c;
		    Reply(msg, pid);
		  }
		else
		  {
		    InputClient[index].requestFlag = TRUE;
		    if (InputClient[index].pid != pid)
		       printf( "BUG! pid=%d, InputClient[%d]=%d, Yale=%d\n",
		             pid, index, InputClient[index].pid, YalePid );
		  }
		break;

	    case Keyboard:
		c = msg->shortbuffer[0];
	        Reply(msg,pid);
		if (c == BrainEscapeCharacter)
		  {
		    manager();
		    break;
		  }
		if (keyboardFlag) return(pid);
		index = CurrentInputClient();
		InputClient[index].c = c;
		if (InputClient[index].requestFlag)
		  {
		    InputClient[index].requestFlag = FALSE;
		    msg->requestcode = (int)CharInput;
		    msg->shortbuffer[0] = (int) InputClient[index].c;
		    Reply(msg, InputClient[index].pid);
		  }
		else
		  {
		    InputClient[index].charAvail = TRUE;
		  }
		break;

	    case MouseHelperMsg:
		index = CurrentInputClient();
		if (mouseFlag) return(pid);
		{ 
		  register struct EventMsg *rep = (struct EventMsg *)msg;
		  InputClient[index].mouseX = rep->x;
		  InputClient[index].mouseY = rep->y;
		  InputClient[index].mouseButtons = rep->buttons;
		  Reply(rep, pid);
		  if (InputClient[index].requestFlag)
		    {
		      InputClient[index].requestFlag = FALSE;
		      rep->requestcode = (int)MouseInput;
		      rep->x = InputClient[index].mouseX;
		      rep->y = InputClient[index].mouseY;
		      rep->buttons = InputClient[index].mouseButtons;
		      Reply(rep, InputClient[index].pid);
		    }
		   else
		    {
		      InputClient[index].mouseAvail = TRUE;
		    }
		  }
		break;
		
	    case Timer:
	        if (InputClient[0].vgt) 
		  {
	            if (InputClient[0].requestFlag)
		        PadCursorBlink(InputClient[0].vgt);
		     else PadCursorOn(InputClient[0].vgt);
		  }
		Reply(msg, pid);
		break;

	    default:
		printf("ERROR in Vgts process.\n");
		break;
	  }
      }
  }


/*******************************************************************/



Select()
  {
  	/*
	 * select a Pad for input - not used in Yale
	 */
  }

/*******************************************************************/


/*
 * LookUpClient:
 * Returns index of client with specified pid.
 * NOTE: Currently only handles one client case.
 */

int LookUpClient(pid)
    ProcessId pid;
  {
    return(0);
  }


/*******************************************************************/


/*
 * CurrentInputClient:
 * Returns index of current input client.
 * Currently this is alway just Yale.
 * In the future it will
 * be the client whose window(s) contain the mouse or some such scheme.
 */

int CurrentInputClient()
  {
    return(0);
  }




/*
 * Yale:
 * This routine invokes the Yale vlsi layout editor program.
 */

Yale()
  {
    InitViewPackage();
    main1(1);
  }

