/********************************************************/
/*							*/
/*		YALE layout editor			*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*		THOMAS R. DAVIS				*/
/*							*/
/********************************************************/

/* FILE: parseexp.c */

/* parseexppost() parses an expression and installs a post-fix version of
 * it in expArray.  The value returned is the beginning location of
 * the expression.  If -1 is returned, then there was a parsing error.
 * parseexp() returns a complete expression, made of both the post-fix
 * form and the string pointer.
 */

/* 9/20/82: TRD -- changed expressions so that they now have the form:
 * E*** E*** ... E*** Eend <string> (zero terminated)
 * expressions will then be turned into a general string pointer, and
 * then such strings can be allocated and de-allocated in the usual way.
 */

#include "aledefs.h"
#define PUSH(X) opStack[opStackPtr++] = X
#define TOP opStack[opStackPtr - 1]
#define POP opStack[--opStackPtr]
#define ADDEXP(X) expArray[FreeExp++] = X
#define ADDADR(X) {ADDEXP(X/256); ADDEXP(X%256);} /* pushes 16 bits */
#define ERR(X) {ErrorPrint(X); return(ERROR);}
#define StackAdjust(Q) while (OpGreaterp(Q,TOP)) ADDEXP(POP);PUSH(Q);
/* token type definitions */

#define OPERAND  1 /* number or ident     */
#define BINOP    2 /* binary operand      */
#define UNOP     3 /* unary operand       */
#define OPENP    4 /* open parenthesis    */
#define CLOSEP   5 /* close parenthesis   */
#define STARTTOK 6 /* to initialize parse */
#define ASSIGNOK 7 /* leading valid ident */

short LastTokType;
BOOLEAN IsAssignable; /* set by ParseIdent */

/* The expression parser works in a somewhat non-standard way.  It
 * basically uses an operator-precedence stack, but it also makes
 * sure that each token is in the correct class to follow the
 * previous token.  The classes are listed above, and include such
 * things as OPERAND, BINOP, UNOP, OPENP, and CLOSEP.  All parsing
 * is done from the point of view of the symbol with number
 * symbolNumber.
 */

	/************************************************/
	/*		parseexp:			*/
	/************************************************/

EXPRESSION parseexp(symbolNumber)
short symbolNumber;
{
EXPRESSION exp, AddExpression();
char *ch;
char string[60];

if (ExpArrayLength - FreeExp < 20)
    {
    TtyBlinkError("Out of expression space");
/*    exp.string = 0; exp.postfix = 0;*/
    exp = NIL_EXPRESSION;
    return(exp);
    }
else if (ExpArrayLength - FreeExp < 100)
    TtyMessage("WARNING: Expression space low!");

LexSaveFlag = TRUE;
LexSavePtr = string;
if (PeekFlag)	/* ++ other stuff might be here later */
    {
    ch = LexString;
    while (*ch)
	*(LexSavePtr++) = *(ch++);
    }

FreeExp = 1;
parseexppost(symbolNumber);
*(--LexSavePtr) = 0;

/* Now add the string to the end of the expression */

  ch = string;
  while (*ch)
      ADDEXP(*ch++);	/* copy the string */
  ADDEXP(0);			/* and terminate with a zero */

LexSaveFlag = FALSE;
exp = AddExpression(&expArray[1]);
return(exp);
}

	/************************************************/
	/*		parseexppost:			*/
	/************************************************/

int parseexppost(symbolNumber)
short symbolNumber;
{
    char opStack[50];
    int opStackPtr;
    int start;
    BOOLEAN startFlag;
    TOKEN_TYPE lex();
    short PossibleToken(), ParseIdent();
    short OpGreaterp();
    short parenDepth;

    parenDepth = 0;
    LastTokType = STARTTOK;
    opStack[0] = Eend;
    opStackPtr = 1;
    start = FreeExp;		/* in case of errors */
    expArray[FreeExp] = Eend;	/* in case there is a null expression */
    if ((LexSource != NIL) && (LexSource != (char *) -1))
	{
	CurrentColumn = 0;
	PeekFlag = FALSE;
	}
    if (!PeekFlag)
	LexToken = L_PAREN;	/* to help lex with unary minus */
    while (TRUE)
      switch(lex())
	{
	case NUMBER:
	    if (!PossibleToken(LastTokType, OPERAND))
		goto parseError;
	    LastTokType = OPERAND;
	    if (LexNumber < 256)
		{
		ADDEXP(E8int);
		ADDEXP(LexNumber);
		}
	    else
		{
		ADDEXP(E16int);
		ADDADR(LexNumber);
		}
	    continue;
	case IDENTIFIER:
	    if (!PossibleToken(LastTokType, OPERAND))
		goto parseError;
	    startFlag = (LastTokType == STARTTOK);
	    if (ParseIdent(symbolNumber, FALSE) == ERROR)
		goto parseError;
	    if (startFlag && IsAssignable)
		LastTokType = ASSIGNOK;
	    else
		LastTokType = OPERAND;
	    continue;
	case L_PAREN:
	    if (!PossibleToken(LastTokType, OPENP))
		goto parseError;
	    LastTokType = OPENP;
	    parenDepth++;
	    PUSH(Elparen);
	    continue;
	case R_PAREN:
	    if (!PossibleToken(LastTokType, CLOSEP))
		goto parseError;
	    if (0 == parenDepth--)
		goto parseDone;
	    LastTokType = CLOSEP;
	    while (TRUE)
	      {
	      switch (TOP)
		{
		case Eend:
		    goto parseError;
		case Elparen:
		    --opStackPtr;
		    break;
		default:
		    ADDEXP(POP);
		    continue;
		}
	      break; /* this aborts the while(TRUE) loop */
	      }
	    continue;
/* all of the rest are just operators */
	case UNARY_MINUS:
	    if (!PossibleToken(LastTokType, UNOP))
		goto parseError;
	    LastTokType = UNOP;
	    StackAdjust(Euminus);
	    continue;
	case PLUS:
	    if (!PossibleToken(LastTokType, BINOP))
		goto parseError;
	    LastTokType = BINOP;
	    StackAdjust(Eplus);
	    continue;
	case MINUS:
	    if (!PossibleToken(LastTokType, BINOP))
		goto parseError;
	    LastTokType = BINOP;
	    StackAdjust(Eminus);
	    continue;
	case TIMES:
	    if (!PossibleToken(LastTokType, BINOP))
		goto parseError;
	    LastTokType = BINOP;
	    StackAdjust(Etimes);
	    continue;
	case QUOTIENT:  /* integer quotient and div are the same */
	case DIV:
	    if (!PossibleToken(LastTokType, BINOP))
		goto parseError;
	    LastTokType = BINOP;
	    StackAdjust(Ediv);
	    continue;
	case MOD:
	    if (!PossibleToken(LastTokType, BINOP))
		goto parseError;
	    LastTokType = BINOP;
	    StackAdjust(Emod);
	    continue;
	case LESS_OR_EQUAL:
	    if (!PossibleToken(LastTokType, BINOP))
		goto parseError;
	    LastTokType = BINOP;
	    StackAdjust(ElessEq);
	    continue;
	case GREATER_OR_EQUAL:
	    if (!PossibleToken(LastTokType, BINOP))
		goto parseError;
	    LastTokType = BINOP;
	    StackAdjust(EgreaterEq);
	    continue;
	case EQUAL:
	    if (!PossibleToken(LastTokType, BINOP))
		goto parseError;
	    LastTokType = BINOP;
	    StackAdjust(Eequal);
	    continue;
	case NOT_EQUAL:
	    if (!PossibleToken(LastTokType, BINOP))
		goto parseError;
	    LastTokType = BINOP;
	    StackAdjust(EnotEq);
	    continue;
	case AND:
	    if (!PossibleToken(LastTokType, BINOP))
		goto parseError;
	    LastTokType = BINOP;
	    StackAdjust(Eand);
	    continue;
	case OR:
	    if (!PossibleToken(LastTokType, BINOP))
		goto parseError;
	    LastTokType = BINOP;
	    StackAdjust(Eor);
	    continue;
	case NOT:
	    if (!PossibleToken(LastTokType, UNOP))
		goto parseError;
	    LastTokType = UNOP;
	    StackAdjust(Enot);
	    continue;
	case ASSIGNMENT:
	    if (LastTokType != ASSIGNOK)
		goto parseError;
	    LastTokType = BINOP;
	    if ((expArray[FreeExp-1] == Edotx) &&
		(expArray[FreeExp-3] != Esymbol))
		expArray[FreeExp-1] = EQdotx;
	    else if ((expArray[FreeExp-1] == Edoty) &&
			(expArray[FreeExp-3] != Esymbol))
		expArray[FreeExp-1] = EQdoty;
	    else if (expArray[start] == Esymbol)
		expArray[start] = EQsymbol;
	    StackAdjust(Eassign);
	    continue;
	default:
parseDone:    while (opStackPtr != 1)
		{
		if (TOP == Elparen)
		    goto parseError;
		ADDEXP(POP);
		}
	    ADDEXP(Eend);
	    UnLex();
	    return(start);
	}
parseError:	ErrorPrint(7);
		FreeExp = start; /* restore lost space */
		return(ERROR);
}

	/************************************************/
	/*		PossibleToken:			*/
	/************************************************/

short PossibleToken(fromtok, totok)
short fromtok, totok;
{
switch (fromtok)
    {
    case OPERAND:
    case ASSIGNOK:
    case CLOSEP:
	switch (totok)
	    {
	    case OPERAND:
	    case UNOP:
	    case OPENP:
		return(FALSE);
	    case CLOSEP:
	    case BINOP:
		return(TRUE);
	    }
    case BINOP:
    case STARTTOK:
    case OPENP:
	switch (totok)
	    {
	    case OPERAND:
	    case UNOP:
	    case OPENP:
		return(TRUE);
	    case CLOSEP:
	    case BINOP:
		return(FALSE);
	    }
    case UNOP:
	switch (totok)
	    {
	    case OPERAND:
	    case OPENP:
		return(TRUE);
	    case BINOP:
	    case UNOP:
	    case CLOSEP:
		return(FALSE);
	    }
    }
return(FALSE);	/* should never get here, but just to be safe ... */
}

/* OpGreaterp returns TRUE if op1 has greater-than or equal
 * precidence to op2.  Basically, if op1 is on the top of the
 * operator stack, and TRUE is returned, the top should be
 * popped and replaced.
 */

	/************************************************/
	/*		OpGreaterp:			*/
	/************************************************/

short OpGreaterp(op1, op2)
short op1, op2;
{
    short OpPrec();
    return(OpPrec(op1) >= OpPrec(op2));
}

	/************************************************/
	/*		OpPrec:				*/
	/************************************************/

short OpPrec(op)
short op;
{
switch (op)
    {
    case Elparen:
    case Eend:
	return(100);
    case Enot:
    case Euminus:
	return(1);
    case Etimes:
    case Ediv:
    case Emod:
	return(2);
    case Eplus:
    case Eminus:
	return(3);
    case EgreaterEq:
    case ElessEq:
	return(4);
    case EnotEq:
    case Eequal:
	return(5);
    case Eand:
	return(6);
    case Eor:
	return(7);
    case Eassign:
	return(8);
    default:
	ERR(8);
    }
}

/* The procedure ParseIdent() assumes that an ident has already been
 * encountered, so it does not need to call lex() for the first
 * identifier.  It returns TRUE if the parse was successful and
 * ERROR otherwise.  This routine parses more than an ident.  It
 * can be subscripted in various ways, as in foo[1,4].bar<7>.x .
 * If locFlag is TRUE, only locally defined symbols are allowed --
 * no foo.bar types.
 */

	/************************************************/
	/*		ParseIdent:			*/
	/************************************************/

short ParseIdent(symbolNumber, locFlag)
short symbolNumber, locFlag;
{
    TOKEN_TYPE token, lex(), LexPeek();
    S_ENTRY_PTR symbolTablePtr, LookupHash(), EnterHash();
    short found; /* boolean */
    int parseexppost();

    IsAssignable = TRUE; /* only instances are not */
    symbolTablePtr = LookupHash(LexString, NIL);
    found = FALSE;
    while (!found)
	{
	if (!symbolTablePtr)
	    ERR(10);
	if (symbolTablePtr->symbolId == symbolNumber ||
	    (symbolTablePtr->symbolId == 0 && !locFlag)) /* got it! */
		found = TRUE;
	else
	    symbolTablePtr =
		LookupHash(LexString, symbolTablePtr);
	}
/*
    if (locFlag && !(symbolTablePtr->data.isExport))
	ERR(67)	
*/

    /* if we reach this point in the code, symbolTablePtr points
     * to the local symbol table entry.
     */

    switch (token = LexPeek())
	{
	case PERIOD: /* it's either a .x, .y, or instance */
	    token = lex();
	    switch (token = lex())
		{
		case X:
		case Y:
		    if (symbolTablePtr->symbolType != SIGNAL_ENTRY)
			ERR(11);
		    ADDEXP(EQsymbol);
		    ADDADR(symbolTablePtr->frameDisplacement);
		    if (locFlag)
/*			ADDEXP(Elocal); */
			ADDEXP(Eplus);
   		    ADDEXP((token == Y) ? Edoty : Edotx);
		    return(TRUE);
		case IDENTIFIER:
		    if (locFlag)
			ERR(12);
		    if (symbolTablePtr->symbolType != INSTANCE0_ENTRY)
			ERR(13);
		    ADDEXP(EQsymbol);
		    ADDADR(symbolTablePtr->frameDisplacement);
		    ADDEXP(Ead0);
		    if (ParseIdent(symbolTablePtr->
		      data.dim0.cellInstPtr->instanceOf->sunInstanceNumber,
				TRUE) == ERROR)
			ERR(14);
		    IsAssignable = FALSE;
		    return(TRUE);
		default:
		    ERR(15);
		}
	case L_BRACKET:  /* a one-or two-dimensional instance */
	    token = lex(); /* read the bracket */
	    if (locFlag)
		ERR(12);
	    ADDEXP(EQsymbol);
	    ADDADR(symbolTablePtr->frameDisplacement);
	    if (parseexppost(symbolNumber) == ERROR)
		goto parseIdentError;
	    FreeExp--; /* clobber the fake Eend */
	    switch (LexToken)
		{
		case R_BRACKET: /* one-dim guy */
		    if (symbolTablePtr->symbolType != INSTANCE1_ENTRY)
			ERR(16);
		    ADDEXP(Ead1);
		    ADDEXP(symbolTablePtr->data.dim1.LB);
		    ADDEXP(symbolTablePtr->data.dim1.UB);
		    ADDEXP(symbolTablePtr->data.dim1.cellInstPtr->instanceOf->exportEnd);
		    if (lex() != PERIOD)
			ERR(17);
		    if (lex() != IDENTIFIER) /* ParseIdent needs it lexed */
			ERR(26);
		    if (ParseIdent(symbolTablePtr->
		      data.dim1.cellInstPtr->instanceOf->sunInstanceNumber,TRUE) == ERROR)
			ERR(18);
		    IsAssignable = FALSE;
		    return(TRUE);
		case COMMA: /* two-dim guy */
		    if (parseexppost(symbolNumber) == ERROR)
			goto parseIdentError;
		    FreeExp--;
		    if (LexToken != R_BRACKET || lex() != PERIOD ||
			symbolTablePtr->symbolType != INSTANCE2_ENTRY)
			    ERR(19);
		    ADDEXP(Ead2);
		    ADDEXP(symbolTablePtr->data.dim2.LB1);
		    ADDEXP(symbolTablePtr->data.dim2.UB1);
		    ADDEXP(symbolTablePtr->data.dim2.LB2);
		    ADDEXP(symbolTablePtr->data.dim2.UB2);
		    ADDEXP(symbolTablePtr->data.dim2.cellInstPtr->instanceOf->exportEnd);
		    if (lex() != IDENTIFIER) /* ParseIdent needs it lexed */
			ERR(26);
		    if (ParseIdent(symbolTablePtr->
		      data.dim2.cellInstPtr->instanceOf->sunInstanceNumber,TRUE) == ERROR)
			ERR(20);
		    IsAssignable = FALSE;
		    return(TRUE);
		default:  /* error */
		    ERR(21);
		}
	case LESS_THAN:  /* a signal vector */
	    token = lex();
	    if (symbolTablePtr->symbolType != SIGNAL_VECTOR_ENTRY)
		ERR(22);
	    ADDEXP(EQsymbol);
	    ADDADR(symbolTablePtr->frameDisplacement);
	    if (parseexppost(symbolNumber) == ERROR)
		goto parseIdentError;
	    FreeExp--;
	    if (LexToken != GREATER_THAN || lex() != PERIOD)
		ERR(23);
	    if (locFlag)
		ADDEXP(Elsigvec);
	    else
		ADDEXP(Esigvec);
	    ADDEXP(symbolTablePtr->data.sig.LB);
	    ADDEXP(symbolTablePtr->data.sig.UB);
	    switch (lex())
	        {
		case X: ADDEXP(Edotx); break;
		case Y: ADDEXP(Edoty); break;
		default:
		    ERR(24);
		}
	    return(TRUE);	    
	default: /* better be xvar, yvar, scalar, or constant */
	    switch (symbolTablePtr->symbolType)
		{
		case CONSTANT_ENTRY:
		case SCALAR_ENTRY:
		case XVAR_ENTRY:
		case YVAR_ENTRY:
		    if (!locFlag)
			ADDEXP(Esymbol);
		    else
			ADDEXP(EQsymbol);
		    ADDADR(symbolTablePtr->frameDisplacement);
		    if (locFlag)
			ADDEXP(Elocal);
		    return(TRUE);
		default: /* garbage */
		    ERR(25);
		}
	}
parseIdentError:
	ERR(9);
}
