/********************************************************/
/*							*/
/*		YALE layout editor			*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*		THOMAS R. DAVIS				*/
/*							*/
/********************************************************/

/* FILE: parse.c */

#include "aledefs.h"

#define ParamList 0
#define Declaration 1

/* This file contains all the parsing routines to parse a single */
/* SILT file.  The main call is to the routine ParseFile. */

	/************************************************/
	/*		ParseFile:			*/
	/************************************************/

ParseFile()
{
TOKEN_TYPE LexPeek();

    /* DEBUG */ TtyCRLF();
    if (LexPeek() == ENDOFFILE)
	return;
    ReadToken(PHYLE);
    ReadToken(IDENTIFIER);
    ReadToken(SEMICOLON);
    ParseSymbolHead(0);
    ParseSymbolDefs();
    ParseSymbolTail(0);
    ReadToken(PERIOD);
}

/* ParseSymbolHead does all of the Declarations but symbols */

	/************************************************/
	/*		ParseSymbolHead:		*/
	/************************************************/

ParseSymbolHead(symbolNumber)
short symbolNumber;
{
TOKEN_TYPE token, lex();
  while (TRUE)
    switch (token = lex())
	{
	case XVAR:
	case YVAR:
	case CONSTANT:
	case SCALAR:
	case SIGNAL:
	    ParseIdentList(symbolNumber, token, Declaration);
	    break;
	case DEFAULT:
	    ParseDefaultList(symbolNumber);
	    break;
	case EXPORT:
	    ParseExportLIst(symbolNumber);
	    break;
	case CONSTRAINT: /* stub +++ */
	    TtyBlinkError("Constraints not implemented yet");
	    SkipToSemicolon();
	    break;
	case SYMBOL:
	case BEGIN:	/* BEGIN and SYMBOL are normal returns */
	    return;
	default:
	    TtyBlinkError("Error in ParseSymbolHead");
	    return;
	}
}

/* ParseDefaultList below assumes that the key-word "default"
 * has already been read.  It will parse expression after expression
 * until another keyword is encountered and will stuff them on the
 * default chain of the OpenCellDefinition.
 */

	/************************************************/
	/*		ParseDefaultList:		*/
	/************************************************/

ParseDefaultList(symbolNumber)
short symbolNumber;
{
EXPRESSION_CONS_PTR eCons, tail;
TOKEN_TYPE LexPeek();
EXPRESSION parseexp();
CONS_PTR GetCons();

/* First throw away the default default list.  May want to garbage-collect
 * this stuff later. ++
 */
OpenCellDefinition->defaultList = NIL;

while (((int)LexPeek()) <= ((int)ALL))	/* Crock! all SILT reserved words */
    {					/* have value less than "ALL"	  */
    eCons = (EXPRESSION_CONS_PTR) GetCons();
    if (eCons == NIL)
	return;	/* out of free storage */
    eCons->value = parseexp(symbolNumber);
    eCons->next = NIL;
    if ((tail = OpenCellDefinition->defaultList) == NIL)
	OpenCellDefinition->defaultList = eCons;
    else
	{
	while (tail->next != NIL)
	    tail = tail->next;
	tail->next = eCons;
	}
    ReadToken(SEMICOLON);
    }
}

	/************************************************/
	/*		ParseExportLIst:		*/
	/************************************************/

ParseExportLIst(symbolNumber)
short symbolNumber;
{
S_ENTRY_PTR SEntry, LookupSTableEntry();
ENTRY_TYPE varType;
TOKEN_TYPE lex();

while (TRUE)
    {
    if (lex() != IDENTIFIER)
	{
	ErrorPrint(63);
	SkipToSemicolon();
	return;
	}
    if ((SEntry = LookupSTableEntry(LexString, symbolNumber)) == NIL)
	ErrorPrint(66);
    else
	switch (varType = SEntry->symbolType)
	    {
	    case XVAR_ENTRY:
	    case YVAR_ENTRY:
	    case SIGNAL_ENTRY:
	    case SCALAR_ENTRY:
		XAddExportVar(LexString, varType);
		break;
	    default:
		ErrorPrint(66);
	    };
    switch (lex())
	{
	case SEMICOLON:
	    return;
	case COMMA:
	    break;
	default:
	    ErrorPrint(63);
	    SkipToSemicolon();
	    return;
	}
    }	
}

/* ParseIdentList insists that there be a list of idents */
/* followed by commas with a semicolon at the end of the */
/* whole thing.  Symbol-table entries are made for each one */
/* and multiple definitions are checked for. */

	/************************************************/
	/*		ParseIdentList:			*/
	/************************************************/

ParseIdentList(symbolNumber, varType, identType)
short identType;
short symbolNumber;
TOKEN_TYPE varType;
{
TOKEN_TYPE token, lex();
S_ENTRY_PTR SPtr, LookupHash();

    while (IDENTIFIER == (token = lex()))
	{
	if (identType == Declaration)
	    {
	    SPtr = 0;
	    while (SPtr = LookupHash(LexString, SPtr))
		{
		if (SPtr->symbolId == symbolNumber)
		    ErrorPrint(60);
		}
	    }
	switch (varType)
	    {
	    case XVAR:
		if (identType == Declaration)
		    XAddLocalVar(LexString, XVAR_ENTRY);
		else
		    XInsRefPt(Vertical, 0, LexString);
		break;
	    case YVAR:
		if (identType == Declaration)
		    XAddLocalVar(LexString, YVAR_ENTRY);
		else
		    XInsRefPt(Horizontal, 0, LexString);
		break;
	    case SCALAR:
		XAddLocalVar(LexString, SCALAR_ENTRY);
		break;
	    case SIGNAL:
		/* stub +++ does not handle signal vectors! */
		XAddLocalVar(LexString, SIGNAL_ENTRY);
		break;
	    case CONSTANT:
		/* stub +++ syntax is wrong! */
		XAddLocalVar(LexString, CONSTANT_ENTRY);
		break;
	    }
	token = lex();
	if (token == SEMICOLON)
	    return;
	if (token == R_PAREN)
	    {
	    UnLex();
	    return;
	    }
	if (token != COMMA) ErrorPrint(61);
	}
    ErrorPrint(61);
    return;
}

/* ParseSymbolDefs returns when 'begin' is found */

	/************************************************/
	/*		ParseSymbolDefs:		*/
	/************************************************/

ParseSymbolDefs()
{
TOKEN_TYPE token, lex();

    token = LexToken;
    while (token != BEGIN)
	{
	/* DEBUG */ TtyPutString(".");
	if (token != SYMBOL)
	    {
	    ErrorPrint(87);
	    token = lex();
	    continue;
	    }
	if ((token=lex()) != IDENTIFIER)
	    {
	    ErrorPrint(88);
	    SkipToSemicolon();
	    }
	if (!XInsCellDef(LexString))
	    return;
	TouchString(LexString);
	ParseParamDecs(OpenCellDefinition->sunInstanceNumber);
	ParseSymbolHead(OpenCellDefinition->sunInstanceNumber);
	ParseSymbolTail(OpenCellDefinition->sunInstanceNumber);
	ReadToken(SEMICOLON);
	token = lex();
	OpenCellDefinition = NIL;
	}
}

	/************************************************/
	/*		ParseParamDecs:			*/
	/************************************************/

ParseParamDecs(sunInst)
int sunInst;
{
TOKEN_TYPE token, lex();

token = lex();
if (token != SEMICOLON && token != L_PAREN)
    {
    ErrorPrint(91);
    SkipToSemicolon();
    return;
    }
if (token == SEMICOLON)
    return;
while (TRUE)
  switch (token = lex())
    {
    case XVAR:
    case YVAR:
    case SCALAR:
	ParseIdentList(sunInst, token, ParamList);
	break;
    case R_PAREN:
	ReadToken(SEMICOLON);
	return;
    default:
	ErrorPrint(57);
	SkipToSemicolon();
	return;
    }
}

	/************************************************/
	/*		ParseSymbolTail:		*/
	/************************************************/

ParseSymbolTail(symbolNumber)
short symbolNumber;
{
TOKEN_TYPE lex();
char *ch, *ch1, nameString[TOKBUFSIZE];

    while (TRUE)
	{
	*nameString = 0;		/* no instance name so far */
	switch (lex())
	    {
	    case IDENTIFIER:		/* must be an instance name -- */
		ch = LexString;	/* process it and drop through */
		ch1 = nameString;
		while (*ch)
		    *(ch1++) = *(ch++);
		*ch1 = 0;
		ReadToken(DOUBLE_COLON);
		ReadToken(PLACE);
	    case PLACE:
		ParsePlace(symbolNumber, nameString);
		break;
	    case BOX:
		ParseBox(symbolNumber);
		break;
	    case BUTT:
		ParseButt(symbolNumber);
		break;
	    case CUT:
		ParseCut(symbolNumber);
		break;
	    case CONNECT: /* stub +++ */
	    case WITH: /* stub */
	    case FOR: /* stub */
	    case CIF_LIST: /* stub */
	    case BEGIN: /* stub */
	    case IF: /* stub */
	    case ROUTE: /* stub */
		SkipToSemicolon();
		break;
	    case END:
		return;
	    case SEMICOLON:
		break; /* skip null statements */
	    default:
		ErrorPrint(62);
		SkipToSemicolon();
		break;
	    } 
	/* at this point, new_cmd_chain_ptr should point to a new */
	/* command which must be linked on to the end of the old */
	/* chain. */
	}
/* remember to hack this so that final ';' is not needed if symb=0 */
}

	/************************************************/
	/*		ParsePlace:			*/
	/************************************************/

ParsePlace(sin, instNameString)
short sin;
char *instNameString;
{
TOKEN_TYPE lex();
TRANS_MATRIX trans;
S_ENTRY_PTR SEntry, LookupSTableEntry();
EXPRESSION_CONS_PTR paramList, eCons;
VARIABLE_CONS_PTR vPtr, vHead;
char *ch, *ch1, cellName[TOKBUFSIZE];

    switch (lex())
        {
	case BOX:
	    ParseBox(sin);
	    return;
	case BUTT:
	    ParseButt(sin);
	    return;
	case CUT:
	    ParseCut(sin);
	    return;
	case IDENTIFIER:
	    /* first, make sure it's a valid symbol name */
	    SEntry = LookupSTableEntry(LexString, 0);
	    if (SEntry == NIL || SEntry->symbolType != SYMBOL_ENTRY)
		{
		ErrorPrint(92);
		return;
		}

	    /* Now save the name for use in the XInsCellInstance routine. */
	    ch = LexString;
	    ch1 = cellName;
	    while (*ch)
		*(ch1++) = *(ch++);
	    *ch1 = 0;

	    /* OK, it's a symbol.  Build a parameter list. */
	    vHead = vPtr = SEntry->data.symbol.cellPtr->parameterList;
	    paramList = NIL;
	    while (vPtr != NIL)
		{
		eCons = (EXPRESSION_CONS_PTR) GetCons();
		if (eCons == NIL)
		    return;	/* out of free storage */
		eCons->value/*.postfix*/ = 0;
		eCons->next = paramList;
		paramList = eCons;
		vPtr = vPtr->next;
		}

	    /* Now we have a dummy parameter list -- fill it in. */
	    ParseActualParamList(paramList, vHead);
	    ParseTransformation(&trans, sin);

	    /* Now make the cell instance. */
	    if (OpenCellDefinition != NIL)
		XInsCellInstance(cellName,instNameString,paramList,&trans);
	    return;
	default:
	    SkipToSemicolon();
	    ErrorPrint(71);
	    return;
	}
}

	/************************************************/
	/*		ParseActualParamList:		*/
	/************************************************/

ParseActualParamList(paramList, vHead)
EXPRESSION_CONS_PTR paramList;
VARIABLE_CONS_PTR vHead;
{
TOKEN_TYPE lex(), LexPeek();
EXPRESSION parseexp();
EXPRESSION_CONS_PTR pPtr;
VARIABLE_CONS_PTR vPtr;
BOOLEAN found;

switch (lex())
    {
    case SEMICOLON:
	return;
    case L_PAREN:
	while (LexPeek() != R_PAREN)
	    {
	    if (lex() != IDENTIFIER)
		{
		ErrorPrint(93);
		SkipToSemicolon();
		return;
		}
	    /* search for the ident in the lists */
	    pPtr = paramList;
	    vPtr = vHead;
	    found = FALSE;
	    while (vPtr != NIL && !found)
		{
		if (strcmp(LexString,vPtr->value->stringPtr))
		    {
		    found = TRUE;
		    ReadToken(EQUAL);
		    pPtr->value =
			parseexp(OpenCellDefinition->sunInstanceNumber);
		    }
		else
		    {
		    vPtr = vPtr->next;
		    pPtr = pPtr->next;
		    }
		}
	    if (!found)
		{
		ErrorPrint(90);
		return;
		}
	    if (LexPeek() == COMMA)
		lex();
	    }
	lex();			/* read the R_PAREN */
	return;
    default:
	ErrorPrint(89);
	SkipToSemicolon();
	return;
    }
}

	/************************************************/
	/*		ParseBox:			*/
	/************************************************/

ParseBox(sin)
short sin;
{
EXPRESSION e1, e2, e3, e4;
PROCESS_LAYER layer;

ParsePoint(sin, &e1, &e2, &layer);
ReadToken(TO);
ParseCoordPair(sin, &e3, &e4);
ReadToken(SEMICOLON);

XInsRectangle(e4, e2, e1, e3, layer);
return;
}

	/************************************************/
	/*		ParsePoint:			*/
	/************************************************/

ParsePoint(sin, ep1, ep2, layerptr)
short sin;
EXPRESSION *ep1, *ep2;
PROCESS_LAYER *layerptr;
{
TOKEN_TYPE lex();
EXPRESSION parseexp();
PROCESS_LAYER ReadLayer();

    ReadToken(L_PAREN);
    *ep1 = parseexp(sin);
    ReadToken(COMMA);
    *ep2 = parseexp(sin);
    switch (lex())
	{
	case R_PAREN:
	    *layerptr = ZZ;
	    break;
	case COMMA:
	    *layerptr = ReadLayer();
	    ReadToken(R_PAREN);
	    break;
	default:
	    ErrorPrint(64);
	    break;
	}
    return;
}

	/************************************************/
	/*		ReadLayer:			*/
	/************************************************/

PROCESS_LAYER ReadLayer()
    {
    TOKEN_TYPE lex();
    switch(lex())
	{
	case BURIED: return(NB);
	case CONTACT: return(NC);
	case DIFF: return(ND);
	case DIFF_METAL: return(NDM);
	case DIFF_POLY: return(NDP);
	case GLASS: return(NG);
	case IMPLANT: return(NI);
	case METAL: return(NM);
	case METAL_POLY: return(NMP);
	case POLY: return(NP);
	case NONE: return(ZZ);
	}
    ErrorPrint(56);
    return(ZZ);
    }


	/************************************************/
	/*		ParseCoordPair:			*/
	/************************************************/

ParseCoordPair(sin, ep1, ep2)
short sin;
EXPRESSION *ep1, *ep2;
{
TOKEN_TYPE lex();
EXPRESSION parseexp();

    ReadToken(L_PAREN);
    *ep1 = parseexp(sin);
    ReadToken(COMMA);
    *ep2 = parseexp(sin);
    ReadToken(R_PAREN);
    return;
}

	/************************************************/
	/*		ParseButt:			*/
	/************************************************/

ParseButt(sin)
short sin;
{
TOKEN_TYPE lex(), LexPeek();
TRANS_MATRIX tMat;

    if (LexPeek() == L_PAREN)
	{
	lex();
	ReadToken(R_PAREN);
	}
    ParseTransformation(&tMat, sin);
    return;
}

	/************************************************/
	/*		ParseCut:			*/
	/************************************************/

ParseCut(sin)
short sin;
{
PROCESS_LAYER layer;
EXPRESSION e1, e2;

layer = DefaultLayerStack[LayerStackTop];
    switch (lex())
	{
	case L_PAREN:
	    switch (lex())
		{
		case DIFF:
		    layer = ND;
		    ReadToken(R_PAREN);
		    ReadToken(AT);
		    break;
		case POLY:
		    layer = NP;
		    ReadToken(R_PAREN);
		    ReadToken(AT);
		    break;
		case R_PAREN:
		    ReadToken(AT);
		    break;
		default:
		    ErrorPrint(72);
		    ReadToken(R_PAREN);
		    ReadToken(AT);
		    break;
		}
	case AT:
	    break;
	default:
	    ErrorPrint(72);
	}
    ParseCoordPair(sin, &e1, &e2);
    return;
}

	/************************************************/
	/*		SkipToSemicolon:		*/
	/************************************************/

SkipToSemicolon()
{
    TOKEN_TYPE lex();
    while (SEMICOLON != lex())
	;
}

	/************************************************/
	/*		ReadToken:			*/
	/************************************************/

ReadToken(tok)
TOKEN_TYPE tok;
{
TOKEN_TYPE lex(), token;
char *expected;
    if ((token = lex()) != tok)
	{
	switch (tok)
	    {
	    case AT: expected = "at"; break;
	    case TO: expected = "to"; break;
	    case R_PAREN: expected = ")"; break;
	    case COMMA: expected = ","; break;
	    case NUMBER: expected = "a number"; break;
	    case PHYLE: expected = "file"; break;
	    case IDENTIFIER: expected = "an identifier"; break;
	    case SEMICOLON: expected = ";"; break;
	    case PERIOD: expected = "."; break;
	    case DOUBLE_COLON: expected = "::"; break;
	    case PLACE: expected = "place"; break;
	    case EQUAL: expected = "="; break;
	    }
	TtyBlinkError("Error in ReadToken. Read token number %d; expected %s",
			(int)token,expected);
	}
}

	/************************************************/
	/*		ParseTransformation:		*/
	/************************************************/

ParseTransformation(tPtr, sin)
TRANS_MATRIX *tPtr;
short sin;
{
EXPRESSION MakeConst();
short tmp;

    TOKEN_TYPE lex();
    tPtr->x11 = tPtr->x22 = 1;
    tPtr->x12 = tPtr->x21 = 0;
    tPtr->x31/*.postfix*/ = NIL_EXPRESSION;
    while (TRUE)
	{
	switch (lex())
	    {
	    case SEMICOLON:
		if (tPtr->x31/*.postfix*/ == NIL_EXPRESSION)
		    {
		    tPtr->x31 = MakeConst(0);
		    tPtr->x32 = MakeConst(0);
		    }
		return;
	    case COMMA:
		break;
	    case FLIPPED:
		switch (lex())
		    {
		    case LR:
		    case RL:
			tPtr->x11 *= -1;
			tPtr->x21 *= -1;
			break;
		    case UD:
			tPtr->x22 *= -1;
			tPtr->x12 *= -1;
			break;
		    default:
			ErrorPrint(75);
			break;
		    }
		break;
	    case AT:
		ReadToken(L_PAREN);
		tPtr->x31 = parseexp(sin);
		ReadToken(COMMA);
		tPtr->x32 = parseexp(sin);
		ReadToken(R_PAREN);
		return;
	    case ROTATED:
		ReadToken(NUMBER);
		if (0 != (LexNumber%6))
		    ErrorPrint(74);
		switch (LexNumber/6)
		    {
		    case 1: /* rotated 3 */
			tmp = tPtr->x11;
			tPtr->x11 = tPtr->x12;
			tPtr->x12 = -tmp;
			tmp = tPtr->x22;
			tPtr->x22 = -tPtr->x21;
			tPtr->x21 = tmp;
			break;
		    case 2: /* rotated 6 */
			tPtr->x11 *= -1;
			tPtr->x22 *= -1;
			tPtr->x21 *= -1;
			tPtr->x12 *= -1;
			break;
		    case 3: /* rotated 9 */
			tmp = tPtr->x11;
			tPtr->x11 = -tPtr->x12;
			tPtr->x12 = tmp;
			tmp = tPtr->x22;
			tPtr->x22 = tPtr->x21;
			tPtr->x21 = - tmp;
			break;
		    }
		break;
	    default:
		return;
	    }
	}
}

#ifdef UNDEFINEDSYMBOL
	/************************************************/
	/*		TestPrint:			*/
	/************************************************/

TestPrint(cmdptr)
CMD_CHAIN_PTR cmdptr;
{
    while (cmdptr)
	{
	switch (cmdptr->type)
	    {
	    case BOX_COMMAND:
		printf("box: %s, %s, %s, %s\n",
		       &string[cmdptr->params.box.urc->x.string],
		       &string[cmdptr->params.box.urc->y.string],
		       &string[cmdptr->params.box.llc->x.string],
		       &string[cmdptr->params.box.llc->y.string]);
		break;
	    case CUT_COMMAND:
		printf("cut: %s, %s\n",
			&string[cmdptr->params.cut.origin->x.string],
			&string[cmdptr->params.cut.origin->y.string]);
		break;
	    case BUTT_COMMAND:
		printf("butting_contact\n");
		PrintTransformation(&(cmdptr->params.butt.trans));
		break;
	    default: printf("unknown command\n");
		break;
	    }
	cmdptr = cmdptr->link;
	}
    }

	/************************************************/
	/*		PrintTransformation:		*/
	/************************************************/

PrintTransformation(tMtx)
TRANS_MATRIX *tMtx;
{
printf("%6d %6d  0\n%6d %6d  0\n%6s %6s  1\n", tMtx->x11, tMtx->x12, tMtx->x21,
		tMtx->x22, &string[tMtx->x31.string], &string[tMtx->x32.string]);
}
#endif
