/********************************************************/
/*							*/
/*		YALE layout editor			*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*		THOMAS R. DAVIS				*/
/*							*/
/********************************************************/

/* FILE: lex.c */

#include "aledefs.h"

# ifdef UNIX
# include <stdio.h>
# else UNIX
# include <Vio.h>
# endif UNIX

#define BLANK 0
#define UC_LETTER 1
#define LETTER 2
#define DIGIT 3
#define SIMPLE 4
#define HARD 5

unsigned char CharClass['~'];
static char lexbuf[2][TOKBUFSIZE];
static short lexbufptr = 0;
char LowerCase();

/* The lexbuf is used to hold a token.  There are two of them
 * so that the last token is also available.  lexbufptr points
 * to the next one to use.
 */

static BOOLEAN isUngotLeaf = FALSE;
static char ungotLeafChar;

/* these last two definitions are used when ungetchar is called and
 * the input source happens to be pointed at the leaf server.
 */

/* The next two routines -- ungetchar() and GetAChar() are the only
 * routines used by lex to pick up characters.  The source for the
 * input is determined by the value of the pointer LexSource.  If it
 * is NIL, input comes from the standard input; if it is positive,
 * input comes from the string pointed to by it, and if it is -1,
 * input comes from a remote file accessed by means of leaf protocols.
 */

	/************************************************/
	/*		ungetchar:			*/
	/************************************************/

ungetchar(ch)
char ch;
{
if (LexSource == NIL)
    {
    ungetc(ch, stdin);
    if (ch == '\n')
	CurrentLine--;
    else
	CurrentColumn--;
    }
else if ((int)LexSource == -1) /* leaf */
    {
    ungotLeafChar = ch; /* ++ bug potential here -- check for flag first */
    isUngotLeaf = TRUE;
    }
else
    {
    LexSource--;
    CurrentColumn--;
    }
if (LexSaveFlag && (ch != ' '))
    --LexSavePtr;
}

	/************************************************/
	/*		GetAChar:			*/
	/************************************************/

char GetAChar()
{
char ch;

if (LexSource == NIL)
    {
    ch = getchar();
    if (LexSaveFlag && (ch != ' '))
	*(LexSavePtr++) = ch;
    if (ch == '\n')
	{
	CurrentLine++;
	CurrentColumn = 0;
	}
    else
	CurrentColumn++;
    return (LowerCase(ch));
    }
else if ((int) LexSource == -1)	/* leaf */
    {
    if (isUngotLeaf)
	{
	isUngotLeaf = FALSE;
	ch = ungotLeafChar;
	}
    else
	ch = LeafGetchar();
    if (LexSaveFlag && (ch != ' '))
	*(LexSavePtr++) = ch;
    if (ch == '\n')
	{
	CurrentLine++;
	CurrentColumn = 0;
	}
    else
	CurrentColumn++;
    return (LowerCase(ch));
    }
else
    {
    CurrentColumn++;
    ch = *(LexSource++);
    if (LexSaveFlag && (ch != ' '))
	*(LexSavePtr++) = ch;
    return(LowerCase(ch));
    }
}

/* lowercasestring: lowers the case of the string pointed to by ptr */

lowercasestring(ptr)
  char *ptr;
{
if (ptr == NIL)
    return;
while (*ptr)
  *ptr++ = LowerCase(*ptr);
}

/* LowerCase: lowercasifies ch */

char LowerCase(ch)
  char ch;
{
if ((ch >= 'A') && (ch <= 'Z'))
    return(ch + 'a' - 'A');
return(ch);
}

	/************************************************/
	/*		LexPeek:			*/
	/************************************************/

TOKEN_TYPE LexPeek()
{
    TOKEN_TYPE token, lex();
    if (PeekFlag)
	return(LexToken);
    token = lex();
    PeekFlag = TRUE;
    return(token);
}

	/************************************************/
	/*		UnLex:				*/
	/************************************************/

UnLex()
{
    if (PeekFlag)
	ErrorPrint(6); /* can't unlex twice in a row! */
    else
	PeekFlag = TRUE; /* makes last lex look like a peek */
}

/* lex returns an integer corresponding to the token type.  All upper-case
 * letters are converted to lower-case.  If the value returned is a string,
 * then sp is left pointing to the first character of the string, which is
 * zero-terminated.  If lex discovers a number, then twice the value of the
 * number is returned in the variable pointed to by nptr.
 * if the token is a reserved word, wptr points to it.
 */

	/************************************************/
	/*		lex:				*/
	/************************************************/

TOKEN_TYPE lex()
{
    char ch, *sp, *sp1, GetAChar();
    int commentDepth, stringLength, numValue;
    static doubleDotFlag = FALSE;
    S_ENTRY_PTR sPtr, LookupHash();
    if (doubleDotFlag)
	{
	doubleDotFlag = FALSE;
	return(LexToken = DOUBLE_PERIOD);
	}
    if (PeekFlag)
	{
	PeekFlag = FALSE;
	return(LexToken);
	}
    while (TRUE) /* only exit is by return or error */
    {
    if (EOF == (ch = GetAChar()))
	{
	ErrorPrint(1);
	return(ENDOFFILE);
	}
    switch (CharClass[ch])
	{
	case BLANK: /* skip it */
	    break;
	case UC_LETTER: /* force lower and drop through to LETTER */
	    ch = ch - 'A' + 'a';
	case LETTER: /* it's an identifier */
	    sp = LexString = &lexbuf[lexbufptr][0];
	    lexbufptr = 1 - lexbufptr;
	    stringLength = 0;
	    sp1 = LexString;
	    *(sp++) = ch;
	    while (stringLength < TOKBUFSIZE)
		{
		switch (CharClass[ch = GetAChar()])
		    {
		    case UC_LETTER:
			ch = ch - 'A' + 'a';
		    case LETTER:
		    case DIGIT:
			*(sp++) = ch;
			stringLength++;
			break;
		    default:
			ungetchar(ch);
			*(sp++) = '\0';
			sPtr = LookupHash(sp1, NIL);
			if(sPtr == 0 || sPtr->symbolType != RESERVED_ENTRY)
			    return(LexToken = IDENTIFIER);
			return(LexToken = sPtr->data.word);
		    }
		}
	    /* must have overflowed the token buffer */
	    ErrorPrint(4);
	    return(LexToken = IDENTIFIER);
	case DIGIT:
	    {
	    numValue = ch - '0';
	    while (DIGIT == CharClass[ch=GetAChar()])
		numValue = numValue*10 + ch - '0';
	    if (ch == '.')
		{
		switch (ch = GetAChar())
		    {
		    case '.':
			doubleDotFlag = TRUE;
			break;
		    case '5':
			LexNumber = numValue*2 + 1;
			return(LexToken = NUMBER);
		    default:
			ErrorPrint(5);
		    }
		}
	    else
		ungetchar(ch);
	    LexNumber = numValue*2;
	    return(LexToken = NUMBER);
	    }
	case SIMPLE:
	    switch (ch)
		{
		case ')': return(LexToken = R_PAREN);
		case ',': return(LexToken = COMMA);
		case ';': return(LexToken = SEMICOLON);
		case '=': return(LexToken = EQUAL);
		case '*': return(LexToken = TIMES);
		case '/': return(LexToken = QUOTIENT);
		case '[': return(LexToken = L_BRACKET);
		case ']': return(LexToken = R_BRACKET);
		case '+': return(LexToken = PLUS);
		};
	case HARD:
	    switch (ch)
		{
		case '-':
		    switch (LexToken)
			{
			case ASSIGNMENT:
			case L_PAREN:
			case L_BRACKET:
			case LESS_THAN:
			case EQUAL:
			case GREATER_THAN:
			case GREATER_OR_EQUAL:
			case LESS_OR_EQUAL:
			case COMMA:
			case PLUS:
			case MINUS:
			case TIMES:
			case QUOTIENT:
			    return(LexToken = UNARY_MINUS);
			default:
			    return(LexToken = MINUS);
			}
		case '(':
		    if ('*' != (ch = GetAChar()))
			{
			ungetchar(ch);
			return(LexToken = L_PAREN);
			};
		    /* if we get here, we're inside a comment */
		    commentDepth = 1;
		    while (commentDepth != 0)
			{
			switch (ch=GetAChar())
			    {
			    case '(': if ('*' == (ch = GetAChar()))
							commentDepth++;
				break;
			    case '*': if (')' == (ch = GetAChar()))
							commentDepth--;
				break;
			    case EOF:
				{
				ErrorPrint(2);
				return(LexToken = PERIOD); /* dummy return */
				};
			    };
			}
		    break;
		case '.':
		    switch (ch = GetAChar())
			{
			case '.': return(LexToken = DOUBLE_PERIOD);
			case '5':
			    {
			    LexNumber = 1;
			    return(LexToken = NUMBER);
			    }
			default:
			    {
			    ungetchar(ch);
			    return(LexToken = PERIOD);
			    };
			};
		case ':':
		    switch (ch = GetAChar())
			{
			case '=': return(LexToken = ASSIGNMENT);
			case ':': return(LexToken = DOUBLE_COLON);
			default:
			    {
			    ungetchar(ch);
			    return(LexToken = COLON);
			    };
			};
		case '<':
		    switch (ch = GetAChar())
			{
			case '>': return(LexToken = NOT_EQUAL);
			case '=': return(LexToken = LESS_OR_EQUAL);
			default:
			    {
			    ungetchar(ch);
			    return(LexToken = LESS_THAN);
			    };
			};
		case '>':
		    switch (ch = GetAChar())
			{
			case '=': return(LexToken = GREATER_OR_EQUAL);
			default:
			    {
			    ungetchar(ch);
			    return(LexToken = GREATER_THAN);
			    };
			};
		case '"':
		    stringLength = 0;
		    sp = LexString = &lexbuf[lexbufptr][0];
		    lexbufptr = 1 - lexbufptr;
		    while (stringLength < TOKBUFSIZE)
			{
			switch (ch = GetAChar())
			    {
			    case EOF:
				ErrorPrint(3);
				return(LexToken = STRING);
			    case '"':
				{
				*(sp++) = '\0';
				return(LexToken = STRING);
				}
			    default:
				{
				stringLength++;
				*(sp++) = ch;
				};
			    }
			}
		    /* must have overflowed the token buffer */
		    ErrorPrint(4);
		    return(LexToken = STRING);
		}
	}
    }
}
/* lexinit initializes the array CharClass for lex */

	/************************************************/
	/*		lexinit:			*/
	/************************************************/

lexinit()
{
    int i;
    CurrentLine = 1;
    CurrentColumn = 0;
    LexSource = NIL;	/* start looking at the terminal */

    for (i=0; i <= '~'; i++)
	CharClass[i] = BLANK;
    for (i='a'; i<= 'z'; i++)
	CharClass[i] = LETTER;
    for (i= 'A'; i <= 'Z'; i++)
	CharClass[i] = UC_LETTER;
    CharClass['_'] = LETTER;
    for (i='0'; i <= '9'; i++)
	CharClass[i] = DIGIT;
    CharClass[')'] = CharClass['+'] = CharClass['/'] = CharClass[';']
     = CharClass['='] = CharClass['*'] = CharClass['[']
     = CharClass[','] = CharClass[']'] = SIMPLE;

    CharClass['('] = CharClass['.'] = CharClass['<'] = CharClass['>']
     = CharClass['"'] = CharClass[':'] = CharClass['-'] = HARD;
}
