/********************************************************/
/*							*/
/*		YALE layout editor			*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*		THOMAS R. DAVIS				*/
/*							*/
/********************************************************/

/* FILE: datamung.c
 *
 * Routines for adding new cell definitions, etc.
 * Lines containing "+++" need to be worked on.
 */

#include "aledefs.h"

char   *xmin = "xmin";		/* make string entries for these guys */
char   *xmax = "xmax";		/* once and for all. */
char   *ymin = "ymin";
char   *ymax = "ymax";

 /* ********************************************** */
 /* 		SelectRectangle:		   */
 /* ********************************************** */

SelectRectangle (rectPtr)
RECTANGLE_PTR rectPtr;
{
    CELL_DEFINITION_PTR cellDef;
    RECTANGLE_CONS_PTR rectConsPtr;
    CONS_PTR GetCons ();

    if (rectPtr -> selectedEdgeset != 0)/* it's already selected */
	return;

    cellDef = rectPtr -> containingCell;
    rectConsPtr = (RECTANGLE_CONS_PTR) GetCons ();
    if (rectConsPtr == NIL)
	return;	/* no free storage */
    rectConsPtr -> value = rectPtr;
    rectConsPtr -> next = cellDef -> selectedRectangleList;
    cellDef -> selectedRectangleList = rectConsPtr;
}

 /* ********************************************** */
 /* 		UnSelectAll:			   */
 /* ********************************************** */

/* UnSelectAll: will de-select all items in a cell which can possibly
 * be selected, except for reference points.  At present, the only thing
 * that this includes is rectangles and cell instances.
 */

UnSelectAll (cellDefPtr)
CELL_DEFINITION_PTR cellDefPtr;
{
    RECTANGLE_CONS_PTR rectConsPtr, saveRectCons;
    INSTANCE_CONS_PTR instConsPtr, saveInstCons;

    rectConsPtr = cellDefPtr -> selectedRectangleList;
    while (rectConsPtr != NIL)
    {
	UnSelect(rectConsPtr -> value -> sunInstanceNumber);
	rectConsPtr -> value -> selectedEdgeset = 0;
	saveRectCons = rectConsPtr;
	rectConsPtr = rectConsPtr -> next;
	FreeCons (saveRectCons);
    }
    cellDefPtr -> selectedRectangleList = NIL;

    instConsPtr = cellDefPtr -> selectedSubcellList;
    while (instConsPtr != NIL)
    {
	UnSelect(instConsPtr -> value -> sunInstanceNumber);
	saveInstCons = instConsPtr;
	instConsPtr = instConsPtr -> next;
	FreeCons (saveInstCons);
    }
    cellDefPtr -> selectedSubcellList = NIL;
}

 /* ********************************************** */
 /* 		MakeRectangle:			   */
 /* ********************************************** */

RECTANGLE_PTR MakeRectangle (cellDef, l, r, t, b, layer)
CELL_DEFINITION_PTR cellDef;
EXPRESSION l, r, t, b;
PROCESS_LAYER layer;
{
    RECTANGLE_PTR rPtr, rTail, GetRectangle ();
    REFERENCE_POINT_ER FindInvolvedRefPt (), refPtr;
    int     sin,
            GetSunInstanceNumber ();

    if (cellDef == NIL)
	return(NIL);		/* bullet-proofing -- TRD */
    rPtr = GetRectangle ();
    if (rPtr == NIL)
	return(NIL);
    rPtr -> selectedEdgeset = 0;
    rPtr -> containingCell = cellDef;
    if (-1 == (sin = (rPtr -> sunInstanceNumber = GetSunInstanceNumber())))
	return(NIL);
    SunInstanceArray[sin].type = SRectangle;
    SunInstanceArray[sin].ptr.rect_ptr = rPtr;
    rPtr -> l = l;
    rPtr -> r = r;
    rPtr -> t = t;
    rPtr -> b = b;
    rPtr -> layer = layer;

    refPtr = FindInvolvedRefPt(l, Vertical);
    LinkDependentRectangle(rPtr, refPtr);
    refPtr = FindInvolvedRefPt(r, Vertical);
    LinkDependentRectangle(rPtr, refPtr);
    refPtr = FindInvolvedRefPt(t, Horizontal);
    LinkDependentRectangle(rPtr, refPtr);
    refPtr = FindInvolvedRefPt(b, Horizontal);
    LinkDependentRectangle(rPtr, refPtr);

/* Now, add it to the end of the rectangle list. */
    rPtr -> next = NIL;
    if ((rTail = cellDef -> rectangles) == NIL)
	cellDef -> rectangles = rPtr;
    else
    {
	while (rTail -> next != NIL)
	    rTail = rTail -> next;
	rTail -> next = rPtr;
    }
    return (rPtr);
}

 /* ********************************************** */
 /* 		RemoveRectangle:		   */
 /* ********************************************** */

RemoveRectangle (rectPtr)
RECTANGLE_PTR rectPtr;
{
    RECTANGLE_CONS_PTR rectConsPtr;
    INSTANCE_CONS_PTR GeneralPointerFind ();

    UnLinkDepRectangle(rectPtr, LeftEdge);
    FreeExpression (rectPtr -> l);
    UnLinkDepRectangle(rectPtr, RightEdge);
    FreeExpression (rectPtr -> r);
    UnLinkDepRectangle(rectPtr, TopEdge);
    FreeExpression (rectPtr -> t);
    UnLinkDepRectangle(rectPtr, BottomEdge);
    FreeExpression (rectPtr -> b);
    FreeSunInstanceNumber (rectPtr -> sunInstanceNumber);
/* first delete it from the selected list if it is selected */

    rectConsPtr = rectPtr -> containingCell -> selectedRectangleList;
    if (GeneralPointerFind (rectConsPtr, rectPtr))
	GeneralConsRemove
	    (&(rectPtr -> containingCell -> selectedRectangleList), rectPtr);

/* and now knock it off the rectangle list and free the space */

    generalInstRemove (&(rectPtr -> containingCell -> rectangles), rectPtr);
    FreeRectangle (rectPtr);
}

 /* ********************************************** */
 /* 		SelectCellInstance:		   */
 /* ********************************************** */

SelectCellInstance (instPtr)
CELL_INSTANCE_PTR instPtr;
{
    INSTANCE_CONS_PTR consPtr;
    CELL_DEFINITION_PTR cellDef;
    CONS_PTR GetCons ();

    cellDef = instPtr -> containingCell;
    /* if it's already selected, just return: */
    if (GeneralPointerFind
	(cellDef->selectedSubcellList, instPtr))
	    return;
    consPtr = (INSTANCE_CONS_PTR) GetCons ();
    if (consPtr == NIL)
	return;	/* no more free storage */
    consPtr -> next = cellDef -> selectedSubcellList;
    consPtr -> value = instPtr;
    cellDef -> selectedSubcellList = consPtr;
}

 /* ********************************************** */
 /* 		RemoveCellInstance:		   */
 /* ********************************************** */

/* RemoveCellInstance() removes the instance from the cell
 * definition, and restores the space to the free_list.  It also
 * frees all of the expression space used, and deletes the symbol
 * table entry for the instance (if any).
 */

RemoveCellInstance (instPtr)
CELL_INSTANCE_PTR instPtr;
{
/* first, free expression space */

/* DEBUG printf ("Remove: %d\n", instPtr->sunInstanceNumber); */

    FreeExpression (instPtr -> l);
    FreeExpression (instPtr -> r);
    FreeExpression (instPtr -> t);
    FreeExpression (instPtr -> b);
    FreeSunInstanceNumber (instPtr -> sunInstanceNumber);
/* free up any lists it keeps */

    FreeSymbolTableEntry (instPtr -> symTblPtr);
    FreeExpressionConsList (instPtr -> parameterList);

/* untangle it from its instance-of and subcell-of lists */
/* first instance-of: 					 */

    GeneralConsRemove (&(instPtr -> instanceOf -> instance_list), instPtr);

/* it might be selected, so remove it from that list */

    if (GeneralPointerFind
	    (instPtr -> containingCell -> selectedSubcellList, instPtr))
      {
	GeneralConsRemove
	    (&(instPtr -> containingCell -> selectedSubcellList), instPtr);
      }

/* And now remove it from the subcell-of list */

    generalInstRemove (&(instPtr -> containingCell -> cells), instPtr);


    UnLinkDepInstance(instPtr, Horizontal);
    UnLinkDepInstance(instPtr, Vertical);
    FreeExpression (instPtr -> trans.x31);
    FreeExpression (instPtr -> trans.x32);

/* and finally, free the record itself */

    FreeCellInstance (instPtr);
}

 /* ********************************************** */
 /* 		MakeConst:			   */
 /* ********************************************** */

EXPRESSION MakeConst (n)
int     n;
{
    char    numb[5];
    EXPRESSION exp;
    EXPRESSION parseexp ();
    TOKEN_TYPE lex ();
/*    unsigned short  AddString (); */
    char   *lexSrcSave;

    sprintf (numb, "%d;", n);
    lexSrcSave = LexSource;
    LexSource = numb;
    exp = parseexp (0);
    lex ();			/* unload dummy semicolon */
    LexSource = lexSrcSave;
    return (exp);
}

 /* ********************************************** */
 /* 		MakeCellInstance:		   */
 /* ********************************************** */

/* MakeCellInstance() creates a new cell instance, and fills in
 * all information possible given just the cell definition.
 */

CELL_INSTANCE_PTR MakeCellInstance (cellDefPtr, cellInclPtr)
CELL_DEFINITION_PTR cellDefPtr,	/* definition of new instance */
cellInclPtr;			/* cell it is to be included in */
{
    CELL_INSTANCE_PTR newCellPtr, tailPtr, GetCellInstance ();
    INSTANCE_CONS_PTR instCons;
    CONS_PTR GetCons ();
    int     sin,
            GetSunInstanceNumber ();
    EXPRESSION MakeConst ();

    newCellPtr = GetCellInstance ();
    if (newCellPtr == NIL)
	return(0);	/* out of free storage */
    if (cellInclPtr != NIL)	/* not a master instance, so */
    {				/* link onto cell def's list */
	newCellPtr -> next = NIL;
	if ((tailPtr = cellInclPtr -> cells) == NIL)
	    cellInclPtr -> cells = newCellPtr;
	else
	{
	    while (tailPtr -> next != NIL)
		tailPtr = tailPtr -> next;
	    tailPtr -> next = newCellPtr;
	}
	instCons = (INSTANCE_CONS_PTR) GetCons ();
	if (instCons == NIL)
	    return(NIL);	/* out of free storage */
	instCons -> value = newCellPtr;
	instCons -> next = cellDefPtr -> instance_list;
	cellDefPtr -> instance_list = instCons;
    };

    newCellPtr -> symTblPtr = NIL;/* we don't know this yet */
    newCellPtr -> l/*.postfix*/ = NIL_EXPRESSION;
    newCellPtr -> r/*.postfix*/ = NIL_EXPRESSION;/* boundaries undefined */
    newCellPtr -> t/*.postfix*/ = NIL_EXPRESSION;
    newCellPtr -> b/*.postfix*/ = NIL_EXPRESSION;

    if (-1 == (sin = (newCellPtr -> sunInstanceNumber = GetSunInstanceNumber())))
	return(NIL);

    SunInstanceArray[sin].type = SCellInstance;
    SunInstanceArray[sin].ptr.instPtr = newCellPtr;
    newCellPtr -> trans.x11 = 1;/* identity transformation */
    newCellPtr -> trans.x12 = 0;
    newCellPtr -> trans.x21 = 0;
    newCellPtr -> trans.x22 = 1;
    newCellPtr -> trans.x31 = MakeConst (0);
    newCellPtr -> trans.x32 = MakeConst (0);

    newCellPtr -> parameterList = NIL;
    newCellPtr -> instanceOf = cellDefPtr;
    newCellPtr -> containingCell = cellInclPtr;

    newCellPtr -> selected = FALSE;
    newCellPtr -> selected_ref = NIL;
    newCellPtr -> expand = TRUE;

    return (newCellPtr);
}

 /* ********************************************** */
 /* 		MakeCellDefinition:		   */
 /* ********************************************** */

CELL_DEFINITION_PTR MakeCellDefinition (symPtr)
S_ENTRY_PTR symPtr;
{
    CELL_DEFINITION_PTR newDefPtr, cellPtr, GetCellDef ();
    REFERENCE_POINT_ER MakeReferencePoint ();
    S_ENTRY_PTR STablePtr, EnterHash ();
    int     sin,
            GetSunInstanceNumber ();

    newDefPtr = GetCellDef ();
    if (newDefPtr == NIL)
	return(NIL);	/* out of free storage */

/* now add the new cell definition to the end of the master list */

    newDefPtr -> next = NIL;
    if (MasterCellList == NIL)
	MasterCellList = newDefPtr;
    else
    {
	cellPtr = MasterCellList;
	while (cellPtr -> next != NIL)
	    cellPtr = cellPtr -> next;
	cellPtr -> next = newDefPtr;
    }

    newDefPtr -> symTblPtr = symPtr;

    newDefPtr -> xSelectedRef =
	(newDefPtr -> xOriginRef = MakeReferencePoint (newDefPtr));
    if (newDefPtr->xOriginRef == NIL)
	return(NIL);	/* out of free storage */
    newDefPtr -> xSelectedRef -> isXvar = TRUE;
    newDefPtr -> ySelectedRef =
	(newDefPtr -> yOriginRef = MakeReferencePoint (newDefPtr));
    if (newDefPtr->yOriginRef == NIL)
	return(NIL);	/* out of free storage */
    newDefPtr -> ySelectedRef -> isXvar = FALSE;

    newDefPtr -> localList = NIL;
    newDefPtr -> parameterList = NIL;
    newDefPtr -> exportList = NIL;

    newDefPtr -> constraint_list = NIL;
    newDefPtr -> defaultList = NIL;
    newDefPtr -> assignment_list = NIL;

    newDefPtr -> exportEnd = 0;/* no local variables so far -- xmin,	 
				*/
    newDefPtr -> param_end = 0;	/* ymin, ... will be added later	 
				*/
    newDefPtr -> frameEnd = 0;

    if (-1 == (sin=newDefPtr->sunInstanceNumber = GetSunInstanceNumber()))
	return(NIL);
    SunInstanceArray[sin].type= SCellDefinition;

    SunInstanceArray[sin].ptr.def_ptr = newDefPtr;
    newDefPtr -> cells = NIL;	/* no cell calls yet */
    newDefPtr -> instance_list = NIL;/* no instances of me yet */
    newDefPtr -> masterInstance = MakeCellInstance (newDefPtr, NIL);

    newDefPtr -> rectangles = NIL;
    newDefPtr -> dependentCells = NIL;

    newDefPtr -> selectedRectangleList = NIL;
    newDefPtr -> selectedSubcellList = NIL;

/* Now we must insert entries for the built-in variables xmin,
 * xmax, ymin, and ymax.
 */

    STablePtr = EnterHash (ymax, sin);
    STablePtr -> symbolType = YVAR_ENTRY;
    STablePtr -> data.isExport = TRUE;
    AddExportVariable (newDefPtr, STablePtr, FALSE);

    STablePtr = EnterHash (xmax, sin);
    STablePtr -> symbolType = XVAR_ENTRY;
    STablePtr -> data.isExport = TRUE;
    AddExportVariable (newDefPtr, STablePtr, FALSE);

    STablePtr = EnterHash (ymin, sin);
    STablePtr -> symbolType = YVAR_ENTRY;
    STablePtr -> data.isExport = TRUE;
    AddExportVariable (newDefPtr, STablePtr, FALSE);

    STablePtr = EnterHash (xmin, sin);
    STablePtr -> symbolType = XVAR_ENTRY;
    STablePtr -> data.isExport = TRUE;
    AddExportVariable (newDefPtr, STablePtr, FALSE);

    return (newDefPtr);
}

 /* ********************************************** */
 /* 		AddRefSon:			   */
 /* ********************************************** */

REFERENCE_POINT_ER AddRefSon (refPt)
REFERENCE_POINT_ER refPt;
{
    REFERENCE_POINT_ER newPt, MakeReferencePoint ();

    newPt = MakeReferencePoint (refPt -> containingCell);
    if (newPt == NIL)
	return(NIL);	/* out of free storage */
    newPt -> next = refPt -> son;
    refPt -> son = newPt;
    newPt -> isXvar = refPt -> isXvar;
    return (newPt);
}

 /* ********************************************** */
 /* 		RemoveReferencePoint:		   */
 /* ********************************************** */

/* The reference point is removed only if all of the stuff
 * dependent upon it has already been removed.  Otherwise,
 * an error is signalled, and nothing happens.  Zero if returned if the
 * removal was unsuccessful; one otherwise.
 */

RemoveReferencePoint (refPt)
REFERENCE_POINT_ER refPt;
{
    REFERENCE_POINT_ER prev, FindPreviousRefPt ();
    if (refPt -> son != NIL ||
	    refPt -> dependentRectangles != NIL ||
	    refPt -> dependentInstances != NIL ||
	    refPt -> dependentConnPts != NIL)
    {
	ErrorPrint (29);
	return(0);
    }
    if (refPt -> containingCell -> xOriginRef == refPt ||
	    refPt -> containingCell -> yOriginRef == refPt)
    {
	ErrorPrint (30);
	return(0);
    }
    if (!(prev = FindPreviousRefPt (refPt -> containingCell -> xOriginRef, refPt)))
	if (!(prev =
		FindPreviousRefPt (refPt -> containingCell -> yOriginRef, refPt)))
	{
	    ErrorPrint (31);
	    return(0);
	}

/* prev's next or son is the refPt we wish to remove */

/*    if (refPt->symTblPtr != NIL)
	FreeSymbolTableEntry(refPt->symTblPtr);
 */
    if (prev -> next == refPt)
    {
	prev -> next = refPt -> next;
	FreeReferencePoint (refPt);
    }
    else
    {
	prev -> son = refPt -> next;
	FreeReferencePoint (refPt);
    }
    FreeSunInstanceNumber (refPt -> sunInstanceNumber);
    return(1);
}

 /* ********************************************** */
 /* 		FindPreviousRefPt:		   */
 /* ********************************************** */

/* This routine does a recursive search of the tree of reference
 * points, and finds the one whose son or next is the target.
 */
REFERENCE_POINT_ER FindPreviousRefPt (start, target)
REFERENCE_POINT_ER start,
target;
{
    REFERENCE_POINT_ER refPt;

    if (start == NIL)
	return (NIL);
    if (start -> next == NIL && start -> son == NIL)
	return (NIL);
    if (start -> next == target || start -> son == target)
	return (start);
    if (refPt = FindPreviousRefPt (start -> son, target))
	return (refPt);
    return (FindPreviousRefPt (start -> next, target));
}

/* FindParentRefPt: finds the reference point which is the parent of
 * the target, starting from target.
 */

REFERENCE_POINT_ER FindParentRefPt(start, target)
REFERENCE_POINT_ER start, target;
{
REFERENCE_POINT_ER temp;

temp = FindPreviousRefPt(start, target);
if (temp->son == target)
    return(temp);
return(FindParentRefPt(start, temp));
}

 /* ********************************************** */
 /* 		MakeReferencePoint:		   */
 /* ********************************************** */

REFERENCE_POINT_ER MakeReferencePoint (cellDef)
CELL_DEFINITION_PTR cellDef;
{
    REFERENCE_POINT_ER newRefPtr, GetReferencePoint ();
    int     sin,
            GetSunInstanceNumber ();

    newRefPtr = GetReferencePoint ();
    if (newRefPtr == NIL)
	return(NIL);
    newRefPtr -> next = NIL;
    newRefPtr -> son = NIL;
    newRefPtr -> symTblPtr = NIL;
    newRefPtr -> containingCell = cellDef;
    if (-1 == (newRefPtr -> sunInstanceNumber = GetSunInstanceNumber()))
	return(NIL);
    SunInstanceArray[(sin = newRefPtr -> sunInstanceNumber)].type
	= SReferencePoint;
    SunInstanceArray[sin].ptr.refPtr = newRefPtr;
    newRefPtr -> dependentRectangles = NIL;
    newRefPtr -> dependentInstances = NIL;
    newRefPtr -> dependentConnPts = NIL;

    return (newRefPtr);
}

 /* ********************************************** */
 /* 		FindInvolvedRefPt:		   */
 /* ********************************************** */

/* This routine will take an expression as input and will search the
 * reference point tree to try to find one whose name is the same as the
 * first string in the expression.  NIL is returned if no such reference
 * point is found.  refType is either Horizontal, Vertical, or 0 (both).
 */

REFERENCE_POINT_ER FindInvolvedRefPt(exp, refType)
EXPRESSION exp;
short refType;
{
REFERENCE_POINT_ER root, involvedRef, refSearch ();
char expId[32], *p1, *p2, *FindExpString();

/* First, dig out the first identifier in the expression. */

/*p1 = &(string[exp.string]);*/
p1 = FindExpString(exp);
p2 = &(expId[0]);

if (!(((*p1) >= 'a' && (*p1) <= 'z') || ((*p1) >= 'A' && (*p1) <= 'Z')))
    return(refType == Vertical ? OpenCellDefinition->xOriginRef :
				 OpenCellDefinition->yOriginRef);
*(p2++) = *(p1++);
while (((*p1) >= 'a' && (*p1) <= 'z') || ((*p1) >= 'A' && (*p1) <= 'Z')
	|| ((*p1) >= '0' && (*p1) <= '9') || (*p1) == '_')
		*(p2++) = *(p1++);
*(p2) = 0;
if (refType == Horizontal || refType == 0)
    {
    root = OpenCellDefinition -> yOriginRef;
    if (involvedRef = refSearch(root->son, expId))
	return(involvedRef);
    if (involvedRef = refSearch(root->next, expId))
	return(involvedRef);
    }

if (refType == Vertical || refType == 0)
    {
    root = OpenCellDefinition -> xOriginRef;
    if (involvedRef = refSearch(root->son, expId))
	return(involvedRef);
    if (involvedRef = refSearch(root->next, expId))
	return(involvedRef);
    }

if (refType == Vertical)
    return(OpenCellDefinition->xOriginRef);	/* couldn't find it anywhere */
else
    return(OpenCellDefinition->yOriginRef);
}

 /* ********************************************** */
 /* 		refSearch:			   */
 /* ********************************************** */

REFERENCE_POINT_ER refSearch(refPt, str)
REFERENCE_POINT_ER refPt;
char *str;
{
REFERENCE_POINT_ER foundRef;

if (refPt == NIL)
    return(NIL);
if (strcmp(str, refPt->symTblPtr->stringPtr))
    return(refPt);
if (foundRef = refSearch(refPt->son, str))
    return(foundRef);
return(refSearch(refPt->next, str));
}

 /* ********************************************** */
 /* 		LinkDependentInstance:		   */
 /* ********************************************** */

/* This routine is essentially identical with LinkDependentRectangle, below.
 */

LinkDependentInstance(instPtr, refPtr)
CELL_INSTANCE_PTR instPtr;
REFERENCE_POINT_ER refPtr;
{
INSTANCE_CONS_PTR instConsPtr;

if (GeneralPointerFind (refPtr->dependentInstances, instPtr))
    return;	/* already linked on */

instConsPtr = (INSTANCE_CONS_PTR) GetCons();
if (instConsPtr == NIL)
    return;	/* out of free storage */
instConsPtr -> next = refPtr -> dependentInstances;
instConsPtr -> value = instPtr;
refPtr -> dependentInstances = instConsPtr;
}

 /* ********************************************** */
 /* 		LinkDependentRectangle:		   */
 /* ********************************************** */

/* This routine adds a cons-cell link onto the dependent rectangle
 * chain of the OpenCellDefinition.  If two of a rectangle's edges are
 * dependent upon a reference point, then there will be only one
 * cons-cell linking it on.
 */

LinkDependentRectangle(rectPtr, refPtr)
RECTANGLE_PTR rectPtr;
REFERENCE_POINT_ER refPtr;
{
RECTANGLE_CONS_PTR rectConsPtr;

if (GeneralPointerFind(refPtr->dependentRectangles, rectPtr))
    return;	/* it's already linked on. */

rectConsPtr = (RECTANGLE_CONS_PTR) GetCons();
if (rectConsPtr == NIL)
    return;	/* out of free storage */
rectConsPtr -> next = refPtr -> dependentRectangles;
rectConsPtr -> value = rectPtr;
refPtr -> dependentRectangles = rectConsPtr;
}

 /* ********************************************** */
 /* 		UnLinkDepInstance:		   */
 /* ********************************************** */

/* This routine is very similar to (but simpler than)
 * UnLinkDepRectangle, below.
 */

UnLinkDepInstance(instPtr, refType)
short refType;	/* Horizontal or Vertical */
CELL_INSTANCE_PTR instPtr;
{
REFERENCE_POINT_ER horizRef, vertRef, FindInvolvedRefPt ();

vertRef = FindInvolvedRefPt(instPtr->trans.x31, 0);
horizRef = FindInvolvedRefPt(instPtr->trans.x32, 0);

if (refType == Horizontal && horizRef != vertRef)
    if (GeneralPointerFind(horizRef->dependentInstances, instPtr))
	GeneralConsRemove(&(horizRef->dependentInstances), instPtr);
if (refType == Vertical && horizRef != vertRef)
    if (GeneralPointerFind(vertRef->dependentInstances, instPtr))
	GeneralConsRemove(&(vertRef->dependentInstances), instPtr);
}

 /* ********************************************** */
 /* 		UnLinkDepRectangle:		   */
 /* ********************************************** */

/* This routine takes a rectangle pointer and an edge specification,
 * and makes sure that the rectangle is not unnecessarily consed onto any
 * dependency list for a reference point.
 */

UnLinkDepRectangle(rectPtr, edge)
RECTANGLE_PTR rectPtr;
short edge;
{
REFERENCE_POINT_ER refPtrTop, refPtrBottom, refPtrRight, refPtrLeft,
		   FindInvolvedRefPt ();

refPtrTop = (rectPtr->t)/*.postfix*/ ? FindInvolvedRefPt(rectPtr->t, 0) : NIL;
refPtrBottom = (rectPtr->b)/*.postfix*/ ? FindInvolvedRefPt(rectPtr->b, 0) : NIL;
refPtrLeft = (rectPtr->l)/*.postfix*/ ? FindInvolvedRefPt(rectPtr->l, 0) : NIL;
refPtrRight = (rectPtr->r)/*.postfix*/ ? FindInvolvedRefPt(rectPtr->r, 0) : NIL;

switch (edge)
    {
    case LeftEdge:
	if (refPtrLeft == refPtrRight ||
	    refPtrLeft == refPtrTop ||
	    refPtrLeft == refPtrBottom)
		return;	/* it's linked elsewhere */
	if (GeneralPointerFind(refPtrLeft->dependentRectangles, rectPtr))
	    GeneralConsRemove(&(refPtrLeft->dependentRectangles),rectPtr);
	break;
    case RightEdge:
	if (refPtrRight == refPtrLeft ||
	    refPtrRight == refPtrTop ||
	    refPtrRight == refPtrBottom)
		return;	/* it's linked elsewhere */
	if (GeneralPointerFind(refPtrRight->dependentRectangles, rectPtr))
	    GeneralConsRemove(&(refPtrRight->dependentRectangles),rectPtr);
	break;
    case TopEdge:
	if (refPtrTop == refPtrBottom ||
	    refPtrTop == refPtrRight ||
	    refPtrTop == refPtrLeft)
		return;	/* it's linked elsewhere */
	if (GeneralPointerFind(refPtrTop->dependentRectangles, rectPtr))
	    GeneralConsRemove(&(refPtrTop->dependentRectangles),rectPtr);
	break;
    case BottomEdge:
	if (refPtrBottom == refPtrTop ||
	    refPtrBottom == refPtrLeft ||
	    refPtrBottom == refPtrRight)
		return;	/* it's linked elsewhere */
	if (GeneralPointerFind(refPtrBottom->dependentRectangles, rectPtr))
	    GeneralConsRemove(&(refPtrBottom->dependentRectangles),rectPtr);
	break;
    }
}

