/*	visual.c: All routines involved in changing the display and	*/
/*	the row descriptors.  The screen and the row descriptors are	*/
/*	always changed closely in parallel.  Only the chunk and cp	*/
/* 	fields are changed anywhere else than in this file.		*/

/* Row descriptors are maintained in an array of Nrows+1 records, one	*/
/* for each row on the screen and one for the next row below the screen.*/
/* They contain:							*/
/* 	chunk and cp	the Mark of the first character in this row	*/
/*	length		length of the row in SCREEN characters		*/
/*	exists		0 when row is beyond end of file, else 1	*/
/*	continues	this row is a long line continuation		*/
/*	continued	this line is continued onto the next row	*/
/* Note that the "length" and "continued" fields of the rows[Nrows]	*/
/* record ARE NOT MAINTAINED and tend to have random values.  Alse note	*/
/* that rows[0].continues is never true: ved guarantees that the screen	*/
/* starts at the start of a logical line.				*/


#include "vedit.h"

OKCR = 1; 	      /* controls displaying: if off, inhibits carriage
			 return to prevent scrolling of the last row */

/* Display: Displays a screenful of text starting at m.  Generates	*/
/* the complete row descriptor information, overwriting any that might	*/
/* have been there before.  In general, wherever there is displaying,	*/
/* there is fiddling with the row descriptors, and nowhere else (except */
/* to update chunk and cp when the text is altered).			*/
Display(m)  Mark m;
{ register int row=0, col=0, eof=0;
  register Rowrec *r;
  register char ch;
  register int prevcol;
  Mark backmark;

  for (r = rows; r <= rows+Nrows; r++)
    r->exists = r->continues = r->continued = 0;
  r = rows;
  r->chunk = m.chunk;  r->cp = m.cp;  r->exists = 1;
  PadClear(pad);

  if (!Atend(m)) {
    while(1) {
      prevcol = col;
      ch = *m.cp++;
      eof = Advance(&m);		/* take care of chunk-crossings */	
      col = ShowChar(ch,col); 	/* display, update col */
      if (col <= 0) {
	  r->length = prevcol;
	  r++; row++;
	  r->exists = 1;
	  if (col < 0) {  /* overrunning line */
		(r-1)->continued = 1;
		r->continues = 1;
		backmark = m;
		Retract(&backmark);
		r->chunk = backmark.chunk;  r->cp = backmark.cp;
		if (row == Nrows) FinalArrow(pad);
		}
	  else {
		r->chunk = m.chunk;  r->cp = m.cp;
		}

	  col = -col;
	  if (row == Nrows - 1)  OKCR = 0;
	  else if (row >= Nrows)  break;
/*^*/myprint(1)("line finished: row %d",row);
	  }
      if (eof)  {r->length = col;  break;}
      }  /* end while(1) */
    }  /* end if !Atend */
/*^*/myprint(1)("Display returning\n");
  OKCR = 1;
  Banner();	/* rewrite the banner which tells the filename */
  }


/* DispRow: display one row, starting at a given mark, at the specified */
/* screen row.  Updates that row's data and the next row's start 	*/
/* chunk and cp.  Leaves the cursor at the end of the area displayed	*/
/* (and ready to overrun if this is an overrunning line), so the caller	*/
/* may need to return it to the proper point.				*/
Mark DispRow(m,row)  Mark m;  register int row;
{ register int col=0, eof=0, prevcol;
  char ch;

/*^*/myprint(0x400)("Disprow (%x, %x), %d  ", m.chunk, m.cp, row);
  rows[row].chunk = m.chunk;  rows[row].cp = m.cp;
  rows[row].exists = 1;
  OKCR = 0;
  do {
    prevcol = col;
    ch = *m.cp++;
    eof = Advance(&m);
    col = ShowChar(ch,col);
    }  while(col>0 && !eof);
  if (col > 0) {	/* stopped because end of file */
    prevcol = col;	/* make sure to count the final character */
    PadClearToEOL(pad);
    }
  else if (col == 0) {
    if (row < Nrows-1) PadCRLF(pad);
    else PadClearToEOL(pad);
    rows[row].continued = rows[row+1].continues = 0;
    }
  else {
    Retract(&m);
    if (row == Nrows-1) FinalArrow(pad);
    rows[row].continued = rows[row+1].continues = 1;
  /*^*/ myprint(0x400)("DispRow overrun on row %d, col=%d\n",row, col);
    }
  rows[row].length = prevcol;
  rows[row+1].chunk = m.chunk;  rows[row+1].cp = m.cp;
  /* next row exists unless we stopped with eof at mid-row */
  rows[row+1].exists = (col > 0) ? 0 : 1;
  OKCR = 1;
/*^*/myprint(0x400)("returns (%x, %x)\n", m.chunk, m.cp);
  return(m);
  }

/* ShowChar: given a char and the current column, display it and return the
   resulting column.  Column==0 means new line; column<0 means this is a
   char which displays as multiple chars and didn't fit; it had to be
   displayed on the next line. */
int ShowChar(ch, col)
  unsigned /* it matters! */ char ch;
  register int col;
{
  register int csize, i, oldcol;

  if (ch == '\n')  {
    if (OKCR) PadCRLF(pad);
    return(0);
    }
  else if (ch == '\t') {
    if (col == Ncols) {
	col = -8;
	if (!OKCR) return(col);
	csize = 8;
	}
    else {
	csize = 1 + (col | 07) - col;
	if (col + csize > Ncols) {
	    col = Ncols;
	    csize = Ncols - col;
	    }
	else col += csize;
	}
    for (i=0; i<csize; i++)  PadSend(pad,' ');
    }
  else  { /* size and content of each char's display is from a table */
    csize = charsize[ch];
    oldcol = col;
    col += csize;
    if (col > Ncols) {
	for (i=oldcol; i<Ncols; i++) PadSend(pad,' ');
	if (!OKCR) return(-csize);
	}
    for (i=0; i<csize; i++) {  PadSend(pad,chartab[ charpt[ch] + i] );
	if (debug&1) putchar(chartab[ charpt[ch] + i] );
	}
    }

  if (col <= Ncols)  return(col);
  else
  return(-csize);
  }

/* CountWidth: the counting aspect of ShowChar without any displaying */
int CountWidth(ch, col)  unsigned char ch;  int col;
{
  if (ch == '\n') return(0);
  else if (ch == '\t') {
    if (col == Ncols) return(-8);
    col = 1 + (col | 07);
    if (col > Ncols) col = Ncols;
    }
  else col += charsize[ch];

  if (col > Ncols) col = - (int)(charsize[ch]);
  return(col);
  }


/* Advance: given a Mark which has just been incremented, take care of 
   the chunk-boundary special case and report end of buffer if it happens */
int Advance(mp)  register Mark *mp;
{
  if (mp->cp >= Chunkend(mp->chunk) ) {
    if (mp->chunk->next == NULL)
	  return(-1);
    mp->chunk = mp->chunk->next;
    mp->cp = mp->chunk->text;
    if (mp->chunk->length == 0) return(-1);
    }
  return(0);
  }

/* Retract: move a Mark back one character.  Not symmetric to Advance, in
   that this one actually does the decrementing.  */
int Retract(mp)  register Mark *mp;
{
  if (mp->cp == mp->chunk->text) {
    if (mp->chunk->prev == NULL)  return(-1);
    mp->chunk = mp->chunk->prev;
    mp->cp = Chunkend(mp->chunk) - 1;
    }
  else mp->cp--;
  return(0);
  }


/* Setcursor: presumes the caller knows both the position and the mark of the
   desired cursor.  If not so, the functions PosSetcursor, MarkSetcursor, and
   LineSetcursor will handle the necessary preprocessing. */
Setcursor(mark, pos)  Pos pos;  Mark mark;
{
  curpos = pos;
  curmark = mark;
  PadPosition(pad, curpos.row, curpos.col);
/*^*/myprint(0x2)("Setcursor (%x, %x), position (%d, %d)\n",
	mark.chunk, mark.cp, curpos.row, curpos.col);
  }

PosSetcursor(pos)  Pos pos;
{ Mark mark;
  mark = Posmark(&pos);
  Setcursor(mark, pos);
  }

/* DispSetcursor: given a mark, which need not be on the screen, get it
   on the screen and set the cursor to it.  If it is not on the screen,
   no efficiencies are used: Display is called.  It is a modified form
   of Markpos.  */
DispSetcursor(mark)  Mark mark;
{ Pos p;
  Mark head;
  register int row;
  register Chunk chunk;

  row = 0;
  /* find first row containing our chunk */
  for (chunk = rows[0].chunk; chunk && chunk != mark.chunk && row < Nrows;
	chunk = chunk->next)
    while (chunk == rows[row].chunk && rows[row].exists && row < Nrows) row++;
/*^*/ myprint(0x2)("DispSetc 1: row = %d   ", row);
  if (chunk != mark.chunk) goto Noton;/* found eof without finding our chunk */

  /* find our row */
  if (row < Nrows) {
    if (mark.cp < rows[row].cp) row--;
    else while (mark.chunk == rows[row+1].chunk && mark.cp >= rows[row+1].cp &&
		rows[row+1].exists)
        row++;
    }
  p.row = row;
/*^*/ myprint(0x2)("DispSetc 2: row = %d\n", row);

  if (p.row < Nrows /* && chunk != NULL */) {
    p.col = Markcol(row,mark);
    Setcursor(mark, p);
    }
  else {
Noton:
    head = Backrows(mark, Nrows/2);
    Display(head);
    MarkSetcursor(mark);
    }
  wishcol = curcol;
  }


/* Scroll: scroll the screen up one LINE - not ROW!  The topmost logical line
   disappears above, and below we add whatever will fit.  If the cursor would
   disappear, we move it downward to the earliest line where it will fit.   */
Scroll()
{ int i,n;
  register Rowrec *r, *r2;
  n = 0;
  for (r = rows; (r->continued) && (r < rows+Nrows); r++) n++;
  r++; n++;
  currow -= n;
  if (currow > 255) {  /* unsigned for < 0 */
    if (selectionexists) Deselect();
    PosSetcursor(Makepos(n,0));
    currow = 0;
    }
/* shift row descriptors */
  for (r2 = rows; r <= rows+Nrows; r++, r2++)  *r2 = *r;
/* shift the actual lines on the screen */
  for (i=n; i>0; i--) PadIndex(pad,0,Nrows-1);
/*^*/ myprint(0x400)("Scrolled %d rows  ",n);
  ScrollAdjust(n);
  }

/* ScrollAdjust: after a scroll or line deletion, displays new material at
   the bottom and adjusts row descriptors.  n is the number
   of rows of material to be displayed.  */
ScrollAdjust(n)  int n;
{ int i;
  Mark m;

  if (rows[Nrows].exists) {
    m = Rowmark(Nrows);	/* former end of page */
    PadPosition(pad, Nrows-n, 0);
    for (i=Nrows-n; i<Nrows && !Atend(m); i++)
        m = DispRow(m,i);  /* figures out existence of row i+1 */
    }
  else  i = Nrows - n;
  if (Atend(m))	 /* if end of file is on screen */
    for (i++; i<=Nrows; i++) rows[i].exists = 0;
  PadPosition(pad,currow,curcol);
  }


/* Backscroll: scroll the screen down one line.  A new line is added at the */
/* top; we scroll down however far it takes to fit it.  If the cursor is    */
/* going to vanish, we move it upward to the bottom row that remains.	    */
Backscroll()
{ Mark head, m;
  register int row;
  register Rowrec *r,*r2;

  head = Rowmark(0);
  if (MarkEQ(head,headmark)) return;
  m = Backrows(head,1);	/* back exactly 1 logical line */
  row = 0;
  PadPosition(pad, 0, 0);
  do {
    PadReverseIndex(pad, row, Nrows-1);
    for (r=rows+Nrows, r2=rows+Nrows-1; r2 >= rows+row; r--, r2--) *r = *r2;
    m = DispRow(m, row);
/*^*/myprint(0x400)("just did row %d, new m=(%x, %x)\n", row, m.chunk,
	m.cp);
    row++;  currow++;
    }
   while (MarkNEQ(m, head) && row<Nrows);
  if (currow >= Nrows)  {
    if (selectionexists) Deselect();
    PosSetcursor(Makepos(Nrows-1,0));
    }
  PadPosition(pad,currow,curcol);
  } /* comment */
/* no comment */


/* The display adjustment system:  Before doing an insertion, deletion,
   or replacement, call PreAdjust on the point at the END of the affected
   section.  Eolpmark must point to a mark on the Marklist, so it will
   survive rearrangement of the text.  Also store the Pos of the beginning
   of the affected region.  Sample call:
	PreAdjust(curmark, curpos, &eolmark, &eolpos);
   After the textual change is made, call AdjustDisplay with the mark of
   the (new) BEGINNING of the affected region, the mark from eolpmark above,
   the stored Pos of the beginning (which we presume is unchanged), and the
   Pos from eolppos of PreAdjust.  Both Pos's are passed by pointer and are
   modified.  The first is rarely modified: only in the case of an overrun
   line which ceases to overrun.  Sample call:
	AdjustDisplay(premark, eolmark, &prepos, &eolpos);
   The eolmark, as generated by PreAdjust, points just BEFORE the \n ending
   the line, not after it.
	If the region being deleted extends off the end of the screen, 
   eolpos stops on row Nrows and eolmark stops accordingly.  AdjustDisplay
   specially detects when eolpos.row==Nrows and in such cases keeps re-
   displaying to the end of the screen, regardless. 
 */


PreAdjust(m, pos, eolpmark, eolppos)
 Mark m;	/* the current cursor or action point */
 Pos pos;	/* its screen position */
 Mark *eolpmark; /* result: the end of this line - BEFORE the \n */
 Pos *eolppos;	 /* result: its screen position */
{ register int row = pos.row,  col = pos.col;
  register char ch;

  ch = *m.cp;
  if (row > Nrows) row = Nrows;
  while ((ch != '\n') && (!Atend(m)) && (row < Nrows)) {
    m.cp++;
    Advance(&m);
    col = CountWidth(ch,col);
    if (col <= 0)  { col = -col;  row++;}
    ch = *m.cp;
    }
  *eolpmark = m;
  eolppos->row = row;  eolppos->col = col;
  }


AdjustDisplay(bmark, eolmark, bppos, eolppos)
 Mark bmark;	/* Beginning of affected region */
 Mark eolmark;	/* End of line containing the end of affected region */
 Pos *bppos;	/* Former position of bmark - may be updated */
 Pos *eolppos;	/* Former position of eolmark - will be updated */
{ register int row = bppos->row,
	       col = bppos->col;
  register char ch, prevch;
  Mark m;
  Pos mypos;	/* A receiver for the value from Markpos */
  register int n;
  register Rowrec *r;
  register int prevcol;
  int fixbppos = 0;	/* a flag */

/*^*/ myprint(0x400)("AdjustD(%x, %x, (%d,%d), (%d,%d))\n",
	bmark.cp, eolmark.cp, bppos->row,bppos->col, eolppos->row,
	eolppos->col);
  m = bmark;
  ch = *m.cp;
  n = Retract(&m);	/* n=-1 if there is nowhere to retract to */
  prevch = *m.cp;
  r = rows+row;

  /* Here we test for a "perched" situation: bppos is at the beginning of
     an overrun which may no longer overrun.  This is the only case where
     bppos gets changed.  We back off and find the position of the previous
     character, and start redisplaying from there.  fixbppos is a signal
     that after displaying one character we should update *bppos to the
     position we find at that point.  */
  if ((col == 0) && (n == 0) && (prevch != '\n')) {
/*^*/myprint(0x400)("perched:  ");
    mypos = Markpos(m);
    row = mypos.row;
    col = mypos.col;
    r = rows + row;
    fixbppos = 1;
    }

/* Here we test the opposite: bppos is at the end of a full line to which
     an overrun has newly been added. */
  else if (col + charsize[ch] > Ncols) {
    r->continued = (r+1)->continues = 1;
    bppos->row = row + 1;
    bppos->col = 0;
    if (row >= Nrows) {
	FinalArrow(pad);
	r->chunk = bmark.chunk;  r->cp = bmark.cp;
	goto Bye;
	}
    m = bmark;
    }
  else m = bmark;

  PadPosition(pad, row, col);
  if (col == 0) {r->chunk = m.chunk;  r->cp = m.cp;}
  OKCR = 0;	/* don't let ShowChar handle end of line, I need to. */
  if (eolppos->row == Nrows) eolmark = endmark;  /* go till the end
						of the page stops us */

 /* the display loop - may happen 0 times.  Inserts lines as needed. */
  while (MarkNEQ(m,eolmark)) {
    prevcol = col;
    ch = *m.cp++;
    Advance(&m);
    col = ShowChar(ch, col);
   /* test for end of physical line */
    if (col <= 0) {
	r->length = prevcol;
	row++; r++;
	if (col < 0) {
		(r-1)->continued = r->continues = 1;
		Retract(&m);
		}
	else  (r-1)->continued = r->continues = 0;
	
       /* test for unusual termination: reached end of screen */
	if (row >= Nrows) {
	    r->chunk = m.chunk;  r->cp = m.cp;  r->exists = 1;
	    if (col < 0) FinalArrow(pad);
	    else PadClearToEOL(pad);
	    goto Eop;
	    }
	PadCRLF(pad);
       /* insert a screen line if necessary */
	if (row > eolppos->row)  InsertRowHere(row);
	if (col < 0) PadPosition(pad, row-1, Ncols);
	r->chunk = m.chunk;  r->cp = m.cp;
	r->exists = 1;
      /*^*/myprint(0x400)("\nNew row %d at cp %x, col=%d\n",row,m.cp, col);
	col = 0;
	}  /* end if col <= 0 */
    if (fixbppos)  fixbppos = Fixpos(bppos, m, row, col);
/*^*/ myprint(0x400)("'%c'->cp %x pos (%d,%d)   ", ch,m.cp,row,col);
    }  /* end while MarkNEQ(m,eolmark) - the display loop */

  PadClearToEOL(pad);
Eop:
  if (fixbppos) fixbppos = Fixpos(bppos, m, row, col);
  r->length = col;
  if (MarkEQ(m,eolmark))  r->continued = (r+1)->continues = 0; 

 /* if we have fallen short, delete lines and display new text below */
  if (row < eolppos->row) {
    if (row < Nrows-1) 	PadCRLF(pad);
    DeleteRowsHere(row+1, eolppos->row - row);
    ScrollAdjust(eolppos->row - row);
    }

Bye:
  eolppos->row = row;
  eolppos->col = col;
  OKCR = 1;
  }


/* Fixpos: a subroutine of AdjustDisplay.  Sets the pos to (row, col),	*/
/* advanced to the next row if it is not a valid pos on this row.	*/
int Fixpos(ppos, m, row, col)
  Pos *ppos;
  Mark m;
  int row, col;
{
  int csize;

  if (Atend(m)) csize = 0;
  else csize = charsize[*m.cp];

  if (col + csize > Ncols) {col = 0; row++;}
  ppos->row = row;
  ppos->col = col;
  return(0);	/* to zero out fixbppos */
  }

/* InsertRowHere: Inserts a row at "row", which must be the row where 	*/
/* the pad's cursor is currently located.  The screen and the row	*/
/* descriptors are affected in parallel.  The new row descriptor is a	*/
/* copy of the one which got pushed downscreen - some other subroutine	*/
/* will have to modify it.						*/
InsertRowHere(row)  int row;
{ register Rowrec *r, *r2;

  PadReverseIndex(pad,row,Nrows-1);
  for (r=rows+Nrows, r2=rows+Nrows-1; r2 >= rows+row; r--, r2--) *r = *r2;
/*^*/ myprint(0x400)("Inserted row at %d\n",row);
  }

/* DeleteRowsHere: Deletes n rows at "row", which must be the pad cursor*/
/* row, as above.  Screen and row descriptors are affected in parallel.	*/
/* Row descriptors are only moved, not altered to agree with the text.	*/
/* The row descriptors at the bottom which are vacated by the move are	*/
/* unchanged.								*/
DeleteRowsHere(row, n)  int row, n;
{ register Rowrec *r, *r2;
  register int i;

  for (i=0; i<n; i++) PadIndex(pad,row,Nrows-1);
  for (r=rows+row, r2=r+n; r2 <= rows+Nrows; r++, r2++) *r = *r2;
/*^*/ myprint(0x400)("Deleted %d rows at %d\n", n, row);
  }


/* FinalArrow: make an overrun arrow appear on the last row of text,	*/
/* which requires writing into the banner line.				*/
FinalArrow(pad)  File *pad;
{
  PadSend(pad, ' ');
  PadCursorBackward(pad);
  PadInverse(pad);
  PadSend(pad, ' ');
  PadNormal(pad);
  }

