/*	file.c:		Functions involving remote file reading and 	*/
/*	writing.  Functions for killbuffer and straytext operations,	*/
/*	and buffer initialization.					*/

#include "vedit.h"
#include <Vioprotocol.h>
#include <Vnaming.h>
#include <Vdirectory.h>

#define MAX_FILENAME_WORD_LEN	255	/* maximum length of any "word" (ie. */
					/* between "/"s) in a file name      */

extern struct ckpdesc *CkpNames;	/* head of ckp file name list */
extern char *ErrorString();
extern SystemCode GetContextName(), GetFileName();
extern File *Open();
extern char *CurrentContextName;

static char iobuffer[2000];
Mark readfile_end;  /* a side-effect output: sometimes wanted, sometimes not*/

int VerifyWriteOption = 1;	/* Signals that writes should be read back
				   in afterwards to verify that they were
				   done correctly. */

/* RJN - 8/12/83 - created to handle absolute path names. */


File *OpenName( AbsoluteName, RelativeName )  
    char AbsoluteName[], RelativeName[];
/*
 * Tries to open "newname", if successful it does a GetFileName on the instance
 * NOTE: this routine presumes that both of the Names may be mangled, ie.
 * 	 it uses RelName for a scratch area.  Also note that they both must
 * 	 be at least MAX_FILE_NAME_LEN in size.
 */
  {
    File *FileDesc;
    SystemCode error;
    ProcessId ServerPid;
    ContextId Context;
    int AbsoluteNameLen, RelativeNameLen;
    ContextPair dummy;    

    /* open and verify existence of the file */
    FileDesc = Open(RelativeName, FREAD | FBLOCK_MODE, &error);
    /*^*/ myprint(0x20)("Open of %s returned %x  error=%x\n",RelativeName,FileDesc,error);
    if (error || (FileDesc==NULL))
      {
	NewMsg("can't read ");
	Msg(": " );
	Msg(RelativeName);
	Msg("  --  ");
	Msg(ErrorString(error));
	/* The file doesn't exist, but the user may want to create it,
	 *  so ask the server to predict what its absolute name will
	 *  be if it is created.
	 */
	strcpy(AbsoluteName, RelativeName);
	error = GetAbsoluteName(AbsoluteName, MAX_FILE_NAME_LEN, &dummy);

	if ( error != OK )
	    AbsoluteName[0] = '\0';

	return( NULL );
      }
    
    RelativeNameLen = MAX_FILE_NAME_LEN;
    ServerPid = FileDesc->fileserver;

    /* get the file name */
    if ( (error = GetFileName( AbsoluteName, MAX_FILE_NAME_LEN, ServerPid,
		    		      FileDesc->fileid )) != OK )
      {
	/* failed to get name */
	NewMsg( RelativeName ); 
	Msg( ": failed to get absolute pathname: " ); 
	Msg( ErrorString( error ) );
	NewMsg( "(Using relative name)" );
	strcpy(AbsoluteName, RelativeName);
      }

    return( FileDesc );

  } /* OpenName */

Chunk  ReadFile(fileid)  File *fileid;
{ 
  Chunk chunk,headchunk;
  register char *cp,*ecp,*bufcp,*bufecp;
  int i, n, blocksize, stop, chunkspace;
  int alldone = 0;

  if ( fileid == NULL ) return( NULL );

  blocksize = fileid->blocksize;
  n = Read(fileid, iobuffer, blocksize);
  fileid->block++;
  bufcp = iobuffer;  bufecp = iobuffer + n;

  headchunk = chunk = NewChunk();
  if (chunk == NULL) goto Nomem;
  chunk->prev = chunk->next = NULL;
  chunk->length = CHUNKFILLPOINT;
  cp = chunk->text;  ecp = cp + CHUNKFILLPOINT;

  while (1) {
    stop = bufecp - bufcp;  chunkspace = ecp - cp;
    if (stop > chunkspace) stop = chunkspace;
  /* tight loop: vast speed advantage */
    for (i = 0; i < stop; i++) *cp++ = *bufcp++;

    if (bufcp == bufecp) {
	if (alldone) {
	    readfile_end = Makemark(chunk,cp);
	    chunk->length = cp - chunk->text;
	    return(headchunk);
	    }
	n = Read(fileid, iobuffer, blocksize);
	fileid->block++;
	if (n < blocksize && fileid->lastexception != OK) {  /* end of file */
	    if (fileid->lastexception != END_OF_FILE)
		printf("\nError in reading: %s\n",
		    ErrorString(fileid->lastexception));
	    Close(fileid);
	    alldone = 1;
	    }
	bufcp = iobuffer;  bufecp = iobuffer + n;
	}

    if (cp == ecp) {
	chunk->next = NewChunk();
	if (chunk->next == NULL) goto Nomem;
	chunk->next->prev = chunk;
	chunk = chunk->next;  cp = chunk->text;
	chunk->length = CHUNKFILLPOINT;
	ecp = cp + CHUNKFILLPOINT;
	}
    }

  /* this is what we do if we run out of memory */
Nomem:
  if (headmark.chunk)  {
    FreeText(headmark.chunk->next);
    NewMsg("File too large!");
    chunk = headmark.chunk;
    chunk->next = NULL;
    chunk->length = 0;
    return(chunk);
    }
  else  return(MakeChunk(NULL, 0));
  }  /* end of ReadFile */


/*
 * ToggleVerifyWrite:
 * Toggles the VerifyWriteOption flag.
 */
ToggleVerifyWrite(ch)
    int ch;
  {
    VerifyWriteOption ^= 1;
    if (VerifyWriteOption)
      {
	NewMsg("Verify-write option on");
      }
    else
      {
	NewMsg("Verify-write option off");
      }
  }


int NCopyProtectionMode(fromfilename, tofile)
    char *fromfilename;
    File *tofile;
    /* (Attempts to) set the protection mode of the file instance "tofile" to
     * be the same as that of the file named "fromfilename".  This is done by
     * reading/writing descriptors.  0 is returned iff this fails.
     */
  {
    Message msg;
    register DescriptorRequest *req = (DescriptorRequest *)msg;
    register DescriptorReply *rep = (DescriptorReply *)msg;
    static struct
      {
	ArbitraryDescriptor fromDescriptor;
	char name[sizeof(ArbitraryDescriptor)]; /* as big as possible */
      } fromSegment;
    ArbitraryDescriptor toDescriptor;

    strcpy(fromSegment.name, fromfilename);
    req->requestcode = NREAD_DESCRIPTOR;
    req->nameindex = ((int)&(fromSegment.name[0])) - (int)(&fromSegment);
    req->dataindex = 0;
    req->segmentptr = (char *)&fromSegment;
    req->segmentlen = sizeof(fromSegment);
    NameSend(req);
    if (rep->replycode != OK)
	return (0);

    req->requestcode = READ_DESCRIPTOR;
    req->fileid = tofile->fileid;
    req->dataindex = 0;
    req->segmentptr = (char *)&toDescriptor;
    req->segmentlen = sizeof(toDescriptor);
    Send(req, tofile->fileserver);
    if (rep->replycode != OK)
	return (0);

    if (fromSegment.fromDescriptor.e.descriptortype
	  != toDescriptor.e.descriptortype)
	return (0);

    switch (fromSegment.fromDescriptor.e.descriptortype)
      {
	case FILE_DESCRIPTOR:
	    toDescriptor.f.perms = fromSegment.fromDescriptor.f.perms;
	    break;

	case UNIXFILE_DESCRIPTOR:
	    toDescriptor.u.st_mode = fromSegment.fromDescriptor.u.st_mode;
	    break;

	default:
	    return (0);
      }

    req->requestcode = WRITE_DESCRIPTOR;
    req->fileid = tofile->fileid;
    req->dataindex = 0;
    req->segmentptr = (char *)&toDescriptor;
    req->segmentlen = sizeof(toDescriptor);
    Send(req, tofile->fileserver);
    if (rep->replycode != OK)
	return (0);

    return (1);
  }


int WriteProtFile(fname, start, end, protfname)
    char *fname, *protfname;
    Mark start, end;
    /* Implements WriteFile() (see below).  The string "protfname", if
     * non-NULL, is the name of a file whose protection mode should be given 
     * to the new file.  (If NULL, then the file is created with the default 
     * protection.)  Note, however, that if "backupoption" is set, then
     * "protfname" is reset inside this routine.
     */
  {
    File *fileid;
    Chunk chunk;
    register char *cp, *bp;
    register int i, k, n;
    SystemCode err;
  
    /* return to editor pad now, because writing takes a long time */
    NewMsg("Writing ");  Msg(fname);  Msg(" ... ");
    if (incommandwindow)  {SelectPad(pad);  incommandwindow = 0;}

    /* First make sure that we are able (allowed) to write the file. */
    fileid = Open(fname, FMODIFY, &err);
    switch (err)
      {
	case NOT_FOUND:
	    break;

	case OK:
	    Close(fileid);
	    if (backupoption) 
	      {
		Msg("Postponed to Back Up.");
		if (!Makebak(fname, &protfname))
		  {
		    NewMsg("Did not write ");  Msg(fname);  Beep();
		    return(0);
		  }
		NewMsg("Resuming writing ");  Msg(fname);  Msg(" ... ");
	      }
	    break;

	default:
	    NewMsg("Can't write ");  Msg(fname);  
	    Msg(" -- ");  Msg(ErrorString(err));
	    Beep();
	    return(0);
      }

    /* Now write out the new file. */
    fileid = Open(fname, FCREATE, &err);
    if (fileid == NULL)
      {
	NewMsg("Can't create ");  Msg(fname);  
	Msg(" -- ");  Msg(ErrorString(err));
	Beep();
	return(0);
      }
    if (protfname != NULL)
	NCopyProtectionMode(protfname, fileid);
  
    chunk = start.chunk;  
    while (chunk != end.chunk)
      {
	if (chunk == start.chunk)
	  {
	    i = chunk->length - (start.cp - chunk->text);
			  /* start.cp is current memory position, chunk->text
			   * is the start of the buffer region 
			   */
	    cp = start.cp;
	  }
	else
	  {
	    i = chunk->length;
	    cp = chunk->text;
	  }
	n = fwrite(cp, i, 1, fileid);
	if (ferror(fileid))
	  {
	    NewMsg("ERROR in fwrite: ");
	    Msg(ErrorString(fileid->lastexception));
	    Beep();
	    return(0);
	  }
	chunk = chunk->next;
      }
  
      if (chunk == start.chunk)
	  cp = start.cp;
      else
	  cp = chunk->text;
      i = end.cp - cp;
      n = fwrite(cp, i, 1, fileid);
      if (ferror(fileid))
	{
	  NewMsg("ERROR in fwrite: ");
	  Msg(ErrorString(fileid->lastexception));
	  Beep();
	  return(0);
	}
  
    fclose(fileid);
    Msg("done.");
  
    if (VerifyWriteOption)
      {
	NewMsg("Verifying write ... ");
	fileid = fopen(fname, "r");
	if (fileid == NULL) {
	  NewMsg("can't write ");  Msg(fname);  Beep();
	  return(0);
	  }
  
	chunk = start.chunk;  
	while (chunk != end.chunk)
	  {
	    if (chunk == start.chunk)
	      {
		i = chunk->length - (start.cp - chunk->text);
			      /* start.cp is current memory position
			      /* chunk->text is the start of the buffer region 
			       */
		cp = start.cp;
	      }
	    else
	      {
		i = chunk->length;
		cp = chunk->text;
	      }
	    n = fread(iobuffer, i, 1, fileid);
	    if (ferror(fileid))
	      {
		NewMsg("ERROR in fread: ");
		Msg(ErrorString(fileid->lastexception));
		Beep();
		return(0);
	      }
	    bp = iobuffer;
	    for (k = 0; k < i; k++)
	      {
		if (*cp++ != *bp++)
		  {
		    NewMsg("ERROR: file was not written back correctly!");
		    Beep();
		    return(0);
		  }
	      }
	    chunk = chunk->next;
	  }
  
	  if (chunk == start.chunk)
	      cp = start.cp;
	  else
	      cp = chunk->text;
	  i = end.cp - cp;
	  n = fread(iobuffer, i, 1, fileid);
	  if (ferror(fileid))
	    {
	      NewMsg("ERROR in fread: ");
	      Msg(ErrorString(fileid->lastexception));
	      Beep();
	      return(0);
	    }
	    bp = iobuffer;
	    for (k = 0; k < i; k++)
	      {
		if (*cp++ != *bp++)
		  {
		    NewMsg("ERROR: file was not written back correctly!");
		    Beep();
		    return(0);
		  }
	      }
  
	fclose(fileid);
	Msg("done.");
      }
    return(1);
  }

/* WriteFile: given an ordered pair of marks and a filename, write the
   text between the marks as the named file.  */
/* Useful if one wants to write out a file that is the internal buffer. */
int WriteFile(fname, start, end, protfname)
    char *fname, protfname;
    Mark start, end;
  {
    WriteProtFile(fname, start, end, NULL);
  }


/* WriteWithAsk: write out the whole buffer to the stored "filename", but if
   that filename is empty then ask for one.  Return value 1 signifies normal,
   0 signifies abort by ^G. */
int WriteWithAsk()
{ int n;

  if (filename[0] == '\0') {
    NewMsg("File name, please: ");  n = Getline(linebuf,72);
    if (n < 0) return(0);
    else if (n == 0) return(1);
    else
      {
	ckpmodified = modified = !WriteFile(linebuf, headmark, endmark);
	if ( !modified )	/* if written get the absolute name */
	   Close( OpenName( filename, linebuf ) );
	else
	   strcpy( filename, linebuf );
      }
    }
  else
    {
      ckpmodified = modified = !WriteFile( filename, headmark, endmark );
    }

  Banner();
  return(1);
  }

/* SaveIfModified: protection against loss of a modified buffer.  If the
   modified bit is on, asks the user whether to save the file.  If user
   responds with ^G, SaveIfModified returns 0, meaning that the calling
   action should abort.  Otherwise it does a WriteWithAsk - write the buffer,
   asking for a filename if necessary.  */
int SaveIfModified()
{ int n;

  if (!modified) return(1);
  else {
    NewMsg("Buffer is modified: save it? ");
    n = Yesno();
    if (n<0) n = 0;				/* ^G abort */
    else if (n > 0)  n = WriteWithAsk();	/* yes */
    else n = 1;					/* no */
    return(n);
    }
  }


RenameFile(fromName, toName)
    char *fromName, *toName;
/*
 * Renames "fromName" as "toName".
 * Returns 0 if unsuccessful, 1 otherwise.
 * NOTE: If the fromName file doesn't exist then no rename is needed and
 * 1 is returned.
 *
 * CRZ 5/16/85 Added messages to tell the user what is going on.
 *
 */
  {
    SystemCode result;

    NewMsg("Renaming file as "); Msg(toName);  Msg(" ... ");

    /* The following code was copied from /xV/cmds/rename/mi/rename.c.
     * This should really be made into a library routine.
     */
      {
	Message msg;
	NameRequest *req = (NameRequest *) msg;
	int i;
	static char bothNames[256];
    
	strcpy(bothNames, fromName);
	i = strlen(fromName) + 1;
	strcpy(&bothNames[i], toName);
    
	req->requestcode = RENAME_FILE;
	req->nameptr = bothNames;
	req->namelength = i + strlen(toName);/* total length of string */
	req->nameindex = 0;
	req->unspecified[0] = i - 1;/* length of old name */
    
	NameSend(req);

	result = req->requestcode;
      }

    Msg(ErrorString(result));
    if (result == OK || result == NOT_FOUND)
	return 1;
    else
	return 0;
  } /* RenameFile */

/* Makebak: Takes a filename and moves that file to a file of that name.
   Returns whether the operation was successful or not.
   The name of the backup file is returned in "bakFileName". */
int Makebak(fileName, bakFileName)
    char *fileName, **bakFileName;
  {
    *bakFileName = Bakname(fileName);
/*^*/ myprint(0x20)("Makebak: \"%s\" from \"%s\"\n",*bakFileName,fileName);
/*^*/ Flush(stdout);

  if (!RenameFile(fileName, *bakFileName))
    {
      NewMsg( "Backup failed for file:" );
      Msg( fileName );
      return(0);
    }
  return(1);

  } /* Makebak */


/* RJN - 8/15/83 - modified to handle context names correctly.  Prior to this
 *		   change, local names for contexts didn't work.
 */

/* CRZ - 5/7/85 -  modified to make the last "word" (ie. between /'s) no
 *		   longer than MAX_FILENAME_WORD_LEN, a new constant defined
 *		   above.  The current UNIX release imposes a 255 char
 *		   maximum, so that is what it is defined as.
 */
		  
/* Bakname: Takes a filename and appends .BAK to it, guaranteeing that the  */
/* result has a filename no longer than MAX_FILENAME_WORD_LEN chars.	    */
/* Bakname returns a static pointer, so call it, use the result, and throw  */
/* it away at once.							    */
char *Bakname(s)  char *s;
{ static char newstring[84];
  register char *beginname, *endname, *p;
  int namelen;

  strcpy(newstring, s );
  p = beginname = newstring;
  while (*p != 0 && p < newstring + 80) {
    if (*p == '/' || *p == '[' ) beginname = p + 1;
    p++;
    }
  endname = p;
  namelen = endname - beginname;
  if (namelen > (MAX_FILENAME_WORD_LEN-4))
      endname -= (namelen - (MAX_FILENAME_WORD_LEN-4));
  strcpy(endname,".BAK");
  return(newstring);
  } /* Bakname */

/* CRZ - 5/7/85 -  modified to make the last "word" (ie. between /'s) no
 *		   longer than MAX_FILENAME_WORD_LEN, a new constant defined
 *		   above.  The current UNIX release imposes a 255 char
 *		   maximum, so that is what it is defined as.
 */

/* CkpName: Takes a filename and appends .CKP to it, guaranteeing that the  */
/* result has a filename no longer than MAX_FILENAME_WORD_LEN chars.	    */
/* CkpName returns a static pointer, so call it, use the result, and throw  */
/* it away at once.							    */
char *CkpName(s)  char *s;
{ static char newstring[84];
  register char *beginname, *endname, *p;
  int namelen;

  strcpy(newstring, s );
  p = beginname = newstring;
  while (*p != 0 && p < newstring + 80) {
    if (*p == '/' || *p == '[' ) beginname = p + 1;
    p++;
    }
  endname = p;
  namelen = endname - beginname;
  if (namelen > (MAX_FILENAME_WORD_LEN-4))
      endname -= (namelen - (MAX_FILENAME_WORD_LEN-4));
  strcpy(endname,".CKP");
  return(newstring);
  } /* CkpName */



/* InitBuffer: given the starting chunk and ending mark of a piece of text,
   turns it into the buffer.  If you already have a buffer,
   free it before calling InitBuffer, or storage will be lost.  */
InitBuffer(headchunk, end_mark)  Chunk headchunk;  Mark end_mark;
{
  headmark = Makemark(headchunk, headchunk->text);
  mousemark = regionmark = headmark;
  endmark = end_mark;
  }


/* ClearBuffer: the other way to initialize a buffer */
ClearBuffer()
{ Chunk chunk;

  chunk = MakeChunk(NULL, 0);
  headmark.chunk = chunk;  headmark.cp = chunk->text;
  mousemark = regionmark = endmark = headmark;
  }


/* Note on creating a buffer:  It requires 5 steps, in order.
   1. Get the contents in straytext form.  (ReadFile)
   2. Give valid values to the standard Marks.  (InitBuffer)
   3. Allocate the row descriptor array, "rows". (in NewBuffer)
   4. Display, establishing row descriptor data.  (Display)
   5. Set the cursor - curmark, curpos  (Setcursor, MarkSetcursor)

   ClearBuffer substitutes for steps 1 and 2, in the special case that the
buffer will be empty.  All these steps should be completed before any other
operations are performed on the buffer.
*/

/***************  killbuffer and straytext operations  ******************/
Chunk killbuffer,killbuftail;

/* KillbufAdd: add a new chunk of text onto the killbuffer */
KillbufAdd(newchunk) Chunk newchunk;
{ register char *p,*q;
  if (killbuftail->length + newchunk->length <= CHUNKSIZE) {
    for (p=Chunkend(killbuftail), q=newchunk->text; q < Chunkend(newchunk);)
	*p++ = *q++;
    killbuftail->length += newchunk->length;
    if (newchunk->next) {
	killbuftail->next = newchunk->next;
	newchunk->next->prev = killbuftail;
	}
    free(newchunk);
    }
  else {
    killbuftail->next = newchunk;  newchunk->prev = killbuftail;
    killbuftail = newchunk;
    }

  while(killbuftail->next)  killbuftail = killbuftail->next;
  }

/* KillbufAddNewline: add a newline character onto the killbuffer */
KillbufAddNewline()
{
  if (killbuftail->length < CHUNKSIZE)
    killbuftail->text[killbuftail->length++] = '\n';
  else {
    killbuftail->next = MakeChunk(killbuftail, 1);
    killbuftail = killbuftail->next;
    killbuftail->text[0] = '\n';
    }
  }


/* CopyText: given the first chunk of a piece of text, return a copy of it */
Chunk  CopyText(startchunk)  Chunk startchunk;
{ Chunk chunk, copychunk, copyhead;
  register char *cp, *ecp, *copycp, *copyecp;
  register int i, stop, copyspace;

  chunk = startchunk;  cp = chunk->text;  ecp = Chunkend(chunk);
  copyhead = copychunk = MakeChunk(NULL, CHUNKFILLPOINT);
  copycp = copychunk->text;  copyecp = copycp + CHUNKFILLPOINT;

  while (1) {
    stop = ecp - cp;  copyspace = copyecp - copycp;
    if (stop > copyspace) stop = copyspace;

    for (i = 0; i < stop; i++) *copycp++ = *cp++;

    if (cp == ecp) {
	if (!chunk->next) {  /* end of material to copy */
	    copychunk->length = copycp - copychunk->text;
	    return(copyhead);
	    }
	chunk = chunk->next;  cp = chunk->text;  ecp = Chunkend(chunk);
	}

    if (copycp == copyecp) {
	copychunk->next = MakeChunk(copychunk, CHUNKFILLPOINT);
	copychunk = copychunk->next;  copycp = copychunk->text;
	copyecp = copycp + CHUNKFILLPOINT;
	}
    }
  }  /* end of CopyText */

/* CopySection: given the starting and ending marks of a part of the buffer,
   return a straytext copy of it. */
Chunk  CopySection(startmk, endmk)  Mark startmk, endmk;
{ Chunk chunk, copychunk, copyhead;
  register char *cp, *ecp, *copycp, *copyecp;
  register int i, stop, copyspace;

  chunk = startmk.chunk;  cp = startmk.cp;  ecp = Chunkend(chunk);
  copyhead = copychunk = MakeChunk(NULL, CHUNKFILLPOINT);
  copycp = copychunk->text;  copyecp = copycp + CHUNKFILLPOINT;

  while (1) {
    stop = ecp - cp;  copyspace = copyecp - copycp;
    if (endmk.chunk == chunk) stop = endmk.cp - cp;  /* guaranteed <= stop */
    if (stop > copyspace) stop = copyspace;

    for (i = 0; i < stop; i++) *copycp++ = *cp++;

    if (cp == ecp) {
	if (!chunk->next) {  /* end of buffer! */
	    ErrorMsg("CopySection: unexpected end of text");
	    copychunk->length = copycp - copychunk->text;
	    return(copyhead);
	    }
	chunk = chunk->next;  cp = chunk->text;  ecp = Chunkend(chunk);
	}

    if (cp == endmk.cp) {  /* proper ending */
	copychunk->length = copycp - copychunk->text;
	return(copyhead);
	}

    if (copycp == copyecp) {
	copychunk->next = MakeChunk(copychunk, CHUNKFILLPOINT);
	copychunk = copychunk->next;  copycp = copychunk->text;
	copyecp = copycp + CHUNKFILLPOINT;
	}
    }
  }  /* end of CopySection */


/* TextString: copy a section out of the buffer into a string of maximum */
/* length len.  Returns the actual length.				 */
int TextString(m1, m2, s, len)
  Mark m1, m2;
  char *s;
  int len;
{
  char *limit = s + len;
  register char *p = s;
  int eof = 0;

  while (MarkNEQ(m1, m2) && p < limit && !eof) {
    *p++ = *m1.cp++;
    eof = Advance(&m1);
    }
  if (p < limit) *p = '\0';
  return(p - s);
  }


/* StringText: copy a string into a new sparetext. 			*/
Chunk StringText(s, len)
  char *s;
  int len;
{
  Chunk chunk, texthead;
  register char *p, *cp, *ecp;
  char *endstr = s + len;

  texthead = chunk = MakeChunk(NULL, CHUNKFILLPOINT);
  cp = chunk->text;  ecp = cp + CHUNKFILLPOINT;
  p = s;

  while (p < endstr) {
    *cp++ = *p++;
    if (cp >= ecp) {
	chunk->next = MakeChunk(chunk, CHUNKFILLPOINT);
	chunk = chunk->next;  cp = chunk->text;
	ecp = cp + CHUNKFILLPOINT;
	}
    }
  chunk->length = cp - chunk->text;
  return(texthead);
  }

  
/* MakeChunk: takes the prev and length fields as parameters, and returns
   a new chunk with the next and line fields NULL.  If no memory is 
   available, it will keep prompting the user until he either makes some
   space or quits. */
Chunk  MakeChunk(prev, length)  Chunk prev;  int length;
{ register Chunk chunk;
  chunk = NewChunk();	/* a simple call to calloc */
  while (chunk == NULL) {
    OutOfMemory();
    chunk = NewChunk();
    }
  chunk->prev = prev;  chunk->length = length;
  chunk->next = NULL;
  return(chunk);
  }

/* FreeText: given a pointer to the start of a straytext, frees its storage */
FreeText(chunk)  Chunk chunk;
{ Chunk temp;
  while (chunk) {
    temp = chunk->next;
    free(chunk);
    chunk = temp;
    }
  }

/* Findend: given a straytext pointer, return a Mark to the end of text. */
Mark  Findend(chunk)  register Chunk chunk;
{ Mark textend;

  if (chunk == NULL)  ErrorMsg("Findend of NULL");
  for (; chunk->next; chunk = chunk->next);  /* seek last chunk */
  textend.chunk = chunk;  textend.cp = Chunkend(chunk);
  return(textend);
  }


/* OutOfMemory: Complain to the user about lack of memory and get him to */
/* do something about it.  Does not check that he has in fact solved	 */
/* the problem.  At the moment OutOfMemory is called only from MakeChunk,*/
/* which does check.							 */
OutOfMemory()
{ char ch;
  register int bufnum, currentbuf;

  NewMsg("Out of memory!  Please do one of the following:");
  NewMsg("  Pick a window to delete");
  NewMsg("  c  - continue (after you free something)");
  NewMsg("  q  - save and quit");
  NewMsg("  ^C - quit without saving");

  while (1) {
    NewMsg("");
    bufnum = PickWindow(1, &ch);
    if (bufnum >= 0) {
	if (bufnum == thisbuffer) 
	    NewMsg("Can't delete the current window without quitting");
	else {
	    currentbuf = thisbuffer;
	    PickBuffer(bufnum);
	    DeleteBuffer();
	    PickBuffer(currentbuf);
	    return;
	    }
	}
    else switch(ch) {
	case 'c': return;
	case 'q': 
	    if (WriteModifiedFiles())
	      {
		exit(1);
	      }
	    else
	      {
		NewMsg("Type q to try again, ^C to quit without saving.");
		break;
	      }
	case '\03': exit(1);
	default:  NewMsg("What?");
	}
    }  /* end while (1) */
  }


/* WriteModifiedFiles: Save all modified buffers, saving thisbuffer LAST.	*/
/* That is so that if the current one is in an inconsistent state,	*/
/* any other buffers will be safely saved first.			*/
/* Returns 1 if all writes were successful, else 0.			*/
/*
 * CRZ 5/16/85 Changed it so that the pads do not flash when they are written.
 * 	       This is done by sending WriteFile the proper headmark and
 * 	       endmark, which is the only reason why PickBuffer is called
 *	       anyway.
 */
int WriteModifiedFiles(ch)
  int ch;
  {
  register int i;
  int succeeded = 1;
  register BufferRec *buf;
  static char NoFileName[] = "(No file)";

  for (i=0; i<Maxbufs; i++) 
    {
      if ( ( i != thisbuffer ) &&
	   ( buffers[i].pad != 0 ) &&
	   ( buffers[i].ptr->fileRec->modified ) )
	{
	  if (buffers[i].ptr )  buf = buffers[i].ptr;
	  else {
	      ErrorMsg("WriteModifiedFiles: nonexistent buffer");
	      return( 0 );
	    }

	  if (!WriteFile(buf->fileRec->filename, buf->headmark, buf->endmark))
	    {
	      succeeded = 0;
	    }
	  else
	    {
	      buf->fileRec->modified = 0;
	      buf->fileRec->ckpmodified = 0;
	      DisplayBanner( buffers[i].pad, buf->fileRec->modified ? "*" : "",
    		*(buf->fileRec->filename) == NULL ? NoFileName :
		 buf->fileRec->filename );
	    }
	}
    }

  if (modified)
    {
      if (!WriteFile(filename, headmark, endmark))
	{
	  succeeded = 0;
	}
      else
	{
	  modified = 0;
	  ckpmodified = 0;
	  Banner();
	}
    }
  return(succeeded);
  }


/* CheckpointModifiedFiles:
 * Checkpoint all modified buffers.
 * Returns 1 if all writes were successful, else 0.
 *
 * By the way, WriteFile normally does Back ups if the backupoption is on, but
 * we disable that for the duration of the procedure.
 *
 * CRZ 5/16/85 Took the backupoption swapping stuff out of mainloop (vedit.c)
 *	       Originally, I fixed the problem here by adding the "PickBuffer"
 *	       calls as above, but then decided to just use the pointers into
 *	       the structures for buffers (BufferRec's) to copy the correct
 *	       buffer into the filename.
 */
int CheckpointModifiedFiles()
  {
  register int i;
  int succeeded = 1;
  register BufferRec *buf;
  int savBackupOption;

  savBackupOption = backupoption;
  backupoption = 0;
  for (i=0; i<Maxbufs; i++) 
    {
      if ( ( i != thisbuffer ) &&
	   ( buffers[i].pad != 0 ) &&
	   ( buffers[i].ptr->fileRec->ckpmodified ) )
	{
	  if (buffers[i].ptr )  buf = buffers[i].ptr;
	  else {
	      ErrorMsg("CheckpointModifiedFiles: nonexistent buffer");
	      return( 0 );
	    }

	  if (!WriteProtFile(CkpName(buf->fileRec->filename),
			     buf->headmark, buf->endmark,
			     buf->fileRec->filename))
	    {
	      succeeded = 0;
	    }
	  else
	    {
	      buf->fileRec->ckpmodified = 0;
	      AddCkpName( CkpName( buf->fileRec->filename ) );
	    }
	}
    }
  if (ckpmodified)
    {
      if (!WriteProtFile(CkpName(filename), headmark, endmark, filename))
	{
	  succeeded = 0;
	}
      else
	{
	  ckpmodified = 0;
	  AddCkpName( CkpName( filename ) );
	}
    }
  backupoption = savBackupOption;
  return(succeeded);
  }

/* CRZ 5/7/85 This is called every time a file is checkpointed.  It maintains a
 * 	      linked list of ckpdescs (defined in vedit.h).  Each ckpdesc
 * 	      contains the name of a checkpointed file.  The list contains
 *	      only those CKP files which were opened during this ved session.
 *	      The idea is to remove all of these on a normal exit, but let
 *	      them remain upon abnormal termination.
 */
AddCkpName( filename )
  char	*filename;
{
  struct ckpdesc	*ctemp;
  int			found = 0;

  		/* CkpNames: global pointer to head of list of ckp files*/
  for( ctemp = CkpNames; ctemp && !found; ctemp = ctemp->nextckpdesc )
      if( Equal( ctemp->ckpfilename, filename ) )
	  found = 1;

		/* if not found, then add to head of CkpNames list */
  if( !found )
    {
      ctemp = ( struct ckpdesc * ) malloc( sizeof( struct ckpdesc ) );
      ctemp->ckpfilename = ( char * ) malloc( strlen( filename ) + 1 );
      strcpy( ctemp->ckpfilename, filename );
      ctemp->nextckpdesc = CkpNames;
      CkpNames = ctemp;
    }
}
      

/* AnyBufsModified: returns 1 if there are any modified buffers, else 0.  */
int AnyBufsModified()
{ register int i, anymod;

  anymod = modified;	/* current buffer's modified bit */
  for (i = 0; i < Maxbufs; i++)
    if (buffers[i].pad != NULL && buffers[i].ptr->fileRec->modified &&
	    i != thisbuffer)
	anymod = 1;
  return(anymod);
  }


/* MarkGT: comparison of Marks.  This is one of the disoptimized operations
   in this data structure system, and takes time proportional to the number
   of chunks in the buffer.  Needed only when a Region operation is to be
   done.  Returns 1 if the first argument follows the second, else 0.
   -1 if the two are in disjoint spaces (error!) and cannot be compared.
*/
int MarkGT(m1, m2)  Mark m1, m2;
{ Chunk chunk;
/* 1. same chunk */
  if (m1.chunk == m2.chunk) {
    if (m1.cp > m2.cp) return(1);
    else return(0);
    }
/* 2. look for m2 in a chunk after m1 */
  for (chunk = m1.chunk->next; chunk; chunk = chunk->next)
    if (chunk == m2.chunk)  return(0);
/* 3. look for m2 in a chunk before m1 */
  for (chunk = m1.chunk->prev; chunk; chunk = chunk->prev)
    if (chunk == m2.chunk)  return(1);
/* error: m2 is nowhere to be found from m1 */
  ErrorMsg("MarkGT on unrelated marks");
  return(-1);
  }
