/*
 * Functions which are like emacs mlisp functions.  Part 1.
 */


#include "vedit.h"
#include "chardef.h"

extern int PrefixArgument;

int AutoLineFeedFlag = 0;


/*
 * ProvidePrefixArgument:
 */

ProvidePrefixArgument(ch)
    int ch;
  {
    ProcessId pid;
    Vedmsg commandMsg;
    int n;
    int c;
    char inBuf[10];
    int inBufPtr = 0;

    while (1)
      {
	pid = ReceiveSpecific(&commandMsg, keyproc);
	Reply(&commandMsg, pid);
	if (commandMsg.type == Mouse)
	  {			/* Ignore the mouse while taking input. */
	    continue;
	  }
	c = commandMsg.x;
	if (Numeric(c))
	  {
	    inBuf[inBufPtr++] = (char) c;
	    if (inBufPtr == 9)
	      {
		NewMsg("WARNING - prefix argument too large.");
		NewMsg("Using the value 1 instead.");
		PrefixArgument = 1;
		return;
	      }
	  }
	else
	  {
	    UngetUserInputBuffer(c);
	    inBuf[inBufPtr] = '\0';
	    break;
	  }
      }
    if (inBufPtr == 0)
      {
	PrefixArgument = 4;
      }
    else
      {
	PrefixArgument = atoi(inBuf);
      }
  }


/*
 * SelfInsert:
 * Insert character at the cursor and advance the cursor.
 */

SelfInsert(ch)
    unsigned char ch;
  {
    PreAdjust(curmark,curpos,&eolmark,&eolpos);
    TextInsertChar(ch);
    if (MarkEQ(curmark,eolmark))
      {
        eolmark.cp++;
        Advance(&eolmark);
      }
    AdjustDisplay(curmark,eolmark,&curpos,&eolpos);
    if (currow == Nrows)
      {
        NextHalfPage();
      }
    Setcursor(curmark,curpos);
    ForwardCharacter();
    if (AutoLineFeedFlag && rows[currow].continues)
      {
	AutoLineFeed();
      }
  }


/*
 * BeginningOfLine:
 */

BeginningOfLine(ch)
    int ch;
  {
    curcol = 0;
    curmark = Rowmark(currow);
    PadPosition(pad, currow, curcol);
    wishcol = 0;
  }


/*
 * EndOfLine:
 */

EndOfLine(ch)
    int ch;
  {
    curmark = Rowsend(currow);
    MarkSetcursor(curmark);
    wishcol = Ncols;
  }


/*
 * NextLine:
 */

NextLine(ch)
    int ch;
  {
    register int i;

    for (i = 0; i < PrefixArgument; i++)
      {
	if (!rows[currow+1].exists)
	  {
	    Beep();
	  }
	else
	  {
	    if ((currow + 1) == Nrows)
	      {
		NextHalfPage();
	      }
	    currow++;
	    PosSetcursor(Makepos(currow, wishcol));
	  }
      }
  }


/*
 * PreviousLine:
 */

PreviousLine(ch)
    int ch;
  {
    register int i;

    for (i = 0; i < PrefixArgument; i++)
      {
	if ((currow == 0) && PageAtHead())
	  {
	    Beep();
	  }
	else
	  {
	    if (currow == 0)
	      {
		PreviousHalfPage();
	      }
	    currow--;
	    PosSetcursor(Makepos(currow, wishcol));
	  }
      }
  }


/*
 * ScrollOneLineUp:
 */

ScrollOneLineUp(ch)
    int ch;
  {
    register int i;

    for (i = 0; i < PrefixArgument; i++)
      {
	Scroll();
      }
  }


/*
 * ScrollOneLineDown:
 */

ScrollOneLineDown(ch)
    int ch;
  {
    register int i;

    for (i = 0; i < PrefixArgument; i++)
      {
	Backscroll();
      }
  }


/*
 * ForwardCharacter:
 * Returns 1 on success, 0 otherwise (at end).
 */

int ForwardCharacter(c)
    int c;
  {
    char ch;
    int newcol;
    int newsize;

    if (Atend(curmark))
      {
        Beep();
	return(0);
      }
    ch = *curmark.cp++;
    Advance(&curmark);
    newcol = CountWidth(ch,curcol);
    if (Atend(curmark))
      {
        newsize = 0;
      }
    else
      {
        newsize = charsize[*curmark.cp];
      }
    if (newcol + newsize > Ncols)
      {
        newcol = 0;
      }
    if (newcol > 0)
      {
	for (; curcol<newcol; curcol++)
	  {
	    PadCursorForward(pad);
	  }
      }
    else
      {
	currow++;
	curcol = -newcol;
	if (currow >= Nrows)
	  {
	    NextHalfPage();
	  }
	PadPosition(pad, currow, curcol);
      }
    wishcol = curcol;
    return(1);
  }


/*
 * BackwardCharacter:
 * Returns 1 if successful, 0 otherwise (at beginning).
 */

BackwardCharacter(c)
    int c;
  {
    char ch;

    if (Atstart(curmark))
      {
        Beep();
	return(0);
      }
    if (currow == 0 && curcol == 0)
      {
        PreviousHalfPage();
      }
    Retract(&curmark);
    ch = *curmark.cp;
    if (ch >= ' ' && ch < '\177' && curcol != 0)
      {  			/* printing character */
        PadCursorBackward(pad);
        curcol--;
      }
    else
      {
	if (curcol == 0)
	  {
	    currow--;
	  }
	curcol = Markcol(currow, curmark);
	PadPosition(pad, currow, curcol);
      }
    wishcol = curcol;
    return(1);
  }


/*
 * ForwardWord:
 */

ForwardWord(ch)
    int ch;
  {
    if (Atend(curmark))
      {
        Beep();
	return;
      }
    curmark = Forwardword(curmark);
    DispSetcursor(curmark);
  }


/*
 * BackwardWord:
 */

BackwardWord(ch)
    int ch;
  {
    if (Atstart(curmark))
      {
        Beep();
	return;
      }
    curmark = BackWord(curmark);
    DispSetcursor(curmark);
  }


/*
 * NextPage:
 */

NextPage(ch)
    int ch;
  {
    int n;
    register int i;

    for (i = 0; i < PrefixArgument; i++)
      {
	PadHome(pad);
	if (!rows[Nrows-2].exists)
	  {
	    Beep();
	    return;
	  }
	n = Nrows-2;
	while (rows[n].continues)
	  {
	    n--;
	  }
	Display(Rowmark(n));	/* in which all Rowmarks are redefined */
	Setcursor(Rowmark(0), zeropos);
	wishcol = 0;
      }
  }


/*
 * PreviousPage:
 */

PreviousPage(ch)
    int ch;
  {
    Mark m;
    register int i;

    for (i = 0; i < PrefixArgument; i++)
      {
	PadHome(pad);
	m = Backrows(Rowmark(0), Nrows-2);
	Display(m);
	Setcursor(m, zeropos);
	wishcol = 0;
      }
  }


/*
 * BeginningOfWindow:
 */

BeginningOfWindow(ch)
    int ch;
  {
    Setcursor(Rowmark(0), zeropos);
    wishcol = curcol;
  }


/*
 * EndOfWindow:
 */

EndOfWindow(ch)
    int ch;
  {
    PosSetcursor(Makepos(Nrows-1,Ncols-1));
    wishcol = curcol;
  }


/*
 * LineToTopOfWindow:
 */

LineToTopOfWindow(ch)
    int ch;
  {
    int oldPrefixArg = PrefixArgument;

    while (currow != 0)
      {
        PrefixArgument = 1;
	ScrollOneLineUp(ch);
  	PrefixArgument = oldPrefixArg;
      }
  }


/*
 * BeginningOfFile:
 */

BeginningOfFile(ch)
    int ch;
  {
    Display(headmark);
    Setcursor(headmark, zeropos);
    wishcol = 0;
  }


/*
 * EndOfFile:
 */

EndOfFile(ch)
    int ch;
  {
    DispSetcursor(endmark);  	/* wishcol is automatic */
  }


/*
 * DeleteNextCharacter:
 */

DeleteNextCharacter(c)
    int c;
  {
    Mark m;
    char ch;

    if (Atend(curmark))
      {
        Beep();
	return;
      }
    ch = *curmark.cp;
    if (ch == '\n')
      {
        m = curmark;
        /* ForwardCharacter would be nice here, but we mustn't Scroll */
        curmark.cp++;
	Advance(&curmark);
        curpos = Makepos(currow+1,0);
        PreAdjust(curmark,curpos,&eolmark,&eolpos);
        BackwardCharacter();
      }
    else
      {
        PreAdjust(curmark,curpos,&eolmark,&eolpos);
      }
    TextDeleteForward();
    AdjustDisplay(curmark,eolmark,&curpos,&eolpos);
    if (currow >= Nrows)
      {
        Scroll();  /* delete \n on full row Nrows-1 */
      }
    Setcursor(curmark,curpos);
  }


/*
 * DeletePreviousCharacter:
 */

DeletePreviousCharacter()
  {
    if (Atstart(curmark))
      {
        Beep();
	return;
      }
    if ((currow == 0) && (curcol == 0))
      {
        Backscroll();
      }
    BackwardCharacter();
    DeleteNextCharacter();
  }


/*
 * BackwardHackingTabs:
 * Same as DeletePreviousCharacter except that tabs are first converted to
 * spaces.
 */

BackwardHackingTabs(c)
    int c;
  {
    char ch;
    register int i;

    if (Atstart(curmark))
      {
        Beep();
	return;
      }
    if ((currow == 0) && (curcol == 0))
      {
        Backscroll();
      }
    BackwardCharacter();
    ch = *curmark.cp;
    DeleteNextCharacter();
    if (ch == '\t')
      {
	for (i = 0; i < 7; i++)
	  {
	    SelfInsert(' ');
	  }
      }
  }


/*
 * DeleteNextWord:
 */

DeleteNextWord(ch)
    int ch;
  {
    Pos p;

    if (Atend(curmark))
      {
        Beep();
        return;
      }
    mousemark = Forwardword(curmark);
				/* a handy Marklist mark */
    p = Markpos(mousemark);
    DelWord(p);
  }


/*
 * DeletePreviousWord:
 */

DeletePreviousWord(ch)
    int ch;
  {
    Pos p;

    if (Atstart(curmark))
      {
        Beep();
	return;
      }
    mousemark = curmark;
    p = curpos;
    curmark = BackWord(curmark);
    DispSetcursor(curmark);
    DelWord(p);
  }


/*
 * TransposeCharacters:
 */

TransposeCharacters(ch)
    int ch;
  {
    Mark m;
    int eof;
    char c1, c2;

    m  = curmark;
    eof = Retract(&m);
    if (eof) return;
    c1 = *m.cp;
    eof = Retract(&m);
    if (eof) return;
    c2 = *m.cp;
    BackwardCharacter();
    DeleteNextCharacter();
    BackwardCharacter();
    DeleteNextCharacter();
    SelfInsert(c1);
    SelfInsert(c2);
  }


/*
 * KillToEndOfLine:
 */

KillToEndOfLine(c)
    int c;
  {
    int ch;
    int eof;
    char *cp;
    Chunk chunk, nchunk;
    extern unsigned int PreviousUserCmd;
    Mark m;

    if (Atend(curmark))
      {
        Beep();
	return;
      }
    ch = *curmark.cp;
    if (ch=='\n')
      {
        if (PreviousUserCmd == CONTROL('K'))
	  {
	    KillbufAddNewline();
	  }
        else
	  {
	    FreeText(killbuffer);
	    killbuffer = killbuftail = NewChunk();
	    killbuffer->prev = killbuffer->next = NULL;
	    killbuffer->length = 1;
	    killbuffer->text[0] = '\n';
	  }
        DeleteNextCharacter();
      }
    else
      {				/* find the block */
        PreAdjust(curmark,curpos,&eolmark,&eolpos);
        if (eolpos.row < Nrows)
	  {
	    m = eolmark;
	  }
        else
	  {
    	    m = curmark;
    	    do {ch= *m.cp++;  eof = Advance(&m);}
	    while (ch !='\n' && !eof);
    	    if (ch == '\n')
	      {
	        Retract(&m);
	      }
    	  }
        nchunk = BlockDelete(curmark,m);
        AdjustDisplay(curmark,eolmark,&curpos,&eolpos);
        Setcursor(curmark, curpos);
        if (PreviousUserCmd == CONTROL('K'))
	  {
	    KillbufAdd(nchunk);
	  }
        else
	  {
    	    FreeText(killbuffer);
       	    killbuffer = killbuftail = nchunk;
    	    while (killbuftail->next)
	      {
	        killbuftail = killbuftail->next;
	      }
    	  }
      }
  }


/*
 * DeleteToKillBuffer:
 */

DeleteToKillBuffer(ch)
    int ch;
  {
    if (MarkEQ(regionmark,curmark))
      {
        return;
      }
    FreeText(killbuffer);
    if (MarkGT(regionmark,curmark))
      { 
        killbuffer = BlockDelete(curmark,regionmark);
      }
    else
      {
        killbuffer = BlockDelete(regionmark,curmark);
      }
    Display(Rowmark(0));
    MarkSetcursor(curmark);
    wishcol = curcol;
  }


/*
 * YankKillBufferBeforeCursor:
 * Take the contents of the kill buffer and insert it before the cursor.
 */

YankKillBufferBeforeCursor(ch)
    int ch;
  {
    YankKillBuffer();
    DispSetcursor(curmark);
  }


/*
 * YankKillBufferAfterCursor:
 * Take the contents of the kill buffer and insert it after the cursor.
 */

YankKillBufferAfterCursor(ch)
    int ch;
  {
    Mark m;

    m = YankKillBuffer();
    regionmark = curmark;
    if (currow == Nrows)
      {
        Scroll();
      }
    Setcursor(m,curpos);
  }


/*
 * ExchangeDotAndMark:
 */

ExchangeDotAndMark(ch)
    int ch;
  {
    Mark m;

    m = curmark;  curmark = regionmark;  regionmark = m;
    DispSetcursor(curmark);
  }


/*
 * IndentRegion:
 */

IndentRegion(ch)
    int ch;
  {
    int oldPrefixArg = PrefixArgument;
    if (PrefixArgument == 0)
      {
	return;
      }
    /* Make both region and cursor marks point to the beginnings of their
       lines. */
    curmark = Backrows(curmark, 0);
    regionmark = Backrows(regionmark, 0);
    /* Make the region mark follow the cursor mark. */
    if (MarkGT(curmark, regionmark))
      {
	ExchangeDotAndMark();
      }

    /* Indent each line, starting at the curmark and stopping at the
       regionmark.*/
    while (MarkNEQ(curmark, regionmark))
      {
	if (PrefixArgument > 0)
	  {
	    ShoveRight(PrefixArgument);
	  }
	else
	  {
	    ShoveLeft(-PrefixArgument);
	  }
	PrefixArgument = 1;
	NextLine(ch);
	PrefixArgument = oldPrefixArg;
	BeginningOfLine();
      }
  }


/*
 * CaseWordUpper:
 */

CaseWordUpper(c)
    int c;
  {
    register char ch;
    Mark m;

    m = curmark;
    Retract(&m);
    if (wordchar(*m.cp))
      {
	BackwardWord();
      }
    for (ch = *curmark.cp; wordchar(ch); ch = *curmark.cp)
      {
	if ((ch >= 'a') && (ch <= 'z'))
	  {
	    DeleteNextCharacter();
	    ch = ch + 'A' - 'a';
	    SelfInsert(ch);
	  }
	else
	  {
	    ForwardCharacter();
	  }
      }
  }


/*
 * CaseWordLower:
 */

CaseWordLower(c)
    int c;
  {
    register char ch;
    Mark m;

    m = curmark;
    Retract(&m);
    if (wordchar(*m.cp))
      {
	BackwardWord();
      }
    for (ch = *curmark.cp; wordchar(ch); ch = *curmark.cp)
      {
	if ((ch >= 'A') && (ch <= 'Z'))
	  {
	    DeleteNextCharacter();
	    ch = ch + 'a' - 'A';
	    SelfInsert(ch);
	  }
	else
	  {
	    ForwardCharacter();
	  }
      }
  }


/*
 * CaseWordCapitalize:
 */

CaseWordCapitalize(c)
    int c;
  {
    register char ch;
    Mark m;

    m = curmark;
    Retract(&m);
    if (wordchar(*m.cp))
      {
	BackwardWord();
      }
    ch = *curmark.cp;
    if (wordchar(ch) && (ch >= 'a') && (ch <= 'z'))
      {
	DeleteNextCharacter();
	ch = ch + 'A' - 'a';
	SelfInsert(ch);
      }
    else
      {
	ForwardCharacter();
      }
    for (ch = *curmark.cp; wordchar(ch); ch = *curmark.cp)
      {
	if ((ch >= 'A') && (ch <= 'Z'))
	  {
	    DeleteNextCharacter();
	    ch = ch + 'a' - 'A';
	    SelfInsert(ch);
	  }
	else
	  {
	    ForwardCharacter();
	  }
      }
  }


/*
 * RedrawDisplay:
 */

RedrawDisplay(ch)
    int ch;
  {
    PadHome(pad);
    Display(Rowmark(0));
    DispSetcursor(curmark);	/* absolutely guarantee a consistent cursor */
  }


/*
 * AutoLineFeed:
 * Splits the current line so that it no longer runs over the right-hand edge
 * of the screen.  Assumes that the cursor is in the continuation line.
 */

AutoLineFeed()
  {
    unsigned short row;
    int ch;

    row = currow;
    /* Back up to the previous display row. */
    while (currow == row)
      {
	BackwardWord(ch);
      }
    /* Back up until we hit white space. */
    row = currow;
    BackwardCharacter(ch);
    ch = *curmark.cp;
    while ((ch != ' ') && (ch != '\t') && (currow == row)) 
				/* If currow != row then the entire line is 
				   contains no white space, so abort. */
      {
	BackwardWord(ch);
	BackwardCharacter(ch);
	ch = *curmark.cp;
	if (Atstart(curmark))
	  {
	    /* Abort the attempted linefeed because we are at the beginning of
	       the file. */
	    EndOfLine(ch);
	    return;
	  }
      }
    if (currow != row)
      {
        NextLine(ch);
	EndOfLine(ch);
	return;
      }
    ForwardCharacter(ch);
    /* HACK: Ved uses lines that are 1 too wide.  Make sure we split the line
       at a point that doesn't make the first line too wide for other 
       editors. */
    if (curcol >= (Ncols - 1))
      {
	BackwardWord(ch);
      }
    /* Split the line, indent it properly, and put the cursor at the eol. */
    NewlineAndIndent(ch);
    EndOfLine(ch);
  }


/*
 * ToggleAutoLineFeed:
 */

ToggleAutoLineFeed(ch)
    int ch;
  {
    AutoLineFeedFlag ^= 1;
    if (AutoLineFeedFlag)
      {
	NewMsg("Automatic linefeed option on");
      }
    else
      {
	NewMsg("Automatic linefeed option off");
      }
  }
