/* unexec -- turn a running program into an executable file 
 *
 * unexec takes as an argument the name or pid of a running team and
 * the name of a file.  The current state of the team space is copied
 * to the file, prefixed by an executable file header which redefines
 * all of the memory beyond the code to be initialized data.  The
 * TeamFileList is cleared, so all open files will be lost.
 *
 * unexec is intended for use with programs which are able to
 * recognize that they have been unexec'ed (e.g., TeX).  Candidate
 * programs are those which spend a lot of time in initialization --
 * especially if that initialization information is read from a file.
 *
 */

#include <Vio.h>
#include <b.out.h>
#include <Vprocess.h>
#include <Vteams.h>
#include <Vdirectory.h>
#include <Vexceptions.h>

/* The basic plan: register as exception handler, then force an exception
 * to gain access to team space (and freeze the target).
 */

ProcessId PidOfTeamFileName();
File *ObjectFileOfTeam();

#define BUFFER_SIZE 2048
char buffer[BUFFER_SIZE];

main(argc, argv)
int argc;
char *argv[];
  {
    ProcessId myPid = GetPid(ACTIVE_PROCESS, LOCAL_PID);
    ProcessId targetPid, exceptionServerPid;
    File *objFile;
    File *dumpFile;
    char *dumpFileName;
    char *progName;
    RegHandlerRequest regReq;
    long teamSize, teamTop, teamSpot;
    struct bhdr header, oldheader;
    Message msg;
    int nread;
    SystemCode error;
    File *nullptr = NULL;
    
    if (argc != 3)
      {
        fprintf(stderr, "Usage: unexec <program> <file>\n");
	exit(1);
      }
      
    progName = argv[1];
    dumpFileName = argv[2];
    
    if ((targetPid = PidOfTeamFileName(progName)) == -1)
      {
        fprintf(stderr, "Couldn't find team named %s\n", progName);
	exit(1);
      }
      
    if ((objFile = ObjectFileOfTeam(targetPid)) == NULL)
      {
        fprintf(stderr, "Couldn't open team's object file\n");
	exit(1);
      }
      
    if ((dumpFile = fopen(dumpFileName, "w")) == NULL)
      {
        fprintf(stderr, "Can't open %s for writing\n", dumpFileName);
	exit(1);
      }
    
    /* register as exception handler (stolen from debugger) */

    exceptionServerPid = GetPid(EXCEPTION_SERVER, LOCAL_PID);
    if (exceptionServerPid == 0)
      {
        fprintf( stderr, "Couldn't find the exception server's pid.\n");
        exit(1);
      }

    regReq.requestcode = REGISTER_HANDLER;
    regReq.regFlag = 1;
    regReq.handlerPid = myPid;
    regReq.regPid = targetPid;

    Send(&regReq, exceptionServerPid);
    if ( regReq.requestcode != OK )
      {
        fprintf( stderr, "Couldn't register to handle exceptions: %s.\n",
		ErrorString(regReq.requestcode));
        exit(1);
      }

    /* Now force an exception, so that we'll have access to the target's
     * team space
     */

    if ((error = ForceException(targetPid)) != OK)
      {
        fprintf(stderr, "Error trying to force exception on %s: %s\n",
		progName, ErrorString(error));
	exit(1);
      }
      
    if (ReceiveSpecific(msg, targetPid) != targetPid)
      {
        fprintf(stderr, "Didn't get exception message\n");
	exit(1);
      }
      
    /* We're all set up, so let's go to work */
    
    teamTop = (long)GetTeamSize(targetPid);
    teamSize = teamTop - TEAM_ORIGIN;
    
    rewind(objFile);
    fread(&oldheader, sizeof(oldheader), 1, objFile);
    header = oldheader;
    header.dsize = teamSize - header.tsize;
    header.bsize = 0;
    printf("Old data size was %x, new size is %x\n",
    		oldheader.dsize, header.dsize);
    printf("Old bss size was %x, new size is %x\n",
    		oldheader.bsize, header.bsize);
    
    fwrite(&header, sizeof(header), 1, dumpFile);
    
    /* go to where the text begins */
    fseek(dumpFile, StartOfText(&header), ABS_BYTE);

    /* Now copy all of team space */
    
    teamSpot = TEAM_ORIGIN;
    while (teamSpot < teamTop)
      {
        int chunkSize = (teamSpot + BUFFER_SIZE > teamTop ?
				teamTop - teamSpot :
				BUFFER_SIZE);
	
	if ((error = MoveFrom(targetPid, buffer, teamSpot, chunkSize)) != OK)
	  {
	    fprintf(stderr, "Error copying team space: %s\n",
	    		ErrorString(error));
	    exit(1);
	  }
	fwrite(buffer, 1, chunkSize, dumpFile);
	
	teamSpot += chunkSize;
      }
      
    /* Now copy the symbol table */
    
    fseek(objFile, StartOfSymbolTable(&oldheader), ABS_BYTE);
    fseek(dumpFile, StartOfSymbolTable(&header), ABS_BYTE);

    while ((nread = fread(buffer, 1, BUFFER_SIZE, objFile)) > 0)
        fwrite(buffer, 1, nread, dumpFile);
	
    if (!feof(objFile))
      {
        fprintf(stderr, "Error reading symbol table: %s\n",
		FileException(objFile));
	exit(1);
      }

    /* clear TeamFileList */

    fseek(dumpFile, OffsetOfSymbol("TeamFileList", objFile, &oldheader),
    	  ABS_BYTE);
    fwrite(&nullptr, sizeof(nullptr), 1, dumpFile);


    fclose(dumpFile);
    fclose(objFile);
    
    /* Reply to the target, letting it continue */
    msg[0] = 0;
    Reply(msg, targetPid);

    printf("Finished!\n");
    exit(0);
    
  }

/* stolen from debugger & hacked */

/*
 * Queries the team server for information about the team with root pid rootPid
 * and returns a file pointer for the file open for reading
 */
File *ObjectFileOfTeam(rootPid)
	ProcessId rootPid;
  {
    Message msg;
    DescriptorRequest *reqMsg = (DescriptorRequest *) msg;
    DescriptorReply *repMsg = (DescriptorReply *) msg;
    char buffer[sizeof(TeamDescriptor)];
    TeamDescriptor *desc = (TeamDescriptor *) buffer;
    ProcessId teamServerPid;
    ProcessId fileServer;
    InstanceId fileId;
    SystemCode error;

    reqMsg->requestcode = NREAD_DESCRIPTOR;
    reqMsg->namecontextid = rootPid;
    reqMsg->segmentptr = buffer;
    reqMsg->segmentlen = sizeof(TeamDescriptor);
    reqMsg->dataindex = 0;
    reqMsg->nameindex = reqMsg->segmentlen;
    teamServerPid = GetPid(TEAM_SERVER, LOCAL_PID);
    Send(msg, teamServerPid);
    if ( repMsg->replycode != OK )
      {
	fprintf( stderr, "unexec:  on access to team server: %s\n", 
		ErrorString(repMsg->replycode));
	exit(1);
      }
    fileServer = desc->fileserver;
    fileId = desc->fileid;
    return(OpenFile(fileServer, fileId, FREAD, &error));
  }

/* stolen from destroy */

/*
 * PidOfTeamFileName:
 * Returns the pid corresponding to the team filename specified by s.
 * Exits the program if s does not correspond to an actual team or specifies
 * a nonunique team filename.
 * s is matched against SUFFIXES of team filenames returned from the 
 * team server.
 */

ProcessId PidOfTeamFileName(s)
    char *s;
  {
    File *serverFile;
    SystemCode error;
    ProcessId pid;
    ArbitraryDescriptor desc;
    int count = 0;

    /* Compare s to the suffix of the filename of each team in the system. */
    serverFile = Open("[team/local]", FREAD|FDIRECTORY|FBLOCK_MODE, &error);
    if (error != OK)
      {
	fprintf(stderr, "ERROR while trying to open the team servers directory: %s\n",
		ErrorString(error));
	exit(1);
      }
    do
      {
	if ( Read(serverFile, &desc, serverFile->blocksize) >=
		sizeof(EmptyDescriptor) &&
	     desc.t.descriptortype == TEAM_DESCRIPTOR )
	  {
	    if (MatchesSuffix(desc.t.fileName, s))
	      {
		count++;
		pid = desc.t.rootPid;
	      }
	  }
	serverFile->block++;
      }
    while (serverFile->lastexception == OK);
    if (serverFile->lastexception != END_OF_FILE)
      {
	fprintf(stderr, "ERROR while reading the team server's directory: %s\n",
		ErrorString(serverFile->lastexception));
	exit(1);
      }
    Close(serverFile);
    if (count == 1)
      {
	return(pid);
      }
    else if (count == 0)
      {
	fprintf(stderr, "unexec: %s not found.\n", s);
	return(-1);
      }
    else
      {
	fprintf(stderr, "unexec: %s is not unique.\n", s);
	return(-1);
      }
  }


/*
 * MatchesSuffix:
 * Returns TRUE if s1 is a suffix of s; else returns FALSE.
 */

int MatchesSuffix(s, s1)
    char *s, *s1;
  {
    char *p;
    char *p1;

    for (p = s + strlen(s) - 1, p1 = s1 + strlen(s1) -1;
    	 (p >= s) && (p1 >= s1);
	 p--, p1--)
      {
	if (*p != *p1)
	  {
	    break;
	  }
      }
    if (p1 < s1)
      {
	return(1);
      }
    else
      {
	return(0);
      }
  }
