/*
 * timeipc - measure V interprocess communication performance
 *
 * Steve Deering
 * October, 1985
 */

#include "timeipc.h"

/* external variables exported to Sender process, */
/* initialized to default values                  */

char		SenderHost[MAX_NAME_LEN]   = "unknown";
char		ReceiverHost[MAX_NAME_LEN] = "sameteam";

ProcessId	RootPid		= 0;
ProcessId	SenderPid	= 0;
ProcessId	ReceiverPid	= 0;

SystemCode	TypeOfTest	= MSG_TEST;
unsigned	InterHost	= FALSE;
unsigned	ReceiverPrio	= LOWER_PRIO;
unsigned	MsgsPerTrial	= 10000;
unsigned	NumberOfTrials	= 10;
char		*SegmentPtr	= NULL;
unsigned	SegmentSize	= 0;

/* other external variables */

static char	ServerProg[]	= "timeipcserver";


main()
  {
    Message		message;
    TimingMsg		*msg = (TimingMsg *) message;
    SystemCode		error;
    char		*senderStack;
    char		inputWord[MAX_NAME_LEN], inputLetter;
    unsigned		inputNumber;
    char		*argv[2];
    ProcessId		teamServerPid;
    SelectionRec	serverSpec;

    extern		Sender(), Receiver();
    extern char		*LastStackAllocated, *ErrorString(), *Lower();
    extern char		*fgets(), *malloc();
    extern ProcessId	MapRemoteHost();

    /* get process id of this process */
    RootPid = GetPid( ACTIVE_PROCESS, LOCAL_PID );

    /* get process id of this host's team server */
    teamServerPid = GetPid( TEAM_SERVER, LOCAL_PID );

    /* get name of this host */
    QueryWorkstationConfig( "name", SenderHost, MAX_NAME_LEN );
    Lower( SenderHost );

    /* set up argument vector and host spec for executing a server team */
    argv[0] = ServerProg;
    argv[1] = NULL;
    DefaultSelectionRec( &serverSpec );

    if( (error = ChangeTeamPriority( RootPid, REAL_TIME1 )) != OK )
	fprintf( stderr,
		 "warning: can't set sender team priority to REAL_TIME1 (%s)\n",
					        Lower( ErrorString( error ) ) );

    SetBreakProcess( stdin, 0 );

    for(;;)	/* main test loop -- get parameters and perform test */
      {
	for(;;)		/* loop until first prompt satisfied */
	  {
	    if( Prompt( inputWord,
		"\nreceiver host name, 'local', 'sameteam', or 'quit'? [%s] ",
					   ReceiverHost ) == NULL ) goto quit;

	    if( Equal( inputWord, "quit" ) || Equal( inputWord, "q" ) )
		goto quit;

	    if( Equal( inputWord, "" ) )
		strcpy( inputWord, ReceiverHost );
	    else if( Equal( inputWord, SenderHost ) )
		strcpy( inputWord, "local" );
	    else if( Equal( inputWord, "l" ) )
		strcpy( inputWord, "local" );
	    else if( Equal( inputWord, "s" ) )
		strcpy( inputWord, "sameteam" );

	    if( Equal( inputWord, ReceiverHost ) && ValidPid( ReceiverPid ) )
		break;

	    if( ValidPid( ReceiverPid ) )    /* destroy previous receiver */
	      {
		if( Equal( ReceiverHost, "sameteam" ) ) Destroy( ReceiverPid );
		else if( (error = DestroyProcess( ReceiverPid )) != OK )
		  {
		    fprintf( stderr, "warning: can't destroy %s on %s (%s)\n",
			     ServerProg,
			     Equal( ReceiverHost, "local" ) ?
				"this host" : ReceiverHost,
			     Lower( ErrorString( error ) ) );
		  }
		ReceiverPid = 0;
	      }
	
	    if( Equal( inputWord, "sameteam" ) )
	      {
		if( (ReceiverPid = Create ( ReceiverPrio, Receiver,
					          RECEIVER_STACK )) == 0 ||
		    Ready( ReceiverPid, 0 ) == 0 )
		  {
		    fprintf( stderr,
			"can't create receiver process on the same team!\n" );
		    continue;
		  }
		InterHost = FALSE;
		strcpy( ReceiverHost, inputWord );
		break;
	      }
	    else /* separate team */
	      {
		if( Equal( inputWord, "local" ) )
		  {
		    serverSpec.teamServerPid = teamServerPid;
		  }
		else if( (serverSpec.teamServerPid =
				MapRemoteHost( inputWord )) == 0 )
		  {
		    fprintf( stderr,
			 "can't find a team server named '%s'\n", inputWord );
		    continue;
		  }
		ReceiverPid = ExecProgram( argv, &serverSpec,
						NULL, NULL, NULL, &error );
		if( error != OK )
		  {
		    fprintf( stderr, "can't execute %s on %s (%s)\n",
			      ServerProg,
		 	      Equal( inputWord, "local" ) ?
					"this host" : inputWord,
			      Lower( ErrorString( error ) ) );
		    continue;
		  }
		msg->timingCode = SET_TEAM_PRIO;
		msg->interHost  = REAL_TIME1;
		Send( msg, ReceiverPid );
		if( msg->timingCode != OK )
		  {
		    fprintf( stderr,
			"warning: can't set receiver team %s (%s)\n",
				     "priority to REAL_TIME1",
				     Lower( ErrorString( msg->timingCode ) ) );
		  }

		msg->timingCode = SET_PROC_PRIO;
		msg->interHost  = ReceiverPrio;
		Send( msg, ReceiverPid );
		if( msg->timingCode != OK )
		  {
		    fprintf( stderr,
			"warning: can't set receiver process %s %u (%s)\n",
				     "priority to", ReceiverPrio,
				     Lower( ErrorString( msg->timingCode ) ) );
		  }

		InterHost = !Equal( inputWord, "local" );
		strcpy( ReceiverHost, inputWord );
		break;
	      }
	  }

	if( !InterHost ) for(;;) /* loop until second prompt satisfied */
	  {
	    if( Prompt( inputWord,
	    "receiver at higher, same, or lower priority than sender? [%s] ",
		   ReceiverPrio == LOWER_PRIO ? "lower" :
		   ReceiverPrio == SAME_PRIO  ? "same" : "higher" ) == NULL )
								     goto quit;
	    if( Equal( inputWord, "" ) ) break;

	    if( Equal( inputWord, "lower" ) || Equal( inputWord, "l" ) )
		inputNumber = LOWER_PRIO;
	    else if( Equal( inputWord, "same" ) || Equal( inputWord, "s" ) )
		inputNumber = SAME_PRIO;
	    else if( Equal( inputWord, "higher" ) || Equal( inputWord, "h" ) )
		inputNumber = HIGHER_PRIO;
	    else continue;

	    if( inputNumber == ReceiverPrio ) break;

	    msg->timingCode = SET_PROC_PRIO;
	    msg->interHost  = inputNumber;
	    Send( msg, ReceiverPid );
	    if( msg->timingCode != OK )
	      {
		fprintf( stderr,
			"can't change receiver process priority to %u (%s)\n",
			inputNumber, Lower( ErrorString( msg->timingCode ) ) );
		continue;
	      }

	    ReceiverPrio = inputNumber;
	    break;
	  }

	for(;;)		/* loop until third prompt satisfied */
	  {
	    if( Prompt( inputWord,
			"segment size in bytes, K bytes, or M bytes? [%u] ",
				(char *) SegmentSize ) == NULL ) goto quit;

	    if( Equal( inputWord, "" ) ) inputNumber = SegmentSize;

	    else if( sscanf( inputWord, "%d%c", &inputNumber, &inputLetter )
								!= 2 ) continue;
	    else
	      {
		if( inputLetter == '\0' )     ;
		else if( inputLetter == 'k' ) inputNumber *= 1024;
		else if( inputLetter == 'm' ) inputNumber *= 1048576;
		else continue;
	      }

	    if( inputNumber == SegmentSize && SegmentPtr != NULL ) break;

	    if( SegmentPtr != NULL ) free( SegmentPtr );

	    if( inputNumber == 0 )
	      {
		SegmentPtr = NULL;
		SegmentSize = 0;
		break;
	      }
	    if( (SegmentPtr = malloc( inputNumber )) != NULL )
	      {
		SegmentSize = inputNumber;
		break;
	      }
	    fprintf( stderr, "can't allocate %u byte segment\n", inputNumber );
	  }

	if( SegmentSize == 0 )
	  {
	    TypeOfTest = MSG_TEST;
	  }
	else for(;;)	/* loop until fourth prompt satisfied */
	  {
	    if( Prompt( inputWord, "read or write? [%s] ",
			(TypeOfTest == SHORT_WRITE_TEST ||
			 TypeOfTest == LONG_WRITE_TEST ) ? "write" : "read" )
							   == NULL ) goto quit;

	    if( Equal( inputWord, "" ) ) strcpy( inputWord,
			(TypeOfTest == SHORT_WRITE_TEST ||
			 TypeOfTest == LONG_WRITE_TEST ) ? "write" : "read" );

	    if( Equal( inputWord, "read" ) || Equal( inputWord, "r" ) )
	      {
		TypeOfTest = (SegmentSize <= MAX_APPENDED_SEGMENT) ?
				SHORT_READ_TEST : LONG_READ_TEST;
		break;
	      }
	    if( Equal( inputWord, "write" ) || Equal( inputWord, "w" ) )
	      {
		TypeOfTest = (SegmentSize <= MAX_APPENDED_SEGMENT) ?
				SHORT_WRITE_TEST : LONG_WRITE_TEST;
		break;
	      }
	  }

	for(;;)		/* loop until fifth prompt satisfied */
	  {
	    if( Prompt( inputWord,
	    		"number of messages per trial? [%u] ",
				(char *) MsgsPerTrial ) == NULL ) goto quit;

	    if( Equal( inputWord, "" ) ) break;

	    if( sscanf( inputWord, "%d", &inputNumber ) != 1 ) continue;

	    if( inputNumber == 0 )
	      {
		fprintf( stderr, "get serious\n" );
		continue;
	      }
	    MsgsPerTrial = inputNumber;
	    break;
	  }

	/* estimate minimum trial time and print a warning if > 1 minute */
	/* "with these parameters, each trial will take at least x minutes" */

	for(;;)		/* loop until sixth prompt satisfied */
	  {
	    if( Prompt( inputWord, "number of trials? [%u] ",
				(char *)NumberOfTrials ) == NULL ) goto quit;

	    if( Equal( inputWord, "" ) ) break;

	    if( sscanf( inputWord, "%d", &inputNumber ) != 1 ) continue;

	    NumberOfTrials = inputNumber;
	    break;
	  }

	if( NumberOfTrials == 0 )
	  {
	    fprintf( stderr, "OK, zero trials it is!\n" );
	    continue;	/* restart main loop */
	  }

	if( ValidPid( SenderPid ) )
	  {
	    SetBreakProcess( stdin, SenderPid );
	    Reply( msg, SenderPid );	/* let Sender perform the test */
	  }
	else	/* create Sender process */
	  {
	    if( (SenderPid = Create( SENDER_PRIO, Sender,
						SENDER_STACK )) == 0 )
	      {
		fprintf( stderr, "can't create a sender process!\n" );
		exit( 1 );
	      }
	    senderStack = LastStackAllocated;
	    SetBreakProcess( stdin, SenderPid );
	    if( Ready( SenderPid, 0 ) == 0 )
	      {
		fprintf( stderr, "can't ready the sender process!\n" );
		exit( 1 );
	      }
	  }

	/* wait for Sender to die or report status */
	ReceiveSpecific( msg, SenderPid );

	SetBreakProcess( stdin, 0 );

	if( !ValidPid( SenderPid ) )
	  {
	    /* Sender killed, presumably from ^C -- clean up stack and I/O */
	    /* (copied from "Destroy" library routine) */
	    if( ((PerProcessArea *) senderStack)->ctxname != NULL )
		free( ((PerProcessArea *) senderStack)->ctxname );
	    free( senderStack );

	    Resynch( stderr );
	    fprintf( stderr, "   test interrupted\n" );
	    msg->timingCode = RESYNCH;
	    Send( msg, ReceiverPid );	/* resynchronize Receiver */
	  }
	else if( msg->timingCode != OK )
	  {
	    error = msg->timingCode | SYS_REPLY_CODE;	/* restore high bits */
	    fprintf( stderr, "   trial aborted (%s)\n",
		     error == BAD_STATE ? "receiver out of sync" :
		     error == NO_MEMORY ? "receiver can't allocate segment" :
		     error == BAD_BYTE_COUNT ?
				          "receiver missed appended segment" :
					  Lower( ErrorString( error ) ) );
	    msg->timingCode = RESYNCH;
	    Send( msg, ReceiverPid );	/* resynchronize Receiver */
	  }
      }


quit:
    if( ValidPid( ReceiverPid ) &&
	!Equal( ReceiverHost, "sameteam" ) &&
	(error = DestroyProcess( ReceiverPid )) != OK )
      {
	fprintf( stderr, "warning: can't destroy %s on %s (%s)\n",
		 ServerProg,
		 Equal( ReceiverHost, "local" ) ? "this host" : ReceiverHost,
		 Lower( ErrorString( error ) ) );
      }

    exit( 0 );
  }


/*
 * Prompt:
 * Prints prompt string on stderr and read first word of input line
 * into "inputWord", a character array of MAX_NAME_LEN characters.
 * Returns FALSE on end-of-file, else TRUE.
 */
Prompt( inputWord, promptstring, defaultarg )
    char *inputWord, *promptstring, *defaultarg;
  {
    char inputLine[MAX_NAME_LEN];

    fprintf( stderr, promptstring, defaultarg );
    fflush( stderr );
    if( fgets( inputLine, MAX_NAME_LEN, stdin ) == NULL ) return( FALSE );
    sscanf( inputLine, "%s", inputWord );
    Lower( inputWord );
    return( TRUE );
  }


/*
 * MapRemoteHost:
 * Maps a string name for a host into its corresponding team server 
 * process-id.  0 is returned if no mapping can be performed.
 */
ProcessId MapRemoteHost(host)
    char *host;
  {
    char namebuf[100];
    ContextPair ctx;
    extern SystemCode GetContextId();

    strcpy(namebuf, "[team/");
    strcat(namebuf, host);
    if (GetContextId(namebuf, &ctx) != OK) return 0;

    return(ctx.pid);
  }
