/************************************************************************/
/*                                                                      */
/*                      (C) COPYRIGHT 1983                              */
/*                      BOARD OF TRUSTEES                               */
/*                      LELAND STANFORD JUNIOR UNIVERSITY               */
/*                      STANFORD, CA. 94305, U.S.A.                     */
/*                                                                      */
/************************************************************************/


/*
 * telnet.c - IP User Telnet for use with VGTS or STS.
 *
 * Marvin Theimer and Bill Nowicki December 1982
 *
 * Rewritten by Bill Nowicki April 1983 for stream package
 *
 * Substantially enhancified by Steve Deering, August 1984, for greater
 * conformance to the Telnet specification (RFCs 854, 855, 856, 857, 858,
 * 859, 860, 861, 884, and 901).  This implementation still deviates from
 * the spec in the following ways:
 *
 *	- We don't send a NUL after every CR, nor do we strip out received
 *	  NULs after CRs, regardless of the state of the TRANSMIT_BINARY
 *	  option.
 *
 *	- We pass full 8-bit data in either direction, regardless of the state
 *	  of the TRANSMIT_BINARY option.  However, if the remote host is not
 *	  echoing, we echo locally only legal NVT characters.
 *
 *	- We never buffer input, regardless of the state of the SUPPRESS-GO-
 *	  AHEAD or ECHO options.  Rather, all characters are transmitted
 *	  immediately when typed.
 *
 *	- We provide no means for the user to send a GO-AHEAD, a DO TIMING-
 *	  MARK, or a WILL TIMING-MARK.
 *
 *	- We always reply to any option request, even if the request does
 *	  not ask for a change of state.  We can get away with this as long
 *	  as remote hosts all behave themselves and we never spontaneously
 *	  generate an option request (i.e. there is never any ambiguity
 *	  between received requests and received replies).
 *
 *	- We reply to a SEND TERMINAL-TYPE subnegotiation request even if
 *	  the TERMINAL-TYPE option has not yet been requested.
 *
 * Received option requests are handled as follows:
 *
 *	WILL TRANSMIT-BINARY   - replies DO TRANSMIT-BINARY.
 *	WILL ECHO              - replies DO ECHO and disables local echoing.
 *	WILL SUPPRESS-GO-AHEAD - replies DO SUPPRESS-GO-AHEAD.
 *	WILL <anything else>   - replies DONT <anything else>.
 *
 *	WONT ECHO              - replies DONT ECHO and enables local echoing.
 *	WONT <anything else>   - replies DONT <anything else>.
 *
 *	DO TRANSMIT-BINARY     - replies WILL TRANSMIT-BINARY.
 *	DO SUPPRESS-GO-AHEAD   - replies WILL SUPPRESS-GO-AHEAD.
 *	DO TIMING-MARK         - replies WILL TIMING-MARK.
 *	DO TERMINAL-TYPE       - replies WILL TERMINAL-TYPE,
 *				         SB TERMINAL-TYPE IS <term type> SE.
 *	DO <anything else>     - replies WONT <anything else>.
 *
 *	DONT <anything>        - replies WONT <anything>.
 *
 *	SB TERMINAL-TYPE SEND SE - replies SB TERMINAL-TYPE IS <term type> SE.
 *	SB <anything else> SE    - ignores.
 *
 * Note that two of the officially recommended options, STATUS and EXTENDED-
 * OPTIONS-LIST, are not implemented in either direction.
 */


# include <Venviron.h>
# include <Vioprotocol.h>
# include <Vtermagent.h>
# include <Vgts.h>
# include <Vnet.h>
# include <ctype.h>
# include <Vspinlock.h>

# define TRUE 1
# define FALSE 0

#define ESCAPE_KEY 0x1E		/* command lead-in character is CTL-^ */

#define MAX_TRIES 3     /* max number of IP addresses to try for one host */

#define HelperStackSize 4000

#define LogFileName "telnet.logfile"

# define TelnetPort 23

/*
 * Some ASCII characters
 */
#define NUL 0x00
#define BEL 0x07
#define BS  0x08
#define HT  0x09
#define LF  0x0A
#define VT  0x0B
#define FF  0x0C
#define CR  0x0D
#define DEL 0x7F

/*
 * Telnet commands
 */
#define SE   240
#define NOP  241
#define DM   242
#define BRK  243
#define IP   244
#define AO   245
#define AYT  246
#define EC   247
#define EL   248
#define GA   249
#define SB   250
#define WILL 251
#define WONT 252
#define DO   253
#define DONT 254
#define IAC  255

#define IS   0
#define SEND 1

/*
 * Telnet option codes.
 */
#define TRANSMIT_BINARY   0
#define ECHO              1
#define SUPPRESS_GO_AHEAD 3
#define STATUS            5
#define TIMING_MARK       6
#define TERMINAL_TYPE     24

/*
 * Telnet option state structure
 */
typedef struct
  {
    int debug;		/* TRUE => print debug trace */
    int logging;	/* TRUE => output to logfile */
    int escapeEnable;	/* TRUE => enable user escapes */
    int synchMode;	/* TRUE => Urgent flag received; awaiting DM */
    int remoteEcho;	/* TRUE => remote host echoes its input */
    int exitFlag;	/* TRUE => user wishes to exit program */
    char *termType;	/* character string name of terminal type */
  }
    TelnetState;

/*
 * Character string names of Telnet commands
 */
char *CmdName[] = { "SE", "NOP", "DM", "BRK", "IP", "AO", "AYT", "EC",
		    "EL", "GA", "SB", "WILL", "WONT", "DO", "DONT", "IAC" };

/*
 * A string to check for to temporarily disable logging
 */
char PassString[] = "\r\nPassword:";
char *PassState;

FILE *logfile, *logopen;
SpinLockType spinlock = 0;
int baton = 0;

Telnet( in, out, myaddr, ihost, debugflag, log, escapes, termtype )
    File *in, *out;
    unsigned long myaddr;
    char *ihost;
    int debugflag;
    int escapes;
    int log;
    char *termtype;
  {
	/*
	 * the main routine here, reads from file in and prints on file out,
	 * and handles one connection.
	 * When the connection is actually open, it starts up a helper to
	 * read from the keyboard.
	 */
     register c, i;
     char host[100], buf[120];
     char *p, *index();
     File *tcpInFile, *tcpOutFile, *OpenTcp();
     unsigned long addr, addrs[MAX_TRIES];
     ProcessId helperPid;
     SystemCode error;
     int TelnetHelperProcess();
     char *malloc();
     int naddrs, port;
     TelnetState *state;
 
     state = (TelnetState *) malloc( sizeof( TelnetState ) );
     state->debug = debugflag;
     state->logging = log;
     state->escapeEnable = escapes;
     state->exitFlag = FALSE;
     state->termType = termtype;
     
     logfile = logopen = NULL;
     if (state->logging)
         logopen = fopen(LogFileName,"w");
     state->logging = (logopen!=NULL);
     if (state->logging)
	logfile = logopen;
     else
	logfile = out;

     while (1)
       {
	 SetVgtBanner(out,"telnet");
	 ModifyPad( out, LineBuffer | Echo | LF_Output | CR_Input );

	 if (ihost == NULL)
	   {
	     fprintf(out,"\nWelcome to the Internet. Enter host name: ");
	     if( fgets( host, 100, in ) == NULL ) doquit();
	   }
	 else
	   {
	     strcpy(host, ihost);
	     ihost = NULL;
	   }
	while( host[0] == ' ' || host[0] == '\t' ) Shift_left( host, 1 );
	if( (p = index( host, '\n' )) != NULL ) *p = '\0';
	if( (p = index( host, '\t' )) != NULL ) *p = '\0';
	if( (p = index( host, ' '  )) != NULL ) *p = '\0';
	if( strlen( host ) == 0 ) continue;

	sprintf(buf,"telnet %s",host);
	SetVgtBanner(out,buf);

	naddrs = GetHostIpAddrs( out, host, myaddr, addrs, MAX_TRIES, &port );

	for( i=0; i<naddrs; ++i )
	  {
	    addr = addrs[i];
	    fprintf( out, (port == 0) ? "Trying %d.%d.%d.%d..." :
					"Trying %d.%d.%d.%d port %d...",
			(addr>>24)&0xFF, (addr>>16)&0xFF,
			(addr>>8)&0xFF, addr&0xFF, port );
	    fflush( out );

	    if( (tcpOutFile = OpenTcp( 0, (port==0) ? TelnetPort : port, addr,
		1, DefaultTcpPrecedence, DefaultTcpSecurity, &error )) != NULL
		&&
		(tcpInFile = OpenFile( tcpOutFile->fileserver,
		tcpOutFile->fileid+1,FREAD|FRELEASE_ON_CLOSE,&error)) != NULL)
	      {
		goto connected;
	      }
	    else
	      {
		fprintf(out,"Cannot connect (%s).\n", ErrorString(error));
	      }
	  }
	continue;

connected:
	if( (helperPid = Create(20, TelnetHelperProcess, HelperStackSize))==0)
	  {
	    fprintf(out, "Failed to create telnet helper process.\n" );
	    SpecialClose( tcpOutFile, REL_STANDARD );
	    SpecialClose( tcpInFile,  REL_STANDARD );
	    continue;
	  }

	state->synchMode   = FALSE;       /* set default Telnet options */
	state->remoteEcho  = FALSE;
	PassState = PassString;

	ModifyPad( out, 0 );            /* disable line buffering in pad */
	Ready( helperPid, 5, in, out, tcpInFile, tcpOutFile, state );

	fprintf(out,"Connected.\r\n");
	fflush(out);

	for(;;)
	  {
	    c = getc(tcpInFile);

	    if( c == EOF )
	      {
		if( Eof( tcpInFile ) == BEGIN_URGENT_DATA )
		  {
		    state->synchMode = TRUE;
		    if( state->debug ) fprintf( logfile, "<BEGIN_URGENT" );
		  }
		else if( Eof( tcpInFile ) == END_URGENT_DATA )
		  {
		    if( state->debug ) fprintf( logfile, "<END_URGENT" );
		  }
		else
		  {
		    break;
		  }
		ClearEof(tcpInFile);
	      }
	    else if( c == IAC )
	      {
		TelnetCommand( in, out, tcpInFile, tcpOutFile, state );
	      }
	    else if( state->debug )
	      {
		if (state->logging)
		    putc( c, out);
		AcquireGlobalSpinLock(spinlock);
		if (baton != -1 && state->logging) 
		    { putc( CR, logfile ); 
		      putc( LF, logfile ); 
		      putc( '<', logfile);
		      baton = -1;
		    };
		/* translate non-printables into an escaped notation */
		if( (i = c) > 0x7F )     { putc( '`', logfile ); i &= 0x7F; }
		if( i < ' ' || i > '~' ) { putc( '^', logfile ); i ^= 0x40; }
		putc( i, logfile );
		if( c == LF ) { putc( CR, logfile ); putc( LF, logfile ); }
		ReleaseGlobalSpinLock(spinlock);
	      }
	    else if( !state->synchMode )
	      {
		putc( c, out );
	      }
	    if (c == *PassState) 
	      {
		if (*PassState == '\0')
		    PassState = PassString;
		else 
		  {
		    PassState++;
		    if (*PassState == '\0' && state->logging)
			fprintf(logfile,"<SECRET>");
		  }
	      }
	    else 
	      {
		/* TOPS-20 password prompt has a trailing space */
		if (c != ' ' || *PassState != '\0')
		    PassState = PassString;
	      }
	    if (BufferEmpty(tcpInFile)) fflush(out);
	  }

	fprintf(out, tcpInFile->lastexception == END_OF_FILE ?
			"\r\nConnection closed.\r\n" :
			"\r\nAbnormal disconnect (%s).\r\n",
			    ErrorString(tcpInFile->lastexception));
	fflush( out );
	SpecialClose(tcpOutFile, REL_STANDARD);
	SpecialClose(tcpInFile, REL_STANDARD);
	Destroy(helperPid);

        if( state->exitFlag ) doquit();
       }
  }


TelnetHelperProcess(in,out,tcpInFile, tcpOutFile, state )
    File *in, *out, *tcpInFile, *tcpOutFile;
    TelnetState *state;
  {
    /*
     * reads characters from the keyboard, and interprets the local
     * Telnet commands
     */
    register c, meta;
    int i;

    meta = 0x00;
     
    for(;;)
      {
	if( (c = getc( in )) == EOF ) goto exit;

        if (state->logging && *PassState != '\0') 
	  {
	    i = c;
	    AcquireGlobalSpinLock(spinlock);
	    if (baton != 1) 
		{ 
		  putc( CR, logfile ); 
		  putc( LF, logfile ); 
		  putc( '>', logfile);
		  baton = 1;
		};
	    /* translate non-printables into an escaped notation */
	    if(  c > 0x7F )          { putc( '`', logfile ); i &= 0x7F; }
	    if( i < ' ' || i > '~' ) { putc( '^', logfile ); i ^= 0x40; }
	    putc( i, logfile );
	    if( i == LF ) { putc( CR, logfile ); putc( LF, logfile ); }
	    ReleaseGlobalSpinLock(spinlock);
	  }
	if( ((c |= meta) != ESCAPE_KEY ) || !state->escapeEnable)
	  {
	    TelnetPut( c, out, tcpOutFile, state );
	    meta = 0x00;
	  }
	else { 
	  if (state->logging) 
		fprintf(logfile,"*LOCAL TELNET ESCAPE*");
	  switch( c = getc( in ) )
	  {
	    case '?':
	    case 'h':
		fprintf( out, "\r\n%s%s%s%s%s%s%s%s%s",
"CTL-^ a  send Are You There         CTL-^ m  metafy next character\r\n",
"CTL-^ b  send Break                 CTL-^ o  send Abort Output\r\n",
"CTL-^ c  close connection           CTL-^ s  send Synch\r\n",
"CTL-^ d  toggle debug mode          CTL-^ ?  print this list\r\n",
"CTL-^ e  close connection and exit\r\n",
"CTL-^ g  toggle logging             CTL-^ BS     send Erase Character\r\n",
"CTL-^ h  print this list  	     CTL-^ DEL    send Erase Character\r\n",
"CTL-^ i  send Interrupt Process     CTL-^ CTL-^  send CTL-^ \r\n",
"CTL-^ l  send Erase Line\r\n");
		fflush( out );
		break;

	    case 'a': c = AYT; goto command;

	    case 'b': c = BRK; goto command;

	    case 'e':
exit:		state->exitFlag = TRUE;	       /* fall through to next case */

	    case 'c':
		SpecialClose(tcpOutFile, REL_STANDARD);
		SpecialClose(tcpInFile, REL_STANDARD);
		/* Note: when the internetserver returns
		         with END_OF_FILE, the main Telnet
			 process will destroy this process and
			 reclaim its stack. */
	        break;

	    case 'g':
		state->logging = !state->logging;
		if (state->logging && logopen == NULL) 
		  {
		    logopen = fopen(LogFileName,"w");
		    state->logging = (logopen != NULL);
		  }
		if (state->logging) 
		  {
		    logfile = logopen;
		    state->debug = TRUE;
		  }
		else 
		  {
		    logfile = out;
		  }
		fprintf( out, "\r\nLogging %s.\r\n", state->logging?"on":"off" );
		fflush( out );
		break;
	    case 'd':
		state->debug = !state->debug;
		fprintf( out, "\r\nDebug %s.\r\n", state->debug?"on":"off" );
		fflush( out );
		break;


	    case 'i': c = IP; goto command;

	    case 'l': c = EL; goto command;

	    case 'm':
		meta = 0x80;
		break;

	    case 'o': c = AO; goto command;

	    case 's': c = DM; goto command;

	    case BS:
	    case DEL: c = EC; goto command;	

	    case ESCAPE_KEY:
		TelnetPut( ESCAPE_KEY, out, tcpOutFile, state );
		break;

	    case EOF:
	        goto exit;

	    default:
		TelnetPut( ESCAPE_KEY, out, tcpOutFile, state );
		TelnetPut( c,          out, tcpOutFile, state );
		break;

command:	if( c == DM ) ToggleUrgent( tcpOutFile );
		putc( IAC, tcpOutFile );
		putc( c,   tcpOutFile );
		if( c == DM ) ToggleUrgent( tcpOutFile ); 

		if( state->debug )
		  {
		    fprintf( logfile, c == DM ? ">BEGIN_URGENT>IAC>DM>END_URGENT"
					  : ">IAC>%s", CmdName[c-SE] );
		    fflush( logfile );
		  }
		break;
	   }
	}
	if (BufferEmpty(in)) fflush(tcpOutFile);
      }
  }


TelnetPut( c, out, tcpOutFile, state )
    int c;
    File *out, *tcpOutFile;
    TelnetState *state;
  {
    /*
     * Put a character to the remote host, editing and echoing as required
     * by the current Telnet options.
     */

    putc( c, tcpOutFile );

    if( c == IAC ) putc( IAC, tcpOutFile );

    if( !state->remoteEcho )
      {
	/* only echo legal NVT Printer characters */
	if( c < ' ' || c > '~' ) switch( c )
	  {
	    case NUL:
	    case LF:
	    case CR:
	    case BEL:
	    case BS:
	    case HT:
	    case FF:
	    case VT:
		break;

	    default:
		return;
	  }
	putc( c, out );
	fflush( out );
      }
  }
    




TelnetCommand( in, out, tcpInFile, tcpOutFile, state )
    File *in, *out, *tcpInFile, *tcpOutFile;
    TelnetState *state;
  {
    /*
     * Handle received Telnet commands and option requests.
     */
    register cmd, opt;
    int reply, tmp;

    if( state->debug ) fprintf( logfile, "<IAC" );

    for(;;)
      {
	cmd = getc( tcpInFile );
    
	if( cmd == EOF )
	  {
	    if( Eof( tcpInFile ) == BEGIN_URGENT_DATA )
	      {
		state->synchMode = TRUE;
		if( state->debug ) fprintf( logfile, "<BEGIN_URGENT" );
	      }
	    else if( Eof( tcpInFile ) == END_URGENT_DATA )
	      {
		if( state->debug ) fprintf( logfile, "<END_URGENT" );
	      }
	    else
	      {
		break;
	      }
	    ClearEof(tcpInFile);
	  }
	else break;
      }

    if( state->debug )
      {
	if( cmd >= SE ) fprintf( logfile, "<%s", CmdName[cmd-SE] );
	else            fprintf( logfile, "<%02x", cmd );
      }

    switch ( cmd )
      {
	case DM:
	    state->synchMode = FALSE;
	    break;

	case WILL:
	case WONT:
	case DO:
	case DONT:
	    opt = getc( tcpInFile );
	    if( state->debug )
	      {
		putc( '<', logfile );
		PrintOptionName( opt, logfile );
	      }
	    switch( cmd )
	      {
		case WILL:
		    reply = DO;
		    switch( opt )
		      {
			case TRANSMIT_BINARY:  break;
			case ECHO:             state->remoteEcho= TRUE; break;
			case SUPPRESS_GO_AHEAD:break;
			default:               reply = DONT; break;
		      }
		    break;

		case WONT:
		    reply = DONT;
		    switch( opt )
		      {
			case ECHO: state->remoteEcho =FALSE; break;
		      }
		    break;

		case DO:
		    reply = WILL;
		    switch( opt )
		      {
			case TRANSMIT_BINARY:  break;
			case SUPPRESS_GO_AHEAD:break;
			case TIMING_MARK:      break;
			case TERMINAL_TYPE:    break;
			default:               reply = WONT; break;
		      }
		    break;

		case DONT:
		    reply = WONT;
		    break;
	      }

	    putc( IAC,   tcpOutFile );
	    putc( reply, tcpOutFile );
	    putc( opt,   tcpOutFile );
	    fflush( tcpOutFile );
	    if( state->debug )
	      {
		fprintf( logfile, ">IAC>%s>", CmdName[reply-SE] );
		PrintOptionName( opt, logfile );
	      }
	    if( opt == TERMINAL_TYPE && reply == WILL ) goto send_term_type;
	    break;

	case SB:
	    opt = getc( tcpInFile );
	    cmd = getc( tcpInFile );
	    if( state->debug )
	      {
		putc( '<', logfile );
		PrintOptionName( opt, logfile );
		fprintf( logfile, cmd == SEND ? "<SEND" :
			      cmd == IS   ? "<IS" :
					    "<%02x", cmd );
	      }
	    for(;;)
	      {
		if( (tmp = getc( tcpInFile )) == IAC )
		  {
		    if( (tmp = getc( tcpInFile )) == SE )
		      {
			if( state->debug ) fprintf( logfile, "<IAC<SE" );
			break;
		      }
		    else if( state->debug ) fprintf( logfile, "<IAC" );
		  }
		if( state->debug ) fprintf( logfile, tmp == IAC ? "<IAC" :
							      "<%02x", tmp );
	      }

	    if( opt == TERMINAL_TYPE && cmd == SEND )
	      {
send_term_type:
		putc( IAC, tcpOutFile );
		putc( SB,  tcpOutFile );
		putc( TERMINAL_TYPE, tcpOutFile );
		putc( IS,  tcpOutFile );
		fprintf( tcpOutFile, state->termType );
		putc( IAC, tcpOutFile );
		putc( SE,  tcpOutFile );
		fflush( tcpOutFile );
		if( state->debug )
		    fprintf( logfile, ">IAC>SB>TERMINAL_TYPE>IS>%s>IAC>SE",
				  state->termType );
	      }
	    break;

	case IAC:
	    if( !state->synchMode )
		putc( IAC, out );
	    break;

	default:
	    break;        /* ignore all other Telnet commands */
      }
  }


PrintOptionName( opt, logfile )
    int opt;
    File *logfile;
  {
    fprintf( logfile, opt == TRANSMIT_BINARY   ? "TRANSMIT-BINARY" :
		  opt == ECHO              ? "ECHO" :
		  opt == SUPPRESS_GO_AHEAD ? "SUPPRESS-GO-AHEAD" :
		  opt == STATUS            ? "STATUS" :
		  opt == TIMING_MARK       ? "TIMING-MARK" :
		  opt == TERMINAL_TYPE     ? "TERMINAL-TYPE" :
					     "%02x", opt );
  }


ToggleUrgent( tcpOutFile )
    File *tcpOutFile;
  {
    Message msg;
    ModifyFileRequest *req = (ModifyFileRequest *) msg;

    fflush( tcpOutFile );

    req->requestcode = MODIFY_FILE;
    req->fileid = tcpOutFile->fileid;
    req->whichParm = ModTcpUrgFlag;

    Send( msg, tcpOutFile->fileserver );
  }

/* 
 * Close the logfile before we quit 
 */
doquit()

{
        if (logopen != NULL)
  	    fclose(logopen);
	exit();
}
