/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * ipnamemap.c - map IP host name to list of IP addresses
 *
 * Marvin Theimer and Bill Nowicki October 1983
 * Extensively modified by Steve Deering  May 1984
 *
 * This version uses a compiled-in cache of Stanford addresses and, if that
 * fails, looks for a UNIX 4.2 host file to search.
 */

#include <Vio.h>
#include <ctype.h>

#define MAX_ADDRS 10	/* maximum number of addresses a host may have */

#define MAX_WORD 100	/* maximimum length of word from GetToken */

#define Arpanet   (10 << 24)	/* Arpanet Network (Class A) */
#define SUNet     (36 << 24)	/* Stanford University Network (Class A) */
#define CSD3Net   (36 << 16)	/* Stanford CSD  3Mb Ethernet subnetwork */
#define CSD10Net  ( 8 << 16)	/* Stanford CSD 10Mb Ethernet subnetwork */
#define CSL3Net   (40 << 16)	/* Stanford CSL  3Mb Ethernet subnetwork */
#define CSL10Net  (10 << 16)	/* Stanford CSL 10Mb Ethernet subnetwork */
#define ERL3Net   (20 << 16)	/* Stanford ERL  3Mb Ethernet subnetwork */
#define ERL10Net  (14 << 16)	/* Stanford ERL 10Mb Ethernet subnetwork */
#define CIS3Net   (35 << 16)	/* Stanford CIS  3Mb Ethernet subnetwork */
#define CIS10Net  (22 << 16)	/* Stanford CIS 10Mb Ethernet subnetwork */
#define CSLI3Net  (18 << 16)	/* Stanford CSLI  3Mb Ethernet subnetwork */
#define CSLI10Net ( 9 << 16)	/* Stanford CSLI 10Mb Ethernet subnetwork */
#define LabNet    ( 2 << 16)	/* Stanford Teaching Lab 10Mb subnetwork */
#define SumexNet  (45 << 16)	/* Stanford Sumex 10Mb Ethernet subnetwork */
#define ITSNet    (54 << 16)    /* Stanford ITS 10Mb Ethernet subnetwork */

struct HostAddrType
  {
    char *hostName;
    unsigned long hostAddr;
  }

Hosts[] =  /* cache of hosts called frequently from Stanford CSD */
  {
	/* first, addresses on the Arpanet */

	"sail",      ( Arpanet | 11 ),
	"sumex",     ( Arpanet | 56 ),
	"score",     ( Arpanet | (3 << 16) | 11 ),

	/* then, addresses on the SUNet 3Mb Ethernet trunks */

/*	"pescadero", ( SUNet | CSD3Net | 101 ),  -- disconnected */
	"gregorio",  ( SUNet | CSD3Net | 102 ),
	"labrea",    ( SUNet | CSD3Net | 113 ),
	"sail",      ( SUNet | CSD3Net | 194 ),
	"score",     ( SUNet | CSD3Net | 195 ),
	"navajo",    ( SUNet | CSD3Net | 203 ),
	"carmel",    ( SUNet | CSD3Net | 212 ),

	"glacier",   ( SUNet | CSL3Net | 205 ),
	"fuji",      ( SUNet | CSL3Net | 206 ),
	"star",      ( SUNet | CSL3Net | 207 ),
	"sierra",    ( SUNet | CSL3Net | 213 ),

	"shasta",    ( SUNet | ERL3Net | 192 ),

	"amadeus",   ( SUNet | CIS3Net | 20 ),

	"whitney",   ( SUNet | CSLI3Net | 204 ),

	/* finally, addresses on the various 10Mb Ethernet branches */

	"pescadero", ( SUNet | CSD10Net | 8 ),
	"carmel",    ( SUNet | CSD10Net | 9 ),
	"gregorio",  ( SUNet | CSD10Net | 11 ),
	"score",     ( SUNet | CSD10Net | 46 ),
	"labrea",    ( SUNet | CSD10Net | 47 ),
	"navajo",    ( SUNet | CSD10Net | 48 ),
	"sushi",     ( SUNet | CSD10Net | 53 ),

	"shasta",    ( SUNet | ERL10Net | 1 ),

	"sumex",     ( SUNet | SumexNet | 87 ),
	"diablo",    ( SUNet | SumexNet | 193 ),

	"leland",    ( SUNet | LabNet | 8 ),
	"glacier",   ( SUNet | LabNet | 2 ),

	"amadeus",   ( SUNet | CIS10Net | 20 ),

	"forsythe",  ( SUNet | ITSNet | 12 ),

	NULL,        0
    };


/*
 * NameToIpAddrs:
 * Maps a character-string host name to an array of IP addresses, returned
 * as an array of unsigned long integers (the component bytes are
 * in big-endian order).  The function's value is the number of addresses
 * found for the given host name.  The returned addresses are ordered from
 * "closest" to "furthest" address, relative to the given local address.
 * One address is considered closer than another if it matches more leading
 * bytes of the local address.  If order is not important, a local address
 * of zero may be specified.  On return, the "name" argument reflects
 * any case conversion performed before looking up the address.
 */

NameToIpAddrs( name, localAddr, addrs, addrsSize )
    char          *name;
    unsigned long localAddr, addrs[];
    unsigned int  addrsSize;
  {
    unsigned int  i, j, k, nAddrs;
    unsigned long a, allAddrs[MAX_ADDRS];
    char *Lower();

    Lower( name );

    if( (nAddrs = NameToCachedIpAddrs( name, allAddrs, MAX_ADDRS )) == 0 &&
	(nAddrs = NameToFiledIpAddrs ( name, allAddrs, MAX_ADDRS )) == 0 )
	return( 0 );

    for( i=0; i<nAddrs-1; ++i )  /* sort by straight selection method */
      {
	a = allAddrs[k = i];
	for( j=i+1; j<nAddrs; ++j )
	  {
	    if( IsCloserIpAddr( allAddrs[j], a, localAddr ) )
		a = allAddrs[k = j];
	  }
	allAddrs[k] = allAddrs[i];
	allAddrs[i] = a;
      }

    for( i=j=0; i<addrsSize && j<nAddrs; ++i )  /* copy and remove dup's */
      {
	addrs[i] = allAddrs[j];
	while( ++j < nAddrs && allAddrs[j] == allAddrs[i] ) ;
      }
    return( i );
  }


/*
 * IsCloserIpAddr:
 * Returns non-zero if addr1 is "closer" to local than addr2.
 * Closer is defined as matching more initial bytes of the local address.
 */

IsCloserIpAddr( addr1, addr2, local )
    unsigned long addr1, addr2, local;
  {
    int i, addr1byte, addr2byte, localbyte;

    for( i=24; i>=0; i-=8 )
      {
	addr1byte = (addr1 >> i) & 0xFF;
	addr2byte = (addr2 >> i) & 0xFF;
	localbyte = (local >> i) & 0xFF;

	if( addr1byte != localbyte ) break;
	if( addr1byte == addr2byte ) continue;
	return( 1 );
      }
    return( 0 );
  }


/*
 * NameToCachedIpAddrs:
 * Returns array of addresses matching given host name in compiled-in cache.
 */

NameToCachedIpAddrs( name, addrs, addrsSize )
    char	  *name;
    unsigned long addrs[];
    unsigned int  addrsSize;
  {
    unsigned int i, j;
    char *Lower();

    for( i=j=0; i<addrsSize && Hosts[j].hostName!=NULL; ++j )
      {
	if( strcmp( name, Lower( Hosts[j].hostName ) ) == 0 )
	  {
	    addrs[i++] = Hosts[j].hostAddr;
	  }
      }
    return( i );
  }


/*
 * NameToFiledIpAddrs:
 * Returns array of addresses matching given host name in well-known file
 * of host names.
 */

NameToFiledIpAddrs( name, addrs, addrsSize )
    char	  *name;
    unsigned long addrs[];
    unsigned int  addrsSize;
  {
    int nAddrs;
    File *hostFile;
    SystemCode error;

    /* look for a 4.2bsd UNIX host file */

    if( (hostFile = Open("[sys]/etc/hosts", FREAD+FRELEASE_ON_CLOSE, &error))
	== NULL )
      {
	fprintf( stderr, "Can't open [sys]/etc/hosts: %s\n",
						     ErrorString( error ) );
	return( 0 );
      }

    nAddrs = NameToUnix42IpAddrs( hostFile, name, addrs, addrsSize );
    Close( hostFile );
    return( nAddrs );
  }


/*
 * NameToUnix42IpAddrs:
 * Returns array of all addresses for the given host name found in a
 * 4.2bsd UNIX host file.  The file contains lines of the following format:
 *
 *	a.b.c.d   name1 name2 name3 ...
 *
 * Comments start with '#' and extend to the next newline character.
 */

NameToUnix42IpAddrs( hostFile, name, addrs, addrsSize )
    File	  *hostFile;
    char	  *name;
    unsigned long addrs[];
    unsigned int  addrsSize;
  {
    int nAddrs, a, b, c, d;
    char hostAddr[MAX_WORD], hostName[MAX_WORD];
    char *GetToken(), *Lower();

    nAddrs = 0;
    for(;;)				/* loop through lines of file */
      {
	/* get the address */
	if( GetToken( hostFile, hostAddr ) == NULL ) return( nAddrs );
	if( hostAddr[0] == '\n' ) continue;

	for(;;)				/* loop through names on one line */
	  {
	    if( GetToken( hostFile, hostName ) == NULL ) return( nAddrs );
	    if( hostName[0] == '\n' ) break;

	    if( strcmp( name, Lower( hostName ) ) == 0 )   /* found a match */
	      {
		if( sscanf(hostAddr, "%d.%d.%d.%d", &a, &b, &c, &d ) != 4 )
		  {
		    fprintf( stderr,
	    "Warning: [sys]/etc/hosts has a malformed address '%s' for %s.\n",
			hostAddr, hostName );
		  }
		else
		  {
		    addrs[nAddrs++] = (a<<24) | (b<<16) | (c<<8) | d;
		    if( nAddrs == addrsSize ) return( nAddrs );
		  }
	      }
	  }
      }
  }


/*
 * GetToken:
 *   Utility function used for parsing 4.1 and 4.2 UNIX files.
 *   On end-of-file, NULL is returned; otherwise, word is returned containing:
 *
 *	"\n" on encountering a newline character.
 *	":" or ";" on encountering a colon or semicolon.
 *	a string of other non-white-space characters (truncated to MAX_WORD).
 *
 *   Blanks, tabs, and comments (from '#' to newline or EOF) are discarded.
 */

char *GetToken( f, word )
    File *f;
    char *word;
  {
    register c, i;

    for(;;)
      {
	switch( c = getc( f ) )
	  {
	    case EOF :
		return( NULL );

	    case ' ' :
	    case '\t':
		continue;	/* skip over blanks and tabs */

	    case '\n':
	    case ':' :
	    case ';' :
		word[0] = c;
		word[1] = '\0';
		return( word );

	    case '#' :		/* skip over comments */
		for(;;)
		  {
		    switch( getc( f ) )
		      {
			case EOF :
			    return( NULL );

			case '\n':
			    word[0] = '\n';
			    word[1] = '\0';
			    return( word );
		      }
		  }
	  }
	break;
    }

    for( i=0; i<MAX_WORD; ++i )		/* collect characters of word */
      {
	word[i] = c;
	switch( c = getc( f ) )
	  {
	    case ' ' :
	    case '\t':
		word[i+1] = '\0';
		return( word );

	    case EOF :
	    case '\n':
	    case ':' :
	    case ';' :
	    case '#' :
		ungetc( c, f );
		word[i+1] = '\0';
		return( word );
	  }
      }
    word[i+1] = '\0';
    return( word );
  }
