/* press/press.c
 *
 * last edited by shore on Shasta.Stanford : Fri May 20 11:56:56 1983
 *
 * Highest-level translation routines.
 *
 * CONTENTS:
 *	DocDir		- read and process document directory
 *	Parts		- process the printed pagesin the pard directory
 */


#include <stdio.h>
#include <text.h>
#include "pressformat.h"
#include "show.h"

/* debugging */
#ifdef DEBUG
    int Debug = 1; /* 0 for none, on for verbose output */
#   define tracem(x) if (Debug) fprintf x
#else
#   define tracem(x)
#endif DEBUG
    

/* press file byte-order information */
#   define ByteOrderAlto	1
#   define ByteOrderVax	0
#   define DefaultByteOrder ByteOrderVax

    int swapBytes = DefaultByteOrder;	/* 1 if we swap, 0 if not */

/* global information about the Press file, from the Document directory */
    extern short TotalPages;	/* number of Parts in the Press file */
    long pdStart;	/* record in which Part Directory starts */
    long pdRecs;	/* length of Pard Directory in records */


/* globals used in processing the printed pages */
    long currentPart;	/* [0..TotalPages) */
    long partStart;	/* record number where current page part begins */

static int	isBuffered = 0;	/* 1 if we have a buffered byte (swap only) */
static long	holdChar;	/* the buffered byte in question */


#define MicaToPixel(x) PixelRound(x)
#define MicaToXpos(x) PixelRound(x)
#define MicaToYpos(y) (500-PixelRound(y))

/* DocDir():
 *	verify that dataFile is a press file, setting byte order flag
 *	extract information from the document directory
 */

DocDir()
  {
    unsigned short pwd; /* press password value from file */
    long fLength;	/* length in bytes of Press file */
    long nRecs;		/* length in records of Press file */
    char fileName[52];  /* file name as in Document directory */
    char creator[32];	/* creator of press file */
    char creationDate[40]; /* creation date of press file */
    
    FileSeek(0L, 2);
    fLength = ftell(dataFile);
    if ((fLength == 0) || 
       ((fLength % BytesPerRecord) != 0) ||
       (FileSeek(-512L, 2) < 0))
      { /* improper Press file, can't read password */
        crash("Neither a Press nor a DVI file!\n");
      }
    pwd = GetUWord();
    if (pwd != PressPassword)
        if (pwd != SwappedPressPassword)
            crash("Bad Press Password\n");
        else swapBytes = (!swapBytes);
    nRecs = GetUWord();
    if (nRecs != (fLength/BytesPerRecord))
        crash("Neither a Press nor a DVI file!\n");
    TotalPages = GetUWord();
    pdStart = GetUWord();
    pdRecs = GetUWord();
    FileSeek(0400L-BytesPerRecord, 2);
    BCPLtoCstring(fileName, 0);
    FileSeek(0464L-BytesPerRecord, 2);
    BCPLtoCstring(creator, 0);
    FileSeek(0524L-BytesPerRecord, 2);
    BCPLtoCstring(creationDate, 0);

    /* Start off a document section:
     *    since we can pack of several Press files into one
     *    Postscipt master, we still leave as clean a division as
     *    possible between them, and identify the boundaries clearly
     */ 

    printf("Press source file: %s\n", fileName);
    printf("Created: %s for: %s\n", creationDate, creator);

    tracem((stderr,"%sByte Press File: %d\n",(swapBytes)?"Alto":"Vax",pwd));
    tracem((stderr,"nRecs= %d, TotalPages= %d\n", nRecs, TotalPages));
    tracem((stderr,"pdStart= %d, pdRecs= %d\n", pdStart, pdRecs));
    tracem((stderr,"File:%s For:%s Date:%s\n",fileName,creator,creationDate));

    FreeJunk();
    PartDirectory();
    ProcessFontDirectory();
  } /* DocDir */

PartDirectory()
  { long *page; int i;
    FileSeek(pdStart*BytesPerRecord, 0);
    Count0 = 0;
    PageStart = (long *)(malloc((TotalPages+2)*4));
    for (i = TotalPages; --i >= 0; ) PageStart[i] = -1;
    page = &PageStart[1];
    for (i = 1; i <= TotalPages; i++)
     {
        ushort partType = GetUWord();
	ushort partStart = GetUWord();
	ushort partRecs = GetUWord();
	ushort padding = GetUWord();
	switch (partType)
	  {
	    case PagePart:
		*page++ = partStart + (partRecs<<16) + (padding<<24);
		break;
	    case FontPart:
		fdStart = partStart; fdRecs = partRecs;
		break;
	    default:
	        fprintf(stderr, "Strange Press part type: %d, number $d.\r\n",
		    partType, i);
	  }
     }
    TotalPages = page - PageStart;
  }


/* Parts():
 *	process each printed page part in the part directory
 *	returns the number of pages (printed page parts) in the Press file
 */

/* Routines to translate a single printed page part.
 *
 * CONTENTS:
 *	TranslatePage	- translate the current page
 *	GetChars	- read a string from the data list
 *	ShowString	- image a character string
 *
 */
 
#include "pressformat.h"
#include "spline.h"


/* current-point positioning information, 
 *	depends on how much information we know
 *	Important Note: AbsoluteXY == AbsoluteX+AbsoluteY (|)
 */

#define Relative	0
#define AbsoluteX	1
#define AbsoluteY	2
#define AbsoluteXY	3


/* necessary globals */
    long dataPos;	/* current read position in DL */
    long commandPos;	/* current read position in EL */
    long commandEnd;	/* end of EL commands */
    char string[256];	/* string for Show commands */
    int fontSet;	/* the font set for the current entity */
    
    long currentX, currentY; /* page coords in micas */
    int positioning;	/* {AbsoluteX, AbsoluteY, AbsoluteXY, Relative} */


/* TranslatePage():
 *	process the entities in the current page part 
 */

TranslatePage(p)
  {
    /* Entity List and "Header" information */
        long entityStack[100];	/* start of each entity on the page */
        register long *entityStackEl;	/* current Entity entry */
        long nextEntity;	/* offset of next entity header */

        long entityType;	/* not used in PS translation */
        long dlStart;		/* start of DL entries for this entity */
        long dlLength;		/* bytes of DL for this entity */
        register long Xe, Ye;	/* entity origin (Micas) */
        long bbLeft, bbBottom, bbWidth, bbHeight;
                        	/* entity BBox info -- not used in PS */
        long length;		/* EL words used by this entity */

    /* general purpose EL command data */
        register long opcode;	/* the current EL command */

    /* command specific data */
        register long n;	/* length for Show's and Skip's */
        long altTypes, altELBytes, altDLBytes; /* alternative info */
        long copy;		/* only-on-copy number */
        register long dx, dy;	/* values for Set(X|Y) in entity coords */
        long objectLength;	/* DL length for ShowObject data */
        long dotsLength;	/* DL length for ShowDots data */
        long width, height;	/* ShowRectangle dimensions */
        long spaceX, spaceY;	/* values for SetSpace commands */
        long font;		/* current font within fontSet */
        long brightness, hue, saturation; /* current color info */

    /* other translation state information */
        int skippingAlternative;	/* flag for skipping Alternatives */

    /* warning messages flags */
        static int onlyOnCopyWarning = 0;
        static int dotsWarning = 0;
        static int colorWarning = 0;

    /* external function references */
        float BCPLtoCfloat();	/* for DrawCurve arguments */
        
    entityStackEl = entityStack;
{ int i; /* compiler error hack!!! */
     i=   PressPart(p);
     i += PageRecs(p);
i*=BytesPerRecord;
i-=EntityPadding(p)*2;
  *entityStackEl = i;
}
    for (;;)
      {
        register long length, nextEntity;
        FileSeek((*entityStackEl)-2, 0); /* get the length */
        if ((length = GetUWord()) == 0) break; /* no more entities */
        nextEntity = (*entityStackEl) - length*2;
        *(++entityStackEl) = nextEntity;
      }

    /* interpret the entities on the stack in stack order */
    do
      {
        entityStackEl--;
        FileSeek((*entityStackEl)-EntityHeadSize, 0);
        entityType = GetUByte();
        fontSet = GetUByte();
        dlStart = GetUDouble();	dlLength = GetUDouble();
        Xe = GetWord();		Ye = GetWord();
        bbLeft = GetWord();	bbBottom = GetWord();
        bbWidth = GetWord();	bbHeight = GetWord();
        length = GetUWord();

        dataPos = PressPart(p)*BytesPerRecord+dlStart;
        commandPos = (*entityStackEl)-length*2;
        commandEnd = (*entityStackEl)-EntityHeadSize;
        
        tracem((stderr," Entity %d: type %d," length, entityType));
        tracem((stderr," fontSet %d, (%ld, %ld)  ",fontSet,dlStart,dlLength));
        tracem((stderr," Xe=%d, Ye=%d, BB[%d %d %d %d]", Xe, Ye, bbLeft, bbBottom, bbWidth, bbHeight));
        tracem((stderr," commands(%ld, %ld) data %ld\n",commandPos,commandEnd,dataPos));
        
        /* Each entity implies: 
         *	brightness = 0; hue = <black>; saturation = 0377;
         * 	Font 0; ResetSpace; SetX 0; SetY 0; OnlyOnCopy 0;
         * so do it!
         */
        brightness = 0;
        hue = saturation = 0377;
        /* SetFont(0) is NOT done now.  It is defered until it is used, since
         * very often, Press files set the font explicitely and we avoid the
         * redundancy, also, sometimes font 0 is not defined (and never used)
         * for this font set */
        CurFont = NULL;
        currentX = Xe;
        currentY = Ye;
        positioning = AbsoluteXY;
        copy = 0;
	Spacing = InvalidWidth;

        FileSeek(commandPos, 0);
        while (commandPos < commandEnd)
          {
            opcode = GetUByte();
            commandPos++;

            if ((ShowShort <= opcode) && (opcode < SkipShort))
              { /* Show-Characters-Short */
                n = opcode-ShowShort+1;
                GetChars(n);
                tracem((stderr,"  ShowShort %d  `%s'\n", n, string));
                ShowString(string);
              }
            else if (opcode < ShowSkip)
              { /* Skip-Characters-Short */
                n = opcode-SkipShort+1;
                dataPos += n;
                tracem((stderr,"  Skip %d\n", n));
              }
            else if (opcode < SetSpXShort)
              { /* Show-Characters-and-Skip */
                n = opcode-ShowSkip+1;
                GetChars(n);
                dataPos++;
                tracem((stderr,"  ShowAndSkip %n %s\n", n, string));
                ShowString(string);
              }
            else if (opcode < SetSpYShort)
              { /* Set-Space-X-Short */
                spaceX = opcode-SetSpXShort*256;
                spaceX += GetUByte();
                commandPos++;
                ZapSpace(spaceX, SpaceX);
                tracem((stderr,"  SpaceXShort %d\n", spaceX));
              }
            else if (opcode < Font)
              { /* Set-Space-Y-Short */
                spaceY = opcode-SetSpYShort*256;
                spaceY += GetUByte();
                commandPos++;
                tracem((stderr,"  SpaceYShore %d\n", spaceY));
                ZapSpace(spaceY, SpaceY);
              }
            else if (opcode < Available)
              { /* Font */
                font = opcode-Font;
                tracem((stderr,"  Font %d\n", font ));
                SetFont(font);
              }
            else if (opcode < SkipImmediate)
              { /* Available and Spare */
                crash("Undefined Press opcode 0%o\n", opcode);
                tracem((stderr,"  UNUSED 0%o\n", opcode));
              }
            else switch (opcode)
              {
                case SkipImmediate:
                    n = GetUByte();
                    tracem((stderr,"  SkipImm %d\n", n));
                    commandPos += 1 + n;
                    break;

                case Alternative:
                    altTypes = GetUWord();
                    altELBytes = GetUDouble();
                    altDLBytes = GetUDouble();
                    tracem((stderr,"  Alternative %d %d %d\n", 
                            altTypes, altELBytes, altDLBytes));
                    commandPos += 10;
                    if (! skippingAlternative)
                      {/* first alternative -- take it */
                        skippingAlternative = 1;
                      }
                    else if ((! altTypes) && (! altELBytes) && (! altDLBytes))
                      {/* end of alternative */
                        skippingAlternative = 0;
                      }
                    else
                      {/* intermediate alternative, skip over it */
                        commandPos += altELBytes;
                        dataPos += altDLBytes;
                        FileSeek(commandPos, 0); /* actually set read ptr */
                      }
                    break;

                case OnlyOnCopy:
                    copy = GetUByte();
                    commandPos += 1;
                    if (! onlyOnCopyWarning)
                      {
                        fprintf(stderr, "Warning: Only-On-Copy not supported -- ignored\n");
                        onlyOnCopyWarning = 1; /* only one warning per run */
                      }
                    tracem((stderr,"  Copy %d\n", copy));
                    break;

                case SetX:
                    dx = GetWord();
                    commandPos += 2;
                    currentX = Xe+dx;
                    positioning |= AbsoluteX; /* we have an X coord */
                    /* note that we do not actually set the current position
                     * here.  Instead, we keep track of it internally so that
                     * we can collapse SetX and SetY into one PS SetXY if 
                     * possible (in Press, these two commands often, but not
                     * always are used together, e.g. to move to the beginning
                     * of a new line of text).  The commands which add Ink to 
                     * paper may emit the commands to set the current position
                     * if they determine that Set's have occured.
                     */
                    tracem((stderr,"  SetX %d\n", dx));
                    break;

                case SetY:
                    dy = GetWord();
                    commandPos += 2;
                    currentY = Ye+dy;
                    positioning |= AbsoluteY;
                    /* see note on positioning in SetX above */
                    tracem((stderr,"  SetY %d\n", dy));
                    break;

                case ShowCh:
                    n = GetUByte();
                    commandPos++;
                    GetChars(n);
                    ShowString(string);
                    tracem((stderr,"  Show %d %s\n", n, string));
                    break;

                case SkipCh:
                    n = GetUByte();
                    commandPos++;
                    dataPos += n;
                    tracem((stderr,"  SkipCh %d\n", n));
                    break;

                case SkipControl:
                    n = GetUWord(); /* type = GetUByte() */
                    commandPos += 3;
                    dataPos += n;
                    tracem((stderr,"  SkipControl %d\n", n));
                    break;

                case ShowImmediate:
                    string[0] = n = GetByte();
                    string[1] = 0;
                    commandPos++;
                    ShowString(string);
                    tracem((stderr,"  ShowImmediate %c\n", n));
                    break;

                case SpaceX:
                    spaceX = GetWord();
                    commandPos += 2;
                    ZapSpace(spaceX, SpaceX);
                    tracem((stderr,"  SpaceX %d\n", spaceX));
                    break;

                case SpaceY:
                    spaceY = GetWord();
                    commandPos += 2;
                    ZapSpace(spaceY, SpaceY);
                    tracem((stderr,"  SpaceY %d\n", spaceY));
                    break;
                
                case ResetSpace:
                    Reset();
                    tracem((stderr,"  ResetSpace\n"));
                    break;

                case Space:
                    ShowString(" ");
                    tracem((stderr,"  Space\n"));
                    break;

                case SetBrightness:
                    brightness = GetUByte();
                    commandPos++;
                    tracem((stderr,"  Brightness %d\n",brightness));
                    if (!colorWarning)
                      {
                        fprintf(stderr, "Hue, Saturation, Brightness not supported -- ignored\n");
                        colorWarning = 1;
                      }
                    break;

                case SetHue:
                    hue = GetUByte();
                    commandPos++;
                    tracem((stderr,"  Hue %d\n", hue));
                    if (!colorWarning)
                      {
                        fprintf(stderr, "Hue, Saturation, Brightness not supported -- ignored\n");
                        colorWarning = 1;
                      }
                    break;

                case SetSaturation:	
                    saturation = GetUByte();
                    commandPos++;
                    tracem((stderr,"  Saturation %d\n",saturation));
                    if (!colorWarning)
                      {
                        fprintf(stderr, "Hue, Saturation, Brightness not supported -- ignored\n");
                        colorWarning = 1;
                      }
                    break;

                case ShowObject:
                  {
                    long	x, y;	/* current point in the path */
		    long	objectOp;	/* object operation */
                    long	ol;		/* amount of remaining data */

                    printf("newpath\n");
                    ol = objectLength = GetUWord();
                    commandPos += 2;
                    tracem((stderr,"  ShowObj %d\n",objectLength));

                    FileSeek(dataPos, 0);
                    while (objectLength > 0)
                      {
                        objectOp = GetUWord();
                        switch (objectOp)
                          {
                            case Moveto:
                                x = GetWord(); y = GetWord();
                                printf(" %d %d moveto\n", x+Xe, y+Ye);
                                objectLength -= 3; 
                                tracem((stderr, "    Moveto %d %d\n", x, y));
                                break;

                            case Drawto:
                                x = GetWord(); y = GetWord();
                                printf(" %d %d lineto\n", x+Xe, y+Ye);
                                objectLength -= 3;
                                tracem((stderr, "    Drawto %d %d\n", x, y));
                                break;

                            case Drawcurve:
                              {
                                Point c0, c1, c2, c3; /* cubic coeffs */
                                Point b0, b1, b2, b3; /* bezier points */

                                c0.x = x;
                                c0.y = y;
                                c1.x = BCPLtoCfloat(GetDouble());
                                c1.y = BCPLtoCfloat(GetDouble());
                                c2.x = BCPLtoCfloat(GetDouble());
                                c2.y = BCPLtoCfloat(GetDouble());
                                c3.x = BCPLtoCfloat(GetDouble());
                                c3.y = BCPLtoCfloat(GetDouble());
                                CubicToBezier(&b0, &b1, &b2, &b3, c0, c1, c2, c3);
                                printf(" %g %g %g %g %g %g curveto\n",
                                    b1.x+Xe, b1.y+Ye, b2.x+Xe, b2.y+Ye, b3.x+Xe, b3.y+Ye);
                                x = b3.x;
                                y = b3.y;

                                tracem((stderr,"    DrawCurve %g %g %g %g %g %g\n",
                                    c1.x,c1.y,c2.x,c2.y,c3.x,c3.y));
                                objectLength -= 13;
                                break;
                              }
                            default:
                                crash("Illegal Object command %d", objectOp);
                          }
                      }
                    if (objectLength != 0)
                      {
                        crash("Out of synch on Object at %d", dataPos);
                      }
                    dataPos += 2*ol;
                    FileSeek(commandPos, 0); /* back to EL */

                    /* fill the outline with Even-Odd rule */
                    printf("closepath eofill\n");
                    break;
                  }

                case ShowDots:
                    dotsLength = GetUDouble();
                    commandPos += 4;
                    dataPos += dotsLength; /* for now */
                    if (!dotsWarning)
                      {
                        fprintf(stderr,"Show-Dots not yet supported -- ignored\n");
                        dotsWarning = 1; 	/* only one per run */
                      }
                    tracem((stderr,"  ShowDots %ld\n", dotsLength));
                    break;

                case ShowOpaque:
                    dotsLength = GetUDouble();
                    commandPos += 4;
                    dataPos += dotsLength; /* for now */
                    if (!dotsWarning)
                      {
                        fprintf(stderr,"Show-Dots-Opaque not yet supported -- ignored\n");
                        dotsWarning = 1;	/* only one per run */
                      }
                    tracem((stderr,"  ShowDotsO %ld\n",dotsLength));
                    break;

                case ShowRectangle:
                    width = GetWord();
                    height = GetWord();
                    commandPos += 4;
		    Rectangle(MicaToXpos(currentX), MicaToYpos(currentY),
			MicaToPixel(width), MicaToPixel(height));

                    break;

                case Nop:
                    tracem((stderr,"  Nop\n"));
                    break;

                default:
                    tracem((stderr,"***** DEFAULT CASE HIT, ERROR\n"));
                    crash("Can't happen, but it did, hit default opcode case\n");
                    break;
              }
          }
      }
    while (entityStackEl != entityStack);

  } /* TranslatePage */


/* GetChars(length):
 *	stuff the next length characters in the DL into string,
 *	update the DL pointer, and prepare to read commands again
 */

GetChars(length)
    register int length;
  {
    register char *s;
    if (length < 0)
        crash("negative length string at %d %d", commandPos, dataPos);
    s = string;
    
    fseek(auxFile, dataPos, 0);
    if ((dataPos & 1) && swapBytes)
      {
        fseek(auxFile, -1L, 1);
        holdChar = getc(auxFile);
        getc(auxFile);
        isBuffered = 1;
      }
    else isBuffered = 0;

    dataPos += length;
    while (length > 0)
      {
	register char ch;
	if (swapBytes)
	    if (isBuffered)
	      {
		isBuffered = 0;
		ch = holdChar;
	      }
	    else
	      {
		holdChar = getc(auxFile);
		isBuffered = 1;
		ch = getc(auxFile);
	      }
	else ch = getc(auxFile);

	if (ch != ' ' || Spacing == InvalidWidth)
            *s++ = ch;
	else {*s++ = SmallHskip; *s++ = Spacing;}
        length--;
      }
    *s = 0;
  } /* GetChars */


/* ShowString:
 *	fix parenthesis nesting of st and Show it at the current
 *	position, adjusting it as necessary
 */

ShowString(st)
    register char *st;
  {
    register char *s;
    register int	pLevel = 0;	/* paren nesting count : ( => +1, ) => -1 */
    register int	Lcount = 0;	/* number of unmatched )'s */
    register int	len = 0;	/* length of st */
    register int	i;		/* loop index for supplying ()'s */

    extern long currentRotation;	/* orientation of the current font */
    static int saveX, saveY;

    if (CurFont == NULL) SetFont(0);
    PutString(MicaToXpos(currentX), MicaToYpos(currentY), CurFont, st);

    if (currentRotation != 0) printf("%d rotate ", currentRotation);

    if (currentRotation != 0) printf(" %d rotate", -currentRotation);

    positioning = Relative;

  } /* ShowString */

#include "spline.h"

/* CubicToBezier(b0, b1, b2, b3, c0, c1, c2, c3):
 *	converts parametric cubic coefficients to Bezier control points
 */

CubicToBezier(b0, b1, b2, b3, c0, c1, c2, c3)
    Point *b0, *b1, *b2, *b3;	/* bezier points */
    Point c0, c1, c2, c3;	/* cubic coefficients */
  {
    b0->x = c0.x;
    b0->y = c0.y;
    b1->x = c0.x+(c1.x/3.0);
    b1->y = c0.y+(c1.y/3.0);
    b2->x = b1->x+(c1.x+c2.x)/3.0;
    b2->y = b1->y+(c1.y+c2.y)/3.0;
    b3->x = c0.x+c1.x+c2.x+c3.x;
    b3->y = c0.y+c1.y+c2.y+c3.y;

  } /* CubicToBezier */

/* press/byteorder.c
 *
 * Stolen from Andy Shore's Press->PostScript program.
 * Hacked up by Per Bothner
 *
 * routines to handle the ugliness of byte-ordering on different hosts
 * with different-ordered Press files (see the paragraph at the beginning of
 * press.c).
 *
 * CONTENTS:
 *	FileSeek		- a byte-order independant fseek
 *	GetUByte, GetByte	- read an (unsigned) byte (8 bits)
 *	GetUWord, GetWord	- read an (unsigned) word (16 bits)
 *	GetUDouble, GetDouble	- read an (unsigned) long (32 bits)
 *
 */


/* FileSeek(where, how):
 *	like unix fseek, but sets up buffered byte if necessary.
 *	always operates on the Press file (stdin)
 */

FileSeek(where, how)
    long where;	/* relative offset */
    int	 how;	/* flag for base */
  {
    register long pos;
    
    if (fseek(dataFile, where, how) < 0) return(-1);
    pos = ftell(dataFile);
    if ((pos % 2) && swapBytes)
      {
        fseek(dataFile, -1L, 1);
        holdChar = getc(dataFile);
        getc(dataFile);
        isBuffered = 1;
      }
    else isBuffered = 0;
    return(0);
  }


/* GetUByte, GetByte, GetUWord, GetWord, GetUDouble, GetDouble
 *	functions to extract data from Press files 
 *	taking into account the byte-swapping and sign bits
 */

long GetUByte() /* get an Unsigned byte */
  {
    if (swapBytes)
        if (isBuffered)
          {
            isBuffered = 0;
            return (unsigned char)holdChar;
          }
        else
          {
            holdChar = getc(dataFile);
            isBuffered = 1;
	    return (unsigned char)(getc(dataFile));
          }
    else return (unsigned char)(getc(dataFile));
  }


long GetByte() /* get a Signed byte */
  {
    if (swapBytes)
        if (isBuffered)
          {
            isBuffered = 0;
            return((char)holdChar);
          }
        else
          {
            holdChar = getc(dataFile);
            isBuffered = 1;
            return((char)getc(dataFile));
          }
    else return((char)getc(dataFile));
  }


long GetUWord()
  {
    register long word;
    word = (GetUByte() << 8);
    return (word | GetUByte());
  }


long GetWord()
  {
    register long word;
    if ((word = GetByte()) > 128) word -= 256;
    return (word << 8) | (GetUByte());
  }


long GetUDouble()
  {
    register long doub;
    doub = (GetUWord() << 16);
    return (doub | GetUWord());
  }


long GetDouble()
  {
    register long doub;
    doub = (GetWord() << 16);
    return (doub | (GetWord() & 0177777));
  }
