#
/* press/bcpl.c
 *
 * last edited by shore on Shasta.Stanford : Fri May 20 12:37:20 1983
 *
 * Routines to hide the Press/BCPL representations of objects.
 *
 *
 * CONTENTS:
 *	BCPLtoCstring	- convert strings
 *	BCPLtoCfloat	- convert floating point numbers
 */

#include <errno.h>
#include <math.h>

#ifdef FPDebug
#  include <stdio.h>
#  define tracem(x) if (Debug) fprintf x
   extern int Debug;
#else FPDebug
#  define tracem(x)
#endif FPDebug

extern errno;


/* BCPLtoCstring(dest, must):
 *	read a BCPL string (from stdin) and convert it to a C string
 *	at dest, return the string length
 *	make sure you read at least must characters
 *
 *	a BCPL string has a length byte, followed by the bytes text.
 *	In Press files, some are fixed-length fields with variable 
 *	length contents (as in the document directory strings).
 */

int BCPLtoCstring(dest, must)
    register char *dest;
    register int must;
  {
    register int len, l;
    register char *os;

    os = dest; l = len = GetUByte();
    while (l-- > 0) *(os++) = GetByte();
    *os = 0;

    /* advance read pointer to the end of area */
    for(l = must-(len+1); l > 0; l--) GetByte(); 

    return(len);
    
  } /* BCPLtoCstring */



/* BCPLtoCfloat(bfl):
 *	takes a long which is the binary representation of 
 * 	a BCPL Floating Point number (described below), and turns it
 * 	into a VAX C float.
 *
 * 	Press files only contain BCPL Reals in DrawCurve parts of Objects 
 *	outlines as the coefficients for parametric cubics.
 *
 *	Available documentation states that a BCPL Real is in the following
 *	format (like the first 32 bits of a PDP-10 Real):
 *	word 1: (high order 16 bits)
 *		1 bit sign (1 if negative)
 *		8 bit exponent (excess 128, ones complement if sign == 1)
 *		. -- binary point is to the left of the mantissa
 *		7 high order mantissa bits
 *	word 2:	16 low order mantissa bits
 *			normalized, 2s complement if sign == 1
 */

float BCPLtoCfloat(bfl)
    register long bfl;
  {
    register long expon;
    register long mantissa;
    register double power;

    expon = (bfl >> 23) & 0377;	/* get rid of sign */
    mantissa = (bfl & 0x007fffffL);	/* 23 bit mantissa */

    if ((bfl >> 31) & 01)
      {/* sign is 1 -- number is negative */
        expon = 127 - expon;		/* unbias complemented exponent */
        mantissa |= 0xff800000L;	/* sign extend so its negative */
      }
    else expon -= 128;			/* unbias positive exponent */

    tracem((stderr, "%lx %d %lx\n", bfl, expon, mantissa));

    power = pow((double) 2.0E0, (double) expon);

    if (((power == HUGE) && (errno == ERANGE))
    || ((power == 0) && (errno == EDOM)))
        fatal("Error on BCPL Float conversion");

    /* mantissa / 2**23 is the true floating point fraction */
    return ((float) (power * (((float) mantissa) / ((float) 0x00800000L))));

  } /* BCPLtoCfloat */
