/*
 * VEC - Vector font package.  Creates a single-page press file with vectors
 * and text.  Provides the following routines:
 *
 * VecInit("PressFile.press")
 *    Initialize the package, and set up for output to PressFile.press.  uts,
 *    fts resp. values for unit table size. 
 *
 * VecNib (nib)
 *    Sets the current nib (Brush shape and size) for VecLine operations.
 *    See splines.h for valid nibs.
 *
 * VecFont("FONTFAMILY",psize,face)
 *    Set the current font to FONTFAMILY (e.g., HELVETICA, NEWVEC,
 *    SNEWVEC), font point size psize (for vector fonts, this is the width of
 *    the vectors in Dover scan lines, @384/in)
 *
 * VecPosn(x,y)
 *    Position to absolute x,y coordinates, in Dover scan lines.
 *
 * VecLine(dx,dy)
 *    Draw a line from the current (x,y) position to (x+dx,y+dy),
 *    updating current (x,y) position. *    All distances in Dover scan lines.
 *
 * VecSkip(dx,dy)
 *    Like VecLine, only just does incremental reposition.
 *
 * VecText("string")
 *    Put the string argument on the page at current position in the current
 *    font, as determined by VecFont.
 *
 * VecRectangle(x,y,endx,endy,addx,addy)
 *    Create a rectangle from (x,y) to (endx,endy), with addx additional to left
 *    and right, addy additional top and bottom.
 *
 * VecEndEntity()
 *    Force the current Press file entity to be terminated
 *
 * VecFinish("filename",xoff,yoff)
 *    Finish up.  filename is the file name which will be printed on the break
 *    page by the Dover.  xoff, yoff offsets for entire image.
 *
 * The global variables GlobalMaxX, GlobalMinX, GlobalMaxY, GlobalMinY are made
 * available.  They record the global extrema, in units of Dover scan lines,
 * excluding widths of lines. 
 *
 *
 * Copyright 1980,1982 Bruce D. Lucas
 *
 * Translated to C with permission from author by Gustavo A. Fernandez
 *
 * 5/1/85 - GAF - Added rectangle optimization.
 */

#include <stdio.h>
#include "vec.h"
#include "press.h"
#include "splines.h"

#define    REVBufSize 240	/* size of reverse buffer = 255 mod 256 */
#define    FORWARD 1	        /* value of Direction for forward */
#define    REVERSE -1		/* value of Direction for reverse */
#define    THICKEN 0		/* amount to thicken in y direction */

#define    max(a,b) ((a)>(b)?(a):(b))
#define    min(a,b) ((a)<(b)?(a):(b))

#define    SCANSperIN 384	/* Dover scans per inch */
#define    MICASperIN 2540	/* micas per inch */

char REVBuf[REVBufSize+20];	/* buffer for reverse characters */
char FWDBuf[REVBufSize+20];	/* Buffer for revering */
enum Nib curnib;
int nibsize;
int BufPos = 0;			/* next char to write in revbuf */
int CurrX = 0;
int CurrY = 0;			/* current x and y positions*/
int StartX = 0;			/* starting x and Y values in a curve */
int StartY = 0;
int Direction = 0;		/* direction (FORWARD or REVERSE) */
int GlobalMaxX,GlobalMinX,
    GlobalMaxY,GlobalMinY; /* global max/min in Dover scans */

VecInit(FileName)
char *FileName;

{

    PressStart(FileName,1);
    BeginPage();
    BufPos = 0;
    Direction=0;
    GlobalMaxX = 0; GlobalMinX = 9*SCANSperIN;	/* 9in. paper width */
    GlobalMaxY = 0; GlobalMinY = 12*SCANSperIN;	/* 12in. paper height */
}

VecFont
(fontname,psize,face)
char *fontname;

{
    FlushRevBuf();
    PressNewFont(fontname,psize,face);
}



VecNib(nb)
enum Nib nb;

{
    static char *families[] = {
	"SNEWVEC",
	"NEWVEC",
	"HNEWVEC",
	"VNEWVEC"
    };
    static short sizes[] = {
	 2,4,8,16
    };

    curnib=nb;
    nibsize=sizes[(int)nb & 3]>>1;
    VecFont(families[(int)nb >> 2],sizes[(int)nb & 3],0);
}	

VecPosn(x,y)

{

    FlushRevBuf();
    CurrX = x;
    CurrY = y;  
    StartX = x;  
    StartY = y;
    if (CurrX > GlobalMaxX) GlobalMaxX = CurrX;
    if (CurrY > GlobalMaxY) GlobalMaxY = CurrY;
    if (CurrX < GlobalMinX) GlobalMinX = CurrX;
    if (CurrY < GlobalMinY) GlobalMinY = CurrY;
}


VecRectangle(x,y,endx,endy,addx,addy)

{
    int minx,miny,maxx,maxy;

    FlushRevBuf();

    x = PixToMicas(x);
    y = PixToMicas(y);
    endx = PixToMicas(endx);
    endy = PixToMicas(endy);
    addx = PixToMicas(addx);
    addy = PixToMicas(addy);

    minx=min(x,endx)-addx;
    miny=min(y,endy)-addy;
    maxx=max(x,endx)+addx;
    maxy=max(y,endy)+addy;
    PressShowRectangle(minx,miny,maxx-minx,maxy-miny);

    minx=MicasToPix(minx);
    miny=MicasToPix(miny);
    maxx=MicasToPix(maxx);
    maxy=MicasToPix(maxy);

    if (maxx > GlobalMaxX) GlobalMaxX = maxx;
    if (maxy > GlobalMaxY) GlobalMaxY = maxy;
    if (minx < GlobalMinX) GlobalMinX = minx;
    if (miny < GlobalMinY) GlobalMinY = miny;
}



char VectoChar(dx,dy)

{ 

    if (dy > 0) {
	return( 160 + dx - dy - 9*max(dx, dy));
	}
    else {
	return( 160 - dx - dy - 7*max(dx,-dy));
	}
}

VecPut (dx,dy)
{
    if (Direction * dx <0) {
        FlushRevBuf();
    }
    if (Direction==0)
    	Direction=(dx>=0?1:-1);
    if (Direction < 0) {
	REVBuf[BufPos++] = VectoChar(-dx,-dy);
	}
    else {
	REVBuf[BufPos++] = VectoChar( dx, dy);
	}

    CurrX += dx;  
    CurrY += dy;

    if (CurrX > GlobalMaxX) GlobalMaxX = CurrX;
    if (CurrY > GlobalMaxY) GlobalMaxY = CurrY;
    if (CurrX < GlobalMinX) GlobalMinX = CurrX;
    if (CurrY < GlobalMinY) GlobalMinY = CurrY;

    if (BufPos >= REVBufSize)
	FlushRevBuf();

}

VecLine(dx,dy)
/*This is the main routine to translate long lines into vector segments*/

#define sgn(x) ((x)>0?1:((x)<0?-1:0))

{
    register int i,flop,cx,cy,bx,by;
    int ccx,ccy,cbx,cby,adx,ady,len,sx,sy,xy,steps;

/* Take care of trivial cases first. */
    adx=abs(dx);
    ady=abs(dy);
    if (((adx==  0 || adx== 1 || adx== 2 || adx== 4 || adx== 8 || adx== 16) &&
          ady<= adx) ||
        ((ady==  0 || ady== 1 || ady== 2 || ady== 4 || ady== 8 || ady== 16) &&
          adx<= ady)) {
	    VecPut(dx,dy);
	}
    else if (dx==0 || dy==0) {
	bx=CurrX;
	by=CurrY;
	cx=bx+dx;
	cy=by+dy;
	switch ((int)curnib>>2) {
	    case 0: /*square*/
		VecRectangle(bx,by,cx,cy,nibsize,nibsize);
		break;
	    case 1: /*circle*/
		VecPut(0,0);
		VecPosn(cx,cy);
		VecPut(0,0);
		if (dx==0) VecRectangle(bx,by,cx,cy,nibsize,0);
		else       VecRectangle(bx,by,cx,cy,0,nibsize);
		break;
	    case 2: /*dash*/
		VecRectangle(bx,by,cx,cy,nibsize,1);
		break;
	    case 3: /*bar*/
		VecRectangle(bx,by,cx,cy,1,nibsize);
	    	break;
	}
	VecPosn(cx,cy);
    }    
    else
	{
	    steps = adx+ady;
	    bx = 0; by = 0;
	    cx = 0; cy = 0;
	    sx = sgn(dx); sy=sgn(dy);
	    flop = 0;
	    for (i=steps; i>=1; i--) {
		if (flop != 0) {
		   xy=(flop > 0);
		}
		else {
		   xy=(adx > ady);
		};
		if (xy) {
		   cx+=sx;
		   bx++;
		   flop-=ady;
		   }
		else {
		   cy+=sy;
		   by++;
		   flop+=adx;
		   };
	    	len = ((bx>by)?bx:by);
	    	if (len>16) {
		    VecPut(ccx,ccy);
		    cx-=ccx; cy-=ccy;
	 	    bx-=cbx; by-=cby;
		    }
	    	else if (len==16||len==8||len==4||len==2||len==1) {
		ccx=cx; ccy=cy;
		cbx=bx; cby=by;
		};
	    };
	VecPut(ccx,ccy);
	if (cx!=ccx || cy != ccy) {
	    VecLine(cx-ccx,cy-ccy);
	}
    }
}
		


VecSkip(dx,dy)

{

    FlushRevBuf();

    CurrX = CurrX + dx; 
    CurrY = CurrY + dy;
    if (CurrX > GlobalMaxX) GlobalMaxX = CurrX;
    if (CurrY > GlobalMaxY) GlobalMaxY = CurrY;
    if (CurrX < GlobalMinX) GlobalMinX = CurrX;
    if (CurrY < GlobalMinY) GlobalMinY = CurrY;

}


VecText(text)
char *text;

{

    FlushRevBuf();
    PutStringontoPage(PixToMicas(CurrX),PixToMicas(CurrY),text);
}


FlushRevBuf()

{
    int i;

    if (BufPos > 0) {

	if (Direction > 0) {
	    PutCharsontoPage(PixToMicas(StartX),PixToMicas(StartY)
		,REVBuf,BufPos);
	    }
	else {
	    for (i=0;i<BufPos;i++) {
		FWDBuf[i]=REVBuf[BufPos-i-1];
	    }
	    PutCharsontoPage(PixToMicas(CurrX),PixToMicas(CurrY)
		,FWDBuf,BufPos);
	    };
	}
    StartX=CurrX;
    StartY=CurrY;
    BufPos=0;
    Direction=0;
}

VecEndEntity()

{

    FlushRevBuf();
    EndEntity();

}


VecFinish(filename,xoff,yoff)
char *filename;

{

    /* Spruce doesn't like an empty font directory, so fake it */
    VecFont("Helvetica",10,0);

    VecEndEntity();
    PressSetBanner(filename);
    EndPage();
    PressFinish();
}
