/************************************************/
/*						*/
/*	       (C)  COPYRIGHT 1983		*/
/*		BOARD OF TRUSTEES		*/
/*	LELAND STANFORD JUNIOR UNIVERSITY	*/
/*	  STANFORD, CA. 94305, U. S. A.		*/
/*						*/
/************************************************/
 
/*
 *  spline.c
 *
 *  This file contains the definitions and code necessary to generate
 *  arbitrary spline curves of order 1  though 5.  The actual blending
 *  functions are precomputed (see splinegen.c) and evaluated, so that
 *  floating point computation can be avoided as much as possible at
 *  run time.  Note that data points are control points, not true knots.
 *  Actually values of the blending function are kept in XYsplinedefs.c.
 *  The only user-callable routines are XYspline() and XYfillSpline();
 *  NB:  If you update XYspline, update XYfillSpline too!!
 *
 *  Update History:
 *	March 1983 -- Written by David Kaelbling.
 *	10/20/84   -- Modified for use with Vec package by Gustavo Fernandez
 *	11/12/84   -- Fixed 2 point spline bug - GAF
 *       5/1/85    -- no longer blend streight lines. Do a single VecLine!
 *	 5/6/85    -- fixed spline offset bug.
 *	5/16/85    -- Attempt to increase accuracy by repositioning
 *			streight lines every third point to minimize 
 *			incremental roundoff error.
 */

#include "Vgts.h"
#include "splines.h"
#include "vec.h"
#include "press.h"

/* hard-wire draw to press conversion factor for now */

  
  
/* Imports from splinedefs */
extern short XYsplineRes;
extern unsigned short spline2[];
extern unsigned short spline3[];
extern unsigned short spline4[];
extern unsigned short spline5[];
    
 
/* Exports */
extern int VecSpline();
    
    
/* Local Definitions */
static POINT knotvec[5];	/* knot vector cache buffer. */
static short scan;		/* scan index in knot array. */

/*
 *  InitKnots will initialize the knot vector and the scan variable.
 *  Scan is used to determine which knot to move next into the cache
 *  (knotvec), based upon the end conditions of the spline.
 *
 *  In addition, the gauge variable is set to the reflect the resolution
 *  of this particular spline.  It is the number of subintervals * order
 *  to skip during evaluation.
 */
 
static InitKnots( closed, order, numVert, vertex, gauge, dx, dy )
	BOOLEAN closed;
	unsigned short order;
	short numVert;
	POINT *vertex;
	short *gauge;
	register short dx, dy;
  {
    register short i;
    
    
    /* For a closed spline, simply include the first 'order' knots. */
    /* For an open spline, duplicate the first data point 'order' times. */
    if (closed)
      {
        for (i = 0;  i < order;  i++)
          {
	    scan = ((numVert<<1) - order + i) % numVert;
            knotvec[i].x = DrawToPix( vertex[scan].x + dx );
	    knotvec[i].y = DrawToPix( vertex[scan].y + dy );
          }
      }
    else
      {
        for (i = 0;  i < order;  i++)
          {
            knotvec[i].x = DrawToPix( vertex[0].x + dx );
	    knotvec[i].y = DrawToPix( vertex[0].y + dy );
          }
      }
    scan = 0;
    
    /* Set the gauge of the spline */
/*    i = (view->zoom + 8) >> 1; */   i=4;  /* for now GAF */
    i = max(i, 2);
    i = min(i, XYsplineRes);
    *gauge = ((order << XYsplineRes) >> i) - order;
  }

/*
 *  NewKnots will shift the knot vector to the left by one, and
 *  insert a new data point in at the end.  If closed end conditions
 *  are used, the vertex points are simply cycled.  Otherwise, the
 *  last end point is duplicated 'order' (actually as often as necessary)
 *  times.
 */
 
static NewKnots( numVert, vertex, knotNum, closed, order, dx, dy )
	short numVert;
	POINT *vertex;
	int knotNum;
	BOOLEAN closed;
	unsigned short order;
	register short dx, dy;
  {
    register short i;
    
    /* Shift existing data points */
    for (i = 1;  i < order;  i++)
      {
        knotvec[i-1].x = knotvec[i].x;
	knotvec[i-1].y = knotvec[i].y;
      }
    
    /* Insert the new point (if open and at end, don't advance scan) */
    knotvec[order-1].x = DrawToPix( vertex[scan].x + dx );
    knotvec[order-1].y = DrawToPix( vertex[scan].y + dy );
    if ((knotNum > (order - 2)) || (closed))
      {
	if (++scan >= numVert)
	    scan -= numVert;
      }
  }

/*
 *  VecSpline
 *       
 *  Generate an open or closed spline curve of order 2, 3, 4, or 5.
 * 
 */
 
VecSpline( numpoints, vertex, order, closed, dx, dy )
	short numpoints;
	POINT *vertex;
	short order;
	BOOLEAN closed;
	register short dx, dy;
  {
    unsigned int x, y;		/* Working end-points. */
    short newx, newy, oldx, oldy;	/* End-points of segment. */
    int knotNum, v;		/* Used to step through the knots */
    short knot;			/* Stepped for each knot in the support */
    int weight;			/* Current weighting factor */
    unsigned short *Blend;	/* Weighting function */
    short gauge;		/* Gauge of the spline */
    
    if (numpoints <= 0)
      {
        printf("VecSpline:  Illegal number of points %d\n", numpoints);
        return( ERROR );
      }
    if (numpoints < order)
	order = numpoints;	/*fix t point spline bug*/
      
    switch (order) {
	case 1: {
	    int i;
	    for (i=0;i<numpoints;i++) {
		VecPosn(DrawToPix(vertex[i].x+dx),DrawToPix(vertex[i].y+dy));
		VecLine(0,0);
	    }
	    return(0);
	}
	case 2: {
	    int i;
	    VecPosn(DrawToPix(vertex[0].x+dx),DrawToPix(vertex[0].y+dy));
	    for (i=0;i<numpoints-1;i++) {
		if ((i+1)%3 == 0)
	            VecPosn(DrawToPix(vertex[i].x+dx),
			    DrawToPix(vertex[i].y+dy));
		VecLine(DrawToPix(vertex[i+1].x-vertex[i].x),
		        DrawToPix(vertex[i+1].y-vertex[i].y));
	    }
	    if (closed)
		VecLine(DrawToPix(vertex[0].x-vertex[i].x),
		        DrawToPix(vertex[0].y-vertex[i].y));
	    return(0);
	}
        case 3:  Blend = spline3;  break;
        case 4:  Blend = spline4;  break;
        case 5:  Blend = spline5;  break;
        default: printf("XYSPLINE:  Illegal order %d.\n", order);
		 return( ERROR );
      }
      
    /* Initialize the knot cache */
    if (InitKnots( closed, order, numpoints, vertex, &gauge, dx, dy )
		== ERROR)
        return( ERROR );
    
    /* If an open spline, first interval is trivial */
    if (!closed)
      {
	knotNum = numpoints + order + (order == 2) - 4;
        oldx = knotvec[0].x;
	oldy = knotvec[0].y;
	VecPosn(oldx,oldy);
	NewKnots( numpoints, vertex, knotNum, closed, order, dx, dy );
	NewKnots( numpoints, vertex, knotNum, closed, order, dx, dy );
      }
    else
      {
        knotNum = numpoints;
	oldx = oldy = -1;
      }
    
    /* Iterate over the range of interesting knots */
    while (knotNum-- > !closed)
      {
	/* Get knots, and compute on subintervals */
	NewKnots( numpoints, vertex, knotNum, closed, order, dx, dy );
	v = 0;
	while (v <= (order << XYsplineRes))
	  {	
	    /* Combine the weighted knot values */
	    x = y = 1<<15;
	    knot = order;
	    while (knot--)
	      {
	        weight = Blend[v++];
	        x += weight * knotvec[knot].x;
	        y += weight * knotvec[knot].y;
	      }
	    v += gauge;
	
	    /* If a new result, do something */
	    newx = x >> 16;  newy = y >> 16;
	    if ((newx != oldx) || (newy != oldy))
	      {
                if (oldy != -1)
		    VecLine(newx-oldx,newy-oldy);
		else
		    VecPosn(newx,newy);
	        oldx = newx;
	        oldy = newy;
	      }
	  }
      }
    
    /* If open, last interval is trivial too. */
    if (!closed)
	VecLine( knotvec[order-1].x-oldx, knotvec[order-1].y-oldy);

    return( 0 );
  }
