/*
 *  Copyright (C) 1980, Ivor Durham
 *
 *  Translated 1980 from SAIL into C by Tom Rodeheffer (tlr)
 *    with permission of the original author.
 *
 *  Modified for V kernel by Gustavo Fernandez
 *
 *------------------------------------------------------------
 *
 *  HISTORY
 *
 * 21-Oct-84 Added automatic spooling if Null passed to filename in
 *	PressStart. Added V support (GAF)
 *
 *  8-Oct-84  Added font tables and font search routines. (GAF)
 *
 *  6-Oct-84  Gus Fernandez (GAF) at Stanford University
 *	Added PutCharsontoPage to allow strings contining 0 bytes
 *      to be output. This is useful for non-alphabetic fonts. Also
 *      allowed an arbitrary (int) number of characters to be output.
 *	Also deleted a number of error states in favor of "logical" 
 *	alternatives. This allows the calling program to not have to
 *	"Shadow" certain state values such as entityopen and pageopen. Note,
 *	however, that the user must still explicitly begin a new entity
 *	as valueable data is passed and the pen position will probably change.
 *      Copied some useful conversion macros from dfont.h
 *
 * 23-Mar-81  Tom Rodeheffer (tlr) at Carnegie-Mellon University
 *	Corrected the byte order of press files.  Press files had
 *	been being written with bytes swapped in pairs.  This was
 *	clearly wrong, but unfortunately it was the way which
 *	existing software expected press files to be written.  Now
 *	that that software has been repaired (notably, cz), we fix
 *	the byte order here.
 *
 * 12-Feb-81  Tom Rodeheffer (tlr) at Carnegie-Mellon University
 *	Added routines to allow the document directory's banner,
 *	creator, and number of copies to be set.
 *
 * 29-Jan-81  Tom Rodeheffer (tlr) at Carnegie-Mellon University
 *	Undid the edit of 25-Jan-81.  Added instead a routine
 *	PressShowRectangleHere which uses the current xpos and ypos.
 *
 * 25-Jan-81  Tom Rodeheffer (tlr) at Carnegie-Mellon University
 *	Changed PressShowRectangle so that it does not demand
 *	xpos and ypos.
 *
 * Dec-80  Tom Rodeheffer (tlr) at Carnegie-Mellon University
 *	Translation from SAIL to C.
 *
 */

#ifdef VAX
#include <stdio.h>
#else
#include <Vio.h>
#endif
/*#include "string.h"*/
#include <pwd.h>
#include "press.h"

extern char * ctime();
extern char * getenv();
extern long time();
extern struct passwd * getpwuid();
extern char * malloc();



#define bool int
#define TRUE 1
#define FALSE 0
#define DPR "cz"	/* program to spool a press file */

#define local static
#define global

#define PressRecordLength 512

struct FontEntry
{
    short fontnum;		/* Font number=16*set+member */
    char  *family;		/* family name */
    short face;			/* face encoding */
    short size;			/* point size */
    struct FontEntry *next;	/* next font in list or null */
};


struct PressRecord
{
    struct PressRecord *next;
    unsigned char data [PressRecordLength];
};

struct PressStream
{
    int recordcount;              /* total press records */
    int totalbytes;               /* total bytes written */
    bool immediate;               /* output each record when full */
    unsigned char * bytep;        /* address to deposit next byte */
    int freebytes;                /* bytes left in current record */
    struct PressRecord * first;   /* first unwritten press record */
    struct PressRecord * current; /* current press record */
};

struct PagePart
{
    struct PressStream DL;        /* data list */
    struct PressStream EL;        /* entity list */
    int firstrecord;
};

struct Entity
{
    int type;
    int fontset;
    int beginbyte;
    int bytelength;
    int xposition;
    int yposition;
    int left;
    int bottom;
    int width;
    int height;
    int length;
    int currentfont;
    long showpending;            /* number of characters on DL whose */
                                 /* show command is still pending    */
};




local FILE * presschannel = NULL;

local char pressfilename [100];
local char pressbanner [100];
local char creatorname [100];
local char pressdate [100];
local long creationtimestamp;

local int totalrecords;
local int totalparts;
local int partdirrecord;
local int totalcopies;
local int totalfonts;

local struct PagePart currentpage;
local struct Entity currententity;
local struct FontEntry *FontList;

local bool pageopen;
local bool entityopen;

local struct PressStream fontdir;
local struct PressStream partdir;
local struct PressStream docdir;

#define _PrintedPagePart 0
#define _FontDirectoryPart 1



quit(string)
char *string;

{
    fprintf(stderr,string);
    exit();
}

/*
 * Attempt to allocate memory and panic if you do not get any.
 */

local char *
PressAlloc (size)

int size;
{
    register char * p;

    p = malloc(size);

    if (p == NULL)
    {
	quit("PressAlloc: insufficient memory\n\r");
    }

    return (p);
}


#define Alloc(thing)  (thing *) PressAlloc(sizeof (thing))



/*
 * Convert the UNIX date and time into Alto format.
 */

local long
AltoDateTime (datetime)

long datetime;
{
    long adjustment = (1970-1901)*365.24*24*60*60;

    return (datetime + adjustment);
}



local
InitPressStream (ps,immediate)

struct PressStream * ps;
bool immediate;
{
    ps->immediate = immediate;
    ps->recordcount = 0;
    ps->totalbytes = 0;
    ps->first = NULL;
    ps->current = NULL;
    ps->freebytes = 0;
}



/*
 * Write a press record into the press file.
 */ 

local
WritePressRecord (pr)

struct PressRecord * pr;
{
    fwrite(pr->data,PressRecordLength,1,presschannel);
}



local
FlushPressStream (ps)

struct PressStream * ps;
{
    register struct PressRecord * pr;
    register struct PressRecord * nextpr;

    for (pr = ps->first;  pr != NULL;  pr = nextpr) {

	WritePressRecord(pr);
	ps->recordcount++;
	totalrecords++;

	nextpr = pr->next;
	free(pr);
    }

    ps->first = pr;
    if (pr == NULL)
	ps->current = NULL;
}



local
NextRecord (ps)

struct PressStream * ps;
{
    register int i;
    register struct PressRecord * pr;

    if (ps->immediate)
	FlushPressStream(ps);

    pr = Alloc(struct PressRecord);

    pr->next = NULL;
    for (i = 0;  i < PressRecordLength;  i++)
	pr->data[i] = 0;

    if (ps->current == NULL) {
	ps->first = pr;
        ps->current = pr;
    } else {
	ps->current->next = pr;
	ps->current = pr;
    }

    ps->freebytes = PressRecordLength;
    ps->bytep = &pr->data[0];
}



local
PutBytetoStream (ps,b)

struct PressStream * ps;
unsigned char b;
{
    if (ps->freebytes <= 0)
	NextRecord(ps);

    *ps->bytep++ = b;
    ps->freebytes--;
    ps->totalbytes++;
}



local
PutWordtoStream (ps,w)

struct PressStream * ps;
unsigned short w;
{
    PutBytetoStream(ps,(w>>8)&0377);
    PutBytetoStream(ps,w&0377);
}



local
PutLongtoStream (ps,l)

struct PressStream * ps;
unsigned long l;
{
    PutWordtoStream(ps,(l>>16)&0177777);
    PutWordtoStream(ps,l&0177777);
}



local
PutBCPLStringtoStream (ps,str,maxwordlen)

struct PressStream * ps;
char * str;
int maxwordlen;
{
    int bytepadding;
    int strlength;
    int i;

    bytepadding = maxwordlen * 2;

    strlength = strlen(str);
    if (strlength >= bytepadding) {
	strlength = bytepadding - 1;
    }

    PutBytetoStream(ps,(unsigned char) strlength);

    for (i=0; i<strlength; i++)
	PutBytetoStream(ps,(unsigned char) str[i]);

    for (; i<bytepadding-1; i++)
	PutBytetoStream(ps,0);
}



local
EnterPartinDirectory (type,beginningrecord,lengthinrecords,value)

int type;
int beginningrecord;
int lengthinrecords;
int value;
{
    PutWordtoStream(&partdir,(unsigned short) type);
    PutWordtoStream(&partdir,(unsigned short) beginningrecord);
    PutWordtoStream(&partdir,(unsigned short) lengthinrecords);
    PutWordtoStream(&partdir,(unsigned short) value);

    totalparts++;
}



local
FinishPartDirectory ()
{
    partdirrecord = totalrecords;

    FlushPressStream(&partdir);
}

/* replace CMU's version */
foldup(dest,src)
char *dest,*src;

{
    for (;*src!='\0';src++,dest++)
	*dest=(*src>='a'&&*src<='z'?*src-'z'+'Z':*src);
    *dest='\0';
}


global
EnterFontinDirectory (fontset,font,firstchar,lastchar,
                      family,face,source,size,rotation)

int fontset;
int font;
int firstchar;
int lastchar;
char * family;
int face;
int source;
int size;
int rotation;
{
    char uppercasefamily [30];

    foldup(uppercasefamily,family);

    PutWordtoStream(&fontdir,(unsigned short) 16);
    PutBytetoStream(&fontdir,(unsigned short) fontset);
    PutBytetoStream(&fontdir,(unsigned short) font);
    PutBytetoStream(&fontdir,(unsigned short) firstchar);
    PutBytetoStream(&fontdir,(unsigned short) lastchar);
    PutBCPLStringtoStream(&fontdir, uppercasefamily, 10);
    PutBytetoStream(&fontdir,(unsigned short) face);
    PutBytetoStream(&fontdir,(unsigned short) source);
    PutWordtoStream(&fontdir,(unsigned short) size);
    PutWordtoStream(&fontdir,(unsigned short) rotation);
}



local
FinishFontDirectory ()
{
    int firstrecord;
    int wordspadding;

    PutWordtoStream(&fontdir,(unsigned short) 0);

    wordspadding = fontdir.freebytes / 2;
    firstrecord = totalrecords;

    FlushPressStream(&fontdir);

    EnterPartinDirectory(_FontDirectoryPart,
	firstrecord,fontdir.recordcount,wordspadding);
}



local
MakeDocumentDirectory ()
{
    int unused;

    PutWordtoStream(&docdir, 27183);
    PutWordtoStream(&docdir, totalrecords+1);
    PutWordtoStream(&docdir, totalparts);
    PutWordtoStream(&docdir, partdirrecord);
    PutWordtoStream(&docdir, partdir.recordcount);
    PutWordtoStream(&docdir, -1);
    PutLongtoStream(&docdir, AltoDateTime(creationtimestamp));
    PutWordtoStream(&docdir, 1);
    PutWordtoStream(&docdir, totalcopies);
    PutWordtoStream(&docdir, -1);
    PutWordtoStream(&docdir, -1);
    PutWordtoStream(&docdir, -1);

    for (unused=13; unused<=0177; unused++)
	PutWordtoStream(&docdir, -1);

    PutBCPLStringtoStream(&docdir, pressbanner, 26);
    PutBCPLStringtoStream(&docdir, creatorname, 16);
    PutBCPLStringtoStream(&docdir, pressdate, 20);

    FlushPressStream(&docdir);
}



local
DLPutByte (value)

int value;
{
    if (!entityopen) {
	quit("DLPutByte: No entity open!");
    }

    PutBytetoStream(&currentpage.DL,value);

    currententity.bytelength++;
}



local
ELPutByte (value)

int value;
{
    if (!entityopen) {
	quit("ELPutByte: No entity open!");
    }

    PutBytetoStream(&currentpage.EL,value);

    currententity.length++;
}



local
ELPutWord (value)

int value;
{
    if (!entityopen) {
	quit("ELPutWord: No entity open!");
    }

    ELPutByte((value>>8)&0377);
    ELPutByte(value&0377);
}



local
ELPutLong (value)

long value;
{
    if (!entityopen) {
	quit("ELPutLong: No entity open!");
    }

    ELPutWord((value>>16)&0177777);
    ELPutWord(value&0177777);
}





/*
 * Generate all of the pending Show Characters commands
 * which have been accumulating, so that a command can
 * be written to the entity list.
 */

local
FlushShowPending ()
{
    register long count;

    count = currententity.showpending;
    currententity.showpending = 0;

    while (count > 255) {
	ELPutByte(ELShowCharacters);
	ELPutByte(255);
	count -= 255;
    }

    if (count > 32) {
	ELPutByte(ELShowCharacters);
	ELPutByte(count);
    }
    else if (count > 0) {
	ELPutByte(ELShowCharactersShort+count-1);
    }
}



/*
 * Put an character onto the data list and increment the
 * count of Show Characters commands which are pending.
 */

global
PressShowCharacter (c)

char c;
{
    DLPutByte(c);
    currententity.showpending++;
}





global
PressSetX (position)

int position;
{

    FlushShowPending();

    ELPutByte(ELSetX);
    ELPutWord(position);
}



global
PressSetY (position)

int position;
{

    FlushShowPending();

    ELPutByte(ELSetY);
    ELPutWord(position);
}

/* This is the highest level set-font routine. The caller need not worry
about redefining fonts. However, the entity might change on the 17th font
so it is not entirely safe. It is thus best to call PressSetX and PressSetY
right after calling PressNewFont */

global
PressNewFont (NFamily,NSize,NFace)
char *NFamily;

{
    register struct FontEntry *curEntry,*oldEntry;

    curEntry = FontList;
    oldEntry = NULL;
    while (curEntry != NULL)
	if (strcmp(NFamily,curEntry->family) == 0 && NFace==curEntry->face &&
	    NSize == curEntry->size) {
	    return(ChangeFont(curEntry->fontnum));
	    }
	else {
	    oldEntry=curEntry;
	    curEntry=curEntry->next;
	    };
    curEntry=Alloc(struct FontEntry);
    if (oldEntry) {
	oldEntry->next = curEntry;
	}
    else {
	FontList = curEntry;
	};
    curEntry->fontnum=totalfonts++;
    curEntry->family=PressAlloc(strlen(NFamily)+1);
    strcpy(curEntry->family,NFamily);
    curEntry->face=NFace;
    curEntry->size=NSize;
    curEntry->next=NULL;
    EnterFontinDirectory(curEntry->fontnum>>4,curEntry->fontnum&15,
	0,255,curEntry->family,curEntry->face,0,curEntry->size,0);
    return(ChangeFont(curEntry->fontnum));
}

/*
 * This is the dangerous part. ChangeFont does what PressFont does except
 * that it will close and re-open the entity if the font is not in the
 * current set, it will close and reopen a new entity. Returns fontnum
 * if entity was not changed or -fontnum if it was changed.
 */

local 
ChangeFont(FontNum)
int FontNum;

{
    if (!entityopen)
	BeginEntity(0,FontNum>>4,0,0,0,0,MicasPageWidth,MicasPageHeight);
    if (currententity.fontset!=FontNum>>4)
	{
	    EndEntity();
	    entityopen=TRUE;
	    currententity.fontset=FontNum>>4;
	    currententity.length=0;
	    currententity.currentfont=0;
	    currententity.showpending=0;
	    PressFont(FontNum);
	    return(-FontNum);
	}
    else {
	    PressFont(FontNum);
	    return(FontNum);
	}
}

/*The lowest level routine. Assumes same font set*/

global
PressFont (fontnumber)

int fontnumber;
{
    fontnumber &= 017;

    if (fontnumber != currententity.currentfont) {
	FlushShowPending();
	currententity.currentfont = fontnumber;
	ELPutByte(ELFont + fontnumber);
    }
}



global
PressShowRectangleHere (width,height)

int width;
int height;
{
    FlushShowPending();

    ELPutByte(ELShowRectangle);
    ELPutWord(width);
    ELPutWord(height);
}



global
PressShowRectangle (xposition,yposition,width,height)

int xposition;
int yposition;
int width;
int height;
{
    PressSetX(xposition);
    PressSetY(yposition);

    ELPutByte(ELShowRectangle);
    ELPutWord(width);
    ELPutWord(height);
}


global
PutStringontoPage (xposition,yposition,text)
int xposition;
int yposition;
char * text;
{
    register int textlength;

    textlength=strlen(text);
    PutCharsontoPage(xposition,yposition,text,textlength);
}

global
PutCharsontoPage (xposition,yposition,text,tl)

int xposition;
int yposition;
char * text;
int tl;
{
    register int i,textlength;


    PressSetX(xposition);
    PressSetY(yposition);

    for(textlength=tl % 256;tl > 0;tl -= 256,text += 256) {

        if (textlength <= 32) {
            ELPutByte(ELShowCharactersShort+textlength-1);
        } else {
	    ELPutByte(ELShowCharacters);
	    ELPutByte(textlength);
        }

        for (i=0; i<textlength; i++)
	    DLPutByte(text[i]);
    }
}



global
BeginEntity (type,fontset,xabsolute,yabsolute,left,bottom,width,height)

int type;
int fontset;
int xabsolute;
int yabsolute;
int left;
int bottom;
int width;
int height;
{
    if (entityopen) {
	EndEntity(); /*GAF*/
    }

    entityopen = TRUE;

    currententity.type = type;
    currententity.fontset = fontset;
    currententity.beginbyte = currentpage.DL.totalbytes;
    currententity.bytelength = 0;
    currententity.xposition = xabsolute;
    currententity.yposition = yabsolute;
    currententity.left = left;
    currententity.bottom = bottom;
    currententity.width = width;
    currententity.height = height;
    currententity.length = 0;
    currententity.currentfont = 0;
    currententity.showpending = 0;
}



global
EndEntity ()
{
    if (!entityopen) {
	return; /*GAF*/
    }

    FlushShowPending();

    if ((currententity.length & 01) != 0) {
	ELPutByte(ELNop);
    }

    ELPutByte(currententity.type);
    ELPutByte(currententity.fontset);
    ELPutLong(currententity.beginbyte);
    ELPutLong(currententity.bytelength);
    ELPutWord(currententity.xposition);
    ELPutWord(currententity.yposition);
    ELPutWord(currententity.left);
    ELPutWord(currententity.bottom);
    ELPutWord(currententity.width);
    ELPutWord(currententity.height);
    ELPutWord(currententity.length/2 + 1);

    entityopen = FALSE;
}



global
BeginPage ()
{
    if (pageopen) {
	EndPage(); /*GAF*/
    }

    pageopen = TRUE;

    InitPressStream(&currentpage.DL,TRUE);
    InitPressStream(&currentpage.EL,FALSE);

    PutWordtoStream(&currentpage.EL,0); /* entity list starts with 0 */
    entityopen = FALSE;

    currentpage.firstrecord = totalrecords;
}



global
EndPage ()
{
    int trailersize;
    int partsize;

    if (!pageopen)
	return; /*GAF*/

    if (entityopen)
	EndEntity(); /*GAF*/

    trailersize = currentpage.EL.freebytes / 2;

    FlushPressStream(&currentpage.DL);
    FlushPressStream(&currentpage.EL);

    partsize = currentpage.DL.recordcount +
               currentpage.EL.recordcount;

    EnterPartinDirectory(_PrintedPagePart,
        currentpage.firstrecord,partsize,trailersize);

    pageopen = FALSE;
}

local char tempname[30];

global bool
PressStart (filename,copies)

char * filename;
int copies;
{
    static char template[] = "/tmp/drawdover XXXXXXXX";
    register char *p;


    if (presschannel != NULL) {
	quit("PressStart: Can only have one Press file open at a time");
	return (FALSE);
    }

    if (filename == NULL) {
	strcpy(tempname,template);
	mktemp(tempname);
	presschannel = (FILE *)fopen(tempname,"w");
	strcpy(pressfilename,tempname);
    }
    else {
	strcpy(tempname,"");
	presschannel = (FILE *)fopen(filename,"w");
	strcpy(pressfilename,filename);
    }
    

    if (presschannel==NULL)
       quit("PressStart: cannot open file\n\r");

    strcpy(pressbanner,pressfilename);

#ifdef VAX
    strcpy(creatorname,getpwuid(getuid())->pw_gecos);
#else
    strcpy(creatorname,"Random V user");
#endif

    for (p = creatorname;  *p!=0 && *p!='(';  p++)
	;
    *p = 0;


    creationtimestamp = time(0);


    strcpy(pressdate,ctime(&creationtimestamp));

    for (p = pressdate;  *p!=0 && *p!=012;  p++)
	;
    *p = 0;



    if (copies <= 0)
	copies = 1;

    totalcopies = copies;

    totalfonts = 0;
    FontList = NULL;

    InitPressStream(&docdir,FALSE);
    InitPressStream(&partdir,FALSE);
    InitPressStream(&fontdir,FALSE);

    totalrecords = 0;
    totalparts = 0;

    return (TRUE);
}




global
PressSetCopies (copies)

int copies;
{
    totalcopies = copies;
}


global
PressSetCreator (creator)

char * creator;
{
    strcpy(creatorname,creator);
}


global
PressSetBanner (banner)

char * banner;
{
    strcpy(pressbanner,banner);
}






global bool
PressFinish ()
{

#ifndef VAX

    FILE *pfile[2];
    static char *argv [] = {
	"-r",
	"/tmp/drawdover XXXXXX      ",
	0};
#endif

    if (presschannel == NULL) {
	quit("PressFinish: No Press file was being generated");
    }

    if (pageopen) {
	quit("PressFinish: The last page is still open.");
    }

    FinishFontDirectory();
    FinishPartDirectory();
    MakeDocumentDirectory();

    fclose(presschannel);
    presschannel = NULL;

    if (strlen(tempname)!=0) {
	printf("Sending output to the Dover...\n\r");
#ifdef VAX
	if (vfork()==0) { 
	    execlp (DPR, DPR, "-r", pressfilename, 0);
            printf ("Couldn't execute %s.  Press file in %s\n",
		 DPR, pressfilename);
	    _exit();
	}
#else
	strcpy(argv[1],pressfilename);
	if (RemoteExecute(pfile,DPR,argv,FREAD)!=OK) {
            printf ("Couldn't execute %s on the Vax.  Press file in %s\n\r",
		 DPR, pressfilename);
	    exit();
	}
#endif
    }
    return (TRUE);
}



