/*
 *  input.c
 *
 *  This file contains the input handling routines -- the main procedure
 *  of interest is GetInput(), which will get an input event (a mouse
 *  click) and decode it into a menuoption.  The routine UnGetInput()
 *  will provide a single element 'undo' feature.  Additional routines
 *  drive the various popup menus.
 *
 *  David Kaelbling, April 1983
 *  Gustavo Fernandez 11/11/84 - Added Helvetica 12 fonts for Altoread
 */
 
/* Includes */
# ifdef VAX
# include "stdio.h"
# else
# include "Vio.h"
# endif
# include "Vgts.h"
# include "splines.h"
# include "draw.h"
    
    
/* Imports */
extern PrintCommand();
extern ITEM_LIST_PTR *FindObject();
 
 
/* Exports */
extern GetInput();		/* Return a command		*/
extern UnGetInput();		/* Push back a previous command	*/
extern GetNewNib();		/* Select a new drawing nib.	*/
extern GetNewFill();		/* Select a new fill pattern.	*/
extern GetNewText();		/* Choose text font/centering	*/
extern enum TemplateTypes GetTemplate();	/* Get a template type */
extern GetGroup();		/* Select an active group.	*/
extern char *GetString();	/* Return a general string	*/
extern GetMisc();		/* Return a misc. command.	*/
 
 
/* Local Definitions for GetInput & UnGetInput */
static int savedclick = 0;
static struct
  {
    enum MenuOptions	cmd;
    short		x, y, but;
  } SavedClicks[MAXCLICK];

/*
 *  This internal routine will process a click in the main drawing
 *  area, converting the command or coordinates as required by the
 *  mouse buttons.
 */
 
CookMouseKeys( cmd, x, y, but )
	enum MenuOptions *cmd;
	short *x, *y, *but;
  {
    short x1, y1;
    int dist;
    
    /* Process the various combinations */
    switch (*but)
      {
	case LeftButton:			/* Click Here */
	    *cmd = CDataPoint;
	    break;
	    
	case MiddleButton:			/* Click at Sticky Point */
	    if (FindObject( -1, *x, *y, &x1, &y1, &dist, 1 ) == NULL)
	      {
		/* Nothing stick around here. */
		printf("No sticky points nearby.  Try again.\n\r");
		*cmd = CNull;
	      }
	    else
	      {
		*x = x1;  *y = y1;
		*cmd = CDataPoint;
	      }
	    break;
	    
	case RightButton:			/* Click at Grid Point */
	    *cmd = CDataPoint;
	    *x = (*x + 8) & ~0xF;
	    *y = (*y + 8) & ~0xF;
	    break;
	    
	case LeftButton+MiddleButton:		/* Almost Done */
	    *cmd = CAlmostDone;
	    break;
	
	case LeftButton+RightButton:		/* Checkpoint */
	    *cmd = CCheckPoint;
	    break;
	    
	case MiddleButton+RightButton:		/* Undo */
	    *cmd = CUndo;
	    break;
	    
	case LeftButton+MiddleButton+RightButton:	/* Abort */
	    *cmd = CAbort;
	    break;
	    
	default:
	    printf("Internal Error:  Bad mouse buttons %d at (%d, %d)\n\r",
	    		*but, *x, *y);
	    *cmd = CNull;
	    break;
      }
  }

/*
 *  This routine will get an input event an classify it as to which
 *  command it represents.  Input events consist solely of mouse
 *  clicks.  If the click is not near any menu item, and is not in
 *  the drawing area, it is considered garbage and rejected.
 *
 *  The specific mouse buttons used to click the mouse are significant
 *  inside the drawing area (within the menu area they are all identical).
 *  The mouse button meanings are:
 *
 *	Buttons		Command
 *	 - - X		Datapoint, force grid alignment.
 *	 - X -		Datapoint, try to stick to an object.
 *	 X - -		Datapoint, right where the mouse is.
 *	 - X X		Undo -- try to undo last command.
 *	 X - X		Checkpoint -- make a checkpoint right now.
 *	 X X -		AlmostDone -- finish the current object but
 *				not the current command.
 *	 X X X		Abort.
 *
 *  NB:  FindMousedObject returns static storage.  DO NOT FREE IT!!
 */
 
GetInput( pcmd, px, py, pbut )
	enum MenuOptions *pcmd;
	short *px, *py, *pbut;
  {
    register LISTTYPE *hitlist;
    register short i, cmdno;
    short clickVgt;
    
    if (Debug)
	printf("GetInput: ");
    
    /* If there are clicks on the undo stact, return one of them. */
    /* Otherwise, wait for a new mouse click. */
    if (savedclick)
      {
	savedclick--;
	*pcmd = SavedClicks[savedclick].cmd;
	*px = SavedClicks[savedclick].x;
	*py = SavedClicks[savedclick].y;
	*pbut = SavedClicks[savedclick].but;
	if (Debug)
	  {
	    printf("Saved cmd ");
	    PrintCommand( *pcmd );
	    printf(" at (%d, %d), buttons %d\n\r", *px, *py, *pbut);
	  }
	return;
      }
    else
      {
	if (Debug)
	  {
	    printf("Mouse click at ");
	  }
	clickVgt = GetMouseClick( px, py, pbut );
	if (Debug)
	    printf("(%d, %d, %d) in %d", *px, *py, *pbut, clickVgt);
      }
    
    /* Menu item or Data point? */
    if (clickVgt == mainVgt)
      {
	CookMouseKeys( pcmd, px, py, pbut );
	if (Debug)
	  {
	    printf(" == ");
	    PrintCommand( *pcmd );
	    printf("\n\r");
	  }
	return;
      }
    
    /* Something ... else? */
    if (clickVgt != menuVgt)
      {
	printf("Missed!  Please select a menu command.\n\r");
	if (Debug)
	    printf("menuVgt = %d, mainVgt = %d\n\r", menuVgt, mainVgt);
	*pcmd = CNull;
	return;
      }
    
    /* In the menu.  Find out which menu item was closest. */
    *pcmd = CNull;
    hitlist = (LISTTYPE *) FindMousedObject( sdf,*px,*py,menuVgt,JustText );
    if (hitlist->NumOfElements == 0)
      {
	printf("Missed.  Try again.\n\r");
	return;
      }
    
    /* Search the various menu items ... */
    for (i = 1, cmdno = 1; (i < NumCommands) && (*pcmd == CNull); i++,cmdno++)
	if (hitlist->Header->item == Commands[i].itemno)
	    *pcmd = (enum MenuOptions) cmdno;
    for (i = 1; (i < NumObjects) && (*pcmd == CNull); i++, cmdno++)
	if (hitlist->Header->item == Objects[i].itemno)
	    *pcmd = (enum MenuOptions) cmdno;
    for (i = 0; (i < NumStaticMenu) && (*pcmd == CNull); i++, cmdno++)
	if (hitlist->Header->item == StaticMenu[i].itemno)
	    *pcmd = (enum MenuOptions) cmdno;
    for (i = 0; (i < NumDefaultsMenu) && (*pcmd == CNull); i++, cmdno++)
	if (hitlist->Header->item == DefaultsMenu[i].itemno)
	    *pcmd = (enum MenuOptions) cmdno;
    
    if (Debug)
      {
	printf(" == ");
	PrintCommand( *pcmd );
	printf("\n\r");
      }
    
    /* Still nothing? */
    if (*pcmd == CNull)
      {
	printf("Missed.  Try again.\n\r");
      }
  }

/*
 *  This routine will push a set of data back onto the saved click
 *  stack, so that the next call to GetInput will return it instead of
 *  some new data.
 */
 
UnGetInput( cmd, x, y, but )
	enum MenuOptions cmd;
	short x, y, but;
  {
    if (Debug)
      {
	printf("UnGetInput:  ");
	PrintCommand( cmd );
	printf(" at (%d, %d), buttons %d\n\r", x, y, but);
      }
    
    if (savedclick >= MAXCLICK)
      {
	printf("Internal error:  Too many saved commands.  Ignored.\n\r");
	return;
      }
    
    SavedClicks[savedclick].cmd = cmd;
    SavedClicks[savedclick].x = x;
    SavedClicks[savedclick].y = y;
    SavedClicks[savedclick].but = but;
    savedclick++;
  }
    
/* Data structure for the Nib selection pop-up menu. */
static PopUpEntry NibMenu[] =
  {
    "Shape: Square",	0,
    "Shape: Circle",	1,
    "Shape: Dash",	2,
    "Shape: Bar",	3,
    "Size: 0",		4,
    "Size: 1",		5,
    "Size: 2",		6,
    "Size: 3",		7,
    NULL,		0,
  };
  
/*
 *  This routine will return get either a new Nib, or return the old one.
 *  The obvious method of having a complete 16 entry menu won't work, since
 *  the VGTS dies horribly if the popup menu is more than 13 entries long.
 */
 
GetNewNib( curr )
	enum Nib *curr;
  {
    short i;
    
    i = popup( NibMenu );
    if (i >= 0)
      {
	switch (i)
	  {
	    case 0:
	    case 1:
	    case 2:
	    case 3:
		*curr = (enum Nib) ( (i<<2) | ((int) DefaultNib & 3) );
		break;
	    case 4:
	    case 5:
	    case 6:
	    case 7:
		*curr = (enum Nib) ( ((int) DefaultNib & ~3) | (i & 3) );
		break;
	  }
      }
  }

/* Data structure for the Fill Pattern pop-up menus. */
static PopUpEntry FillTopMenu[] =
  {
    NULL,			0,	/* Opaque/Transparent toggle */
    "Striping patterns",	1,
    "Gray tones/area patterns",	2,
    "Textured patterns",	3,
    NULL,			0,
  };
    
static PopUpEntry Fill0[] =
  {	/* striping patterns */
    "Stripes left",		(int) PatStripeLeft,
    "Big Stripes left",		(int) PatBigStripeLeft,
    "Stripes right",		(int) PatStripeRight,
    "Big Stripes right",	(int) PatBigStripeRight,
    "Crosshatch",		(int) PatCrossHatch,
    "Chain-link",		(int) PatChainLink,
    "Columns",			(int) PatColumns,
    "Rows",			(int) PatRows,
    "Matrix crosshatch",	(int) PatMatrix,
    NULL,			0,
  };
    
static PopUpEntry Fill1[] =
  {	/* Gray tones/area patterns */
    "White",			(int) PatWhite,
    "Black",			(int) PatBlack,
    "Gray",			(int) PatGray,
    "Dots",			(int) PatDots,
    "Checkers",			(int) PatCheckers,
    "Square Dots",		(int) PatSquareDots,
    "Pinwheels",		(int) PatPinwheels,
    "Math symbols",		(int) PatMath,
    "Desktop gray",		(int) PatDeskTop,
    NULL,			0,
  };
    
static PopUpEntry Fill2[] =
  {	/* textured patterns */
    "Roses",			(int) PatRoses,
    "Indian Rug",		(int) PatRug,
    "Cubes",			(int) PatCubes,
    "Wicker Texture",		(int) PatWicker,
    "Herringbone",		(int) PatHerringbone,
    "Tangle pattern",		(int) PatTangle,
    "Web pattern",		(int) PatWeb,
    "Weave pattern",		(int) PatWeave,
    "Abstract Trees",		(int) PatTrees,
    NULL,			0,
  };
  
  
static PopUpEntry *FillMenu[] =
  {
    Fill0,
    Fill1,
    Fill2,
  };

/*
 *  This routine will prompt the user for a new fill pattern, and return it.
 *  Once again, the thirteen line limit on popup menus precludes the obvious
 *  single level menu.
 */
 
GetNewFill( curr, opaque )
	enum Pattern *curr;
	short *opaque;
  {
    short i;
    
    /* Diddle the opacity menu entry. */
    if (*opaque)
	FillTopMenu[0].string = "Use translucent filling";
    else
	FillTopMenu[0].string = "Use opaque filling";
    
    /* Get a selection */
    i = popup( FillTopMenu );
    if (i > 0)
      {
	i = popup( FillMenu[i-1] );
	if (i < 0)
	    return;
	*curr = (enum Pattern) i;
      }
    else if (i == 0)
	*opaque = !(*opaque);
  }

/* Data structures for the text default pop-up menus */
static PopUpEntry TextMenu[] =
  {
    "Right Aligned",	PositionRight,
    "Center Aligned",	PositionCenter,
    "Left Aligned",	PositionLeft,
    "Standard Fonts",	3,
    "Unusual Fonts",	4,
    NULL,		0,
  };

/*
 * WARNING - Altoread accesses certain font numbers directly. Check perm.c
 * and init.c as well - GAF
 */
static PopUpEntry Font0Menu[] =
  {
    "Clarity 12",		0,
    "Cream 12",			1,
    "Helvetica 10",		2,
    "Helvetica 10 Bold",	3,
    "Helvetica 18",		4,
    "Helvetica 7",		5,
    "Helvetica 7 Bold",		6,
    "Helvetica 7 Bold Italic",	7,
    "Helvetica 7 Italic",	8,
    "Helvetica 12",		9,
    "Helvetica 12 Bold",	10,
    "Helvetica 12 Italic",	11,
    "TimesRoman 12",		12,
    "TimesRoman 12 Bold",	13,
    "TimesRoman 12 Italic",	14,
    NULL,			0,
  };
    
static PopUpEntry Font1Menu[] =
  {
    "APL 14",			15,
    "Ascii 12",			16,
    "CMR 12",			17,
    "Cyrillic 12", 		18,
    "Gates 32",			19,
    "Greek 12",			20,
    "Hebrew 12",		21,
    "Math 12",			22,
    "Old English 18",		23,
    "Template 64",		24,
    NULL,			0,
  };

/*
 *  This routine will modify one of text centering or default font.
 *
 *  Future work:  Let the user name his own font.
 */
 
GetNewText( center, fontnum )
	short *center;
	short *fontnum;
  {
    short i;
    
    /* Get a selection */
    i = popup( TextMenu );
    if (i < 0)
	return;
    
    /* Changing the text centering default? */
    if (i < 3)
      {
	*center = i;
	return;
      }
    
    /* Changing the default font? */
    if (i == 3)
	i = popup( Font0Menu );
    else
	i = popup( Font1Menu );
    if (i < 0)
	return;
    *fontnum = i;
  }

/* Data structure for the template popup menu */
static PopUpEntry TemplateMenu[] =
  {
    "Wide Open Arrowhead",	(short) WideArrowO,
    "Wide Closed Arrowhead",	(short) WideArrowC,
    "Narrow Open Arrowhead",	(short) NarArrowO,
    "Narrow Closed Arrowhead",	(short) NarArrowC,
    "Rectangle",		(short) Rectangle,
    "Oval",			(short) Oval,
    "Circle",			(short) Circle,
    NULL,			0,
  };

/*
 *  This routine will get and return a template type.
 */
 
enum TemplateTypes GetTemplate()
  {
    register short type;
    
    /* Get a selection */
    type = popup( TemplateMenu );
    if (type < 0)
	return( NullTemplate );
    else
	return( (enum TemplateTypes) type );
  }

/*
 *  This routine will prompt the user for a group name.
 */
 
GetGroup( gptr )
	ITEM_LIST_PTR **gptr;
  {
    PopUpEntry groupmenu[MAXGROUP + 1];
    int i;
    
    if (Groups[0] == NULL)
      {
	printf("No groups to select.\n\r");
	*gptr = NULL;
	return;
      }
    
    for (i = 0;  i < MAXGROUP;  i++)
      {
	groupmenu[i].menuNumber = i;
	if (Groups[i])
	    groupmenu[i].string = Groups[i]->itemdesc->name;
	else
	    groupmenu[i].string = NULL;
      }
    groupmenu[MAXGROUP].string = NULL;
    groupmenu[MAXGROUP].menuNumber = 0;
    
    i = popup( groupmenu );
    if (i < 0 )
	*gptr = NULL;
    else
	*gptr = Groups[i];
  }

/*
 *  This routine will read in a string from the user, possibly
 *  containing control characters, and return a pointer to it.
 *  NULL is returned for an empty string.  ^Q is the quote character.
 */
 
char *GetString()
  {
    char *ptr, c;
    short i, typing;
    
    /* Allocate a buffer */
    if ((ptr = (char *) malloc( MAXLEN + 1 )) == NULL)
      {
	printf("Out of memory.  Couldn't allocate a string.\n\r");
	return( NULL );
      }

    /* Okay, off we go. */
    ResetTTY();
    printf("Enter a string.  Use ^Q to quote characters.\n");
    fflush( stdout );
    typing = 1;
    for (i = 0; (i < MAXLEN) && (typing);)
      {
	ptr[i++] = (c = getchar());
	switch (c)
	  {
	    case '\021':		/* control-Q */
		printf("\b \b");
		fflush( stdout );
		ptr[i - 1] = getchar();	
		break;
		
	    case '\177':		/* delete */
	    case '\b':			/* backspace */
		printf(" \b");
		i = (i > 1 ? i - 2 : 0);
		fflush( stdout );
		break;
	
	    case '\025':		/* control-U */
		i = 0;
		printf("\r");
		fflush( stdout );
		break;
	
	    case '\n':
		ptr[i - 1] = 0;
		typing = 0;
		break;
	
	    default:
		break;
	  }
      }
    
    /* Return the results. */
    if ((i == 0) || (*ptr == 0))
      {
	/* If user aborted, return NULL */
	free( ptr );
	ptr = NULL;
      }
    else
      {
	/* Zero out the rest of the string. */
	while( i <= MAXLEN )
	    ptr[i++] = 0;
      }
	
    GetTTY();
    return( ptr );
  }

/* Data structure for the miscellaneous popup menu. */
static PopUpEntry MiscMenu[] =
  {
    "Clear Screen",		0,
    "Write to a File",		1,
    "Read from a File",		2,
    "Generate a Press File",	3,
    "Debug",			4,
    NULL,			0,
  };

/*
 *  This routine will get the results of a MiscMenu popup.
 */
 
GetMisc( i )
	short *i;
  {
    /* Get a selection, and return it */
    *i = popup( MiscMenu );
    if (*i < 0)
	*i = -1;
  }
