/*
 *  find.c
 *
 *  This file contains the routines used to select an object on the
 *  screen.  The primary routine of interest is FindObject(), which
 *  will locate an object of the given type, based upon the distance
 *  to its sticky points.  The object with the minimum distance to a
 *  control point is selected.
 *
 *  SetCurrentObject() is the standard routine used to hilight and unhilight
 *  existing objects
 *
 *  David Kaelbling, May 1983
 *  Gus Fernandez, May 1985
 */
 
/* Includes */
# ifdef UNIX
# include "stdio.h"
# else
# include "Vio.h"
# endif
# include "Vgts.h"
# include "splines.h"
# include "draw.h"
# include <math.h>
 
 
/* Imports */
extern short GetINum();		/* Get an item number */
extern short GetSeqINum();	/* Get a sequential set of item numbers */
extern short LastINum();	/* Return the last item number gotten
extern short FreeINum();	/* Return an item number to the freelist */

 
/* Exports */
extern ITEM_LIST_PTR *FindObject();
extern SetCurrentObject();
extern SelectObject(); 
extern OpenMainSymbol();
extern CloseMainSymbol();
 
/* Local Definitions */

static BOOLEAN Selecting = FALSE;


/*
 *  This routine will return the square of the distance between two points.
 */
 
int PointDistance( x1, y1, x2, y2 )
	register short x1, y1, x2, y2;
  {
    register short x, y;
    
    x = (x1 - x2);
    y = (y1 - y2);
    return( x * x  +  y * y );
  }

/*
 *  This internal routine is strictly a helper process for SplineDistance.
 *  It will return the proper knot from the simulated knot vector in order
 *  to calculate the sticky points on a spline.  End conditions are dealt
 *  with properly.
 */
 
short Knot( i, order, closed, x, numvert, vert )
	short i, order, closed, x, numvert;
	POINT *vert;
  {
    register short j;			/* Index variable */
    
    /* Compute the proper index into the vertex array. */
    if (closed)
      {
	/* Just circulate the vertices */
	j = i % numvert;
      }
    else
      {
	/* Duplicate beginning and ending knots appropriately. */
	i = i + 3 - order;
	if (i <= 0)
	    j = 0;
	else if (i >= numvert)
	    j = numvert - 1;
	else
	    j = i;
      }
    
    /* Return the value of this control point. */
    if (x)
	return( vert[j].x );
    else
	return( vert[j].y );
  }

/*
 *  This gives the sticky points given the control points
 */
    Sticky(numvert,vert,order,closed,i,cp,xval,yval)
	short numvert;		/* Number of control points. */
	POINT *vert;		/* Array of control points. */
	short order;		/* Order of the spline. */
	BOOLEAN closed;		/* End conditions. */
	short i;		/* index */
	short *xval, *yval;	/* sticky point */
	short cp;		/* Measure from control points too? */
{
    if (cp || order<3 || order>5) {
	*xval=Knot(i,order,closed,1,numvert,vert);
	*yval=Knot(i,order,closed,0,numvert,vert);
    }
    else switch (order)
        {
	    case 3:	/* Sticky point is (p1 + p2) / 2 */
	        *xval = (Knot(i, order, closed, 1, numvert, vert) +
			Knot(i+1, order, closed, 1, numvert, vert)) / 2;
		*yval = (Knot(i, order, closed, 0, numvert, vert) +
			Knot(i+1, order, closed, 0, numvert, vert)) / 2;
		break;
	    case 4:	/* Sticky point is (p1 + 4*p2 + p3) / 6 */
		*xval = (Knot(i, order, closed, 1, numvert, vert) +
			4*Knot(i+1, order, closed, 1, numvert, vert) +
			Knot(i+2, order, closed, 1, numvert, vert)) / 6;
		*yval = (Knot(i, order, closed, 0, numvert, vert) +
			4*Knot(i+1, order, closed, 0, numvert, vert) +
			Knot(i+2, order, closed, 0, numvert, vert)) / 6;
		break;		
	    case 5:	/* Sticky point is (p1 + 11*p2 + 11*p3 + p4) / 24 */
		*xval = (Knot(i, order, closed, 1, numvert, vert) +
			11*Knot(i+1, order, closed, 1, numvert, vert) +
			11*Knot(i+2, order, closed, 1, numvert, vert) +
			Knot(i+3, order, closed, 1, numvert, vert)) / 24;
		*yval = (Knot(i, order, closed, 0, numvert, vert) +
			11*Knot(i+1, order, closed, 0, numvert, vert) +
			11*Knot(i+2, order, closed, 0, numvert, vert) +
			Knot(i+3, order, closed, 0, numvert, vert)) / 24;
		break;
	  }
}

/*
 *  This internal routine, along with its helper Knot(), will compute
 *  the square of the distance from a point to the nearest sticky
 *  point on a spline.
 */
 
int SplineDistance( numvert, vert, order, closed, x, y, dx, dy, cp )
	short numvert;		/* Number of control points. */
	POINT *vert;		/* Array of control points. */
	short order;		/* Order of the spline. */
	BOOLEAN closed;		/* End conditions. */
	short x, y, *dx, *dy;	/* Current point, closes point */
	short cp;		/* Measure from control points too? */
  {
    int curdist;
    register int dist;
    short xval, yval, i;
    short bestx, besty;
    
    /* Initialize */
    order = ((order == 2) ? 3 : order);
    curdist = MAXDIST << 6;
    
    /* Check each sticky point. */
    for (i = -(!closed); i < numvert; i++)
      {
        Sticky(numvert,vert,order,closed,i,0,&xval,&yval);
	
	/* Compute the distance from this sticky point. */
	dist = PointDistance( x, y, xval, yval );
	if (dist < curdist)
	  {
	    bestx = xval;
	    besty = yval;
	    curdist = dist;
	  }
	
	/* Check Control points too? */
	if (cp)
	  {
	    dist = PointDistance( x, y, vert[i].x, vert[i].y );
	    if (dist < curdist)
	      {
		bestx = vert[i].x;
		besty = vert[i].y;
		curdist = dist;
	      }
	  }
      }
    
    /* Return the results */
    *dx = bestx;
    *dy = besty;
    return( curdist );
  }

/*
 *  This internal routine, will determine
 *  whether a point lies inside a spline. (or very close to an edge
 *  We use the parity algorithem which returns TRUE if the spline cuts
 *  a ray leading in a streight line from the given point an odd number of
 *  times.
 */
 
BOOLEAN PointInSpline( numvert, vert, order, closed, x, y, cp)
	short numvert;		/* Number of control points. */
	POINT *vert;		/* Array of control points. */
	short order;		/* Order of the spline. */
	BOOLEAN closed;		/* End conditions. */
	short x, y;		/* Current point, closes point */
	short cp;		/* Measure from control points too? */
  {
    BOOLEAN parity;
    short xval, yval,xval2,yval2, i;
    double fract;
    
    /* Initialize */
    order = ((order == 2) ? 3 : order);
    parity=0;
    
    /* Check each sticky point. */
    for (i = -(!closed); i < numvert; i++) {
        Sticky(numvert,vert,order,closed,i  ,cp,&xval ,&yval );
	Sticky(numvert,vert,order,closed,i+1,cp,&xval2,&yval2);
        if (y>yval && y>yval2) continue;
	if (y<yval && y<yval2 && (x-xval)*(x-xval2) <=0) {
	    parity = !parity;
	    continue;
	}
	if (x<xval && x<xval2) continue;
	if (x>xval && x>xval2) continue;
	if (xval==xval2) continue;
	fract = (double)(x-xval)/(double)(xval2-xval);
	if (y<(yval+fract*(yval2-yval))) {
	    parity = !parity;
	    continue;
	}
    }
    return( parity );
  }

/*
 *  This procedure will return the coordinates of the minimum distance
 *  sticky points on an object, measured from a given point.  The value
 *  of the procedure is the square of the distance.
 */
 
int ObjectDistance( x, y, px, py, ptr, expand, deltax, deltay )
	short x, y, *px, *py;
	register ITEM_DESCRIPTOR *ptr;
	short expand, deltax, deltay;
  {
    short xtmp, ytmp;
    register int dist, curdist = MAXDIST << 6;
    SPLINE *sptr;
    POINT *pptr;
    ITEM_LIST_PTR *gptr;
    
    /* Subtract offset from target point now, */
    /*  add it in to returned point later.  */
    x -= deltax;  y-= deltay;
    switch (ptr->type)
      {
	case TextObj:
	    /* Sticky points are the corners and midpoints of	*/
	    /* the edges of the bounding box.			*/
	    if ((pptr = (POINT *) malloc( sizeof(POINT) * 4 )) == NULL)
	      {
		printf("Out of memory.  Can't compute text distance.\n\r");
		break;
	      }
	    pptr[0].x = ptr->xmin;
	    pptr[0].y = ptr->ymin;
	    pptr[1].x = ptr->xmin;
	    pptr[1].y = ptr->ymax;
	    pptr[2].x = ptr->xmax;
	    pptr[2].y = ptr->ymax;
	    pptr[3].x = ptr->xmax;
	    pptr[3].y = ptr->ymin;
	    dist = SplineDistance( 4, pptr, 2, 1, x, y, &xtmp, &ytmp, 1 );
	    if (dist < curdist)
	      {
		curdist = dist;
		*px = xtmp;  *py = ytmp;
	      }
	    free( pptr );
	    break;
	
	case OpenSplineObj:
	case ClosedSplineObj:
	case OpenPolygonObj:
	case ClosedPolygonObj:
	case TemplateObj:
	    /* A spline.  Measure distance from sticky points. */
	    sptr = (SPLINE *) ptr->data;
	    pptr = &(sptr->head);
	    if (sptr->numvert == 1)
	      {
		/* Only one possible point.  An easy case. */
		dist = PointDistance( x, y, xtmp=pptr[0].x, ytmp=pptr[0].y );
		if (dist < curdist)
	          {
		    *px = xtmp;  *py = ytmp;
		    curdist = dist;
	          }
	      }
	    else
	      {
		/* A hard case.  Call the more elaborate routines. */
		dist = SplineDistance(
			sptr->numvert, pptr, sptr->order, sptr->closed,
			x, y, &xtmp, &ytmp, (sptr->order == 2) );
		if (dist < curdist)
	          {
		    *px = xtmp;  *py = ytmp;
		    curdist = dist;
	          }
	      }
	    break;
	
	case GroupObj:
	    /* If necessary, check all of the objects in the group */
	    if (expand)
	      {
		gptr = (ITEM_LIST_PTR *) ptr->data;
		while (gptr)
		  {
		    dist = ObjectDistance( x, y, &xtmp, &ytmp,
			gptr->itemdesc, 1, gptr->dx, gptr->dy );
		    if (dist < curdist)
		      {
			*px = xtmp;  *py = ytmp;
			curdist = dist;
		      }
		    gptr = gptr->next;
		  }
	      }
	    break;
	    
	default:
	    printf("Internal Error:  Illegal object type %d in ObjDist.\n\r",
	    		ptr->type);
	    break;
      }
    
    /* Compensate for displacements, and return results */
    *px += deltax;  *py += deltay;
    return( curdist );
  }

/*
 *  SelectObject is used to select objects in the field given
 *  the button state and the x/y coordinates.
 */

SelectObject(x,y,but)
short x,y,but;
{
    short dx,dy;
    int distance;

    Selecting = TRUE; /* Kluge to make the VGTS outline optimization work */
    SetCurrentObject(FindObject(-1,x,y,&dx,&dy,&distance,1,1),1);
    Selecting = FALSE;
}

/*
 *  This routine will find the closest suitable object to a given point.
 *  Groups are examined only if the 'expand' flag is set.
 *  Inside will enable point-inside-spline checking which does not
 *  find sticky points.
 */
 
ITEM_LIST_PTR *FindObject( type, x, y, dx, dy, distance, expand, inside )
	short type;
	short x, y;
	short *dx, *dy;
	int *distance;
	short expand,inside;
  {
    register ITEM_LIST_PTR *p, *object;
    int bestDistance, tempDistance;
    short bestx, besty;
    short gotit;
    SPLINE *sptr;
    ITEM_DESCRIPTOR *ptr;
    
    /* Initialize for no object found. */
    object = NULL;
    bestDistance = MAXDIST << 6;
    gotit = 0;
    
    /* Scan the active object list */
    p = activelist->prev;
    while (p)
      {
	/* Shall we even try? */
	if ( ((int) p->itemdesc->type == type) || (type == -1) )
	  {
	    /* Right type.  Right place? */
	    tempDistance = ObjectDistance( x, y, dx, dy,
	    		p->itemdesc, expand, p->dx, p->dy );
	    
	    if (Debug&DebugFind)
		printf("    distance = %d, object number = %d\n\r",
			tempDistance, p->itemdesc->number );
		
	    /* Is this the one we want? */
	    if (tempDistance < bestDistance)
	      {
		bestx = *dx;
		besty = *dy;
		bestDistance = tempDistance;
		object = p;
	      }
	    if (bestDistance <= MAXDIST) gotit=1;
	    ptr=p->itemdesc;
	    if (!gotit && ptr->type != TextObj && ptr->type != GroupObj) {
		sptr = (SPLINE *)ptr->data;
		if (PointInSpline(sptr->numvert,
				  &(sptr->head), sptr->order,sptr->closed,
				  x-p->dx, y-p->dy, 0)) {
		    gotit = 1;
		    object = p;
		}
	    }
	  }
	p = p->next;
      }
    
    /* Debugging Information? */
    if (Debug&DebugFind)
      {
	printf("    Best distance = %d, object number = ", bestDistance);
	if (object)
	    printf("%d\n\r", object->itemdesc->number);
	else
	    printf(" 0\n\r");
      }

    /* Report the results */
    *dx = bestx;  *dy = besty;
    *distance = bestDistance;
    return( gotit ? object : NULL);
  }

/*
 * This routine will set the current object and frame it, or set no
 * current object if NULL
 *
 */


SetCurrentObject(q,refresh)
	ITEM_LIST_PTR *q;
	short refresh;

  {
    ITEM_DESCRIPTOR *p;
    SPLINE *s;
    static short CurrFrameNo;
    if (q==CurrentObject) return;
    if (CurrentObject != NULL)
    UnFrameObject(&CurrFrameNo,refresh);
    if (Selecting)
	CloseMainSymbol(1); /* otherwise the object being un-framed will be
				redrawn */
    if (q != NULL && refresh) {
	FrameObject(q,&CurrFrameNo,1);
	/* We now set the default menu parameters to reflect the newly selected
	   object */
	p = q->itemdesc;
	switch(p->type) {
	    case TextObj:
		if (p->typedata != 3) /* Ignore PressEdit */
		    SetCurrentFont(p->subtype,p->typedata);
		break;
	    case GroupObj:
		break;
	    default:
		s = (SPLINE *) p->data;
		if (s->border)
		    SetCurrentNib(s->nib);
		else
		    SetCurrentNib((enum Nib) 0);
		if (s->filled)
		    SetCurrentPat((int)s->pat + 1);
		else
		    SetCurrentPat(1);
		break;
	}
    }
    CurrentObject = q;
  }

/*
 *  This routine will put a frame of some kind around an object, thus
 *  highlighting it.
 */
 
FrameObject( q, frameno , refresh)
	ITEM_LIST_PTR *q;
	short *frameno;
	short refresh;
  {
    register ITEM_DESCRIPTOR *id = q->itemdesc;
    ITEM_LIST_PTR *p;
    SPLINE *sptr;
    POINT *svert;
    short i, saveSymbol, saveVgt;
    
    if (id == NULL)
	return;
    
    switch (id->type)
      {
	case TextObj:
	    /* Show an expanded bounding box. */
	    OpenMainSymbol();
	    *frameno = AddItem( sdf, GetINum(),
		id->xmin - 3 + q->dx, id->xmax + 3 + q->dx,
		id->ymin - 3 - id->base + q->dy, id->ymax + 3 + q->dy,
		AllEdges, SDF_OUTLINE, NULL );
	    break;
	
	case GroupObj:
	    /* A group.  Frame all of the objects inside the group. */
	    saveSymbol = mainSymbol;
	    saveVgt = mainVgt;
	    CloseMainSymbol(1);
	    *frameno = DefineSymbol(sdf, GetSeqINum(2), "group frame symbol" );
	    EndSymbol( sdf, *frameno, 0 );
	    mainSymbol = *frameno;
	    mainVgt = 0;
	    for (p = (ITEM_LIST_PTR *) id->data;  p;  p = p->next)
		FrameObject( p, &i,0 );
	    CloseMainSymbol(0);
	    mainSymbol = saveSymbol;
	    mainVgt = saveVgt;
	    OpenMainSymbol();
	    AddCall( sdf, (*frameno) + 1, q->dx, q->dy, *frameno );
	    *frameno = - (*frameno);
	    break;
	
	default:
	    /* Some form of spline */
	    sptr = (SPLINE *) id->data;
	    
	    /* For degenerate splines, simply show bounding box. */
#if 0
	    if ((sptr->order == 2) || (sptr->numvert <= 2) ||
	        (id->type == TemplateObj))
	      {
#endif 0
		OpenMainSymbol();
		if (sptr->order >=5) {
		    double d1,d2;
		    short x0,y0,r,r2;
		    /* HACK assumes that splines order 5 are circles */
	            svert = &(sptr->head);
		    d1=(double)(svert[2].x-svert[0].x);
		    d2=(double)(svert[2].y-svert[0].y);
		    r = (short)(sqrt(d1*d1+d2*d2) * .35);
		    d1=(double)(svert[3].x-svert[1].x);
		    d2=(double)(svert[3].y-svert[1].y);
		    r2 = (short)(sqrt(d1*d1+d2*d2) * .35);
		    if (r2>r) r=r2;
		    x0=(svert[2].x+svert[0].x)/2;
		    y0=(svert[2].y+svert[0].y)/2;
		    *frameno = AddItem(sdf, GetINum(),
			x0 + q->dx - r,
			x0 + q->dx + r,
			y0 + q->dy - r,
			y0 + q->dy + r,
			AllEdges, SDF_OUTLINE, NULL );
		    }
		else
		    *frameno = AddItem( sdf, GetINum(),
			id->xmin - 2 + q->dx, id->xmax + 2 + q->dx,
			id->ymin - 2 + q->dy, id->ymax + 2 + q->dy,
			AllEdges, SDF_OUTLINE, NULL );
		break;
#if 0
	      }
	      
	    /* For REAL splines, show a frame */
	    tptr = (SPLINE *) malloc ( sizeof(SPLINE) +
		(sptr->numvert - 1) * sizeof(POINT) );
	    if (tptr == NULL)
	      {
		printf("Out of Memory.  Can't frame anything.\n\r");
		break;
	      }
	    tptr->order = 2;
	    tptr->numvert = sptr->numvert;
	    tptr->nib = NibCircle1;
	    tptr->border = 1;
	    tptr->closed = sptr->closed;
	    tptr->filled = 0;
	    tptr->pat = PatWhite;
	    svert = &(sptr->head);
	    tvert = &(tptr->head);
	    for (i = sptr->numvert;  i--;)
	      {
		tvert[i].x = svert[i].x + q->dx;
		tvert[i].y = svert[i].y + q->dy;
	      }
	    OpenMainSymbol();
	    *frameno = AddItem( sdf, GetINum(), 0, 0, 0, 0, 0, SDF_SPLINE, tptr);
	    free( tptr );
	    break;
#endif 0
      }
  }

/*
 *  This routine will remove the frame from around an item.
 */
 
UnFrameObject( frameno, refresh )
	short *frameno, refresh;
  {
    /* If we haven't done so already, delete the frame. */
    if (*frameno > 0)
      {
	OpenMainSymbol();
	DeleteItem( sdf, *frameno );
	FreeINum(*frameno);
	*frameno = 0;
      }
    else if (*frameno < 0)
      {
	/* Unframe a group.  Delete the symbol call and definition. */
	OpenMainSymbol();
	DeleteItem( sdf, (- (*frameno)) + 1 );
	CloseMainSymbol(1);
	DeleteSymbol( sdf, -(*frameno) );
	FreeINum( (- (*frameno)) + 1 );
	FreeINum(  -(*frameno) );
	*frameno = 0;
	CloseMainSymbol(0);
      }
  }

