/*
 *  create.c
 *
 *  This file contains the routines which deal with the 'Draw' or create
 *  command.  The top routine parses the desired object type, and then
 *  invokes a specific routine to deal with the details.
 */
 
 
/* Includes */
# ifdef VAX
# include "stdio.h"
# else
# include "Vio.h"
# endif
# include "Vgts.h"
# include "splines.h"
# include "draw.h"
 
 
/* Imports */
extern double sin();
extern double cos();
extern double atan2();
extern double sqrt();
    
extern DefNib();
extern DefPattern();
extern DefFilling();
extern DefText();
extern PermMisc();
extern PermHelp();
extern PermExit();
    
extern Highlight();
extern HighlightStatic();
extern FrameObject();
extern UnFrameObject();
extern SelectMenuObject();
extern UnSelectMenuObject();
extern GetInput();
extern UnGetInput();
extern char *GetString();
extern enum TemplateTypes GetTemplate(); 
extern GetGroup();
extern NewGroup();
extern Checkpoint();
extern RevertToCKP();
extern NewObject();
 
extern ITEM_LIST_PTR *FindObject();
 
 
/* Exports */
extern ActDraw();
 
 
/* Local Definitions */
# define ARROWLEN	8	/* Length of the side of an arrowhead */

/*
 *  This routine will create a spline-type object.  It will return 1
 *  if the user has indicated that another similar object is to be
 *  drawn.  This is done via the CAlmostDone command, entered on the
 *  mouse buttons.
 */
 
short CreateSpline( closedP, orderP, typeP )
	short closedP, orderP;
	enum ObjTypes typeP;
  {
    enum MenuOptions cmd;
    short x, y, but;
    short adding, frame;
    register SPLINE *sptr;
    register POINT *vert;
    
    if ((sptr = (SPLINE *) malloc( sizeof(SPLINE) )) == NULL)
      {
	printf("Whoa!  Out of memory.  Can't add ANYTHING right now.\n\r");
	return( 0 );
      }
    sptr->order = orderP;
    sptr->numvert = 0;
    sptr->border = 1;
    sptr->closed = closedP;
    
    /* Prepare to display the frame */
    frame = DefineSymbol( sdf, item++, "Temporary Spline Frame" );
    EndSymbol( sdf, frame, 0 );
    EditSymbol( sdf, mainSymbol );
    AddCall( sdf, item++, 0, 0, frame );
    EndSymbol( sdf, mainSymbol, mainVgt );
    
    adding = 1;
    printf("Place the control points.  Hit Done when done.\n\r");
    while (adding)
      {
	GetInput( &cmd, &x, &y, &but );
	if (cmd == CDataPoint)
	  {
	    sptr = (SPLINE *) realloc( sptr,
	    		sizeof(SPLINE) + (sptr->numvert) * sizeof(POINT) );
	    if (!sptr)
	      {
		printf("Whoa! Out of memory. Can't add any more points.\n\r");
		adding = 0;
		break;
	      }
	    vert = (POINT *) &(sptr->head);
	    vert[sptr->numvert].x = x;
	    vert[sptr->numvert].y = y;
	    EditSymbol( sdf, frame );
	    AddItem( sdf, item++, x - 2, x + 2, y - 2, y + 2,
	    	    AllEdges, SDF_OUTLINE, NULL );
	    if (sptr->numvert > 0)
		AddItem( sdf, item++, vert[sptr->numvert - 1].x, x,
		    vert[sptr->numvert - 1].y, y, 0, SDF_GENERAL_LINE, NULL );
	    else
	        AddItem( sdf, item++, x - 2, x + 2, y - 2, y + 2,
		    AllEdges, SDF_OUTLINE, NULL );
	    EndSymbol( sdf, frame, mainVgt );
	    sptr->numvert += 1;
	  }
	else if ((cmd == CDone) || (cmd == CAlmostDone))
	  {
	    HighlightStatic( &(StaticMenu[0]) );
	    adding = 0;
	    sptr->nib = DefaultNib;
	    sptr->pat = DefaultPat;
	    sptr->filled = closedP & DefaultFilling;
	    sptr->opaque = DefaultOpaque;
	    HighlightStatic( &(StaticMenu[0]) );
	  }
	else if (cmd == CAbort)
	  {
	    HighlightStatic( &(StaticMenu[1]) );
	    sptr->numvert = 0;
	    adding = 0;
	    HighlightStatic( &(StaticMenu[1]) );
	  }
	else if (cmd == CUndo)
	  {
	    HighlightStatic( &(StaticMenu[2]) );
	    if (sptr->numvert > 0)
	      {
		sptr->numvert -= 1;
		EditSymbol( sdf, frame );
		DeleteItem( sdf, --item );
		DeleteItem( sdf, --item );
		EndSymbol( sdf, frame, mainVgt );
	      }
	    else
		printf("No more points to undo!\n\r");
	    HighlightStatic( &(StaticMenu[2]) );
	  }
	else if (cmd == CNib)		DefNib();
	else if (cmd == CPattern)	DefPattern();
	else if (cmd == CFilling)	DefFilling();
	else if (cmd == CTextDefault)	DefText();
	else if (cmd == CMisc)
	  {
	    HighlightStatic( &(StaticMenu[3]) );
	    PermMisc();
	    HighlightStatic( &(StaticMenu[3]) );
	  }
	else if (cmd == CHelp)
	  {
	    HighlightStatic( &(StaticMenu[4]) );
	    PermHelp();
	    HighlightStatic( &(StaticMenu[4]) );
	  }
	else if (cmd == CCheckPoint)
	  {
	    printf("Creating a Checkpoint.\n\r");
	    Checkpoint();
	  }
	else if ((sptr->numvert == 0) &&
	         ((int) cmd >= (int) CText) &&
	         ((int) cmd <= (int) CTemplate))
	  {
	    /* Permit change of object types if nothing pending. */
	    UnGetInput( cmd, x, y, but );
	    adding = 0;
	  }
	else if (cmd == CExit)
	  {
	    HighlightStatic( &(StaticMenu[5]) );
	    PermExit();
	    HighlightStatic( &(StaticMenu[5]) );
	  }
	else if (cmd != CNull)
	  {
	    printf("That command is meaningless right now.\n\r");
	  }
      }
    
    /* Okay, the spline data structure is built.  Display it. */
    EditSymbol( sdf, mainSymbol );
    DeleteItem( sdf, frame + 1 );
    EndSymbol( sdf, mainSymbol, mainVgt );
    DeleteSymbol( sdf, frame );
    if (sptr->numvert > 0)
      {
	NewObject( sptr, typeP, 0, 0, 0, 0, 0, 0, 0, NULL, 1, 0, 0 );
      }
    else
	free( sptr );
    
    return( cmd == CAlmostDone );
  }

/*
 *  This routine will create an object of type text.
 */
 
CreateText()
  {
    short working;
    char *string;
    short x1, x2, slength, bmin, bmax;
    enum MenuOptions cmd;
    short x, y, but;
    
    /* Get the string to place */
    printf("(%s)  ", FontData[DefaultFont].fontname );
    string = GetString();
    working = !(string == NULL);
    while (working)
      {
	/* Prompt the user. */
	switch (DefaultCentering)
	  {
	    case PositionRight:
		printf("Bottom-right corner of the destination?\n\r");
		break;
	    case PositionCenter:
		printf("Bottom-center of the destination?\n\r");
		break;
	    case PositionLeft:
		printf("Bottom-left corner of the destination?\n\r");
		break;
	  }
    
	/* Get down to business */
	GetInput( &cmd, &x, &y, &but );
	switch (cmd)
	  {
	    case CDataPoint:
		/* Load the font if necessary. */
		if (!(FontData[DefaultFont].loaded))
		  {
		    printf("\n\rLoading font '%s', please wait ..",
			    FontData[DefaultFont].fontname);
# ifndef VAX
		    RedrawPad( stdout );
# endif
		    FontData[DefaultFont].refnumber =
			DefineFont( FontData[DefaultFont].fontname, NULL );
		    printf(".. done.\n\r");
		    if ((FontData[DefaultFont].refnumber < 0) ||
		        (FontData[DefaultFont].refnumber == 255))
		      {
			printf("Internal Error. Can't load font %s (%d).\n\r",
				FontData[DefaultFont].fontname, DefaultFont );
			printf("    return code = %d\n\r",
				FontData[DefaultFont].refnumber );
			break;
		      }
		    else
		        FontData[DefaultFont].loaded = 1;
		  }
		
		/* Compute the true baseline reference point */
		EditSymbol( sdf, mainSymbol );
		AddItem( sdf, item, 0, 0, 0, 0,
			FontData[DefaultFont].refnumber, SDF_TEXT, string );
		if (InquireItem( sdf,item,0,&slength,&bmin,&bmax,0,0,0 ) == 0)
		  {
		    printf("Internal Error: InquireItem failed (slength).\n\r");
		    working = 0;
		    free( string );
		    DeleteItem( sdf, item );
		    EndSymbol( sdf, mainSymbol, 0 );
		    break;
		  }
		DeleteItem( sdf, item );
		EndSymbol( sdf, mainSymbol, 0 );
		switch (DefaultCentering)
		  {
		    case 0:
		        x1 = x - slength;
			x2 = x;
			break;
		    
		    case 1:
			x1 = x - (slength >> 1);
			x2 = x + (slength >> 1);
			break;
			
		    case 2:
			x1 = x;
			x2 = x + slength;
			break;
		  }
		NewObject( string, TextObj, DefaultFont, DefaultCentering,
			x1, x2, y, y+bmax-bmin, -bmin, NULL, 1, 0, 0 );
		working = 0;
		break;
	
	    case CDone:
		HighlightStatic( &(StaticMenu[0]) );
		printf("No need for done, just position the string.\n\r");
		HighlightStatic( &(StaticMenu[0]) );
		break;
		
	    case CAbort:
		HighlightStatic( &(StaticMenu[1]) );
	        printf("Abort.\n\r");
		RevertToCKP();
		free( string );
		working = 0;
		HighlightStatic( &(StaticMenu[1]) );
		break;
		
	    case CUndo:
		HighlightStatic( &(StaticMenu[2]) );
		free( string );
		HighlightStatic( &(StaticMenu[2]) );
		printf("(%s)  ", FontData[DefaultFont].fontname );
		string = GetString();
		working = !(string == NULL);
		break;
		
	    case CNib:		DefNib();  break;
	    case CPattern:	DefPattern();  break;
	    case CFilling:	DefFilling();  break;
	    case CTextDefault:	DefText();  break;
	
	    case CMisc:
		HighlightStatic( &(StaticMenu[3]) );
		PermMisc();
		HighlightStatic( &(StaticMenu[3]) );
		break;
		
	    case CHelp:
		HighlightStatic( &(StaticMenu[4]) );
		PermHelp();
		HighlightStatic( &(StaticMenu[4]) );
		break;
	    
	    case CExit:
		HighlightStatic( &(StaticMenu[5]) );
		PermExit();
		HighlightStatic( &(StaticMenu[5]) );
		break;
	
	    case CNull:		break;
	    
	    default:
		printf("That command doesn't make any sense here.\n\r");
		break;
	  }
      }
  }

/*
 *  This routine will enable the user to point at various objects,
 *  and have them incorporated into a group.
 */
 
CreateGroup()
  {
    ITEM_LIST_PTR *groupobjects, *tempptr;
    ITEM_LIST_PTR *object, *tmpobj;
    enum MenuOptions cmd;
    short x, y, but, bad;
    short xtmp, ytmp;
    int dist;
    char *name;
    short type, selected, grouping;
    
    /* Can this be done at all? */
    if (Groups[MAXGROUP-1])
      {
	printf("Sorry, only %d groups can be created.\n\r", MAXGROUP);
	return;
      }
    
    /* Warn the user what is going to happen */
    printf("To build a group, select an object type and some objects.\n\r");
    printf("    Hit 'Done' when finished.\n\r");
    groupobjects = NULL;
    
    /* Select the objects to put into the group. */
    type = -1;
    selected = 0;
    grouping = 1;
    while (grouping == 1)
      {
	GetInput( &cmd, &x, &y, &but );
	if (cmd == CDataPoint)
	  {
	    if (selected == 0)
		printf("Select the object type first, then the object.\n\r");
	    else
	      {
		/* Try to find something. */
		object = FindObject( type, x, y, &xtmp, &ytmp, &dist, 0 );
		if (object)
		  {
		    tempptr = groupobjects;
		    bad = 0;
		    while (tempptr)
		      {
		        if (tempptr->itemdesc == object->itemdesc)
			    bad = 1;
		        tempptr = tempptr->prev;
		      }
		    if (bad)
		        printf("That item is already in this group.\n\r");
		    else if (object)
		      {
		        /* We got an object.  Frame it to make it obvious. */
		        if ((tempptr = (ITEM_LIST_PTR *)
			    malloc( sizeof(ITEM_LIST_PTR) )) == NULL)
			  {
			    printf("Out of memory.  Can't add that.\n\r");
			    return;
			  }
		        tempptr->itemdesc = object->itemdesc;
			tempptr->dx = object->dx;
		        tempptr->dy = object->dy;
		        tempptr->prev = groupobjects;
		        groupobjects = tempptr;
		        FrameObject( object, &(tempptr->flag) );
		      }
		  }
		else
		    printf("Nothing appropriate near there.  Try again.\n\r");
	      }
	  }
	else if (cmd == CUndo)
	  {
	    HighlightStatic( &(StaticMenu[2]) );
	    if (groupobjects)
	      {
		/* Deselect an object */
		UnFrameObject( &(groupobjects->flag), 1 );
		tempptr = groupobjects->prev;
		free( groupobjects );
		groupobjects = tempptr;
	      }
	    else
		printf("No objects are selected!!\n\r");
	    HighlightStatic( &(StaticMenu[2]) );
	  }
	else if ((cmd == CText) || (cmd == COpenCurve) ||
		 (cmd == CClosedCurve) || (cmd == COpenPolygon) ||
		 (cmd == CClosedPolygon) || (cmd == CTemplate))
	  {
	    if (selected)
		UnSelectMenuObject( selected );
	    type = (int) cmd - (int) CText;
	    selected = type + 2;
	    if ((int) cmd >= (int) CCurrentObj)
		type--;
	    SelectMenuObject( selected );
	  }
	else if (cmd == CAllObj)
	  {
	    if (selected)
		UnSelectMenuObject( selected );
	    type = -1;
	    selected = 1;
	    SelectMenuObject( selected );
	    for (tmpobj = activelist->prev; tmpobj; tmpobj = tmpobj->next)
	      {
		/* Insert new object from the activelist. */
		tempptr = groupobjects;
		bad = 0;
		while (tempptr)
		  {
		    bad = bad || (tempptr->itemdesc == tmpobj->itemdesc);
		    tempptr = tempptr->prev;
		  }
		if (!bad)
		  {
		    if ((tempptr = (ITEM_LIST_PTR *)
			malloc( sizeof(ITEM_LIST_PTR) )) == NULL)
		      {
			printf("Out of memory.  Can't add more items.\n\r");
			return;
		      }
		    tempptr->itemdesc = tmpobj->itemdesc;
		    tempptr->dx = tmpobj->dx;
		    tempptr->dy = tmpobj->dy;
		    tempptr->prev = groupobjects;
		    groupobjects = tempptr;
		    FrameObject( tmpobj, &(tempptr->flag) );
		  }
	      }
	  }
	else if (cmd == CCurrentObj)
	  {
	    if (CurrentObject == NULL)
		printf("No current object.\n\r");
	    else
	      {
		tempptr = groupobjects;
		bad = 0;
		while (tempptr)
	          {
		    if (tempptr->itemdesc == CurrentObject->itemdesc)
		        bad = 1;
		    tempptr = tempptr->prev;
	          }
		if (bad)
	            printf("That item is already in this group.\n\r");
		else
	          {
		    if ((tempptr = (ITEM_LIST_PTR *)
			malloc( sizeof(ITEM_LIST_PTR) )) == NULL)
		      {
			printf("Out of memory.  Can't do that.\n\r");
			return;
		      }
		    tempptr->itemdesc = CurrentObject->itemdesc;
		    tempptr->dx = CurrentObject->dx;
		    tempptr->dy = CurrentObject->dy;
		    tempptr->prev = groupobjects;
		    groupobjects = tempptr;
		    FrameObject( CurrentObject, &(tempptr->flag) );
		    if (selected)
			UnSelectMenuObject( selected );
		    type = (int) CurrentObject->itemdesc->type;
		    selected = type + 2;
		    if ((int) CurrentObject->itemdesc->type >=
			(int) OpenPolygonObj)
			selected++;
		    SelectMenuObject( selected );
	          }
	      }
	  }
	else if (cmd == CGroup)
	  {
	    GetGroup( &tmpobj );
	    if (tmpobj)
	      {
		tempptr = groupobjects;
		bad = 0;
		while (tempptr)
		  {
		    if (tempptr->itemdesc == tmpobj->itemdesc)
			bad = 1;
		    tempptr = tempptr->prev;
		  }
		if (bad)
		    printf("That item is already in this group.\n\r");
		else
		  {
		    if ((tempptr = (ITEM_LIST_PTR *)
			malloc( sizeof(ITEM_LIST_PTR) )) == NULL)
		      {
			printf("Out of memory.  Can't do anything.\n\r");
			return;
		      }
		    tempptr->itemdesc = tmpobj->itemdesc;
		    tempptr->prev = groupobjects;
		    groupobjects = tempptr;
		    FrameObject( tmpobj, &(tempptr->flag) );
		  }
	      }
	  }
	else if ((cmd == CDone) || (cmd == CAlmostDone))
	  {
	    HighlightStatic( &(StaticMenu[0]) );
	    grouping = 0;
	    HighlightStatic( &(StaticMenu[0]) );
	  }
	else if (cmd == CAbort)
	  {
	    HighlightStatic( &(StaticMenu[1]) );
	    grouping = -1;
	    HighlightStatic( &(StaticMenu[1]) );
	  }
	else if (cmd == CNib)		DefNib();
	else if (cmd == CPattern)	DefPattern();
	else if (cmd == CFilling)	DefFilling();
	else if (cmd == CTextDefault)	DefText();
	else if (cmd == CMisc)
	  {
	    HighlightStatic( &(StaticMenu[3]) );
	    PermMisc();
	    HighlightStatic( &(StaticMenu[3]) );
	  }
	else if (cmd == CHelp)
	  {
	    HighlightStatic( &(StaticMenu[4]) );
	    PermHelp();
	    HighlightStatic( &(StaticMenu[4]) );
	  }
	else if (cmd == CExit)
	  {
	    HighlightStatic( &(StaticMenu[5]) );
	    PermExit();
	    HighlightStatic( &(StaticMenu[5]) );
	  }
	else if (cmd != CNull)
	  {
	    printf("That command doesn't make sense here.\n\r");
	  }
      }
    
    /* Unframe and deselect everything, while filling in links */
    tempptr = groupobjects;
    while (tempptr)
      {
	UnFrameObject( &(tempptr->flag), 0 );
	if (tempptr->prev)
	    tempptr->prev->next = tempptr;
	tempptr = tempptr->prev;
        if ((grouping == -1) && (tempptr != NULL))
	    free( tempptr->next );
      }
    if (groupobjects)
        DisplayItem( sdf, mainSymbol, mainVgt );
    if (selected)
	UnSelectMenuObject( selected );
    
    /* If not aborted, create the group. */
    if ((grouping == -1) || (groupobjects == NULL))
	return;
    groupobjects->next = NULL;
    tempptr = groupobjects;
    while (tempptr->prev)
	tempptr = tempptr->prev;
    printf("Name that group.  (NULL to abort)\n\r");
    name = GetString();
    if (name)
      {
	NewObject( tempptr, GroupObj, 0, 0, 0, 0, 0, 0, 0, name, 0, 0, 0 );
	NewGroup( CurrentObject, 1, 1 );
      }
    else
      {
	while (tempptr)
	  {
	    groupobjects = tempptr;
	    tempptr = tempptr->next;
	    free( groupobjects );
	  }
      }
  }

/*  Local Definitions for CreateTemplate */
static char *TempPrompts0[] =
  {
    "Enter the tip of the Arrowhead.",
    "Enter the tip of the Arrowhead.",
    "Enter the tip of the Arrowhead.",
    "Enter the tip of the Arrowhead.",
    "Enter a corner of the bounding box for the Oval.",
    "Enter the center of the circle.",
    "Enter a corner of the rectangle.",
    "Internal Error:  Trying to get control points of NullTemplate.",
  };
 
static char *TempPrompts1[] =
  {
    "Enter the 'root' of the arrow.",
    "Enter the 'root' of the arrow.",
    "Enter the 'root' of the arrow.",
    "Enter the 'root' of the arrow.",
    "Enter a diagonally opposite corner.",
    "Enter a point on the circle.",
    "Enter a diagonally opposite corner.",
    "Internal Error:  Trying to get control points of NullTemplate.",
  };

/*
 *  This routine will create an object of one of the standard
 *  template types.
 */
 
CreateTemplate( temptype )
	enum TemplateTypes temptype;
  {
    short x1, y1, x2, y2, numpnt;
    enum MenuOptions cmd;
    short but;
    SPLINE *sptr;
    POINT *vert;
    int radius;
    double theta;
    
    /* Get the template type desired */
    if (temptype == NullTemplate)
	temptype = GetTemplate();
    if (temptype == NullTemplate)
	return;
    
    /* Okay, get the two control points. */
    numpnt = 2;
    while (numpnt)
      {
	/* Prompt the user and get a point */
	if (numpnt == 2)
	  {
	    printf("%s\n\r", TempPrompts0[(int) temptype]);
	    GetInput( &cmd, &x1, &y1, &but );
	  }
	else
	  {
	    printf("%s\n\r", TempPrompts1[(int) temptype]);
	    GetInput( &cmd, &x2, &y2, &but );
	  }
	
	if (cmd == CDataPoint)
	  {
	    if (numpnt == 2)
	      {
		EditSymbol( sdf, mainSymbol );
		AddItem( sdf, item, x1 - 2, x1 + 2, y1 - 2, y1 + 2,
			AllEdges, SDF_OUTLINE, NULL );
		EndSymbol( sdf, mainSymbol, mainVgt );
		numpnt--;
	      }
	    else if ((x1 != x2) || (y1 != y2))
	      {
		EditSymbol( sdf, mainSymbol );
		DeleteItem( sdf, item );
		EndSymbol( sdf, mainSymbol, mainVgt );
		numpnt--;
	      }
	    else
	      {
		printf("Don't use the same point twice.  It doesn't work.\n\r");
	      }
	  }
	else if (cmd == CDone)
	  {
	    HighlightStatic( &(StaticMenu[0]) );
	    printf("Done??  Use abort to quit.\n\r");
	    HighlightStatic( &(StaticMenu[0]) );
	  }
	else if (cmd == CAbort)
	  {
	    HighlightStatic( &(StaticMenu[1]) );
	    if (numpnt == 1)
	      {
		EditSymbol( sdf, mainSymbol );
		DeleteItem( sdf, item );
		EndSymbol( sdf, mainSymbol, mainVgt );
	      }
	    printf("Aborted.\n\r");
	    RevertToCKP();
	    HighlightStatic( &(StaticMenu[1]) );
	    return;
	  }
	else if (cmd == CUndo)
	  {
	    HighlightStatic( &(StaticMenu[2]) );
	    if (numpnt == 1)
	      {
		EditSymbol( sdf, mainSymbol );
		DeleteItem( sdf, item );
		EndSymbol( sdf, mainSymbol, mainVgt );
	      }
	    else
		printf("Nothing to Undo.\n\r");
	    numpnt = 2;
	    HighlightStatic( &(StaticMenu[2]) );
	  }
	else if (cmd == CNib)		DefNib();
	else if (cmd == CPattern)	DefPattern();
	else if (cmd == CFilling)	DefFilling();
	else if (cmd == CTextDefault)	DefText();
	else if (cmd == CMisc)
	  {
	    HighlightStatic( &(StaticMenu[3]) );
	    PermMisc();
	    HighlightStatic( &(StaticMenu[3]) );
	  }
	else if (cmd == CHelp)
	  {
	    HighlightStatic( &(StaticMenu[4]) );
	    PermHelp();
	    HighlightStatic( &(StaticMenu[4]) );
	  }
	else if (cmd == CExit)
	  {
	    HighlightStatic( &(StaticMenu[5]) );
	    PermExit();
	    HighlightStatic( &(StaticMenu[5]) );
	  }
	else if (cmd != CNull)
	    printf("That command doesn't make sense here.\n\r");
      }
    
    /* We have the control points, now define the object. */
    switch (temptype)
      {
	case NarArrowO:
	case NarArrowC:
	case WideArrowO:
	case WideArrowC:
	    if ((sptr = (SPLINE *) malloc( sizeof(SPLINE) + 2*sizeof(POINT) ))
		    == NULL)
	      {
		printf("Out of memory.  Can't allocate arrowhead.\n\r");
		return;
	      }
	    sptr->order = 2;
	    sptr->numvert = 3;
	    sptr->nib = (enum Nib) ((int) DefaultNib % 4 + (int) NibCircle0);
	    sptr->border = 1;
	    sptr->closed = ((temptype == NarArrowC) ||
			(temptype == WideArrowC));
	    sptr->filled = sptr->closed;
	    sptr->opaque = DefaultOpaque;
	    sptr->pat = PatBlack;
	    vert = &(sptr->head);
	    
	    /* Calculate the angles needed, using (groan) floating point */
	    theta = atan2( (double) (x2 - x1), (double) (y2 - y1) );
		
	    /* Place the real control points of the spline */
	    if ((temptype == NarArrowO) || (temptype == NarArrowC))
	      {
		vert[0].x = ARROWLEN * sin( theta - 0.5 ) + x1 + 0.5;
		vert[0].y = ARROWLEN * cos( theta - 0.5 ) + y1 + 0.5;
		vert[1].x = x1;
		vert[1].y = y1;
		vert[2].x = ARROWLEN * sin( theta + 0.5 ) + x1 + 0.5;
		vert[2].y = ARROWLEN * cos( theta + 0.5 ) + y1 + 0.5;
	      }
	    else
	      {
		vert[0].x = ARROWLEN * sin( theta - 0.75 ) + x1 + 0.5;
		vert[0].y = ARROWLEN * cos( theta - 0.75 ) + y1 + 0.5;
		vert[1].x = x1;
		vert[1].y = y1;
		vert[2].x = ARROWLEN * sin( theta + 0.75 ) + x1 + 0.5;
		vert[2].y = ARROWLEN * cos( theta + 0.75 ) + y1 + 0.5;
	      }
	    
	    NewObject( sptr, TemplateObj, (short) temptype, 0,
		    0, 0, 0, 0, 0, NULL, 1, 0, 0 );
	    break;
	
	case Oval:
	case Rectangle:
	    if ((sptr = (SPLINE *) malloc( sizeof(SPLINE) + 3*sizeof(POINT) ))
		    == NULL)
	      {
		printf("Out of memory.  Can't allocate a Template.\n\r");
		return;
	      }
	    sptr->order = (temptype == Oval ? 3 : 2);
	    sptr->numvert = 4;
	    sptr->nib = DefaultNib;
	    sptr->border = 1;
	    sptr->closed = 1;
	    sptr->filled = DefaultFilling;
	    sptr->opaque = DefaultOpaque;
	    sptr->pat = DefaultPat;
	    vert = &(sptr->head);
	    vert[0].x = min(x1, x2);
	    vert[0].y = min(y1, y2);
	    vert[1].x = min(x1, x2);
	    vert[1].y = max(y1, y2);
	    vert[2].x = max(x1, x2);
	    vert[2].y = max(y1, y2);
	    vert[3].x = max(x1, x2);
	    vert[3].y = min(y1, y2);
	    NewObject( sptr, TemplateObj, (short) temptype, 0,
		    0, 0, 0, 0, 0, NULL, 1, 0, 0 );
	    break;
	    
	case Circle:
	    if ((sptr = (SPLINE *) malloc( sizeof(SPLINE) + 3*sizeof(POINT) ))
		    == NULL)
	      {
		printf("Out of memory.  Can't allocate a Circle.\n\r");
		return;
	      }
	    sptr->order = 5;
	    sptr->numvert = 4;
	    sptr->nib = DefaultNib;
	    sptr->border = 1;
	    sptr->closed = 1;
	    sptr->filled = DefaultFilling;
	    sptr->opaque = DefaultOpaque;
	    sptr->pat = DefaultPat;
	    radius = 1.6970562 *
	    		sqrt( (double) ((x2-x1)*(x2-x1) + (y2-y1)*(y2-y1)) );
	    vert = &(sptr->head);
	    vert[0].x = x1;
	    vert[0].y = y1 + radius;
	    vert[1].x = x1 + radius;
	    vert[1].y = y1;
	    vert[2].x = x1;
	    vert[2].y = y1 - radius;
	    vert[3].x = x1 - radius;
	    vert[3].y = y1;
	    NewObject( sptr, TemplateObj, (short) temptype, 0,
		    0, 0, 0, 0, 0, NULL, 1, 0, 0 );
	    break;
	    
	default:
	    printf("Internal Error:  Template = %d.\n\r", (int) temptype);
	    break;
      }
  }

/*
 *  This is the main routine to handle the 'draw' action.
 */
 
ActDraw( cmd )
	enum MenuOptions cmd;
  {
    short x, y, but;
    short drawing;
    
    /* Make ready */
    Checkpoint();
    drawing = 1;
    if (cmd == CNull)
      {
	printf("DRAW:  Select an object type.\n\r");
	GetInput( &cmd, &x, &y, &but );
      }
    
    /* Parse the command */
    while (drawing)
      {
	switch (cmd)
	  {
	    case CNull:
		break;
		
	    case CAllObj:
		printf("You must choose a specific object type.");
		printf("  Try again.\n\r");
		break;
	    
	    case CText:
		Highlight( &(Objects[2]) );
	    	CreateText();
		Highlight( &(Objects[2]) );
		drawing = 0;
		break;
		
	    case COpenCurve:
		Highlight( &(Objects[3]) );
		while (drawing)
		    drawing = CreateSpline( 0 /* Open */, 3, OpenSplineObj );
		Highlight( &(Objects[3]) );
		break;
		
	    case CClosedCurve:
		Highlight( &(Objects[4]) );
		while (drawing)
	            drawing = CreateSpline( 1 /* Clsd */, 3, ClosedSplineObj);
		Highlight( &(Objects[4]) );
		break;
		
	    case CCurrentObj:
		if (CurrentObject)
		  {
		    switch (CurrentObject->itemdesc->type)
		      {
			case TextObj:
			    Highlight( &(Objects[2]) );
			    CreateText();
			    Highlight( &(Objects[2]) );
			    break;
			case OpenSplineObj:
			    Highlight( &(Objects[3]) );
			    while (drawing)
			        drawing = CreateSpline( 0, 3, OpenSplineObj );
			    Highlight( &(Objects[3]) );
			    break;
			case ClosedSplineObj:
			    Highlight( &(Objects[4]) );
			    while (drawing)
				drawing = CreateSpline( 1,3,ClosedSplineObj );
			    Highlight( &(Objects[4]) );
			    break;
			case OpenPolygonObj:
			    Highlight( &(Objects[6]) );
			    while (drawing)
			        drawing = CreateSpline( 0,2, OpenPolygonObj );
			    Highlight( &(Objects[6]) );
			    break;
			case ClosedPolygonObj:
			    Highlight( &(Objects[7]) );
			    while (drawing)
				drawing = CreateSpline(1,2,ClosedPolygonObj);
			    Highlight( &(Objects[7]) );
			    break;
			case GroupObj:
			    Highlight( &(Objects[8]) );
			    CreateGroup();
			    Highlight( &(Objects[8]) );
			    break;
			case TemplateObj:
			    Highlight( &(Objects[9]) );
			    CreateTemplate(CurrentObject->itemdesc->subtype);
			    Highlight( &(Objects[9]) );
			    break;
		      }
		    drawing = 0;
		  }
		else
		    printf("No current object!  Try again.\n\r");
		break;
		
	    case COpenPolygon:
		Highlight( &(Objects[6]) );
		while (drawing)
		    drawing =
			CreateSpline( 0 , 2 /* Order */, OpenPolygonObj );
		Highlight( &(Objects[6]) );
		break;
		
	    case CClosedPolygon:
		Highlight( &(Objects[7]) );
		while (drawing)
		    drawing =
		        CreateSpline( 1, 2 /* Order */, ClosedPolygonObj );
		Highlight( &(Objects[7]) );
		break;
		
	    case CGroup:
		Highlight( &(Objects[8]) );
		CreateGroup();
		Highlight( &(Objects[8]) );
		drawing = 0;
		break;
	    
	    case CTemplate:
		Highlight( &(Objects[9]) );
		CreateTemplate( NullTemplate );
		Highlight( &(Objects[9]) );
		drawing = 0;
		break;
		
	    case CDone:
		HighlightStatic( &(StaticMenu[0]) );
		drawing = 0;
		HighlightStatic( &(StaticMenu[0]) );
		break;
	
	    case CUndo:
		HighlightStatic( &(StaticMenu[2]) );
		drawing = 0;
		HighlightStatic( &(StaticMenu[2]) );
		break;
	    
	    case CAbort:
		HighlightStatic( &(StaticMenu[1]) );
		RevertToCKP();
		drawing = 0;
		HighlightStatic( &(StaticMenu[1]) );
		break;
	
	    case CNib:
		DefNib();
		break;
		
	    case CPattern:
		DefPattern();
		break;
	
	    case CFilling:
		DefFilling();
		break;
	
	    case CTextDefault:
		DefText();
		break;
	
	    case CMisc:
		HighlightStatic( &(StaticMenu[3]) );
		PermMisc();
		HighlightStatic( &(StaticMenu[3]) );
		break;
		
	    case CHelp:
		HighlightStatic( &(StaticMenu[4]) );
		PermHelp();
		HighlightStatic( &(StaticMenu[4]) );
		break;
		
	    case CExit:
		HighlightStatic( &(StaticMenu[5]) );
		PermExit();
		HighlightStatic( &(StaticMenu[5]) );
		break;

		
	    default:
		printf("That command doesn't make sense here.\n\r");
		break;
          }
	
	/* If we need one, get a new command */
        if (drawing)
	  {
	    printf("DRAW:  Select an object type.\n\r");
	    GetInput( &cmd, &x, &y, &but );
	  }
      }
  }
