/* bitpress - converts a bit map to a press file.
 * Per Bothner, 1982/March.
 * Based on cz.
 *
 * HISTORY
 * STANFORD MODS:
 *
 * 03-Apr-81  Jeffrey Mogul (jcm) at Stanford University
 *	Made location of temporary files a compile-time parameter.
 *	Added byte-swapping option -S.
 *	In general, made program "smart" about byte-order; automatically
 *	 determines input order, always produces output order according
 *	 to state of -S flag independent of input order.
 *	Added "File Under" heuristic.
 *	
 * 26-Mar-81  James Gosling (jag) at Carnegie-Mellon University
 *	Changed the beast to generate press files in the
 *	opposite byte order from what it had been.  It
 *	now generates them as 8 bit byte streams rather
 *	than as 16 bit byte streams -- this is just a
 *	byte swap of what it used to do.  This was done
 *	to be compatible with the rest of the world and
 *	to make the shipping of press files easier.  I
 *	also changed the press file copying to accept
 *	press files in either byte order -- it checks the
 *	password of the Document Directory to determine
 *	the byte order and byte swaps if necessary.
 */

/* Abbrev: Mcs = Micas  --  C needs long identifiers! */
#define McsPerInch 2540
#define PageLength McsPerInch*(8*11-3)/8/* 10.625 inches */
#define PageWidth  McsPerInch*17/2

# define MAXPARTS 500

#include <stdio.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "pressparams.h"

/*
 * filenames defined here for easy portability.
 */
#define TEMPNAME "/tmp/Dover XXXXXX"	/* normal temporary file name */
#define PRTEMPNAME "/tmp/DoverC XXXXXX"	/* alternate temporary file name */

/*
 * Byte order options: BYTEORDERPDP11 or BYTEORDERALTO
 *	(must be logical complements!)
 *
 * Straight Dope on Byte Order:  This program has some messy sections
 *	wrt byte order; i.e., it used to assume too much.  Mostly, it
 *	means that if you fread or fwrite, you may have to swab as well
 *	depending on ByteOrder or ByteOrderInput.
 */

#define BYTEORDERALTO	0
#define BYTEORDERPDP11	1
#define DEFBYTEORDER	BYTEORDERPDP11

/* face encodings: */
#define ITALIC	1
#define BOLD	2
short bitmap[128][1024];	/* For reformatting bitmap */
struct stat S;
FILE *fopenp();
char   *ctime ();
char   *getenv ();
long    time ();
struct passwd  *getpwuid ();

int     McsPerLine = McsPerInch / 7;/* lines of a line */
int     McsPerHLine = McsPerInch / 7;/* lines of a header line */
char	TempName[50] = TEMPNAME;
int     SeenText = 1;		/* true if we've seen some text on this
				   page */
int     Copies = 1;		/* number of copies to make */
char   *Banner = 0;		/* label to appear on the cover page */
char    Names[54] = "";		/* list of file names */
int     NoTitle = 0;		/* true if the title line is to be
				   suppressed */
int     Pressed = 0;		/* true if we've converted a file to
				   press format */
int	ByteOrder = DEFBYTEORDER;	/* true if PDP-11 byte order wanted */
int	ByteOrderInput = DEFBYTEORDER;	/* true if PDP-11 byte order input */
int     SeenFile = 0;		/* true if a file has been processed */

char   *FileName;		/* name of file currently being pressed 
				*/
char   *FileDate;		/* last mod date of file being pressed 
				*/
int     wrdsleft;		/* number of words left in index entry 
				*/
int     prevbyte;		/* last byte written to the press file 
				*/
int     havebyte = 0;		/* true iff there is a valid byte in
				   prevbyte -- used for byte-swapping
				   strings */

int     cX,
        cY;			/* current page positions */
int     dX,
        dY;			/* desired page positions */
int     lX,
        lY;			/* page positions of the start of the
				   line */
int     crX,
        crY;			/* X and Y increments to apply to
				   carriage returns */

/* Press entity list commands */
#define ShowChShort	0000
#define Font		0160
#define SetX		0356
#define SetY		0357
#define ShowCh		0360
#define ShowDots	0374
#define ShowRectangle	0376
#define Nop		0377

long    PartList[MAXPARTS];	/* the lengths of each part in bytes */
int     nparts = 0;		/* the current number of parts */
int	nprinted = 0;		/* the actual number of parts
				   printed */
long    PartLength;		/* the length of the current part */
int     showpending;		/* number of characters waiting to be
				   shown */
int     column;			/* column number in input file */
char   *UsersHeader = 0;	/* user specified heading */
char   *UsersName = 0;		/* user specified name for filing output
				   */
int     Page = 0;		/* current page number */
int     TotalPages = 0;		/* total number of pages printed */

FILE   *PressFile = 0;		/* output press file */

unsigned char   *DL;		/* scratch storage for Data List */
int	LenDL = 0,		/* number of bytes used in DL */
	UsedDL;			/* number of bytes in DL that
				   contain useful stuff */
unsigned char   *EL;		/* scratch storage for Entity List */
int	LenEL = 0,		/* number of bytes used in EL */
	UsedEL;			/* number of bytes in EL that
				   contain useful stuff */
int width = 0, height = 0;	/* Size of bitmap */
enum formatType { bits, SunFormat};
enum formatType inFormat = bits;
int scale = 32;			/* Micas per pixel */

#define NotThere 0100003

/* add a character to the press file */
PRputc (c)
char    c; {
    if (havebyte) {
	if (ByteOrder == BYTEORDERALTO) {
	    putc (c, PressFile);
	    putc (prevbyte, PressFile);
	}
	else	{	/* PDP-11 byte order */
	    putc (prevbyte, PressFile);
	    putc (c, PressFile);
	}
	havebyte = 0;
    }
    else {
	prevbyte = c;
	havebyte++;
    }
    PartLength++;
}

/* add an integer (16 bit) to the press file */
PRputi (i)
int     i; {
    PRputc (i >> 8, PressFile);
    PRputc (i, PressFile);
/*    PartLength += 2;  <-- Done by PRputc */
}

/* put the current date into the press file in Alto format */
PRputDate () {
	long now = time(0) + (1970-1901)*365.24*24*60*60;
	PRputi (now>>16);
	PRputi (now);
}

/* Pad the press file to the indicated boundary (eg. 512 gets you to
 * the next block.
 */
Pad (boundary) {
    if (havebyte)
	PRputc (Nop);
    while ((PartLength & boundary - 1) != 0) {
	PRputc (Nop, PressFile);
/*	PartLength++; <-- Done by PRputc */
    }
}


/* put a charcter into the data list at the desired X and Y positions.
 * If the current position doesn't agree with the desired position, put out
 * cursor movement directives.  Leave the current cursor position updated
 * to account for the character.
 */
Dputc (c){}

/* put out a string to the press file data part */
Dputs (s)
register char  *s; {
    while (*s) {
	if (*s >= 040)
	    Dputc (*s);
	s++;
    }
}

/* add a character to the entity list */
Eputc (c) {
    register    i;
    if (i = showpending) {
	showpending = 0;
	while (i > 255) {
	    Eputc (ShowCh);
	    Eputc (255);
	    i -= 255;
	}
	if (i > 32) {
	    Eputc (ShowCh);
	    Eputc (i);
	}
	else
	    Eputc (ShowChShort + i - 1);
    }
    if (c >= 0){
	UsedEL++;

	if(UsedEL>LenEL) SetEL (UsedEL);
	EL[UsedEL-1] = c;
    }
}

/* add an integer to the entity list */
Eputi (i) {
    Eputc (i >> 8);
    Eputc (i);
}

/* Get a 16-bit integer from standard input */
geti()
  { int i = getchar() << 8;
    return ( (i & 0xFF00) + (getchar() & 0xFF));
}

/* put out a page heading to the press file,
 * initialize the data & entity parts.
 */
InitPage () {
    char    header[200];

    if (!(height || width))
        if (getchar() == 157 && getchar() == 3)
	    {
	    getchar(); getchar();
	    height = geti();
	    width = geti();
	    width = (width+15) & 0xFFF0;	/* Press problem */
fprintf(stderr,"SunFormat: h=%d,w=%d\r\n",height,width);
	    inFormat = SunFormat;
	    }
	else quit (1, "x and y switches not given and file not in sun format\n");        

    SeenText = 0;
    cX = cY = -1;
    PartLength = UsedEL = showpending = 0;
    lX = dX = McsPerInch;
    lY = dY = PageLength - McsPerHLine * 3 / 2;
    TotalPages++;
    if (!NoTitle) {
	if (UsersHeader)
	    Dputs (UsersHeader);
	else {
	    sprintf (header, "%s             %s",
		    FileName ? FileName : "              ",
		    FileDate);
	    Dputs (header);
	}
	dX = lX = lX + crX * 4;
	dY = lY = lY + crY * 4;
    };
    Eputc(SetX); Eputi(1270);
    Eputc(SetY); Eputi(1270);
}

/* terminate a page.  Put out the entity list and entity trailer,
 * pad out the block, and add an entry to the parts list.
 */
ClosePage () {
    register long   EntityStart,
                    i;

    Eputc (-1);
    Pad (2);
    EntityStart = PartLength;
    PRputi (0);
    for (i = 0; i < UsedEL; i++)
	PRputc (EL[i]);
    Pad (2);
    PRputc (0);			/* entity type */
    PRputc (1);			/* font set */
    PRputi (0);			/* begin byte */
    PRputi (0);			/*   " low */
    PRputi (0);			/* byte length */
    PRputi (EntityStart);	/*   " low */
    PRputi (0);			/* X origin */
    PRputi (0);			/* Y origin */
    PRputi (0);			/* X bound corner */
    PRputi (0);			/* Y bound corner */
    PRputi (PageWidth);		/* X bound size */
    PRputi (PageLength);	/* Y bound size */
    PRputi ((PartLength - EntityStart) / 2);/* entity length */
fprintf(stderr,"CloseP.Part[%d]=%d,\r\n",nparts,PartLength);
    PartList[nparts++] = PartLength;
    if (nparts>MAXPARTS) 
      {
        printf( "Too many Parts (%d) in press file\n", MAXPARTS);
	Exit(-1);
      }
    Pad (512);
    height = width = 0;
    inFormat = bits;
    
}

/* set the size of the DL scratch segment */
SetDL (n) {
    if (n > LenDL) {
	register    size = LenDL * 3 / 2;
	if (n > size)
	    size = n + 1000;
	if (DL)
	    DL = (unsigned char *) realloc (DL, size);
	else
	    DL = (unsigned char *) malloc (size);
	LenDL = size;
    }
    UsedDL = n;
/*    printf("SetDL(%d) => Len=%d, DL=%o\n",n,LenDL,DL); */
}

/* set the size of the EL scratch segment */
SetEL (n) {
    if (n > LenEL) {
	register    size = LenEL * 3 / 2;
	if (n > size)
	    size = n + 1000;
	if (EL)
	    EL = (unsigned char *) realloc (EL, size);
	else
	    EL = (unsigned char *) malloc (size);
	LenEL = size;
    }
    UsedEL = n;
/*    printf("SetEL(%d) => Len=%d, EL=%o\n",n,LenEL,EL);*/
}


/* byte swap a short */
unsigned short swab1 (s)
register unsigned short s; {
    return ((s>>8) + (s<<8));
}


blt (s, d, l)			/* block transfer */
register char  *s,
               *d; {
    register char  *lim = d + l;
    if(l<=0) return;
    do

	*d++ = *s++;
    while (d < lim);
}


/* Copy the standard input file to the press file, assumes that the last
 * entity (page) has been flushed, and leaves the system in the same state
 * when it leaves.
 */
CopyFile () {
    register char   c;
    register int    i, j;

    Pressed++;

    if (PressFile == 0)
	PressFile = fopen (TempName, "w");
    if (PressFile == NULL)
	quit (1, "Can't create press file\n");
    Page = 0;
fprintf(stderr,"Before InitPg.\r\n");
    InitPage ();

    PRputc(1);			/* Set-coding */
    PRputc(0);			/* - plain bitmap */
    PRputi(width);		/* - bitmap width */
    PRputi(height);		/* - bitmap height */
    PRputc(2);			/* Set-mode */
    PRputc(3);			/* - dots to right; lines down */
    PRputi(2);			/* Set-size */
    PRputi(width*scale);	/* - width */
    PRputi(height*scale);	/* - height */
/*    PRputi(1);			/* Set window */
/*    PRputi(0);			/* - pass up dots */
/*    PRputi(width);		/* - display dots */
/*    PRputi(0);			/* - pass up lines */
/*    PRputi(height);		/* - display lines */
    PRputi(3);			/* Dots-follow */
fprintf(stderr,"H=%d,W=%d,PartLen=%d.\r\n",height,width,PartLength);
    if (inFormat = SunFormat) {
	for (i = 0; i < ((width+15)>>4); i++)
	    for (j = 0; j < height; j++)
	        bitmap[i][j] = geti();
	fprintf(stderr,"After reading bitmap\r\n");
	for (j = 0; j < height; j++)
	    for (i = 0; i < ((width+15)>>4); i++)
	        PRputi(bitmap[i][j] ^ 0xFFFF);
	fprintf(stderr,"After outputting bitmap\r\n");
	i = 0;
    } else
        for (i = height*width/8; i > 0 && !feof(stdin); i--)
	    PRputc(getchar());
fprintf(stderr,"After put.PartLen=%d,i=%d.",PartLength,i);
    while (i-- > 0) PRputc(0);
fprintf(stderr,"After nulls.PartLen=%d.\r\n",PartLength);
    Eputc(ShowDots);
    Eputi(PartLength>>17);
    Eputi((PartLength>>1) & 0xFFFF);
fprintf(stderr,"PartLen=%d=%x/%x.Close.\r\n",PartLength,PartLength>>16,PartLength&0xFFFF);
    ClosePage ();
}


/* Spit out a bcpl string to the press file.  The string will be n
 * words long, and will be case-folded if desired.
 */
Bcpl (n, fold, s)
register char  *s; {
    long    limit;

    limit = PartLength + n * 2;
    PRputc (strlen (s));
    while (*s && PartLength < limit)
	PRputc ('a' <= *s && *s <= 'z' && fold ? *s++ - 040 : *s++);
    while (PartLength < limit)
	PRputc (0);
}

/* add a string to the name list */
addstr (s)
char   *s; {
    if (s && strlen (Names) + strlen (s) <= 52)
	strcat (Names, s);
}

/* Dump the final part directory and the document directory.
 * (Assumes that the font direcory is the last part)
 */
ClosePress () {
    register long   i,
                    RecNo,
                    RecLen,
                    PadLen;
    long    t;

/* dump the font directory to the press file */
    PartLength = 0;
    PRputi (16);
    PRputc (1);		/* font set */
    PRputc (0);		/* font number within set */
    PRputc (0);		/* first character in font */
    PRputc (127);		/* last character in font */
    Bcpl (10, 1, "HELVETICA\0");/* font family name */
    PRputc (BOLD);		/* encoded face information */
    PRputc (0);		/* start character */
    PRputi (12);		/* font size in points */
    PRputi (0);		/* rotation */

    PRputi (0);		/* Indicate end of font directory */
    PartList[nparts++] = PartLength;
    Pad (512);

/* The part directory */
    RecNo = 0;
    PartLength = 0;
    for (i = 0; i < nparts; i++) {
	RecLen = (PartList[i] + 511) / 512;
	RecNo += RecLen;
	nprinted++;
	PRputi (i == nparts - 1 ? 1 : 0);
	PRputi (RecNo-RecLen);
	PRputi (RecLen);
	PadLen = (512 - PartList[i] % 512) / 2;
	PRputi (PadLen == 256 ? 0 : PadLen);
    }
    TotalPages = nprinted-1;	/* correct for the font directory
				   part */
    Pad (512);
 /* end of part directory */

 /* Document directory */
    PRputi (PRESSPASSWORD);		/* General Password */
    PRputi (RecNo + (PartLength + 511) / 512);
 /* total records in file */
    PRputi (nprinted);		/* total number of parts */
    PRputi (RecNo);		/* record number of start of part
				   directory */
    PRputi (PartLength / 512);	/* Length of part directory in records 
				*/
    PRputi (-1);		/* obsolete */
    PRputDate ();		/* date */
    PRputi (1);			/* first copy to print */
    PRputi (Copies);		/* last copy to print */
    PRputi (-1);		/* first page */
    PRputi (-1);		/* last page */
    PRputi (-1);		/* printing mode */
    Pad (0400);

    Bcpl (FILENAMELEN, 0, Banner ? Banner : UsersHeader ? UsersHeader : Names);
    if (UsersName)
	Bcpl (USERNAMELEN, 0, UsersName);
    else {
	register char  *p,
	               *name;
	name = p = getpwuid (getuid ()) -> pw_gecos;
	while (*p && *p != '(')
	    p++;
	*p = '\0';
	FileUnder(name);
    }
    t = time (0);
    Bcpl (20, 0, ctime (&t));
    Pad (512);
}

ProcessArg (p)
register char  *p; {
    static  enum State {
	normal, pressname, grabname,
	getwidth, getheight, getscale,
	grabheader, getbanner
    }
            state = normal;
    static int *whichfont;

    switch (state) {
	case pressname: 
	    strcpy (TempName, p);
	    state = normal;
	    break;
	case grabname: 
	    UsersName = p;
	    state = normal;
	    break;
	case grabheader: 
	    UsersHeader = p;
	    state = normal;
	    break;
	case getbanner: 
	    Banner = p;
	    state = normal;
	    break;
	case getwidth:
	    width = atoi(p);
	    state = normal;
	    break;
	case getheight:
	    height = atoi(p);
	    state = normal;
	    break;
	case getscale:
	    scale = atoi(p);
	    state = normal;
	    break;
	default: 
	    if (*p == '-')
		while (*++p)
		    switch (*p) {
			case 'x': case 'w':
			    state = getwidth;
			    break;
			case 'y': case 'h':
			    state = getheight;
			    break;
			case 'm':
			    state = getscale;
			    break;
			case 'b': 
			    state = getbanner;
			    break;
			case 'H':
			    state = grabheader;
			    break;
			case 'n': 
			    state = grabname;
			    break;
			case 'p':
			    state = pressname;
			    break;
			case 'S':
			     ByteOrder = (!ByteOrder);	/* toggle flag */
			     if (SeenFile)
			     	quit(1,"Byte Swapping must be specified before any files are processed\n");
			     break;
			case 't': 
			    NoTitle++;
			    break;
			default: 
			    printf ("Unknown option: %c\n", *p);
			    SeenFile++;
			    break;
		    }
	    else if (*p >= '0' && *p <= '9') {
		width = (atoi(p) + 15) / 16 * 16;
	    }
	    else {
		FileName = p;
		if (freopen (FileName, "r", stdin) == NULL) {
		    printf ("Can't open %s\n", FileName);
		    Exit(1);
		}
		fstat (fileno (stdin), &S);
		FileDate = ctime (&S.st_mtime);
		if (*Names)
		    addstr (", ");
		addstr (FileName);
		CopyFile ();
		fclose (stdin);
		SeenFile = 1;
	    }
    }
}

main (argc, argv)
char  **argv; {
    register char  *p,
                   *arg;

    mktemp (TempName);
    while (argc > 1) {
	argc--;
	ProcessArg (*++argv);
    }
    if (!SeenFile) {
	addstr ("It came from out of the blue");
	FileName = 0;
	FileDate = "";
	fstat (fileno (stdin), &S);

	if ((S.st_mode & S_IFMT) == S_IFREG)
	    FileDate = ctime (&S.st_mtime);
	CopyFile ();
    }
    if (Pressed) {
	ClosePress ();
	fclose (PressFile);
        printf ("press file left on %s\n", TempName);
    };
}

char FilingRule();	/* forward declaration */

/*
 * FileUnder -- use system dependent heuristic to guess at filing
 *	bin, print nice message asking for proper filing.
 *	(If this works out to more than 32 chars, we have to
 *	 shorten this message.)
 */
FileUnder(Uname)
char *Uname;
{
	char filechar;
	char filebuf[100];
	
	if (filechar = FilingRule(Uname)) {
		sprintf(filebuf, "%s: File under %c", Uname, filechar);
		if (strlen(filebuf) > (USERNAMELEN*2))
		    sprintf(filebuf, "%c: %s", filechar, Uname);
		Bcpl(USERNAMELEN,0,filebuf);
	}
	else Bcpl(USERNAMELEN, 0, Uname);
}

char *index();
char *rindex();

/* FilingRule -- change this to suit your filing rules */

/* rather than remove old rules, just define a new constant and
 * conditionally compile */

#define	SU_FRULE

char FilingRule(Unm)
char *Unm;
#ifdef	SU_FRULE
{
	char *firstcomma;
	char RealName[100];
	char *lastspace;
	char *endname;
	
	strcpyn(RealName,Unm,sizeof(RealName));

	firstcomma = index(RealName,',');
	
	if (firstcomma)
	    *firstcomma = 0;
	    
	endname = &RealName[strlen(RealName) - 1];
	
	while (*endname == ' ') *endname-- = 0;	/* strip trailing spaces */

	/* We now have a good guess as to the actual user's name,
	 * since we've stripped off everything after the first
	 * comma.
	 */

	lastspace = rindex(RealName,' ');
	
	if (!lastspace)
	    lastspace = RealName;	/* point to first char if no spaces */
	else
	    lastspace++;		/* skip to char after last space */

	return(*lastspace);
}
#endif	SU_FRULE

/* SwabInPlace -- does a byteswap on every word in the buffer */
/*	this could be swab() if we wanted to trust it
 *	that far.
 */
SwabInPlace(Adr,Len)
unsigned short *Adr;
int Len;
{
	register unsigned short *sptr = Adr;
	register unsigned short *bufend = &Adr[Len/2];
	register unsigned short s;
	
	while (sptr < bufend) {
		s = *sptr;
		*sptr++ = ( (s>>8) & 0xFF) | ( (s&0xFF) << 8);
		}
}
quit(i, s)
char *s;
  { fprintf(stderr,"Err:%d, %s", i, s); abort();
}
