/*
 * Logical Host Migration Program
 *
 * Usage:
 * migrateprog [-h <host-to-migrate-to>] [<pid-from-lhost-to-migrate>]*
 *
 * Specifying no logical hosts to migrate will cause all guest logical hosts
 * to be migrated.
 */

#include "Vio.h"
#include "Vmigrate.h"
#include "Vteams.h"
#include "Vprocess.h"
#include "Vdirectory.h"
#include <Vquerykernel.h>
#include <Vgroupids.h>



#define MAX_STR_LEN 80


#define MAX_DIRTY_PAGE_COPIES 4
#define CHANGE_DIRTY_PAGE_LIMIT 30

#define DIRTY_PAGE_BUFFER_SIZE 1024
char DirtyPageBuffer[DIRTY_PAGE_BUFFER_SIZE];

#define MOVE_SIZE 8192
char MoveBuf[MOVE_SIZE];

#define DEAD_TEAM 0
#define LIVE_TEAM 1
#define ALREADY_COPIED 2


#define TEAM_SERVER_DIRECTORY "[team/local]"
#define MAX_DIR_SIZE 64		/* largest number of programs we can handle */
ArbitraryDescriptor *DescriptorList[MAX_DIR_SIZE] = {0};


extern char *GetTeamSize(), *SetTeamSize();
extern char *malloc();

SystemCode GetGuestPrograms();
ProcessId MapRemoteHost(), GetRemoteHost();
SystemCode CopyAddressSpaceSegment(), CopyAddressSpace(), InitialPrecopy();

char *queryBuffer;
char *queryBuffer1;
unsigned queryBufferLen;
int *Flags;
unsigned pageSize;

ProcessId TeamServer;
ProcessId MyPid;
ProcessId SystemLHost;

int dflag = 0;
int pflag = 0;
int tflag = 0;


#define SameHost(a,b) ((a & LOGICAL_HOST_PART) == (b & LOGICAL_HOST_PART))


main(argc, argv)
    int argc;
    char **argv;
  {
    int hflag = 0;
    int nflag = 0;
    register int i;
    ProcessId pid, remHostPid;
    static char fileName[MAX_STR_LEN], remHostName[MAX_STR_LEN];
    int t1, t2, c1, c2;
    int maxTds;
    SystemCode status;
    TeamDescriptor *td;
    int success;

    while ((argc > 1) && (argv[1][0] == '-'))
      {
	if (Equal(argv[1], "-h"))
	  {
	    hflag = 1;
	    argc -= 2;
	    if (strncmp(argv[2], "0x", 2) == 0)
	      {
		sscanf(argv[2], "0x%x", &remHostPid);
		remHostPid = remHostPid << 16;
		remHostPid = (LOGICAL_HOST_PART & remHostPid) | 
						LTEAM_SERVER_GROUP;
		GetRemoteHostName(remHostPid, remHostName);
	      }
	    else
	      {
		remHostPid = MapRemoteHost(argv[2]);
		strcpy(remHostName, argv[2]);
	      }
	    argv += 2;
	    if (remHostPid == 0)
	      {
		printf("ERROR: Couldn't find team server of specified host.\n");
		exit(1);
	      }
	  }
	else if (Equal(argv[1], "-p"))
	  {
	    pflag = 1;
	    argc--;
	    argv++;
	  }
	else if (Equal(argv[1], "-d"))
	  {
	    dflag = 1;
	    pflag = 1;
	    argc--;
	    argv++;
	  }
	else if (Equal(argv[1], "-n"))
	  {
	    nflag = 1;
	    argc--;
	    argv++;
	  }
	else if (Equal(argv[1], "-t"))
	  {
	    tflag = 1;
	    argc--;
	    argv++;
	  }
	else
	  {
	    printf("Illegal option specified: %s\n", argv[1]);
	    exit(1);
	  }
      }

    TeamServer = GetPid(TEAM_SERVER, LOCAL_PID);
    MyPid = GetPid(ACTIVE_PROCESS, LOCAL_PID);
    /*
     * NOTE: The exception server should also be joining groups!
     * For now it isn't and we use this to get a system logical host id.
     */
    SystemLHost = GetPid(EXCEPTION_SERVER, LOCAL_PID);

    QueryKernelForInfo(&maxTds, &pageSize);
    queryBufferLen = sizeof(HostResourcesRec) + maxTds * sizeof(ProcessId)
					     + maxTds * sizeof(unsigned)
					     + maxTds * sizeof(ProcessId);
    queryBuffer = malloc(queryBufferLen);
    if (queryBuffer == NULL)
      {
	if (pflag) printf("ERROR in allocating a query buffer.\n");
	exit(1);
      }
    queryBuffer1 = malloc(queryBufferLen);
    if (queryBuffer1 == NULL)
      {
	if (pflag) printf("ERROR in allocating a query buffer.\n");
	exit(1);
      }
    Flags = (int *) malloc(maxTds * sizeof(int));
    if (Flags == NULL)
      {
	if (pflag) printf("ERROR in allocating a Flags buffer.\n");
	exit(1);
      }

    if (argc > 1)
      {
        /* Migrate specified programs off the machine. */
	for (i = 1; i < argc; i++)
	  {
	    /* Find which logical host to migrate. */
	    if (strncmp("0x", argv[i], 2) == 0)
	      {
		sscanf(argv[i], "0x%x", &pid);
		GetTeamName(pid, fileName);
	      }
	    else
	      {
		pid = MapTeamName(argv[i]);
		strcpy(fileName, argv[i]);
	      }
	    if (!ValidPid(pid))
	      {
		if (pflag) printf("Invalid process specified: %s.\n", argv[i]);
		continue;
	      }
	    if (SameHost(pid, SystemLHost))
	      {
		if (pflag) 
		  {
		    printf("Can't migrate a program residing in the system ");
		    printf("logical host: %s.\n", argv[i]);
		  }
		continue;
	      }
	    if (SameHost(pid, MyPid))
	      {
		if (pflag)
		  {
		    printf("Can't migrate the migrateprog program!\n");
		  }
		continue;
	      }

	    /* Find which remote host to migrate to. */
	    if (!hflag)
	      {
		remHostPid = GetRemoteHost(&status);
		if (status != OK)
		  {
		    if (pflag) printf(
			"ERROR: couldn't find a remote host to migrate to.\n");
		    exit(1);
		  }
		GetRemoteHostName(remHostPid, remHostName);
	      }

	    if (pflag) printf("Migrate logical host %s (%x) to %s (%x).\n", 
					fileName, pid,
					remHostName, remHostPid);
	    t1 = GetTime(&c1);
	    success = MigrateHost(pid, remHostPid);
	    if (!success && nflag)
	      {
		DestroyProcess(pid);
	      }
	    t2 = GetTime(&c2);
	    t2 = (t2 - t1) * 100 + (c2 - c1);
	    if (pflag) 
		printf("Time required to migrate the logical host: %d clicks.\n", 
					t2);
	    fflush(stdout);
	  }
      }
    else
      {
	/* Migrate all guest programs off the machine. */
	status = GetGuestPrograms();
	if (status != OK)
	  {
	    if (pflag) 
	      {
		printf("ERROR: couldn't query the team server for guest ");
		printf("programs to migrate: %s\n", ErrorString(status));
	      }
	    exit(1);
	  }

	for (i = 0; DescriptorList[i] != NULL; i++)
	  {
    
	    /* Find which remote host to migrate to. */
	    if (!hflag)
	      {
		remHostPid = GetRemoteHost(&status);
		if (status != OK)
		  {
		    if (pflag) printf(
			"ERROR: couldn't find a remote host to migrate to.\n");
		    exit(1);
		  }
		GetRemoteHostName(remHostPid, remHostName);
	      }

	    td = (TeamDescriptor *) DescriptorList[i];
	    if (pflag) printf("Migrate logical host %s (%x) to %s (%x).\n", 
					td->fileName, td->rootPid,
					remHostName, remHostPid);
	    t1 = GetTime(&c1);
	    success = MigrateHost(td->rootPid, remHostPid);
	    if (!success && nflag)
	      {
		DestroyProcess(td->rootPid);
	      }
	    t2 = GetTime(&c2);
	    t2 = (t2 - t1) * 100 + (c2 - c1);
	    if (pflag) 
		printf("Time required to migrate the logical host: %d clicks.\n",
					t2);
	    fflush(stdout);
	  }
      }
  }


/*
 * MigrateHost:
 */

int MigrateHost(lHostPid, remHostPid)
    ProcessId lHostPid;
    ProcessId remHostPid;
  {
    int i, j, k;
    int nT, nP;
    int redoFlag = 0;
    int nT1;
    unsigned totalMemSize, *memSizes;
    HostResourcesRec *hostRec;
    unsigned descSize;
    ProcessId *processes, *remoteProcesses;
    ProcessId *processes1, *remoteProcesses1;
    char *tmp;
    ProcessId myPid;
    SystemCode status;
    Message msg;
    MsgStruct *req = (MsgStruct *) msg;
    ProcessId migrationId;
    int dirtyPageNumber, oldDirtyPageNumber, teamDirtyPageNumber;
    char *descBuffer;

    if (tflag) InitTimingPackage(stdout);

    /*
     * Initialize a migration request and set up a dummy logical host on the
     * remote machine to migrate to.
     */

    /* Find out the resource requirements of the logical host. */
    status = ExtractHost(lHostPid, QUERY_HOST_CASE, 
    				queryBuffer, queryBufferLen);
    if (status != OK)
      {
	if (pflag) printf("ERROR in ExtractHost query operation: %s\n", 
					ErrorString(status));
	return(0);
      }
    hostRec = (HostResourcesRec *) queryBuffer;
    nT = hostRec->numTds;
    nP = hostRec->numPds;
    totalMemSize = hostRec->totalMemory;
    descSize = hostRec->descSize;
    processes = (ProcessId *) (queryBuffer + sizeof(HostResourcesRec));
    memSizes = (unsigned *) (((char *)processes) + nT * sizeof(ProcessId));
    remoteProcesses = (ProcessId *) (((char *)memSizes) + 
						    nT * sizeof(unsigned));
    for (j = 0; j < nT; j++)
      {
	Flags[j] = LIVE_TEAM;
      }

    if (tflag)
      {
	printf("Number of teams in lhost: %d,  number of processes: %d\n", 
						nT, nP);
	printf("Total memory requirements: %d\n", totalMemSize);
	MarkTime("Time for ExtractHost query");
      }

redo:

    /* Send a migration request message to the remote host's program manager
       asking it if it will accept the migrating logical host.  Send along
       the resource requirements.  If accepted, the remote program manager 
       will create a dummy logical host with dummy teams in it to be used for 
       copying the address spaces over to. */
    if (redoFlag)
	req->sysCode = REDO_MIGRATION_REQUEST;
    else
	req->sysCode = MIGRATION_REQUEST;
    req->unspecified[0] = (unsigned) nT;
    req->unspecified[1] = (unsigned) nP;
    req->unspecified[2] = totalMemSize;
    if (redoFlag)
	req->unspecified[3] = migrationId;
    req->segmentPtr = (char *) processes;
    req->segmentSize = nT * (sizeof(ProcessId) + sizeof(unsigned) + 
						 sizeof(ProcessId));
    Send(msg, remHostPid);
    if (req->sysCode != OK)
      {
	if (pflag) 
	    printf("Migration request failed: %s\n", ErrorString(req->sysCode));
        return(0);
      }
    migrationId = req->unspecified[0];
    if (tflag) MarkTime("Time to send initial migration request");

    /*
     * Precopy the address spaces of the logical host.
     */
	  
    myPid = GetPid(ACTIVE_PROCESS, LOCAL_PID);
    for (j = 0; j < nT; j++)
      {
        if (Flags[j] == ALREADY_COPIED)
	  {
	    continue;
	  }
        status = InitialPrecopy(processes[j], &memSizes[j], 
					remoteProcesses[j], myPid);
	if (status != OK)
	  {
	    if (status == NONEXISTENT_PROCESS)
	      {
		Flags[j] = DEAD_TEAM;
	      }
	    else
	      {
		AbortMigration(remHostPid, lHostPid);
		if (pflag) printf("Error in Initial pre-copy operation: %s.\n", 
			    ErrorString(status));
		return(0);
	      }
	  }
      }
    if (tflag) MarkTime("Time to do first full copy of address spaces");

    /* Iteratively copy the dirty pages left in each address space until
       the change in the number of dirty pages is less than 
       CHANGE_DIRTY_PAGE_LIMIT or MAX_DIRTY_PAGE_COPIES iterations have been 
       performed. */
    oldDirtyPageNumber = CHANGE_DIRTY_PAGE_LIMIT * 2;
				/* Make sure the first iteration will always
				   happen. */
    for (i = 0; i < MAX_DIRTY_PAGE_COPIES; i++)
      {
        dirtyPageNumber = 0;
	for (j = 0; j < nT; j++)
	  {
	    if (Flags[j] == DEAD_TEAM)
	      {
		continue;
	      }
	    status = CopyAddressSpace(processes[j], remoteProcesses[j],
	    		&teamDirtyPageNumber, pageSize, &memSizes[j]);
	    if (status != OK)
	      {
	        if (status == NONEXISTENT_PROCESS)
		  {
		    Flags[j] = DEAD_TEAM;
		  }
		else
		  {
		    AbortMigration(remHostPid, lHostPid);
		    if (pflag) printf("Error in copying address spaces: %s\n",
				ErrorString(status));
		    return(0);
		  }
	      }
	    dirtyPageNumber += teamDirtyPageNumber;
	  }
	if ((oldDirtyPageNumber - dirtyPageNumber) < CHANGE_DIRTY_PAGE_LIMIT)
	  {
	    break;
	  }
	oldDirtyPageNumber = dirtyPageNumber;
      }
    if (tflag) MarkTime("Time to precopy address spaces");

    /* 
     * Check to see if any teams have died or been created. 
     */

    /* Destroy the dummy teams of teams that have died. */
    for (j = 0; j < nT; j++)
      {
        if (Flags[j] == DEAD_TEAM)
	  {
	    if (dflag) 
	      {
		printf("Team %x DIED before it could be migrated.\n",
			processes[j]);
	      }
	    DestroyDummyTeam(remoteProcesses[j], remHostPid);
	  }
      }

    /* Save the old descriptor info. */
    nT1 = nT;
    processes1 = processes;
    remoteProcesses1 = remoteProcesses;
    tmp = queryBuffer;
    queryBuffer = queryBuffer1;
    queryBuffer1 = tmp;

    /* Get current info. */
    status = ExtractHost(lHostPid, QUERY_HOST_CASE, 
    				queryBuffer, queryBufferLen);
    if (status != OK)
      {
	if (pflag) 
	    printf("ERROR in ExtractHost query operation: %s\n", 
				ErrorString(status));
	return(0);
      }
    hostRec = (HostResourcesRec *) queryBuffer;
    nT = hostRec->numTds;
    nP = hostRec->numPds;
    totalMemSize = hostRec->totalMemory;
    descSize = hostRec->descSize;
    processes = (ProcessId *) (queryBuffer + sizeof(HostResourcesRec));
    memSizes = (unsigned *) (((char *)processes) + nT * sizeof(ProcessId));
    remoteProcesses = (ProcessId *) (((char *)memSizes) + 
						    nT * sizeof(unsigned));

    /* Determine if any new teams have been created. */
    redoFlag = 0;
    for (i = 0; i < nT; i++)
      {
        Flags[i] = LIVE_TEAM;
        for (j = 0; j < nT1; j++)
	  {
	    if (processes[i] == processes1[j])
	      {
		Flags[i] = ALREADY_COPIED;
		remoteProcesses[i] = remoteProcesses1[j];
		break;
	      }
	  }
	if (Flags[i] != ALREADY_COPIED)
	  {
	    if (dflag)
	      {
		printf("NEW team %x creating during migration.\n",
			processes[i]);
	      }
	    redoFlag = 1;
	  }
      }
    if (redoFlag)
      {
	goto redo;
      }

    /*
     * Freeze the logical host and make a final copy of its address spaces.
     */

    /* Freeze the logical host. */
    status = FreezeHost(lHostPid);
    if (status != OK)
      {
	AbortMigration(remHostPid, lHostPid);
	if (pflag) printf("Error in FreezeHost: %s\n", ErrorString(status));
	return(0);
      }
    if (tflag) MarkTime("Time to freeze host");

    /* Make a final copy of the address spaces. */
    for (j = 0; j < nT; j++)
      {
	status = CopyAddressSpace(processes[j], remoteProcesses[j],
		    &teamDirtyPageNumber, pageSize, &memSizes[j]);
	if (status != OK)
	  {
	    AbortMigration(remHostPid, lHostPid);
	    if (pflag) printf("Error in final copy operation: %s.\n", 
				ErrorString(status));
	    return(0);
	  }
      }

    /*
     * Create a descriptor for the kernel information for this
     * logical host.
     */

    /* Set up the descriptor buffer. */
    descBuffer = malloc(descSize);
    if (descBuffer == NULL)
      {
	AbortMigration(remHostPid, lHostPid);
	if (pflag) printf("Error in allocating descriptor buffer\n");
	return(0);
      }

    /* Extract the logical host's kernel state. */
    status = ExtractHost(lHostPid, EXTRACT_HOST_CASE, descBuffer, descSize);
    if (status != OK)
      {
	if (pflag) printf("Error in ExtractHost: %s\n", ErrorString(status));
	AbortMigration(remHostPid, lHostPid);
	free(descBuffer);
	return(0);
      }
    if (tflag) MarkTime("Time to extract host");

    /* Send the logical host descriptor information to the remote host. */
    req->sysCode = TRANSFER_HOST_REQUEST;
    req->unspecified[0] = migrationId;
    req->unspecified[1] = TeamServer;
    req->segmentPtr = (char *) descBuffer;
    req->segmentSize = descSize;
    Send(msg, remHostPid);
    if (req->sysCode != OK)
      {
	if (pflag) 
	    printf("TransferHost request failed: %s\n", 
					ErrorString(req->sysCode));
	AbortMigration(remHostPid, lHostPid);
	free(descBuffer);
        return(0);
      }
    free(descBuffer);
    if (tflag) MarkTime("Time to send TransferHost request");

    /*
     * There are now two copies of the logical host.
     * Destroy the local one via the team server.  NOTE: We can't just let
     * the team server try to find out later about the destroyed host because
     * it will find the remote, new copy instead!
     */
    
    req->sysCode = DESTROY_HOST;
    req->unspecified[0] = MIGRATED_HOST;
    req->unspecified[1] = lHostPid;
    Send(msg, TeamServer);
    if (req->sysCode != OK)
      {
	if (pflag) printf("ERROR in Team Server DestroyHost req.: %s\n", 
					ErrorString(req->sysCode));
	return(0);
      }
    if (tflag) MarkTime("Time to DestroyHost");

    return(1);
  }


/*
 * InitialPrecopy:
 */

SystemCode InitialPrecopy(localPid, memSizePtr, remotePid, myPid)
    ProcessId localPid;
    unsigned *memSizePtr;
    ProcessId remotePid;
    ProcessId myPid;
  {
    ProcessId pid;
    unsigned memSize;
    char *newMemSize;
    SystemCode status;
    Message msg;
    MsgStruct *req = (MsgStruct *) msg;

    if (!ValidPid(localPid))
      {
	return(NONEXISTENT_PROCESS);
      }
    memSize = *memSizePtr;
    /* Gain access to the address space of the team
       by becoming its owner. */
    pid = SetObjectOwner(localPid, myPid);
    /* Receive access to the address space of the corresponding dummy 
       team on the remote host. */
    pid = ReceiveSpecific(msg, remotePid);
    /* Do the initial copy of the address spaces over to the remote dummy 
       teams after clearing all the page map dirty bits. */
    ClearModifiedPages(localPid);
    newMemSize = GetTeamSize(localPid);
    if (memSize != ((unsigned)(newMemSize - TEAM_ORIGIN)))
      {
	if (SetTeamSize(remotePid, newMemSize) != newMemSize)
	  {
	    if (pflag) printf("Error in setting dummy team size.\n");
	    return(NO_MEMORY);
	  }
	memSize = (unsigned) (newMemSize - TEAM_ORIGIN);
      }
    status = CopyAddressSpaceSegment(localPid, remotePid, 
	    TEAM_ORIGIN, memSize);
    if ((status != OK) && (status != NONEXISTENT_PROCESS))
      {
	if (pflag) 
	    printf("Error in copying address spaces: %s\n",
				ErrorString(status));
      }
    *memSizePtr = memSize;
    return(status);
  }


/*
 * CopyAddressSpace:
 */

SystemCode CopyAddressSpace(fromPid, toPid, dirtyPageNumberPtr, 
						pageSize, memSize)
    ProcessId fromPid;
    ProcessId toPid;
    unsigned *dirtyPageNumberPtr;
    unsigned pageSize;
    unsigned *memSize;
  {
    SystemCode status;
    unsigned dirtyPageNumber, dpn;
    int k;
    ProcessId pid;
    unsigned *dpPtr;
    unsigned ps;
    char *teamSize;
    unsigned newMemSize;

    status = RETRY;
    dirtyPageNumber = 0;
    while (status == RETRY)
      {
        dpn = DIRTY_PAGE_BUFFER_SIZE;
	status = ReturnModifiedPages(fromPid, DirtyPageBuffer,
		&dpn);
	if (status != OK)
	  {
	    if (status != NONEXISTENT_PROCESS)
	      {
		if (pflag) printf("ERROR in ReturnModifiedPages: %s\n", 
						ErrorString(status));
	      }
	    return(status);
	  }
        dirtyPageNumber += dpn;
	/*
	 * NOTE: This could be speeded up by copying contiguous
	 * dirty pages regions in single copy operations.
	 */
	teamSize = GetTeamSize( fromPid );
	newMemSize = (unsigned) (teamSize - TEAM_ORIGIN);
	if (newMemSize > *memSize)
	  {
	    *memSize = newMemSize;
	    if (SetTeamSize(toPid, teamSize) != teamSize)
	      {
		if (pflag) 
		    printf("ERROR: couldn't SetTeamSize the %x dummy team.\n", 
								toPid);
		return(NO_MEMORY);
	      }
	  }
        ps = pageSize;
	for (k = 0, dpPtr = (unsigned *) DirtyPageBuffer; 
	     k < dpn; 
	     k++, dpPtr++)
	  {
	    if (k == (dpn - 1))
	      {
		if (teamSize < ((char *)(*dpPtr + pageSize)))
		  {
		    ps = ((unsigned)teamSize) - ((unsigned)*dpPtr);
		  }
	      }
	    status = CopyAddressSpaceSegment(fromPid, toPid,
			*dpPtr, ps);
	    if (status != OK)
	      {
		return(status);
	      }
	  }
      }
    *dirtyPageNumberPtr = dirtyPageNumber;
    return(OK);
  }


/*
 * CopyAddressSpaceSegment:
 * Copy the specified pages between the address
 * spaces designated by fromPid and toPid.
 */

SystemCode CopyAddressSpaceSegment(fromPid, toPid, start, length)
    ProcessId fromPid, toPid;
    char *start;
    unsigned length;
  {
    char *mp, *endP;
    unsigned size;
    SystemCode status;

    endP = start + length;
    for (mp = start; 
	    (mp + MOVE_SIZE) <= endP;
	    mp += MOVE_SIZE)
      {
	status = MoveFrom(fromPid, MoveBuf, mp, MOVE_SIZE);
	if (status != OK)
	  {
	    if (status != NONEXISTENT_PROCESS)
	      {
		if (pflag) 
		    printf("ERROR in MoveFrom of an address space: %s\n",
		    		ErrorString(status));
	      }
	    return(status);
	  }
	status = MoveTo(toPid, mp, MoveBuf, MOVE_SIZE);
	if (status != OK)
	  {
	    if (status != NONEXISTENT_PROCESS)
	      {
		if (pflag) 
		    printf("ERROR in MoveTo of an address space: %s\n",
		    		ErrorString(status));
	      }
	    return(status);
	  }
      }
    if (mp != endP)
      {
	size = endP - mp;
	status = MoveFrom(fromPid, MoveBuf, mp, size);
	if (status != OK)
	  {
	    if (status != NONEXISTENT_PROCESS)
	      {
		if (pflag) 
		    printf("ERROR in last MoveFrom of an address space: %s\n",
		    ErrorString(status));
	      }
	    return(status);
	  }
	status = MoveTo(toPid, mp, MoveBuf, size);
	if (status != OK)
	  {
	    if (status != NONEXISTENT_PROCESS)
	      {
		if (pflag) 
		    printf("ERROR in last MoveTo of an address space: %s\n",
		    ErrorString(status));
	      }
	    return(status);
	  }
      }
    return(OK);
  }


/* 
 * QueryKernelForInfo:
 */

QueryKernelForInfo(maxTds, pageSize)
    int *maxTds;
    unsigned *pageSize;
  {
    /* For now, fake it. */
    *maxTds = 64;
    *pageSize = 0x800;
  }


/*
 * AbortMigration:
 */

AbortMigration(remHostPid, lhost)
    ProcessId remHostPid;
    ProcessId lhost;
  {
    SystemCode status;
    Message msg;
    MsgStruct *req = (MsgStruct *) msg;

    /* Unfreeze the local copy so it can continue on its way. */
    UnfreezeHost(lhost);

    /* Tell the remote team server to destroy its copy. */
    req->sysCode = DESTROY_HOST;
    req->unspecified[0] = KILL_HOST;
    req->unspecified[1] = lhost;
    Send(msg, remHostPid);
    if (req->sysCode != OK)
      {
	if (pflag) 
	    printf("ERROR in AbortMigration req. to remote t.server: %s\n", 
					ErrorString(req->sysCode));
      }
  }


DestroyDummyTeam(dummyPid, remHostPid)
    ProcessId dummyPid;
    ProcessId remHostPid;
  {
    SystemCode status;
    Message msg;
    ExitTeamRequest *req = (ExitTeamRequest *) msg;

    req->requestcode = TERMINATE_TEAM;
    req->requestType = SpecificTeam;
    req->rootPid = dummyPid;
    req->status = -1;
    Send(msg, remHostPid);
  }


extern SystemCode GetContextId();

/*
 * MapRemoteHost:
 * Maps a string name for a host into its corresponding team server 
 * process-id.  0 is returned if no mapping can be performed.
 */
ProcessId MapRemoteHost(host)
    char *host;
  {
    char namebuf[100];
    ContextPair ctx;

    strcpy(namebuf, "[team/");
    strcat(namebuf, host);
    if (GetContextId(namebuf, &ctx) != OK) return 0;

    return(ctx.pid);
  }


SystemCode GetGuestPrograms()
  {
    register File *serverFile;
    register ArbitraryDescriptor *desc;
    int lastdesc = 0;
    SystemCode error;
    int size, rsize;

    serverFile = Open(TEAM_SERVER_DIRECTORY, FREAD|FDIRECTORY|FBLOCK_MODE, 
								 &error);
    if (error != OK)
      {
	return(error);
      }
    size = serverFile->blocksize;

    desc = (ArbitraryDescriptor *) malloc(sizeof(ArbitraryDescriptor));

    do
      {
	rsize = Read( serverFile, desc, size );
        if (rsize > 0)
	  {
	    if ( desc->e.descriptortype != EMPTY_DESCRIPTOR )
	      {
	        if (NewLogicalHost(desc->t.rootPid) &&
		    (desc->t.runPriority == GUEST))
		  {
		    DescriptorList[lastdesc++] = desc;
		    if (lastdesc > MAX_DIR_SIZE)
		      {
			break;
		      }
		    desc = (ArbitraryDescriptor *) 
			malloc(sizeof(ArbitraryDescriptor));
		  }
	      }
	    serverFile->block++;
	  }
      }
    while ( serverFile->lastexception == OK );

    if ( serverFile->lastexception == END_OF_FILE )
      {
	error = OK;
      }
    else
      {
	error = serverFile->lastexception;
      }

    Close(serverFile);
    return(error);
  }


int NewLogicalHost(pid)
    ProcessId pid;
  {
    register int i;

    if (SameHost(MyPid,pid) || SameHost(TeamServer,pid))
      {
	return(0);
      }
    for (i = 0; DescriptorList[i] != NULL; i++)
      {
	if (SameHost(((TeamDescriptor *) DescriptorList[i])->rootPid, pid))
	  {
	    return(0);
	  }
      }
    return(1);
  }


ProcessId GetRemoteHost(error)
    SystemCode *error;
  {
    static SelectionRec foundHostSpec[2];
    static SelectionRec hostSpec;
    int n, indx;
    Message msg;
    QueryKernelRequest *request = (QueryKernelRequest *)msg;
    MachineConfigurationReply *mcReply = (MachineConfigurationReply *)msg;
    unsigned procType;

    DefaultSelectionRec(&hostSpec);

    QueryKernel(0, MACHINE_CONFIG, mcReply);
    if (mcReply->replycode != OK) 
      {
        *error = mcReply->replycode;
        return(0) ;
      }
    procType = (mcReply->processor & PROC_FAMILY) >> 4;
    hostSpec.procMachType &= ~PROC_FAMILY_IRRELEVANT;
    hostSpec.procMachType |= procType;

    n = QueryHosts(&hostSpec, foundHostSpec, 2, error);
    if (*error != OK || n == 0)
      {
	*error = SERVER_NOT_RESPONDING;
	return(0);
      }
    if (TeamServer != foundHostSpec[0].teamServerPid)
      {
	indx = 0;
      }
    else if (n > 1)
      {
	indx = 1;
      }
    else
      {
	*error = SERVER_NOT_RESPONDING;
	return(0);
      }
    return(foundHostSpec[indx].teamServerPid);
  }


GetRemoteHostName(pid, name)
    ProcessId pid;
    char *name;
  {
    Message msg ;
    UpdateHostStatusRequest * updaterequest = (UpdateHostStatusRequest *) msg;
    static HostStatusRecord hoststatus;

    updaterequest->requestcode = UpdateRequest ;
    updaterequest->requesttype = GetHostStatus ;
    updaterequest->bufferptr = (char *) &hoststatus ;
    updaterequest->bytecount = sizeof (HostStatusRecord) ;
    Send (updaterequest, pid);
    if (updaterequest->requestcode != OK)
      {
	name[0] = '\0';
      }
    else
      {
	strcpy(name, hoststatus.hostName);
      }
  }


GetTeamName(pid, name)
    ProcessId pid;
    char *name;
  {
    Message msg;
    DescriptorRequest *reqMsg = (DescriptorRequest *) msg;
    DescriptorReply *repMsg = (DescriptorReply *) msg;
    static TeamDescriptor desc;

    desc.fileName[0] = '\0';	/* Team server is queried by team root, 
				   not by name. */
    reqMsg->requestcode = NREAD_DESCRIPTOR;
    reqMsg->nameindex = ((int) &(desc.fileName[0])) - (int)(&desc);
    reqMsg->dataindex = 0;
    reqMsg->fileid = 0;
    reqMsg->namecontextid = (ContextId) pid;
    reqMsg->segmentptr = (char *) (&desc);
    reqMsg->segmentlen = sizeof(desc);
    Send(msg, TeamServer);
    if (repMsg->replycode != OK)
      {
	name[0] = '\0';
      }
    else
      {
	strcpy(name, desc.fileName);
      }
  }
