/* The code in this file handles the checking for the arrival of new mail
 * (a feature enabled by the -c option).
 */
#include <Venviron.h>
#include <Vio.h>

extern int checkInterval;

ProcessId IO_Helper, MailChecker = 0;

#define READ_LINE 0x0036
#define IO_Helper_Priority 11
#define MailChecker_Priority 12
#define HelperStackSize 1000


typedef struct
  {
#ifndef LITTLE_ENDIAN
    SystemCode requestcode;  /* should be READ_LINE. */
    short padding;
#else LITTLE_ENDIAN
    short padding;
    SystemCode requestcode;  /* should be READ_LINE. */
#endif LITTLE_ENDIAN
    File *ibuf;
    char *linebuf;
    int unused[5];
  }
ReadLineRequest;

typedef struct
  {
#ifndef LITTLE_ENDIAN
    SystemCode replycode;  /* should be OK. */
    short padding;
#else LITTLE_ENDIAN
    short padding;
    SystemCode replycode;  /* should be OK. */
#endif LITTLE_ENDIAN
    int result;
    int unused[6];
  }
ReadLineResponse;


IO_HelperProcess()
    /* Reads lines of input, on behalf of the client (the main program). */
  {
    Message msg;
    ReadLineRequest *req = (ReadLineRequest *)msg;
    ReadLineResponse *rep = (ReadLineResponse *)msg;
    ProcessId fromPid;

    while (1)
      {
	fromPid = Receive(msg);
	rep->result = readline(req->ibuf, req->linebuf);
	rep->replycode = OK;
	Reply(rep, fromPid);
      }
  }

File *checkerFbuf = NULL;

MailCheckerProcess()
    /* Periodically checks for the arrival of new mail.  If new mail is 
     * found, then this process signals this fact by destroying the IO_Helper 
     * process.
     */
  {
    extern char mailname[];
    extern int mailsize, commandInProgress;

    int delaySeconds;

    delaySeconds = checkInterval*60;
    while (1)
      {
	Delay(delaySeconds, 0);
	if (commandInProgress)
	    continue;

	checkerFbuf = fopen(mailname, "r");
	if (checkerFbuf == NULL)
	  {
	    printf("new mail check - couldn't open mailfile!\n");
	    continue;
	  }
	if ((fsize(checkerFbuf) > mailsize) || (checkerFbuf == NULL))
	    Destroy(IO_Helper); /* signal arrival of new mail */
	fclose(checkerFbuf);
	checkerFbuf = NULL;
      }
  }


/* Read a command line.  If periodic checking for new mail is turned on and
 * "ibuf" is stdin, then this causes a message to be sent to the IO_Helper 
 * process (which calls readline()), otherwise (the default case) readline() 
 * is invoked directly.
 */
int GetLine(ibuf, linebuf)
	FILE *ibuf;
	char *linebuf;
  {
    if (!checkInterval || (ibuf != stdin))
	/* The usual case. */
	return (readline(ibuf, linebuf));
    else
      {
	Message msg;
	ReadLineRequest *req = (ReadLineRequest *)msg;
	ReadLineResponse *rep = (ReadLineResponse *)msg;

	/* If the IO helper process has been nuked by the mail checker (or 
	 * has yet to be started), then restart it.
	 */
	if (!ValidPid(IO_Helper))
	  {
	    IO_Helper
	       = Create(IO_Helper_Priority, IO_HelperProcess, HelperStackSize);
	    Ready(IO_Helper, 0);
	  }

	req->requestcode = READ_LINE;
	req->ibuf = ibuf;
	req->linebuf = linebuf;
	Send(req, IO_Helper);
	/* Note: this Send() will return either if the IO_Helper process 
	 * reads a command and does a Reply() (the usual case), or if the 
	 * IO_Helper process is destroyed by the MailChecker() (to signal the
	 * arrival of new mail).
	 */
	if (rep->replycode != OK)
	  {
	    /* We assume that the IO_Helper got destroyed, because the 
	     * MailChecker found new mail.
	     */
	    Flush(stdout);
	    Flush(ibuf); /* == stdin */
	    linebuf[0] = 'U'; /* Enter a "Update" command into the buffer. */
	    linebuf[1] = '\0';
	    return(2);
	  }
	else
	    return (rep->result);
      }
  }


enableMailChecker()
  {
    if (checkInterval)
      {
	MailChecker
	   = Create(MailChecker_Priority, MailCheckerProcess, HelperStackSize);
	Ready(MailChecker, 0);
      }
  }

disableMailChecker()
  {
    if (checkInterval)
      {
	Destroy(MailChecker);
	/* If the mail checker process had the mail file open at the time, 
	 * then close it.
	 */
	if (checkerFbuf != NULL)
	  {
	    fclose(checkerFbuf);
	    checkerFbuf = NULL;
	  }
      }
  }
