/*
 * files.c : V-System Iconiface file manipulation routines. (or rather,
 *   routines that manipulate the file icons on the screen...
 *
 * AddDirectory, FileHit are included.
 *
 * CS246 - Advanced OS
 * Robert Neilon, Madhavan Srinivasan, Matthew Zekauskas
 * June 10, 1985
 *
 * 26May85 (Matt Zekauskas): first coherent module.
 * 30May85 (mjz): Refresh & scrolling
 */
/* due to funny V-sys definitions, sys/types must be early in include
 * list ... I haven't bothered to figure out how early...
 */
#include <sys/types.h>		/* sys/stat is dependent on this... */
#include <sys/stat.h>		/* UNIXFILE descriptor stuff (S_IFDIR) */
#include <Vstorage.h>		/* FILE descriptor stuff (SS_DIR) */

#include <Vio.h>		/* for stderr */
#include <Vgts.h>		/* for SDF_RASTER */
#include <bitmaps.h>		/* for MemoryRaster */
#include <Vnaming.h>		/* for ContextPair */
#include <Vdirectory.h>		/* Arbitrary Descriptors *//* Vio dependent */
#include "iconstants.h"		/* our global include file */

#define	WIDTH	100		/* space for an icon in x */
#define HEIGHTH	108		/* space for icon + descr in y */

/* imports */
extern char i_generic[], i_folder[], i_program[], i_computer[], i_text[];
extern char i_up[], i_down[];	/* for scrolling */
#define SCROLLX 	17	/* width of scroll icon */
#define SCROLLY		32	/* depth of scroll icon */
extern char i_watch[], i_hourglass[];	/* for reassurance while waiting */
extern int DebugFlag;
extern int StatsFlag;
extern File *StatsFile;
extern long GetTime();

/* static structures local to this "module" */
#define MAX_FILE_DESCRIPTORS	1024	/* max # of files we can handle */
static ArbitraryDescriptor *FileDescriptors[MAX_FILE_DESCRIPTORS];
static int NumFiles;		/* actual # descs in above */
static int StartFile;		/* starting file # on screen */


static char *TypeIcon[FILETYPE_NUM];	/* map filetypes to icons */
				/* filled in on first AddDir call */
static int Initialized = 0;	/* to fill above, and more! */

static struct {			/* map icons to names, types */
	int 	descnum;	/* descriptor in FileDescriptors */
	FileType ftype;		/* type of this file */
    } FileTable[NO_OF_FILES];
static int NumOnScreen = 0;	/* Number of file icons on screen */
static int MaxOnScreen = 0;	/* Max in given bounding box on screen */

static char OldContext[500] = ""; /* Last context added to screen */

/*
 ***************************************************************************
 *                        A d d D i r e c t o r y			   *
 *-------------------------------------------------------------------------*
 *    Add in the directory icons.  They are added within the given bounding*
 *   box.  If noInvert, then don't invert the context line.  If noRead,    *
 *   then the directory could not possibly have been updated, so don't     *
 *   bother to read the descriptors again.  startNum is where in the       *
 *   directory to start displaying files, if negative start from the       *
 *   current position.							   *
 ***************************************************************************
 */

AddDirectory(sdf,xmin,xmax,ymin,ymax,noInvert,noRead,startNum)
	short sdf;
	int xmin,xmax,ymin,ymax;
	int noInvert;		/* do not rev video context */
	int noRead;		/* skip reading of directory (hack?) */
	int startNum;		/* starting file number to display. */
				/* (if negative, use current value) */
  {
    int numAcross;		/* how many fit across? */
    int numDown;		/* how many fit down? */
    int topMargin, leftMargin;  /* like it says */
    static char context[500];	/* current context, get this from elsewhere?*/
    ContextPair ctxpr;		/* dummy, to satisfy GetAbsoluteName */
    SystemCode error;
    int num;			/* number of directory entries */
    int limit, x, y, i, j;
    int itable;			/* index into FileTable */
    int loc;			/* loc of .c in file name */
    char *rindex();		/* @#$% function returns c pointer */
    char *fname;		/* file name of current file */
    short item;			/* item number source for icons */
    short filefont;		/* font used for file names */

    filefont = DefineFont("Helvetica10",NULL);
    fprintf(stderr,"Font Helvetica10 is %d.\n\r",filefont);

    if (!Initialized)
      {
	Initialized = 1;	/* now true */
	/* associate icons with file types */
	TypeIcon[FILETYPE_OTHER] = i_generic;
	TypeIcon[FILETYPE_C] = i_program;
	TypeIcon[FILETYPE_DIR] = i_folder;
	TypeIcon[FILETYPE_TEXT] = i_text;
	TypeIcon[FILETYPE_EXE] = i_computer;
      }

    xmin += 20;			/* allow room for scroll boxes */
    if (startNum >= 0)		/* figure starting file # */
      StartFile = startNum;

    numAcross = (xmax - xmin) / WIDTH;
    leftMargin = ((xmax - xmin) % WIDTH)/2;	/* center */
    numDown = (ymax - ymin) / HEIGHTH;
    topMargin = ((ymax - ymin) % HEIGHTH)/2;
    MaxOnScreen = numAcross * numDown;	/* max possible on screen */
    if (MaxOnScreen > NO_OF_FILES) 
      MaxOnScreen = NO_OF_FILES;	/* but no more than array allows */

    context[0] = '\0';			/* null string => get current ctx */
    error = GetAbsoluteName(context, sizeof(context), &ctxpr);	/* temp */
    fprintf(stderr,"Context: %s\n\r",context);
    /* following so cd to new directory starts us at the beginning */
    if (strcmp(context,OldContext) != 0)
      {
	StartFile = 0;		/* force start to beginning */
	noRead = 0;		/* force read of directory */
	strcpy(OldContext,context);	/* save new one */
      }



    if (!noInvert)
      {
	char *cptr;	/* to invert */
	for (cptr = context; *cptr != '\0'; *(cptr++) = *cptr+128);
      }
    AddItem(sdf,FILE_CONTEXT, xmin+20, xmax, ymax-20, ymax, 0, 
	SDF_SIMPLE_TEXT, context);
    if (!noInvert)	/* must undo - perhaps get our own copy??? */
      {			/* (otherwise ReadDirectory Fails) */
	char *cptr;	/* to invert */
	for (cptr = context; *cptr != '\0'; *(cptr++) = *cptr ^ 0x80);
      }

    if (!noRead)
      {
	if (StatsFlag)
	  fprintf(StatsFile, "%ld About to call ReadDirectory\n",GetTime(0));
	fprintf(stderr,"About to call ReadDirectory\n\r");
	/* hack warning -- follwing is temp and must be fixed up */
	num = ReadDirectory(context,FileDescriptors,MAX_FILE_DESCRIPTORS,
			&error);
	if (StatsFlag)
	  fprintf(StatsFile,"%ld Back from ReadDirectory\n",GetTime(0));
	if (num <= 0) 
	  {
	    fprintf(stderr,"Couldn't read directory!\n\r");
	    fprintf(stderr,"Error (%d): %s\n",error,ErrorString(error));
	    Quit();
	  }
	NumFiles = num;		/* save in static area */
      } /* end of if not (noRead) */

    if (NumFiles <= StartFile)	/* then bogus Start number */
      {			/* skew to place last files on screen */
	StartFile = NumFiles - (MaxOnScreen);
	if (StartFile < 0) 
	  StartFile = 0;
      }

    if (DebugFlag) fprintf(stderr, "About to add files..\n\r");
    if (DebugFlag)
	fprintf(stderr, "numacross=%d, numdown=%d\n\r",numAcross,numDown);
    if (StatsFlag)
	fprintf(StatsFile,"%ld Adding icons to screen...\n",GetTime(0));

    limit = NumFiles - StartFile;	/* max # we want on screen */
    limit = limit < MaxOnScreen ? limit : MaxOnScreen; /* max # we can have */
    NumOnScreen = limit;	/* save in static area */
    limit += StartFile;		/* adjust to StartFile origin */

    item = FILE_LOWLIMIT;	/* first file item number */
    for (i = StartFile, itable = 0,  y = ymax - 20 - topMargin - (ICON_Y + 1);
	i < limit;
	y -= HEIGHTH) 
      {
	for (x = leftMargin + xmin, j = 0;
	    (j < numAcross) && (i < limit);
	     j++, i++, itable++, x += WIDTH)
	  {
	    if (DebugFlag) fprintf(stderr, "Adding file %d",i);
	    /* decide file type */
	    switch (FileDescriptors[i]->e.descriptortype)
	      {
		case FILE_DESCRIPTOR:
		    if (FileDescriptors[i]->f.perms & SS_DIR)
		      FileTable[itable].ftype = FILETYPE_DIR;
		    else
		      FileTable[itable].ftype = FILETYPE_OTHER;
		    fname = FileDescriptors[i]->f.name;
		    break;

		case UNIXFILE_DESCRIPTOR:
		    if (FileDescriptors[i]->u.st_mode & S_IFDIR)
		      FileTable[itable].ftype = FILETYPE_DIR;
		    else
		      FileTable[itable].ftype = FILETYPE_OTHER;
		    fname = FileDescriptors[i]->u.name;
		    break;

		default:
		    FileTable[itable].ftype = FILETYPE_OTHER; /* hmm */
		    break;
	      } /* switch */
	    if (FileTable[itable].ftype == FILETYPE_OTHER) 
	      {			/* see if it's something special */
		if ((loc = (rindex(fname,'.') - fname)) > 0)
		  { /* loc is pointing to . in fname */
		    if (DebugFlag) fprintf(stderr,"-c-%d-",loc);
		    if (loc+2 == strlen(fname))
		      {
			if (fname[loc+1] == 'c')
			  FileTable[itable].ftype = FILETYPE_C;
			else if (fname[loc+1] == 'h')
			  FileTable[itable].ftype = FILETYPE_C;
			else if (fname[loc+1] == 'b')
			  FileTable[itable].ftype = FILETYPE_EXE;
			else if (fname[loc+1] == 'r')
			  FileTable[itable].ftype = FILETYPE_EXE;
		      }
		  }
	      }
	    /* since we may use the Generic icon for the Blank File Icon, */
	    /* for now, use FILETYPE_TEXT for those we don't know */
	    if (FileTable[itable].ftype == FILETYPE_OTHER)
		FileTable[itable].ftype = FILETYPE_TEXT;

	    /* end of deciding file type */

	    FileTable[itable].descnum = i; 
	    AddItem(sdf, item++, x, x+ICON_X, y, y+ICON_Y,
		    MemoryRaster, SDF_RASTER,
		    TypeIcon[FileTable[itable].ftype]);
	    AddItem(sdf, item++, x, x+90, y-20, y-10,
		    filefont, SDF_TEXT, fname);
	    if (DebugFlag) fprintf(stderr, " filename=%s\n\r",fname);
	  }
      }

    /* now add in the scroll boxes if necessary */
    if (StartFile)		/* Files before start file */
      AddItem(sdf, FILE_SCROLLUP, xmin-(SCROLLX-1), xmin, ymax-20-(SCROLLY-1),
		ymax-20,MemoryRaster, SDF_RASTER, i_up);
    if (limit < NumFiles)	/* files after last on screen */
      AddItem(sdf, FILE_SCROLLDOWN, xmin-(SCROLLX-1), xmin, ymin, 
		ymin+SCROLLY-1,MemoryRaster, SDF_RASTER, i_down);

    if (StatsFlag)
      fprintf(StatsFile,"%ld Added directory icons to screen.\n",GetTime(0));

  } /* AddDirectory */

/*
 ***************************************************************************
 *                             F i l e H i t				   *
 *-------------------------------------------------------------------------*
 * Take care of a file icon hit.  Right now, we need to:                   *
 *  - if a file has already been chosen, de-highlight it.		   *
 *  - highlight new file						   *
 *  - return filename and type to caller.				   *
 ***************************************************************************
 */

FileHit(sdf,mainSymbol,vgt,icon,cIcon,fname,ftype)
  short sdf, mainSymbol, icon, vgt;
  short *cIcon;			/* highlighted Icon on screen. */
  char **fname;
  FileType *ftype;
  {
    short x,y,xmax,ymax;
    char type, typedata;
    int iicon, i;		/* to index into FileTable */
    ArbitraryDescriptor *desc;

    iicon = icon;		/* convert short to integer */
    iicon = (iicon >> 1) << 1;  /* make filenames and icons equivalent */
    icon = iicon;		/* local copy for comparison later */

    fprintf(stderr,"FileHit, sdf=%d,mS=%d,vgt=%d,icon=%d\n\r",
		sdf,mainSymbol,vgt,icon);

    EditSymbol(sdf,mainSymbol);
    /*
     * ok, here we implement file hit symantics. 
     * If there was a highlighted icon, it is taken out of reverse video.
     * If the new icon matches the old icon, then say that there
     *   is now no current icon or file.
     * If the new icon does not match the old, then display the new
     *   icon in reverse video, and set the current icon, file, and 
     *   file type.
     */

    /* If old icon, display normally */

    if (*cIcon)
      {
	/* find index into file table (see current <> new icon below) */
	iicon = *cIcon;		/* convert short to integer */
	i = (iicon - FILE_LOWLIMIT) >> 1;	/* convert to index */

	/* note: InqItem will not return "string", even if given.. */
	InquireItem(sdf,*cIcon,&x,&xmax,&y,&ymax,0,&type,0);
/*    printf("Returned x=%d,xm=%d,y=%d,%ym=%d\n\r",x,xmax,y,ymax);*/
	/* Fudge Brownies:
	 * InqItem seems to return an x that is 1 less than what I 
	 * originally gave, and an xmax that is 1 more than org.
	 * likewise for y...
	 */
	x++; y++; xmax--; ymax--; /* compensate for error */

	DeleteItem(sdf,*cIcon);

	AddItem(sdf, *cIcon, x, xmax, y, ymax, MemoryRaster,
		type, TypeIcon[FileTable[i].ftype]);
      } /* if (*cIcon) */

    /* new match old? */
    if (*cIcon == icon)
      {
	*fname = "";		/* zap old name */
	*cIcon = 0;		/* and old Icon */
      }
    else /* current icon didn't match new one */
      {
	/* here, icons are an even number, and the file names are odd, */
	/* we want to map FILE_LOWLIMIT and F_L+1 to 0, FL+2 & FL+3 to 1 */
	/* and so forth... */
	iicon = icon;		/* convert new icon to integer */
	i = (iicon - FILE_LOWLIMIT) >> 1;  /* index so that F_L is 0 */

	*ftype = FileTable[i].ftype;

	InquireItem(sdf,icon,&x,&xmax,&y,&ymax,0,&type,0);
	x++; y++; xmax--; ymax--;
	DeleteItem(sdf,icon);
	AddItem(sdf,icon,x,xmax,y,ymax,MemoryRaster|InverseRaster,
		type, TypeIcon[*ftype]);

	desc = FileDescriptors[FileTable[i].descnum]; 	/* grab desc */
	switch (desc->e.descriptortype)	/* find file name */
	  {
	    case FILE_DESCRIPTOR:		/* V storage server */
	      *fname = desc->f.name;
	      break;

	    case UNIXFILE_DESCRIPTOR:		/* V unix server */
	      *fname = desc->u.name;
	      break;
	  } /* switch */
	*cIcon = icon;		/* remember new current icon */
      } /* current didn't match new icon */

    EndSymbol(sdf,mainSymbol,vgt);

  } /* FileHit */

/*
 ***************************************************************************
 *		    	     C l e a r I c o n                             *
 *-------------------------------------------------------------------------*
 *  Un-reverse-video's the given icon item.				   *
 ***************************************************************************
 */
ClearIcon(sdf,mainSymbol,vgt,icon)
  short sdf,mainSymbol,vgt,icon;
  {
    short x,xmax,y,ymax;
    char type, typedata;
    int iicon,i;

    iicon = icon;		/* convert short to integer */
    /* here, icons are an even number, and the file names are odd, */
    /* we want to map FILE_LOWLIMIT and F_L+1 to 0, FL+2 & FL+3 to 1 */
    /* and so forth... */
    i = (iicon - FILE_LOWLIMIT) >> 1;  /* index so that F_L is 0 */

    EditSymbol(sdf,mainSymbol);
    InquireItem(sdf,icon,&x,&xmax,&y,&ymax,&typedata,&type,0);
    /* Fudge warning: see FileHit above for description... */
    x += 1; y += 1; xmax -= 1; ymax -= 1;
    DeleteItem(sdf,icon);
    AddItem(sdf,icon,x,xmax,y,ymax,MemoryRaster,type,
	TypeIcon[FileTable[i].ftype]);
    EndSymbol(sdf,mainSymbol,vgt);
  } /* ClearIcon */

/*
 ***************************************************************************
 *                        C l e a r D i r e c t o r y			   *
 *-------------------------------------------------------------------------*
 *   Removes the directory icons from the screen.			   *
 ***************************************************************************
 */
ClearDirectory(sdf, mainSymbol, vgt)
    short sdf, mainSymbol, vgt;
  {
    short item;
    int i;

     for (i = 0, item = FILE_LOWLIMIT; i < NumOnScreen; i++)
      {
	DeleteItem(sdf,item++);
	DeleteItem(sdf,item++);
      }

    /* Clear off scroll boxes (if any) too! */
    DeleteItem(sdf,FILE_SCROLLUP);
    DeleteItem(sdf,FILE_SCROLLDOWN);

    /* don't forget context message */
    DeleteItem(sdf,FILE_CONTEXT);

   } /* ClearDirectory */

/*
 ***************************************************************************
 *                     R e f r e s h D i r e c t o r y			   *
 *-------------------------------------------------------------------------*
 *   Clean out the current directory and get a new copy... useful when     *
 *   a command may have changed some files.				   *
 ***************************************************************************
 */
RefreshDirectory(sdf,mainSymbol,vgt,xmin,xmax,ymin,ymax, noInvert)
    short sdf, mainSymbol, vgt;
    int xmin, xmax, ymin, ymax;
  {
    /* Add nice reassuring prompt, because directory updates take no  */
    /* more than 20 seconds. */
    EditSymbol(sdf,mainSymbol);
    AddItem(sdf,FILE_WAITING, xmax-ICON_X-2, xmax-2, ymin-ICON_Y-4, ymin-4,
		MemoryRaster, SDF_RASTER, i_watch);
    EndSymbol(sdf,mainSymbol,vgt); 

    EditSymbol(sdf,mainSymbol);
    ClearDirectory(sdf,mainSymbol,vgt);
    FreeDescriptor(FileDescriptors, NumFiles);
    AddDirectory(sdf, xmin, xmax, ymin, ymax, noInvert, 0 , -1);
    DeleteItem(sdf,FILE_WAITING);	/* delete reassurance */
    EndSymbol(sdf,mainSymbol,vgt);	/* redraw screen */
  } /* RefreshDirectory */

/*
 ***************************************************************************
 *                        S c r o l l H i t				   *
 *-------------------------------------------------------------------------*
 *   Deal with a scroll box hit...					   *
 ***************************************************************************
 */
ScrollHit(sdf, mainSymbol, vgt, xmin, xmax, ymin, ymax, noInvert, scroll)
    short sdf, mainSymbol, vgt, scroll;
    int xmin, xmax, ymin, ymax;
  {
    int start;

    /* Add nice reassuring prompt, because screen updates take up to */
    /* 4 seconds (much too long) */
    EditSymbol(sdf,mainSymbol);
    AddItem(sdf,FILE_WAITING, xmax-ICON_X-2, xmax-2, ymin-ICON_Y-4, ymin-4,
		MemoryRaster, SDF_RASTER, i_watch);
    EndSymbol(sdf,mainSymbol,vgt); 


    EditSymbol(sdf,mainSymbol);

    ClearDirectory(sdf,mainSymbol,vgt);

    switch (scroll)
      {
	case FILE_SCROLLUP:
	  start = (StartFile < MaxOnScreen) ? 0 : (StartFile - MaxOnScreen);
	  break;

	case FILE_SCROLLDOWN:
	  start = ((StartFile + (2 * MaxOnScreen)) > NumFiles) ?
			(NumFiles - MaxOnScreen) : (StartFile + MaxOnScreen);
	  break;

	default:
	  fprintf(stderr,"ScrollHit: bogus scroll code: %d\n\r",scroll);
	  start = 0;
	  break;
      }  /* switch */

    AddDirectory(sdf, xmin, xmax, ymin, ymax, noInvert, 1, start);
    DeleteItem(sdf,FILE_WAITING);	/* delete reassurance */
    EndSymbol(sdf,mainSymbol,vgt);
  } /* ScrollHit */
