/*
 *  objects.c
 *
 *  Objects are the highest level data elements in Draw. Object point to parts.
 *  More than one object can point to the same part. Each object carries an
 *  independent x/y offset. The Checkpoint/Undo mechanism works by copying
 *  the current main object list (The ActiveList) and making multiple 
 *  references to the same parts that do not change between checkpoints.
 *  When an object DOES change, FreshCopy is called to insure that there are
 *  no references to the part that would invalidate the undo and that it is
 *  safe for one of the part manipulation routines to modify.
 */
 
 
/* Includes */
#ifdef UNIX
#include "stdio.h"
#else
#include "Vio.h"
#endif
#include "Vgts.h"
#include "splines.h"
#include "draw.h"
 
 
/* Imports */
extern DebugItemDescriptor();
extern DebugLinks();
extern SetCurrentObject();
extern double sin();
extern double cos();
extern short GetINum();		/* Get an item number */
extern short GetSeqINum();	/* Get a sequential set of item numbers */
extern short LastINum();	/* Return the last item number gotten
extern short FreeINum();	/* Return an item number to the freelist */
extern PART *CreatePart();	/* Create a new part given all parameters */
extern PART *CopyPart();	/* Duplicate a part */
extern char *malloc();
 
 
/* Exports */
extern RedrawActivelist();	/* Redisplay the main drawing area. */
extern OBJECT *NewObject();	/* Add a new object to the display list */
extern DecrementRef();		/* Decrement reference count on an item */
extern RotateObject();		/* Rotate an object about a point. */
extern ScaleObject();		/* Expand/Contract an object about a point. */
extern CopyObject();		/* Make a copy of an object. */
extern MoveObject();		/* Move an object. */
extern EraseObject();		/* Erase (delete) an object. */
extern LowerObject();		/* Lower an object. */
extern RaiseObject();		/* Raise an object. */
extern AttribObject();		/* set nib, fill, and opaque attributes. */

/*
 *  Add a new object to an object list, given a pre-made part.
 *  The defer flag is used when building groups. It will not call the
 *  VGTS. This is done when CreatePart is called with the object list
 */
 
OBJECT *AddObject( id, deltax, deltay, refresh, defer, oblist )
	PART *id;
	OBJECT_HEADER *oblist;		/* The object list to put this in. */
	BOOLEAN refresh,defer;
	short deltax, deltay;
  {
    OBJECT *ob;
    
    /* Append to object list */
    if ((ob = (OBJECT *) malloc( sizeof(OBJECT) )) == NULL)
      {
	printf("Out of memory.  Couldn't allocate object.\n\r");
	SetCurrentObject(NULL,FALSE,FALSE,refresh);
	DecrementRef(id);
	return(NULL);
      }
    id->refs += 1;
    ob->prev = oblist->last;
    ob->part = id;
    ob->next = NULL;
    ob->parent = oblist;
    ob->frame = 0;
    ob->dx = deltax;
    ob->dy = deltay;
    if (oblist->first == NULL)
	oblist->first = ob;
    if (oblist->last)
	(oblist->last)->next = ob;
    oblist->last = ob;
    if (!defer)
        DisplayObjectSymbol( ob, GetINum(), oblist->symbol, refresh );
    if (Debug&DebugObjects)
      {
	printf("AddObject:  object = %x; prev = %x; next = %x; parent = %x\n\r"
			    ,ob,ob->prev,ob->next,ob->parent);
        printf("            part = %x; call = %d; dx = %d; dy=%d\n\r"
			    ,ob->part,ob->call,ob->dx,ob->dy);
	DebugLinks( oblist );
      }
    return(ob);
  }

/*
 *  Create a new object/part pair and make it the current object. 
 */
 
OBJECT *NewObject( dataptr, typeP, subtypeP, td, x1, x2, y1, y2, b,
	    refresh, deltax, deltay, oblist )
	char *dataptr;			/* New itemdesc->data entry. */
	enum ObjTypes typeP;		/* itemdesc->type entry. */
	short subtypeP, td;		/* subtype and typedata fields. */
	short x1, x2, y1, y2, b;	/* Bounding box and baseline */
	OBJECT_HEADER *oblist;		/* The object list to put this in. */
	BOOLEAN refresh;	/* refresh mode */
	short deltax, deltay;		/* display offset */
  {
    OBJECT *ob;
    PART *id;
    
    if (Debug&DebugObjects)
      {
	printf("NewObject:");
      }
    id = CreatePart( dataptr, typeP, subtypeP, td, x1, x2, y1, y2, b);
    if (id == NULL)
	return(NULL);
    ob = AddObject( id, deltax, deltay, refresh, FALSE, oblist );
    if (ob == NULL)
	DecrementRef(id);
    
    if (typeP != GroupObj) {
       SetCurrentObject(ob,FALSE, FALSE,refresh);
    }
    else 
	printf("NewObject was used to create a group. (obsolete)\n\r");
    return(ob);
  }

/*
 *  FreshCopy makes sure that the part of an object has but one reference,
 *  that it is thus safe to modify. If independent is FALSE then multiple
 *  references to the part within the current object list are allowed.
 *  FreshCopy returns a count of the number of parts which need to be
 *  switched. The OldID parameter is the old part which is passed to
 *  BringOver, which finishes the job that FreshCopy started. The idea
 *  is that whenever a part needs to be modified, call FreshCopy FIRST,
 *  modify the part, and then call BringOver to clean things up. BringOver
 *  can be called immediately after FreshCopy if no part manipulation
 *  is being made.
 *  
 *  The goose parameter is used only if the part is a group. If one of the 
 *  objects in the group equals goose, then the value of goose will be updated
 *  to reflect the new copy. Same for goose2.
 */
short FreshCopy(object,independent,oldid,goose,goose2)
    OBJECT *object;
    BOOLEAN independent;
    PART **oldid;
    OBJECT **goose, **goose2;

{
    OBJECT_HEADER *oblist = object->parent;
    OBJECT *ob;
    PART *id, *newid;
    BOOLEAN okflag;
    short count;

    if (Debug&DebugObjects)
	printf("FreshCopy: object = %x; oblist = %x; independent=%s\n\r",
	       object,oblist,independent?"TRUE":"FALSE");
    id = object->part;
    *oldid = id;
    /* First make sure that the copy is actually necessary */
    if (independent) {
	count = 1;
    }
    else {
    	/* we need to chase down the object list and find how many local 
	   references there are. Other references may have come from the
	   backup lists (UNDO) or from other groups. */
	count = 0;
	okflag = FALSE;
	for (ob = oblist->first;ob;ob=ob->next) {
	    if (ob == object) okflag = TRUE;
	    if (ob->part==object->part) count++;
	}
	if (!okflag) {
	    printf("Internal error: FreshCopy; object not in object list\n\r");
	    return(0);
	}
    }
    if (Debug&DebugObjects) {
	printf("FreshCopy: object refs = %d; Allowable count = %d\n\r",
	    object->part->refs, count);
    }
    if (object->part->refs < count) {
	printf("internal error: Freshcopy; invalid part reference count\n\r");
	return(0);
    }
    if (object->part->refs == count) 
	return(count);
    /* At this point, we know we have to make the copy */

    if (Debug&DebugObjects) {
	printf("FreshCopy: starting copy\n\r");
    }
    newid = CopyPart(id,goose,goose2);

    /* Now switch over to the new copy */

    DecrementRef(id);
    object->part = newid;
    newid->refs++;
    return(count);
}

/*
 * This routine tells the VGTS about the new part and also changes any 
 * dependent copies. See FreshCopy for more details.
 */
BringOver(object,count,refresh,oldid)
    OBJECT *object;
    short count;
    BOOLEAN refresh;
    PART *oldid;
{
    OBJECT *ob;
    PART *newid = object->part;

    if (Debug&DebugObjects)
	printf("BringOver: object=%x; count=%x; refresh=%s; oldid=%x\n\r",
	       object,count,refresh?"TRUE":"FALSE",oldid);
    ReplaceObjectSymbol(object,refresh);
    if (count <= 1) return;
    for (ob=object->parent->first;ob;ob=ob->next) 
	if (ob->part==oldid) {
	    DecrementRef(oldid);
	    ob->part = newid;
	    newid->refs++;
	    ReplaceObjectSymbol(ob,refresh);
	}
  }
	
/*
 *  This routine will delete an object from an object list 
 */
 
EraseObject( victim, refresh)
	OBJECT *victim;
	BOOLEAN refresh;
  {
    
    OBJECT_HEADER *oblist = victim->parent;

    /* Print Debugging Information */
    if (Debug&DebugObjects)
      {
	printf("Erase Object %x\n\r",victim);
      }
    
    /* Update the screen */
    OpenSymbol(oblist->symbol);
    DeleteItem( sdf, victim->call );
    CloseSymbol(refresh);
    FreeINum(victim->call);

    /* Remove any dangling pointers. */
    SetCurrentObject(NULL,TRUE,FALSE,refresh);
    
    /* Update the activelist */
    DecrementRef( victim->part );
    if (victim->prev)
	victim->prev->next = victim->next;
    if (victim->next)
	victim->next->prev = victim->prev;
    if (oblist->first == victim)
	oblist->first = victim->next;
    if (oblist->last == victim)
	oblist->last = victim->prev;
    /* The next three lines are crowbars */
    victim->part = (PART *) -3;
    victim->prev = (OBJECT *) -5;
    victim->next = (OBJECT *) -7;
    victim->parent = (OBJECT_HEADER *) -23;
    free( victim );
  }

/*
 *  This routine will change the display offset of an object in the
 *  object list
 */
 
MoveObject( ob, deltax, deltay, refresh )
	OBJECT *ob;
	short deltax, deltay;
	BOOLEAN refresh;
  {
    OBJECT_HEADER *oblist = ob->parent;
    BOOLEAN SaveFake;

    /* Don't die because of bad pointers. */
    if (ob == NULL)
      {
	printf("Internal Error:  Move NULL object.\n\r");
	return;
      }
    
    /* Debugging information requested? */
    if (Debug&DebugObjects)
      {
	printf("Move Object %x by (%d, %d)\n\r", ob, deltax, deltay );
      }
    
    /* Do the actual work */
    ob->dx += deltax;
    ob->dy += deltay;

    SaveFake = FakeGroup;
    SetCurrentObject(NULL,TRUE,FALSE,refresh);
    ReplaceObjectSymbol( ob, refresh);
    SetCurrentObject(ob,FALSE,SaveFake,refresh);
  }

/*
 *  This primitive will raise an object to the top of the display.
 *  This operation is useful when opaque filling is used.
 */
 
RaiseObject( ob, refresh )
	OBJECT *ob;
	BOOLEAN refresh;
  {
    OBJECT_HEADER *oblist = ob->parent;

    if (Debug&DebugObjects)
      {
	printf("Raise object %x\n\r",ob);
      }
    
    if (oblist->last == NULL)
      {
	printf("Internal Error:  Null last cell in object list.\n\r");
	return;
      }
      
    /* If it's already there, don't bother playing with the links. */

    if (oblist->last == ob)
	return;
	
    /* Update the links in the object list */
    if (ob->prev)
        ob->prev->next = ob->next;
    else
	oblist->first = ob->next;
    if (ob->next)
	ob->next->prev = ob->prev;
    else
	printf("Internal Error:  Raise already raised object.\n\r");
    oblist->last->next = ob;
    ob->prev = oblist->last;
    ob->next = NULL;
    oblist->last = ob;
    ReplaceAndRaiseObjectSymbol(ob, refresh);
  }

/*
 *  This primitive will lower an object to the bottom of the display list.
 *  This operation is useful when opaque filling is used.
 */
 
LowerObject( ob, refresh)
	OBJECT *ob;
	BOOLEAN refresh;

  {
    OBJECT_HEADER *oblist = ob->parent;
    BOOLEAN SaveFake;

    if (Debug&DebugObjects)
      {
	printf("Lower object %x\n\r",ob);
      }
    
    if (oblist->first == NULL)
      {
	printf("Internal Error:  Null start cell in object list.\n\r");
	return;
      }
      
    /* If it's already there, don't bother playing with the links. */
    if (oblist->first == ob)
	return;
	
    /* Update the links in the activelist */
    if (ob->prev)
        ob->prev->next = ob->next;
    else
	printf("Internal Error:  Lower already lowered object.\n\r");
    if (ob->next)
	ob->next->prev = ob->prev;
    else
	oblist->last = ob->prev;
    oblist->first->prev = ob;
    ob->next = oblist->first;
    ob->prev = NULL;
    oblist->first = ob;
    SaveFake = FakeGroup;
    SetCurrentObject(NULL,TRUE,FALSE,refresh);
    NewActiveList();
    SetCurrentObject(ob,FALSE,SaveFake,refresh);
  }

/*
 *  This routine will copy an object. All copies are shallow. If an object is
 *  later manipulated, then a fresh copy will be made. 
 */
CopyObject( ob, deltax, deltay, refresh )
	OBJECT *ob;
	short deltax, deltay;
	BOOLEAN refresh;
  {
    OBJECT_HEADER *oblist = ob->parent;
    OBJECT *tmp;

    /* Don't die because of bad pointers. */
    if (ob == NULL)
      {
	printf("Internal Error:  Copy NULL object.\n\r");
	return;
      }
    
    /* Debugging information requested? */
    if (Debug&DebugObjects)
      {
	printf("Copy Object %x by (%d, %d)\n\r", ob, deltax, deltay );
      }
    
   
    /* Do the actual work */
    tmp = AddObject(ob->part,ob->dx+deltax,ob->dy+deltay,refresh,FALSE,oblist);
    SetCurrentObject(tmp,FALSE,FakeGroup,refresh);
  }
 

/*
 *  This internal routine is a helper routine for RotateObject().
 *  It will rotate a single point about (x0, y0), given the sine and
 *  cosine of the angle.
 */
 
RotatePoint( x, y, sine, cosine, x0, y0 )
	short *x, *y;
	double sine, cosine;
	short x0, y0;
  {
    register short newx, newy;
    
    if (Debug&DebugObjects)
	printf("    (%d, %d) rotates to ", *x, *y );
    
    /* Compute the distance of the point from the center of rotation. */
    newx = x0 + (cosine * (*x - x0) + sine * (*y - y0));
    newy = y0 + (cosine * (*y - y0) - sine * (*x - x0));
    *x = newx;
    *y = newy;
    
    if (Debug&DebugObjects)
	printf(" (%d, %d)\n\r", *x, *y );
  }

/* 
 *  This internal routine is a helper routine for ScaleObject().
 *  It will scale a single point about (x0, y0), given the scale factor.
 */
 
ScalePoint( x, y, factorx, factory, x0, y0 )
	short *x, *y;
	double factorx, factory;
	short x0, y0;
  {
    register short newx, newy;
    
    if (Debug&DebugObjects)
	printf("    (%d, %d) scales to ", *x, *y );
    
    /* Scale the point. */
    newx = ((*x - x0) * factorx) + x0;
    newy = ((*y - y0) * factory) + y0;
    *x = newx;
    *y = newy;
    
    if (Debug&DebugObjects)
	printf(" (%d, %d)\n\r", *x, *y );
  }

/*
 *  This routine will transform an object.  Currently, the two
 *  transformation are rotation about a point, and scaling.
 */
 
TransformObject( ob, fact1, fact2, x0, y0, ptrans,independent )
	OBJECT *ob;
	double fact1, fact2;	/* Point Transform factors. */
	short x0, y0;		/* Center of transformation. */
	register (*ptrans)();	/* Point transformation routine. */
	BOOLEAN independent;	/* TRUE if this object only */
  {
    PART *id,*oldid;
    short count;
    BOOLEAN SaveFake;

    if (Debug&DebugObjects) 
	printf("TransformObject %x: fact1=%f; fact2=%f; x0=%d; y0=%d; ptrans=%x; independent=%s\n\r",ob,fact1,fact2,x0,y0,ptrans,independent?"TRUE":"FALSE");
    DupCheck++;
    count = FreshCopy(ob,independent,&oldid,NULL,NULL);
    SaveFake = FakeGroup;
    FakeGroup = FALSE;
    SetCurrentObject(NULL,TRUE,FALSE,TRUE);
    id = ob->part;
    TransformPart(id, fact1, fact2, x0 - ob->dx, y0 - ob->dy, ptrans,SaveFake);
    BringOver(ob,count,TRUE,oldid);
    SetCurrentObject(ob,FALSE,SaveFake,TRUE); /* re-compute highlight box */
  }

/*
 *  This routine will rotate an object.
 */
 
RotateObject( p, theta, x0, y0, independent )
	OBJECT *p;
	double theta;		/* Angle of rotation. */
	short x0, y0;		/* Center of rotation. */
	BOOLEAN independent;
  {
    double sine, cosine;
    
    if (Debug&DebugObjects) 
	printf("RotateObject %x: theta=%f; x0=%d; y0=%d; independent=%s\n\r",
	    p,theta,x0,y0,independent?"TRUE":"FALSE");
    /* Check parameters. */
    if (p == NULL)
	return;
    
    if (Debug&DebugObjects)
      {
	printf("Rotate object %d through %f about (%d, %d)\n\r",
		p, theta, x0, y0);
      }
    
    /* Transform the object. */
    sine = sin( theta );
    cosine = cos( theta );
    TransformObject( p, sine, cosine, x0, y0, RotatePoint, independent );
  }

/*
 *  This routine will scale and raise an object.  To do this, it invokes
 *  TransformObject(), which will transform an object based on the
 *  point transformation routine passed to it.
 */
 
ScaleObject( p, factorx, factory, x0, y0, independent )
	OBJECT *p;
	double factorx,factory;	/* Scale factor */
	short x0, y0;		/* Center of transformation */
	BOOLEAN independent;
  {
    if (Debug&DebugObjects) 
	printf("ScaleObject %x: factorx=%f; factory=%f; x0=%d; y0=%d; ptrans=%x; independent=%s\n\r",
	    p,factorx,factory,x0,y0,independent?"TRUE":"FALSE");
    /* Check parameters. */
    if (p == NULL)
	return;
    
    if (Debug&DebugObjects)
      {
	printf("Scale object %d by %f,%f about (%d, %d)\n\r",
		p, factorx,factory, x0, y0);
      }
    
    /* Transform the object. */
    TransformObject( p, factorx, factory, x0, y0,ScalePoint, independent );
  }


/*
 *  This internal routine will change the filling, nib, and opaque attributes
 *  of spline objects and the font and centering attributes of text objects.
 *
 *   togopaque - 0 = don't toggle, 1 = toggle opaque bit
 *   nib - 0-15 = nib type, -1 = don't change
 *   pat - -1 = don't change, 0 = clear, 1 = white, 2 = black, etc.
 *   newfont - -1 = don't change, 0,1,.... font number
 *   newcenter - -1 = don't change, 0,1,2 = set centering.
 *
 *  (Actually, most of the work is done by AttributePart in parts.c)
 */
 
AttribObject( ob, togopaque, nib, pat, newfont, newcenter, independent)
	OBJECT *ob;
	short togopaque;
	enum Nib nib;
	enum Pattern pat;
	short newfont,newcenter;
	BOOLEAN independent;
  {
    PART *id,*oldid;
    short count;
    BOOLEAN SaveFake;

    if (Debug&DebugObjects) 
	printf("AttribObject %x: independent=%s\n\r",
	    ob,independent?"TRUE":"FALSE");
    SaveFake = FakeGroup;
    FakeGroup = FALSE;
    SetCurrentObject(NULL,TRUE,FALSE,TRUE);
    DupCheck++;
    count = FreshCopy(ob,independent,&oldid,NULL,NULL);
    id = ob->part;
    AttribPart( id, togopaque, nib, pat, newfont, newcenter);
    BringOver(ob,count,TRUE,oldid);
    SetCurrentObject(ob,FALSE,SaveFake,TRUE); /* re-compute highlight box for text*/
 }

/*
 *  This routine will wipe the drawing area and redisplay the
 *  activelist from scratch.
 */
NewActiveList()
  {
    register OBJECT *ob;
    register short symbol = activelist->symbol;

    if (Debug&DebugObjects)
	printf("NewActiveList\n\r");
    /* Delete the old drawing area. */
    CloseAllSymbols();
    DeleteSymbol( sdf, symbol );
    
    /* Create the new one. */
    InitFrame(FALSE);
    
    /* Update the Vgts */
    for (ob = activelist->first;  ob;  ob = ob->next)
      {
	DisplayObjectSymbol( ob, ob->call, activelist->symbol,FALSE );
      }
    CloseSymbol(FALSE);
    /* Update the screen */
    MajorChange();
  }

