/*
 *  load.c
 *
 *  This file contains the code the deal with loading a file into
 *  memory. Alto draw, SIL, and native files are supported.
 */
 
 
/* Includes */
# ifdef UNIX
# include "stdio.h"
# else
# include "Vio.h"
# endif
# include "Vgts.h"
# include "splines.h"
# include "draw.h"
 
 
/* Imports */
extern Quit();
extern DebugObjectList();
extern char *GetString();
extern OBJECT *NewObject();
extern OBJECT *CreateGroup();
extern Checkpoint();
extern RevertToCKP();
extern PART *CreateGroupPart();
 
 
/* Exports */
extern ReadFile();
 
 
/* Local Definitions */
# define tolower(c)	(((c >= 'A') && (c <= 'Z')) ? (c - 'A' + 'a') : c)
# define SILHEADER	034562	/* SIL format header (password) */

#define PART_CACHE_SIZE	100

static struct PART_CACHE {
    short c_label;
    PART  *c_part;
} partcache[PART_CACHE_SIZE];

#define hash(x) (x%PART_CACHE_SIZE)

static short LabelCount;


/*
 *  This routine flushes the part cache and sets all the filnum values in parts
 *  back to 0;
 */
FlushPartCache()
  {
    short i;
    PART *id;
    if (Debug&DebugIO) printf("FlushPartCache\n\r");
    for (i=0;i<PART_CACHE_SIZE;i++) {
	partcache[i].c_label = 0;
	partcache[i].c_part = NULL;
    }
    id = FirstPart;
    while (id) {
	id->filnum = 0;
	id= id->next;
    }
    LabelCount = 0;
    DupCheck = 1;
  }

/*
 *  This routine finds part with a given label. Returns TRUE if found.
 */
BOOLEAN FindLabeledPart(label,part)
    short label;
    PART **part;
  {
    struct PART_CACHE *h = &partcache[hash(label)];
    PART *id;

    if (Debug&DebugIO) printf("FindLabeledPart (%d)...",label);
    if (label <= LabelCount && h->c_label != 0) {
        if (h->c_label == label) {
	    *part = h->c_part;
            if (Debug&DebugIO) printf("found part %x (cache hit)\n\r",*part);
	    return(TRUE);
	}
	else {
	    /* We missed on the cache. Find it by traversing the part list. */
	    id = FirstPart;
	    while (id) {
		if (id->filnum == label) {
		    h->c_label = label;
		    h->c_part  = id;
		    *part = id;
                    if (Debug&DebugIO) printf("found part %x\n\r (no hit)",*part);
		    return(TRUE);
		}
		id = id->next;
	    }
            if (Debug&DebugIO) printf("not found. (not in list)");
	    return(FALSE);
	}
    }
    else
        if (Debug&DebugIO) printf("not found. (out of range)\n\r");
	return(FALSE);
  }

/*
 *  Put a part in the part cache
 */

PutLabeledPart(label,part)
    short label;
    PART *part;

{
    struct PART_CACHE *h = &partcache[hash(label)];
    if (Debug&DebugIO) printf("PutLabeledPart: label = %d; part = %x\n\r",
	label,part);
    h->c_label = label;
    h->c_part  = part;
    if (label > LabelCount) 
	LabelCount = label;   
}

/*
 *  This internal routine is a helper process for ReadFile().
 */
 
ScanRead( dx, dy, label, fp, oblist )
	short dx, dy, label;
	FILE *fp;
	OBJECT_HEADER *oblist;
  {
    enum ObjTypes type;
    short subtype, typedata, xmin, xmax, ymin, ymax, base;
    char *data;
    unsigned short order, numvert, border, closed, filled, opaque;
    enum Nib nib;
    enum Pattern pat;
    short i,len;
    short cnt, newdx, newdy;
    SPLINE *sptr;
    POINT *pptr;
    OBJECT *ob;
    OBJECT_HEADER *glist;
    PART *part;
    
    /* First see if we already read in the part. If so, don't expect a full
	description */
    if (FindLabeledPart(label,&part)) {
	AddObject(part,dx,dy,Debug&DebugIO,FALSE,oblist);
	return;
    }
    /* Read in the basic data. */
    fscanf( fp, "%d %hd %hd %hd %hd %hd %hd %hd ",
	&type, &subtype, &typedata,
	&xmin, &xmax, &ymin, &ymax, &base );
    switch (type)
      {
	case TextObj:
	    /* Text.  Read the string. */
	    if ((data = (char *) malloc( MAXLEN+1 )) == NULL)
	      {
		printf("\n\rOut of memory.  Can't read a text string.\n\r");
		return;
	      }
	    fscanf( fp, "%hd ", &len);
	    for (i = 0; i < len; i++)
	        data[i] = getc( fp );
	    data[i]=0;
	    ob = NewObject( data, type, subtype, typedata, xmin, xmax,
		    ymin, ymax, base, Debug&DebugIO, dx, dy, oblist);
	    PutLabeledPart(LabelCount+1,ob->part);
	    break;

	case GroupObj:
	    while (getc( fp ) != '\n'); /* ignore old group name */
	    
            if ((part = CreateGroupPart())==NULL)
		return;	/* CreateGroupPart gave out of mem warning */
	    glist = (OBJECT_HEADER *)part->data;
	    PutLabeledPart(LabelCount+1,part);
	    fscanf( fp, "%hd %hd %hd ", &newdx, &newdy, &cnt );
	    while( cnt != -1 )
	      {
		ScanRead( newdx, newdy, cnt, fp, glist );
		fscanf( fp, "%hd %hd %hd ", &newdx, &newdy, &cnt );
	      }
	    CloseAllSymbols();
    	    InquireItem( sdf, part->symbol, &(part->xmin), &(part->xmax),
		&(part->ymin), &(part->ymax), 0, 0, 0 );
	    AddObject(part,dx,dy,FALSE,FALSE,oblist);
	    break;
	
	default:
	    /* Splines.  Read the data points. */
	    fscanf( fp, "%hd %hd %d %hd %hd %hd %hd %d",
		&order, &numvert, &nib, &border,
		&closed, &filled, &opaque, &pat );
	    if ((sptr = (SPLINE *) malloc( sizeof(SPLINE) +
		(numvert - 1) * sizeof(POINT) )) == NULL)
	      {
		printf("\n\rOut of memory.  Can't read a spline.\n\r");
		return;
	      }
	    sptr->order = order;
	    sptr->numvert = numvert;
	    sptr->nib = nib;
	    sptr->border = border;
	    sptr->closed = closed;
	    sptr->filled = filled;
	    sptr->opaque = opaque;
	    sptr->pat = pat;
	    pptr = &(sptr->head);
	    for (i = 0; i < numvert; i++)
		fscanf( fp, "    %hd %hd ", &(pptr[i].x), &(pptr[i].y) );
	    ob =NewObject( sptr, type, subtype, typedata, xmin, xmax,
		ymin, ymax, base, Debug&DebugIO, dx, dy, oblist);
	    PutLabeledPart(LabelCount+1,ob->part);
	    break;
      }
    
  }

/*
 *  This internal routine will read a file in the internal V format
 */

drawread(fp,mx,my)
    FILE *fp;
    short mx,my;
  {
    short count;
    short dx, dy;
    
	FlushPartCache();
	MajorChange();
        /* Read and load all of the referenced items in SUN format*/
	fscanf( fp, "%hd %hd %hd ", &dx, &dy, &count );
	while( count != -1 )
	  {
	    dx += mx;
	    dy += my;
	    ScanRead( dx, dy, count, fp, activelist );
	    fscanf( fp, "%hd %hd %hd ", &dx, &dy, &count );
	  }
	DupCheck = LabelCount + 1;
	CurrentObject = NULL;
	FakeGroup = FALSE;
    }

/*----------------------------------SIL READ-------------------------------*/

/*
 * This routine will read in a SIL format file 
 */

silread(fp)
    FILE *fp;

  {
    SPLINE *sptr;
    POINT *pptr;
    char *data;
    int xmin,xmax,ymin,ymax;
    unsigned short aword;
    short font;
    unsigned short getword();
    int getfloat();
    short descent;
    short link;

    if ((aword=getword(fp)) != SILHEADER) { 
	printf("\r\nERROR:It looked like a SIL file but it wasn't %o\n\r",aword);
	RevertToCKP();
        return;
    }
    MajorChange();
    while (!feof(fp)) {
        for (link=getword(fp);!feof(fp) && link!=-1 && (link<=32 || link>=127);
	link = getword(fp))
	    printf("Skipping SIL junk - %04X\n\r",link);
	if (feof(fp)) break;
	xmin=getword(fp)&1023;
	ymax=getword(fp)&1023;
	xmax=getword(fp)&1023;
	ymin=getword(fp);
	font=ymin>>12;
	ymin &= 1023;
	ymax=800-ymax;
	ymin=801-ymin;
	xmax--;
	if (Debug&DebugIO)
	    printf("link = %04X; font = %d; bbox= %d - %d,  %d - %d",
		link,font,xmin,xmax,ymin,ymax);
	if (font>=14) {
	    if (xmin<=xmax && ymin<= ymax) {
	    if ((sptr=(SPLINE*)malloc(sizeof(SPLINE)+4*sizeof(POINT))) == NULL)
		  {
		    printf("\n\rOut of memory.  Can't read a spline.\n\r");
		    return;
		  }
	    sptr->order = 2;
	    sptr->numvert = 4;
	    sptr->nib = (enum Nib) 0;
	    sptr->border = 1;
	    sptr->closed = 1;
	    sptr->filled = 1;
	    sptr->opaque = 1;
	    sptr->pat = PatBlack;
	    pptr = &(sptr->head);
	    pptr[0].x=xmin; pptr[0].y=ymin;
	    pptr[1].x=xmax; pptr[1].y=ymin;
	    pptr[2].x=xmax; pptr[2].y=ymax;
	    pptr[3].x=xmin; pptr[3].y=ymax;
		NewObject( sptr,SILObj,0,0,0,0,0,0,0,Debug&DebugIO,0,0,activelist);
	    } }
	else { /*text*/
	    if ((data = (char *) malloc( MAXLEN+1 )) == NULL)
	      {
		printf("\n\rOut of memory.  Can't read a text string.\n\r");
		return;
	      }
	    getstring(fp,0,data);
	    if (Debug&DebugIO)
		printf("; text = (%s)",data);
	    switch(font) {
		case 0: /*Helvetica10*/
		    font=2;
		    descent=Descenders(data);
		    break;
		case 1: /*Helvetica10B*/
		    font=3;
		    descent=Descenders(data);
		    break;
		case 2: /*Helvetica7*/
		    font=5;
		    descent=Descenders(data);
		    break;
		case 3: /*Helvetica7b*/
		    font=6;
		    descent=Descenders(data);
		    break;
		case 4:
		case 5:
		case 6:
		case 7:
		case 12:
		case 13: /*Template64*/
		    font=24;
		    descent=ymax-ymin-1;
		    break;
		case 8: /*Helvetica10I*/
		case 9: /*Helvetica10BI*/
		    font= 8;
		    descent=Descenders(data);
		    printf("Helvetica 10 italic fonts not supported.\n\r");
		    break;
		case 10: /*Helvetica7I*/
		    font=8;
		    descent=Descenders(data);
		    break;
		case 11: /*Helvetica7BI*/
		    font=7;
		    descent=Descenders(data);
		    break;	
		default:
		    printf("Invalid SIL font number :%d\n\r",font);
		    font= 7;
		    descent=Descenders(data);
		    break;
	    }
	    if (xmin <= xmax && ymin <= ymax)
	    NewObject( data, TextObj, font, 2, xmin,xmax,ymin+descent,ymax,
		descent,Debug&DebugIO,0,0,activelist);
	    else
		free(data);

	}
	if (Debug&DebugIO)
	    printf("\n\r");
    }
	CurrentObject = NULL;
	FakeGroup = FALSE;
}	

/*
 * This is SILEDIT's descent kluge algorithm
 */
Descenders(string)
  char *string;
 {
 	/*
	 * return true if there are any characters with descenders
	 * in the string.
	 */
    return( (index(string,'g') ||
    	    index(string,'j') ||
    	    index(string,'p') ||
    	    index(string,'q') ||
    	    index(string,'y') ||
    	    index(string,']') ||
    	    index(string,'}') ||
    	    index(string,'[') ||
    	    index(string,'{')) ? 4 : 0 
	  );
 }
 


/*---------------------------------ALTO READ-------------------------------*/
/*
 * This routine will read in an Alto format file 
 */

#define ALTO_BOTTOM_MARGIN 160
#define ALTO_LEFT_MARGIN 80

altoread(fp)
    FILE *fp;

  {
    SPLINE *sptr;
    POINT *pptr;
    char *data;
    int dx,dy;
    unsigned short aword;
    short nsplines,nstrings,font;
    unsigned short getword();
    int getfloat();
    int i,j;
    short shape,thickness,numvert,descent;
    enum Nib nib;

    if ((i=(getword(fp)<<16)+getword(fp))!=0xffffffff) { 
	printf("ERROR:It looked like an Alto Draw file but it wasn't\n\r");
	RevertToCKP();
        return;
    }
    MajorChange();
    nsplines=getword(fp);
    for (j=1;j<=nsplines;j++) {
	aword=getword(fp);
	if ((aword&0x1000)!=0)
	    printf("Can't do Alto dashed lines.\n\r");
	shape=(aword&0x0c00)>>10;
        if (shape<2) shape ^= 1;
	thickness=(aword&0x0300)>>8;
	nib=(enum Nib)(thickness+(shape<<2));
	if ((aword&0x0007)!=7)
	     printf("Colors are being ignored.\n\r");
	aword=getword(fp);
	numvert=aword&0x7fff;
	if ((sptr = (SPLINE *) malloc( sizeof(SPLINE) +
	    (numvert - 1) * sizeof(POINT) )) == NULL)
	      {
		printf("\n\rOut of memory.  Can't read a spline.\n\r");
		return;
	      }
	sptr->order = 3;
	if (numvert < 3)
	    sptr->order = numvert;
	sptr->numvert = numvert;
	sptr->nib = nib;
	sptr->border = 1;
	sptr->closed = (aword&0x8000!=0);
	sptr->filled = 0;
	sptr->opaque = 0;
	sptr->pat = PatWhite;
	pptr = &(sptr->head);
	    for (i = 0; i < numvert; i++)
		pptr[i].x=getfloat(fp)+ALTO_LEFT_MARGIN;
	    for (i = 0; i < numvert; i++)
		pptr[i].y=getfloat(fp)+ALTO_BOTTOM_MARGIN;
	    NewObject( sptr, numvert>2?
		(sptr->closed?ClosedSplineObj:OpenSplineObj):
		(sptr->closed?ClosedPolygonObj:OpenPolygonObj),
		0,0,0,0,0,0,0,Debug&DebugIO,0,0,activelist);
	}
    nstrings=getword(fp);
    for (i=1;i<=nstrings;i++) {
	if ((data = (char *) malloc( MAXLEN+1 )) == NULL)
	  {
	    printf("\n\rOut of memory.  Can't read a text string.\n\r");
	    return;
	  }
	dx=getword(fp)+ALTO_LEFT_MARGIN;
	dy=getword(fp)+ALTO_BOTTOM_MARGIN;
        font=getword(fp);
	switch(font) {
	    case 0:
		font=9;
		descent=5;
		break;
	    case 1:
		font=10;
		descent=5;
		break;
	    case 2:
		font=5;
		descent=7;
		printf("Helvetica 7 being substituted for Helvetica 8\n\r");
		break;
	    case 3:
		font= 9;
		descent=5;
		printf("Alto arrows font not supported.\n\r");
		break;
	    default:
		printf("Invalid Alto font number :%d\n",font);
		font= 9;
		descent=5;
		break;
	}
	aword=getword(fp);	
	if ((aword&0x0007)!=7) printf("Color is bing ignored in text\n\r");
	aword=getword(fp);
	getstring(fp,aword,data);
	NewObject( data, TextObj, font, 2, dx, dx+strlen(data)*5,dy-16,dy,
	    descent,Debug&DebugIO,0,0,activelist);
    }
	CurrentObject = NULL;
	FakeGroup = FALSE;
}

/* 
 * get a BCPL word from the standard input and return as a short
 */

unsigned short getword(fp)

FILE *fp;
{
    unsigned short i;
    i=(getc(fp)&255)<<8;
    return(i|(getc(fp)&255));
    }
    
/* 
 * get a BCPL packed floating point number from standard input and return
 * integer portion
 */

int getfloat(fp)
FILE *fp;

{
    int num;
    int mantisa;
    short exponent;
    short sign;
    short aword;
    
    aword=getword(fp);
    sign=aword&0x8000;
    exponent=((aword&0x7f80)>>7)-128;
    mantisa=getword(fp)+((aword&0x007f)<<16);
    if (exponent<0) return(0);
    else {
	num=mantisa>>(23-exponent);
	if (sign!=0) num= -num;
	return(num);
    }
}

/* 
 * get a BCPL string, but read at least (word) words.
 */

getstring(fp,words,str)
FILE *fp;
int words;
char *str;


{
   int numbytes,strlen,i;
   strlen=getc(fp);
   numbytes=2*words-1-strlen;
   if (numbytes<0) numbytes=0;
   if (((numbytes+strlen)&1)==0) numbytes++;
   for (i=1;i<=strlen;i++) 
       *(str++)=(char) getc(fp);
   *(str++)='\0';
   for (i=1;i<=numbytes;i++)
	getc(fp);
}

	


/*
 *  This internal routine will read a file, appending it to the activelist.
 *  This routine currently supports four file types --
 *	V Draw files - V internal files - preferred format
 *	Alto Draw files - For compatibility with an old ancestor
 *	SIL files - For compatibility with V and Ato SilEdit
 *	Journal files - a low level event recording mechanism for crash recovery
 *
 */

ReadFile()
  {
    char *fname;
    FILE *fp, *fopen();
    short aword;
    
    /* Get the file pointer */
    mprintf(1,"Input file name?  ");
    if ((fname = GetString()) == NULL)
      {
	mprintf(2,"Read File Aborted.");
	return;
      }
    if ((fp = fopen( fname, "r" )) == NULL)
      {
	mprintf(2,"Couldn't open file '%s'.", fname);
	free( fname );
	return;
      }
    mprintf(1,"Reading file '%s'...", fname);
    Checkpoint();

    /* Check the file header. */
    aword=getword(fp);
    fseek(fp,0,0);
    if (aword==0xffff) 
        altoread(fp);
    else if (aword==SILHEADER)
	silread(fp);
    else { 
	fgets( fname, MAXLEN, fp );
	if (strcmp( fname, FILEHEADER ) == 0)
	    drawread(fp,0,0);
	else if (strcmp( fname, OLDFILEHEADER) == 0)
/*
 * This is for compatibility with the old coordinate system. The old boundaries
 * were (bot left - top right - x,y) (34,50) - (632,808). The new coordinate
 * system has the same orientation but the boundaries are (18,18) - (594-768)
 * This ensures that anything visible in the default window is printable.
 * The new coordinate system is smaller than the old, and no scaling is done.
 * The X,Y offset below reflets a shift so that a point at the top-center of
 * the drawing will remain stationary. Some lines may be clipped on either side
 * and at the pottom, but the top margin will be the same.
 * Other possible values are - to keep the specified point constant...
 * 	-27, -36 	center
 *	-16, -32	bottom left
 *	-16, -40	top left
 *	-38, -32	bottom right
 *	-16, -40	top right
 */
	    drawread(fp,-27,-40);
	else if (strcmp ( fname, JOURNALHEADER) == 0) {
	    if (Journal == JournalPlay || OldJournal != JournalOff ||
	      JournalFIn != NULL) {
		mprintf(5,"\007\007Already reading a journal file.");
		free( fname);
	 	fclose(fp);
		return;
	    }
	    else {	/* Start reading the journal file */
		OldJournal = Journal;
		Journal = JournalPlay;
		OldJournalSteps = JournalSteps;
		JournalSteps = 0;
		JournalFIn = fp;
		free (fname);
		return;
	    }
	}
	else {
	    mprintf(2,"Draw cannot read this file\007");
	    free( fname );
	    fclose( fp );
	    RevertToCKP();
	    return;
	}
    }

    /* Each of the load routines did a MajorChange which will cause the screen
	to be re-drawn just before the next input event. */

    free( fname );
    fclose( fp );
    mprintf(2,"done.");
  }
