/*
 *  input.c
 *
 *  This file contains the input handling routines -- the main procedures
 *  of interest is GetInput(), GetString, GetStr2, and GetPopup.
 *  GetInput is the main mouse click event routine. SetString returns a string,
 *  GetStr2 is like GetString except that the caller allocates memory. GetPopup
 *  is the pop-up menu routine. All of these routines go through the Draw 
 *  Monkey and Journal facilities. The monkey generates random events. The
 *  Journal records events for later playback. This is useful for crash
 *  recovery, and to aid bug reproducability.
 *
 */
 
/* Includes */
# ifdef UNIX
# include "stdio.h"
# else
# include "Vio.h"
# endif
# include "Vgts.h"
# include "splines.h"
# include "draw.h"
# include "icons.h"
# include "menu.h"    
    
/* Imports */
extern PrintCommand();
extern OBJECT *FindSticky();
extern ICON_DESCRIPTOR IconList[];
 
 
/* Exports */
extern GetInput();		/* Return a command		*/
extern short getpopup();	/* get a pop-up menu */
extern char *GetString();	/* Return a general string	*/
extern char *GetStr2();		/* Non-memory allocating GetString */
 
static short GetMonkey();


/*
 *  This internal routine will process a click in the main drawing
 *  area, converting the command or coordinates as required by the
 *  mouse buttons.
 */
 
CookMouseKeys( cmd, x, y, but )
	enum MenuOptions *cmd;
	short *x, *y, *but;
  {
    short x1, y1;
    int dist;
    
    /* Process the various combinations */
    switch (*but)
      {
	case LeftButton:			/* Click Here */
	    *cmd = CDataPoint;
	    break;
	    
	case MiddleButton:			/* Click at Sticky Point */
	    if (FindSticky(*x, *y, &x1, &y1) == NULL)
	      {
		/* Nothing stick around here. */
		mprintf(2,"No sticky points nearby.  Try again.\n\r");
		*cmd = CNull;
	      }
	    else
	      {
		*x = x1;  *y = y1;
		*cmd = CDataPoint;
	      }
	    break;
	    
	case RightButton:			/* Click at Grid Point */
	    *cmd = CDataPoint;
	    *x = (*x + 8) & ~0xF;
	    *y = (*y + 8) & ~0xF;
	    break;

	case LeftButton+MiddleButton:		/* Again */
	    *cmd = CAgain;
	    break;

	case LeftButton+RightButton:		/* toggle current selection */
	    *cmd = CToggleSelect;
	    break;

	case MiddleButton+RightButton:		/* Undo */
	    *cmd = CUndo;
	    break;
	    
	case LeftButton+MiddleButton+RightButton:	/* Abort */
	    *cmd = CAbort;
	    break;

	case 77:			/* Kluge for Monkey - See GetMonkey */
	    *cmd = CDone;
	    break;
	    
	case 66:			/* likewise */
	    *cmd = CGroup;
	    break;

	default:
	    printf("Internal Error:  Bad mouse buttons %d at (%d, %d)\n\r",
	    		*but, *x, *y);
	    *cmd = CNull;
	    break;
      }
  }

/*
 *  This routine will get an input event an classify it as to which
 *  command it represents.  Input events consist solely of mouse
 *  clicks.  If the click is not near any menu item, and is not in
 *  the drawing area, it is considered garbage and rejected.
 *
 *  The specific mouse buttons used to click the mouse are significant
 *  inside the drawing area (within the menu area they are all identical).
 *  The mouse button meanings are:
 *
 *	Buttons		Command
 *	 - - X		Datapoint, force grid alignment.
 *	 - X -		Datapoint, try to stick to an object.
 *	 X - -		Datapoint, right where the mouse is.
 *	 - X X		Undo -- try to undo last command.
 *	 X - X		ToggleSelect -- Add/remove an item from the selection
 *	 X X -		Again -- Finish command, and restart it.
 *	 X X X		Abort.
 *
 *  NB:  FindMousedObject returns static storage.  DO NOT FREE IT!!
 */
 
GetInput( pcmd, px, py, pbut )
	enum MenuOptions *pcmd;
	short *px, *py, *pbut;
  {
    register LISTTYPE *hitlist;
    register short i, cmdno;
    short clickVgt;
    
   
	DeleteMessage(1); /* Age the messages */

	if (Journal == JournalPlay) {
	    char jtype,jvgt;
	    int ioresult,jstep;
	    ioresult = fscanf(JournalFIn,"%c %d %hd %hd %hd %c\n",
				&jtype,&jstep,px,py,pbut,&jvgt);
	    if (jtype != 'M' || ioresult != 6 || jstep != JournalSteps+1) {
		KillJournalIn(ioresult);
		GetInput(pcmd,px,py,pbut);
		return;
	    }
	    else {
		JournalSteps++;
		if (jvgt == 'D')
		    clickVgt = mainVgt;
		else
		    clickVgt = menuVgt;
	    }
	}
	else if (Monkey) {
	    clickVgt = GetMonkey( px, py, pbut );
	    if (MonkeySteps > 0) {
		if (--MonkeySteps == 0) {
		    printf("**Last Monkey Step**\n\r");
		    Monkey = 0;
		}
		else if (MonkeySteps > 0 && 
			 (MonkeySteps % 20 == 0 || MonkeySteps<20)) {
		    printf("%d steps remain\n\r",MonkeySteps);
	 	}
	    }
	    else if (MonkeySteps < 0) {
		MonkeySteps--;
		if ((-MonkeySteps)%20==0)
		    printf("%d Monkey steps\n\r",-MonkeySteps);
	    }
	}
	else {
	    clickVgt = GetMouseClick( px, py, pbut );
	}

  	if (Debug&DebugInput) {
	    printf("Mouse click at ");
	    printf("(%d, %d, %d) in %d\n\r", *px, *py, *pbut, clickVgt);
	}

	if (Journal == JournalRecord) {
	    char jvgt;
	    JournalSteps++;
	    jvgt = (clickVgt == mainVgt ? 'D' : 'M');
	    fprintf(JournalFOut,"M %d %d %d %d %c\n",
				JournalSteps,*px,*py,*pbut,jvgt);
	}
    
    /* Menu item or Data point? */
    if (clickVgt == mainVgt)
      {
	CookMouseKeys( pcmd, px, py, pbut );
	return;
      }
    
    /* Something ... else? */
    if (clickVgt != menuVgt)
      {
	if (!Monkey)
	    mprintf(2,"Missed!  Please select a menu command.\n\r");
	*pcmd = CNull;
	return;
      }
    
    /* In the menu.  Find out which menu item was closest. */
    *pcmd = CNull;
    if (*px>=FontXMin && *px<=FontXMax && *py>=FontYMin && *py<=FontYMax) {
	*pcmd = CCenter;
	*px=(*px<FontTick1 ? PositionLeft : 
		(*px<FontTick2 ? PositionCenter : PositionRight));
	return;
    }
    if (*px>=MesgXMin && *px<=MesgXMax && *py>=MesgYMin && *py<=MesgYMax
	&& *pbut==7) {
	*pcmd = CDebug;
	return;
    }
    hitlist = (LISTTYPE *) FindMousedObject( sdf,*px,*py,menuVgt,All );
    if (hitlist->NumOfElements == 0)
      {
	if (!Monkey)
	    mprintf(2,"Missed.  Try again.\n\r");
	return;
      }
    
    /* Search the various menu items ... */

    for (i=FIRST_NOUN,cmdno=C_NOUNS;(i<=LAST_NOUN) && (*pcmd == CNull); 
		i++,cmdno++)
        if (hitlist->Header->item == IconList[i].itemnum)
	    *pcmd = (enum MenuOptions) cmdno;
    for (i=FIRST_VERB,cmdno=C_VERBS;(i<=LAST_VERB) && (*pcmd == CNull); 
		i++,cmdno++)
        if (hitlist->Header->item == IconList[i].itemnum)
	    *pcmd = (enum MenuOptions) cmdno;
    for (i=0,cmdno=C_COMMANDS;(i< NumCommands) && (*pcmd == CNull); 
		i++,cmdno++)
        if (hitlist->Header->item == Commands[i].itemno)
	    *pcmd = (enum MenuOptions) cmdno; 
    for (i=0;(i< MAXNIB) && (*pcmd == CNull);i++)
        if (hitlist->Header->item == Nibs[i])
	  {
	    *pcmd = CNib;
	    *px = i;
	  }; 
    for (i=0;(i< MAXPAT) && (*pcmd == CNull);i++)
        if (hitlist->Header->item == Patterns[i])
	  {
	    *pcmd = CPattern;
	    *px = i;
	  }; 
    
    /* Still nothing? */
    if (*pcmd == CNull)
      {
	*pcmd = CDone;
      }
  }

/*
 *  This routine will read in a string from the user, possibly
 *  containing control characters, and return a pointer to it.
 *  This is the lower level routine which does not provide a prompt and
 *  does not allocate memory. Called by GetString below
 */
 
char *GetStr2(ptr,maxlen)
   char *ptr;
   short maxlen;
  {
    char c;
    short i, typing;
    
    if (Journal == JournalPlay) {
	    char jtype;
	    int ioresult,jstep,jlen;
	    ioresult = fscanf(JournalFIn,"%c %d %d %s\n",
				&jtype,&jstep,&jlen,ptr);
	    if (jtype != 'S' || jstep != JournalSteps+1) {
		if (ioresult != EOF)
		    mprintf(2,"Error in journal file after step %d",
			JournalSteps);
		else
		    mprintf(2,"Journal complete after step %d",
			JournalSteps);
		KillJournalIn(ioresult);
		return(GetStr2(ptr,maxlen));
	    }
	    else {
		JournalSteps++;
		if (jlen==0) {
		    printf("\n\r");
		    ptr = NULL;
		}
		else
		    printf("%s\n\r",ptr);
	    }
	}
    else { 
    fflush( stdout );
    ResetTTY();
    typing = 1;
	for (i = 0; (i < maxlen) && (typing);)
      {
	if (Monkey) {
	   if (rand()%4 == 0) c = '\n';
	   else c = 'a' + rand()%26;
	   putchar(c);
	}
	else c = getchar();
	ptr[i++] = c;
	switch (c)
	  {
	    case '\021':		/* control-Q */
		printf("\b \b");
		fflush( stdout );
		ptr[i - 1] = getchar();	
		break;
		
	    case '\177':		/* delete */
	    case '\b':			/* backspace */
		printf(" \b");
		i = (i > 1 ? i - 2 : 0);
		fflush( stdout );
		break;
	
	    case '\025':		/* control-U */
		i = 0;
		printf("\r");
		fflush( stdout );
		break;
	
	    case '\n':
		ptr[i - 1] = 0;
		typing = 0;

		break;
	
	    default:
		break;
	  }
      }

    fflush(stdin);
    /* Return the results. */
    if ((i == 0) || (*ptr == 0))
      {
	/* If user aborted, return NULL */
	ptr = NULL;
      }
    else
      {
	/* Zero out the rest of the string. */
	while( i <= maxlen )
	    ptr[i++] = 0;
      }
    GetTTY();
    }
    if (Debug&DebugInput) 
	printf("String Input :%s\n\r",ptr);

    if (Journal == JournalRecord) {
	JournalSteps++;
	    if (ptr == NULL)
	        fprintf(JournalFOut,"S %d 0 <null>\n");
	    else
		fprintf(JournalFOut,"S %d %d %s\n",
			JournalSteps,strlen(ptr),ptr);
	}
    return( ptr );
  }
/*
 *  This routine will read in a string from the user, possibly
 *  containing control characters, and return a pointer to it.
 *  NULL is returned for an empty string.  ^Q is the quote character.
 *  This is the higher level routine which allocates memory.
 */
 
char *GetString()
  {
    char *ptr, *ptr2;
    
    /* Allocate a buffer */
    if ((ptr = (char *) malloc( MAXLEN + 1 )) == NULL)
      {
	printf("Out of memory.  Couldn't allocate a string.\n\r");
	return( NULL );
      }

    /* Okay, off we go. */
    printf("Enter a string.  Use ^Q to quote characters.\n\r");
    ptr2 = GetStr2(ptr,MAXLEN);
    
    /* Return the results. */
    if (ptr2 == NULL)
      {
	/* If user aborted, return NULL */
	free( ptr );
	ptr = NULL;
      }
    return( ptr );
  }

static short GetMonkey(x,y,but)
    short *x,*y,*but;

{
    ICON_DESCRIPTOR *p = &IconList[FIRST_VERB+(int)CGroup-C_VERBS];
    static enum MenuOptions LastCommand;
/*
 * These are the monkey's command weight tables. The current values are
 * set ad-hoc. Ideally, statistics should be taken to see what they are
 * in practical use. Entries are coded as follows...
 *
 *   Left, Middle, Right = 1, 4, 2 respectively from Vgts.h 
 *   1 = normal, 2 = grid, 3 = ToggleSelect, 4 = sticky, 5 = Again,
 *   6 = undo, 7 = abort, 77 = Done , 66 = Group
 */
    /*
     * This is the monkey choice map for field clicks when NOT in any 
     * particular command.
     */
#define CHOICES1 30
    static unsigned char choicemap1[CHOICES1] = {
	 3, 3, 3, 3, 3, 3, 3, 3, 3, 1,
	 1, 3, 3, 3, 3, 3, 3, 3, 3, 3,
	 2, 4, 5, 5, 5, 1, 1, 3, 3, 1,
    };
    /*
     * This is the monkey choice map for field clicks when in an indeinite 
     * point command
     */
#define CHOICES2 30
    static unsigned char choicemap2[CHOICES2] = {
	 1, 1, 1, 1, 1, 1, 1, 1, 2, 2,
	 2, 2, 4, 4, 4, 4, 4, 1, 2, 4,
	77,77,77,77, 4,77, 2, 1, 4, 2,
    };
    /*
     * This is the monkey choice map for field clicks when in a deinite point
     * command
     */
#define CHOICES3 30
    static unsigned char choicemap3[CHOICES3] = {
	 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
	 2, 2, 2, 2, 2, 4, 4, 4, 4, 4,
	 4,77, 1, 1, 2, 4, 1, 1, 4, 2,
    };
	
    if (rand () % 5 == 0) {	/* Menu click */
	if (CurrentObject!=NULL && CurrentObject->part->type==GroupObj &&
	    rand() % 3 == 0) {
	    printf("G");
	    *but = 66;	/* CGroup */
	}
	else if (rand()%20 == 0) {
	    *but = (rand()%3 ? 6 : 7);	/* undo or abort */
	    LastCommand = CurrentCommand;
	    return (mainVgt);
	}
	else {
	    *x = rand () % (Mxmax - Mxmin);
	    *y = rand () % (Mymax - Mymin);
	    *but = LeftButton;
	}
	LastCommand = CurrentCommand;
	return (menuVgt);
    }
    else {			/* Field click */
	if (CurrentCommand==CNull || LastCommand==CNull) {
	    *x = rand () % (680);
	    *y = rand () % (880);
	}
	else {
	    *x += (rand () % 200) * (rand () % 200) / 200 - 100;
	    *y += (rand () % 200) * (rand () % 200) / 200 - 100;
	    if (*x < 0)
		*x = 100;
	    if (*x > 680)
		*x = 580;
	    if (*y < 0)
		*y = 100;
	    if (*y > 880)
		*y = 780;
	}
	if (CurrentCommand == CNull)
	    *but = choicemap1[rand() % CHOICES1];
	else if (DoneFlag)
	    *but = choicemap2[rand() % CHOICES2];
	else
	    *but = choicemap3[rand() % CHOICES3];
	LastCommand = CurrentCommand;
	/* Note that this is not PrevCommand since that is used for the "again"
	 * function and thus does not remember operations that don't set 
	 * CurrentCommand to anything but CNull. 
	 */
	return (mainVgt);
    }
}

/*
 * This is an interface to the VGTS popup() function which goes through the
 * journal and monkey hooks.
 */
GetPopup(menu)
PopUpEntry menu[];

{
    short   menusize;
    int     result;

    for (menusize = 0; menu[menusize].string != NULL; menusize++);
    if (Journal == JournalPlay) {
	char    jtype;
	int     ioresult,
	        jstep;
	ioresult = fscanf (JournalFIn, "%c %d %d\n",
		&jtype, &jstep, &result);
	if (jtype != 'P' || ioresult != 3 || jstep != JournalSteps + 1) {
	    KillJournalIn (ioresult);
	    return (GetPopup (menu));
	}
	else {
	    JournalSteps++;
	}
    }
    else if (Monkey) {
	    int     i;
	    i = rand () % (menusize + 1);
	    if (i == menusize)
		result = -1;
	    else
		result = menu[i].menuNumber;
	}
    else
	result = popup (menu);

    if (Debug&DebugInput) {
	if (result<0)
	    printf("Pop-up aborted");
	else {
	    int i;
	    for (i=0;menu[i].menuNumber!=result&&menu[i].string!=NULL;i++);
	    printf ("Pop-up: %d (%s)\n\r",result,menu[i].string);
	}
    }


    if (Journal == JournalRecord) {
	JournalSteps++;
	fprintf (JournalFOut, "P %d %d\n", JournalSteps, result);
    }
    return(result);
}

/*
 * KillJournalIn is called when any of the input routines detext an error or
 * an end-of-file on the input journal file.
 */

KillJournalIn(io)
int io;

{
    if (io != EOF)
	mprintf (2, "Error in journal file after step %d",
		JournalSteps);
    else
	mprintf (2, "Journal complete after step %d",
		JournalSteps);
    fclose (JournalFIn);
    JournalFIn = NULL;
    Journal = OldJournal;
    JournalSteps = OldJournalSteps;
    OldJournal = JournalOff;
    OldJournalSteps = 0;
}
