/*
 *  find.c
 *
 *  This file contains the routines used to find an object on the
 *  screen.  The primary routine of interest is FindObject(), which
 *  will locate an object, based upon the distance
 *  to its sticky points.  The object with the minimum distance to a
 *  control point is selected.
 *
 */
 
/* Includes */
# ifdef UNIX
# include "stdio.h"
# else
# include "Vio.h"
# endif
# include "Vgts.h"
# include "splines.h"
# include "draw.h"
# include <math.h>
 
 
/* Imports */
extern short GetINum();		/* Get an item number */
extern short GetSeqINum();	/* Get a sequential set of item numbers */
extern short LastINum();	/* Return the last item number gotten
extern short FreeINum();	/* Return an item number to the freelist */

 
/* Exports */
extern OBJECT *FindObject();
 
BOOLEAN PointInObject();

/*
 *  This routine will return the square of the distance between two points.
 */
 
int PointDistance( x1, y1, x2, y2 )
	register short x1, y1, x2, y2;
  {
    register short x, y;
    
    x = (x1 - x2);
    y = (y1 - y2);
    return( x * x  +  y * y );
  }

/*
 *  This internal routine is strictly a helper process for SplineDistance.
 *  It will return the proper knot from the simulated knot vector in order
 *  to calculate the sticky points on a spline.  End conditions are dealt
 *  with properly.
 */
 
short Knot( i, order, closed, x, numvert, vert )
	short i, order, closed, x, numvert;
	POINT *vert;
  {
    register short j;			/* Index variable */
    
    /* Compute the proper index into the vertex array. */
    if (closed)
      {
	/* Just circulate the vertices */
	j = i % numvert;
      }
    else
      {
	/* Duplicate beginning and ending knots appropriately. */
	i = i + 3 - order;
	if (i <= 0)
	    j = 0;
	else if (i >= numvert)
	    j = numvert - 1;
	else
	    j = i;
      }
    
    /* Return the value of this control point. */
    if (x)
	return( vert[j].x );
    else
	return( vert[j].y );
  }

/*
 *  This gives the sticky points given the control points
 */
    Sticky(numvert,vert,order,closed,i,xval,yval)
	short numvert;		/* Number of control points. */
	POINT *vert;		/* Array of control points. */
	short order;		/* Order of the spline. */
	BOOLEAN closed;		/* End conditions. */
	short i;		/* index */
	short *xval, *yval;	/* sticky point */
{
    if (order<3 || order>5) {
	*xval=Knot(i,order,closed,1,numvert,vert);
	*yval=Knot(i,order,closed,0,numvert,vert);
    }
    else switch (order)
        {
	    case 3:	/* Sticky point is (p1 + p2) / 2 */
	        *xval = (Knot(i, order, closed, 1, numvert, vert) +
			Knot(i+1, order, closed, 1, numvert, vert)) / 2;
		*yval = (Knot(i, order, closed, 0, numvert, vert) +
			Knot(i+1, order, closed, 0, numvert, vert)) / 2;
		break;
	    case 4:	/* Sticky point is (p1 + 4*p2 + p3) / 6 */
		*xval = (Knot(i, order, closed, 1, numvert, vert) +
			4*Knot(i+1, order, closed, 1, numvert, vert) +
			Knot(i+2, order, closed, 1, numvert, vert)) / 6;
		*yval = (Knot(i, order, closed, 0, numvert, vert) +
			4*Knot(i+1, order, closed, 0, numvert, vert) +
			Knot(i+2, order, closed, 0, numvert, vert)) / 6;
		break;		
	    case 5:	/* Sticky point is (p1 + 11*p2 + 11*p3 + p4) / 24 */
		*xval = (Knot(i, order, closed, 1, numvert, vert) +
			11*Knot(i+1, order, closed, 1, numvert, vert) +
			11*Knot(i+2, order, closed, 1, numvert, vert) +
			Knot(i+3, order, closed, 1, numvert, vert)) / 24;
		*yval = (Knot(i, order, closed, 0, numvert, vert) +
			11*Knot(i+1, order, closed, 0, numvert, vert) +
			11*Knot(i+2, order, closed, 0, numvert, vert) +
			Knot(i+3, order, closed, 0, numvert, vert)) / 24;
		break;
	  }
}

/*
 *  This internal routine, along with its helper Knot(), will compute
 *  the square of the distance from a point to the nearest sticky
 *  point on a spline.
 */
 
int SplineDistance( numvert, vert, order, closed, x, y, dx, dy, cp )
	short numvert;		/* Number of control points. */
	POINT *vert;		/* Array of control points. */
	short order;		/* Order of the spline. */
	BOOLEAN closed;		/* End conditions. */
	short x, y, *dx, *dy;	/* Current point, closes point */
	short cp;		/* Measure from control points too? */
  {
    int curdist;
    register int dist;
    short xval, yval, i;
    short bestx, besty;
    
    /* Initialize */
    order = ((order == 2) ? 3 : order);
    curdist = MAXDIST << 6;
    
    /* Check each sticky point. */
    for (i = -(!closed); i < numvert; i++)
      {
        Sticky(numvert,vert,order,closed,i,&xval,&yval);
	
	/* Compute the distance from this sticky point. */
	dist = PointDistance( x, y, xval, yval );
	if (dist < curdist)
	  {
	    bestx = xval;
	    besty = yval;
	    curdist = dist;
	  }
	
	/* Check Control points too? */
	if (cp)
	  {
	    dist = PointDistance( x, y, vert[i].x, vert[i].y );
	    if (dist < curdist)
	      {
		bestx = vert[i].x;
		besty = vert[i].y;
		curdist = dist;
	      }
	  }
      }
     /* Also, do the center of a circle */
    if (order == 5 && closed && numvert==4) {
	    dist = PointDistance( x, y, (vert[0].x+vert[2].x)/2, 
					(vert[0].y+vert[2].y)/2);
	    if (dist < curdist)
	      {
		bestx = (vert[0].x+vert[2].x)/2;
		besty = (vert[0].y+vert[2].y)/2;
		curdist = dist;
	      }
    }
    
    /* Return the results */
    *dx = bestx;
    *dy = besty;
    return( curdist );
  }

/*
 *  This procedure will return the coordinates of the minimum distance
 *  sticky points on an object, measured from a given point.  The value
 *  of the procedure is the square of the distance.
 */
 
int ObjectDistance( x, y, px, py, id, deltax, deltay, insideOnly )
	short x, y, *px, *py;
	register PART *id;
	short deltax, deltay;
	BOOLEAN insideOnly;
  {
    short xtmp, ytmp;
    register int dist, curdist = MAXDIST << 6;
    SPLINE *sptr;
    POINT *pptr;
    OBJECT *gptr;
    
    if (Debug&DebugFind) {
	printf("ObjectDistance (%d ,%d): part=%x...",x,y,id);
    }
    /* Subtract offset from target point now, */
    /*  add it in to returned point later.  */
    x -= deltax;  y-= deltay;
    switch (id->type)
      {
	case TextObj:
	    /* Sticky points are the corners and midpoints of	*/
	    /* the edges of the bounding box.			*/
	    if ((pptr = (POINT *) malloc( sizeof(POINT) * 4 )) == NULL)
	      {
		printf("Out of memory.  Can't compute text distance.\n\r");
		break;
	      }
	    pptr[0].x = id->xmin;
	    pptr[0].y = id->ymin;
	    pptr[1].x = id->xmin;
	    pptr[1].y = id->ymax;
	    pptr[2].x = id->xmax;
	    pptr[2].y = id->ymax;
	    pptr[3].x = id->xmax;
	    pptr[3].y = id->ymin;
	    dist = SplineDistance( 4, pptr, 2, 1, x, y, &xtmp, &ytmp, 1 );
	    if (dist < curdist)
	      {
		curdist = dist;
		*px = xtmp;  *py = ytmp;
	      }
	    free( pptr );
	    break;
	
	case OpenSplineObj:
	case ClosedSplineObj:
	case OpenPolygonObj:
	case ClosedPolygonObj:
	case SILObj:
	case TemplateObj:
	    /* A spline.  Measure distance from sticky points. */
	    sptr = (SPLINE *) id->data;
	    pptr = &(sptr->head);
	    if (sptr->numvert == 1)
	      {
		/* Only one possible point.  An easy case. */
		dist = PointDistance( x, y, xtmp=pptr[0].x, ytmp=pptr[0].y );
		if (dist < curdist)
	          {
		    *px = xtmp;  *py = ytmp;
		    curdist = dist;
	          }
	      }
	    else
	      {
		/* A hard case.  Call the more elaborate routines. */
		dist = SplineDistance(
			sptr->numvert, pptr, sptr->order, sptr->closed,
			x, y, &xtmp, &ytmp, (sptr->order == 2) );
		if (dist < curdist)
	          {
		    *px = xtmp;  *py = ytmp;
		    curdist = dist;
	          }
	      }
	    break;
	
	case GroupObj:
	    gptr = ((OBJECT_HEADER *) id->data)->first;
	    while (gptr) {
		if (!insideOnly || PointInObject(gptr,x,y,TRUE,0)) {
		    dist = ObjectDistance( x, y, &xtmp, &ytmp,
			gptr->part, gptr->dx, gptr->dy,insideOnly );
		    if (dist < curdist) {
			*px = xtmp;  *py = ytmp;
			curdist = dist;
		    }
		}
		gptr = gptr->next;
	    }
	    break;
	    
	default:
	    printf("Internal Error:  Illegal object type %d in ObjDist.\n\r",
	    		id->type);
	    break;
      }
    
    /* Compensate for displacements, and return results */
    *px += deltax;  *py += deltay;
    if (Debug&DebugFind) {
	printf("distance=%d (%d,%d)\n\r",curdist,*px,*py);
    }
    return( curdist );
  }

/*
 *  This macro tests whether a point is within the bounds of a rectangle
 */
#define PointInRect(x,y,x1,x2,y1,y2) (x>=x1&&x<=x2&&y>=y1&&y<=y2)

/*
 *  This internal routine, will determine
 *  whether a point lies inside a spline. (or very close to an edge
 *  We use the parity algorithem which returns TRUE if the spline cuts
 *  a ray leading in a streight line from the given point an odd number of
 *  times.
 */
 
BOOLEAN PointInSpline( numvert, vert, order, closed, x, y)
	short numvert;		/* Number of control points. */
	POINT *vert;		/* Array of control points. */
	short order;		/* Order of the spline. */
	BOOLEAN closed;		/* End conditions. */
	register short x, y;	/* Current point, closes point */
  {
    BOOLEAN parity;
    short xval, yval,xval2,yval2, i;
    double fract;
    if (Debug&DebugFind) 
	printf("PointInSpline: numvert=%d; order=%d; closed=%d; point=(%d,%d)\n\r",numvert,order,closed,x,y);
    
    
    /* Initialize */
    if (numvert <= 2) return(TRUE);
    parity=0;
    
    /* Check each sticky point. */
    for (i = -(!closed); i < numvert; i++) {
        Sticky(numvert,vert,order,closed,i  ,&xval ,&yval );
	Sticky(numvert,vert,order,closed,i+1,&xval2,&yval2);
	if (y==yval && x==xval) return(TRUE);
        if (y>yval && y>yval2) continue;
	if (y<yval && y<yval2 && (x-xval)*(x-xval2) <=0) {
	    parity = !parity;
	    continue;
	}
	if (x<xval && x<xval2) continue;
	if (x>=xval && x>=xval2) continue;
	if (xval==xval2) continue;
	fract = (double)(x-xval)/(double)(xval2-xval);
	if (y<(yval+fract*(yval2-yval))) {
	    parity = !parity;
	    continue;
	}
	if (Debug&DebugFind)
	    printf("    (%d,%d) - (%d,%d) parity = %d",xval,yval,xval2,yval2,parity);
    }
    return( parity );
  }

/*
 *  This procedure will return TRUE if the given point is inside the given
 *  object.
 */
 
BOOLEAN PointInObject( ob, x, y, quick, area)
 	register OBJECT *ob;
	short x, y;
	BOOLEAN quick;
	int *area;
  {
    register PART *id = ob->part;
    SPLINE *sptr;
    POINT *pptr;
    OBJECT *gptr;
    
    if (Debug&DebugFind) {
	printf("PointInObject %x at (%d, %d) %s",ob,x,y,quick?"QUICK":"SLOW");
	printf("x[%d %d] y[%d %d]\n\r",id->xmin,id->xmax,id->ymin,id->ymax);
    }
    /* Subtract offset from target point now, */
    x -= ob->dx;  y-= ob->dy;
    switch (id->type) {
	case TextObj:
	    if (area)
		*area = (id->xmax-id->xmin)*(id->ymax-id->ymin+id->base);
	    return(PointInRect(x,y,id->xmin,id->xmax,id->ymin-id->base,id->ymax));
	case GroupObj:
	    if (!PointInRect(x,y,id->xmin,id->xmax,id->ymin,id->ymax))
		return(FALSE);
	    if (quick)
		return(TRUE);
	    gptr = ((OBJECT_HEADER *) id->data)->first;
	    while (gptr)  {
		if (PointInObject(gptr,x,y,quick,area))
		    return(TRUE);
		gptr = gptr->next;
	    }
	    return(FALSE);
	default:
	    if (!PointInRect(x,y,id->xmin,id->xmax,id->ymin,id->ymax))
		return(FALSE);
	    if (quick)
		return(TRUE);
	    if (area)
		*area = (id->xmax-id->xmin)*(id->ymax-id->ymin);
	    if ((area == 0 || *area > 900) && (id->xmax - id->xmin < 30 || id->ymax - id->ymin < 30))
		return(TRUE);
	    if (id->type==SILObj) 
		return(TRUE);
	    /* A spline.  Measure distance from sticky points. */
	    sptr = (SPLINE *) id->data;
	    pptr = &(sptr->head);
	    /* SIL objects are very narrow rectangles which don't pass the */
	    /*	PointInSpline test. */
	    return(PointInSpline( sptr->numvert, pptr, sptr->order, 
		sptr->closed, x, y));
      }
  }

/*
 *  SectRect returns TRUE if two rectangles intersect
 */
BOOLEAN SectRect(xmin1,xmax1,ymin1,ymax1,xmin2,xmax2,ymin2,ymax2)
    short xmin1,xmax1,ymin1,ymax1,xmin2,xmax2,ymin2,ymax2;
{
    return(!(xmax2<xmin1 || xmax1 < xmin2 || 
	     ymax2<ymin1 || ymax1 < ymin2));
}


/*
 *  This procedure will return TRUE if the given rectangle intersects the
 *  object.
 */
 
BOOLEAN RectInObject( ob, xmin, xmax, ymin, ymax)
 	register OBJECT *ob;
	short xmin, xmax, ymin, ymax;
  {
    register PART *id = ob->part;
    OBJECT *gptr;
    
    /* Subtract offset from target point now, */
    xmin -= ob->dx;  ymin -= ob->dy;
    xmax -= ob->dx;  ymax -= ob->dy;
    if (Debug&DebugFind) {
	printf("RcInObj %x at x[%d %d] y[%d %d]",ob,xmin,xmax,ymin,ymax);
	printf("ob= x[%d %d] y[%d %d] dx=%d; dy=%d\n\r",
		id->xmin,id->xmax,id->ymin,id->ymax,ob->dx,ob->dy);
    }
    switch (id->type) {
	case TextObj:
	    return(SectRect(xmin,xmax,ymin,ymax,
		id->xmin,id->xmax,id->ymin-id->base,id->ymax));
	case GroupObj:
	    if (!SectRect(xmin,xmax,ymin,ymax,
		id->xmin,id->xmax,id->ymin,id->ymax))
		return(FALSE);
	    gptr = ((OBJECT_HEADER *) id->data)->first;
	    while (gptr)  {
		if (RectInObject(gptr,xmin,xmax,ymin,ymax))
		    return(TRUE);
		gptr = gptr->next;
	    }
	    return(FALSE);
	default:
	    return(SectRect(xmin,xmax,ymin,ymax,
		id->xmin,id->xmax,id->ymin,id->ymax));
      }
  }

/*
 *  This routine will find the closest suitable object to a given point.
 *  Preference is given to objects which surround the point. Point distance
 *  is a strictly secondary criterium. 
 *
 *  If FakeFlag is TRUE, then objects within the fake group will also be 
 *  considered.
 *  Otherwise, the fake group will be taken as a whole like all other groups.
 *
 *  Flipflop reverses the search on every other search to help avoid ties
 *  locking out an object. (of course, we are dead on a three-way tie!)
 */

static BOOLEAN flipflop = FALSE;
 
OBJECT *FindObject(x, y, fakeFlag)
	short x, y;
	BOOLEAN fakeFlag;
  {
    register OBJECT *p, *object;
    BOOLEAN gotit,inside;
    short pass;
    int area,maxarea;
    
    /* Initialize for no object found. */
    object = NULL;
    gotit = FALSE;
    flipflop = !flipflop;
    maxarea = 10000000;
    
    /* 
     *	Scan the active object list  on the first pass and the fake group on
     *  the second
     */
    for (pass = 0; pass<=fakeFlag; pass++) {
	if (pass) {
	    if (FakeGroup) {
		if (!CurrentObject)
		    printf("ERROR*** FakeGroup but CurrentObject == NULL\n\r");
		x -= CurrentObject->dx;
		y -= CurrentObject->dy;
		if (flipflop)
	            p = ((OBJECT_HEADER *)CurrentObject->part->data)->first;
		else
	            p = ((OBJECT_HEADER *)CurrentObject->part->data)->last;
		}
	    else
		p = NULL;
        }
	else
	    if (flipflop)
	    	p = activelist->first;
	    else
		p = activelist->last;
	while (p) {
	    if (fakeFlag && FakeGroup && p == CurrentObject) {
		if (flipflop)
		    p = p->next;
		else
		    p = p->prev;
		continue;
	    }
	    area = maxarea;
	    inside = PointInObject(p,x,y,FALSE,&area);
	    if (Debug&DebugFind) {
		printf("inside = %s area = %d\n\r",inside?"TRUE":"FALSE",area);
	    }
	    if (inside && (!gotit || area<maxarea || object==CurrentObject)) {
			/* Try not to re-select the current object ^^^^ */
			gotit = TRUE;
			maxarea = area;
			object = p;
		    }
	    if (flipflop)
	        p = p->next;
	    else
		p = p->prev;
	}
    }
    /* Debugging Information? */
    if (Debug&DebugFind)
      {
	if (object) 
	    printf("    Best area = %d, object  = %x\n\r", maxarea, object);
	else
	    printf("    no object found\n\r");
      }

    /* Report the results */
    return( object );
  }


/*
 *  This routine will find the closest sticky point to a given point.
 */
 
OBJECT *FindSticky(x, y, dx, dy)
	short x, y;
	short *dx, *dy;
  {
    register OBJECT *p, *object;
    int bestDistance, tempDistance;
    register short bestx, besty;
    
    /* Initialize for no object found. */
    bestDistance = MAXDIST+1;
    
    p = activelist->first;
	while (p) {
		    tempDistance = ObjectDistance( x, y, dx, dy,
				p->part, p->dx, p->dy, FALSE );
		    if (tempDistance < bestDistance) {
			bestx = *dx;
			besty = *dy;
			bestDistance = tempDistance;
			object = p;
		    }
	    p = p->next;
	}
    /* Debugging Information? */
    if (Debug&DebugFind)
      {
	printf("    Best distance = %d, object  = ", bestDistance);
	if (object)
	    printf("%x\n\r", object);
	else
	    printf(" <none found>\n\r");
      }

    /* Report the results */
    *dx = bestx;  *dy = besty;
    return( bestDistance<=MAXDIST ? object : NULL);
  }

