/*
 *  debug.c
 *
 *  This file contains routines useful for making debugging printouts.
 *
 *  David Kaelbling, May 1983
 */
 
 
/* Includes */
# ifdef UNIX
# include "stdio.h"
# else
# include "Vio.h"
# endif
# include "Vgts.h"
# include "splines.h"
# include "draw.h"
 
 
/* Imports */
 
 
/* Exports */
extern PrintObjType();
extern PrintTemplate();
extern PrintCommand();
extern DebugSpline();
extern DebugItemDescriptor();
extern DebugObjectList();
extern DebugLinks();
 
 
/* Local Definitions */

/*
 *  This routine will print a ObjTypes variable.
 */
 
PrintObjType( type )
	enum ObjTypes type;
  {
    printf("%d ", (int) type);
    switch (type)
      {
	case TextObj:		printf("(Text)");  break;
	case OpenSplineObj:	printf("(OpenSpline)");  break;
	case ClosedSplineObj:	printf("(ClosedSpline)");  break;
	case OpenPolygonObj:	printf("(OpenPolygon)");  break;
	case ClosedPolygonObj:	printf("(ClosedPolygon)");  break;
	case GroupObj:		printf("(Group)");  break;
	case TemplateObj:	printf("(Template)");  break;
	case SILObj:		printf("(SIL line)"); break;
	default:		printf("(unknown)");  break;
      }
  }

/*
 *  This routine will print a TemplateTypes variable.
 */
 
PrintTemplate( type )
	enum TemplateTypes type;
  {
    printf("%d ", (int) type);
    switch (type)
      {
	case NarArrowO:		printf("(Narrow Open Arrowhead)");  break;
	case NarArrowC:		printf("(Narrow Closed Arrowhead)");  break;
	case WideArrowO:	printf("(Wide Open Arrowhead)");  break;
	case WideArrowC:	printf("(Wide Closed Arrowhead)");  break;
	case Oval:		printf("(Oval)");  break;
	case Circle:		printf("(Circle)");  break;
	case Rectangle:		printf("(Rectangle)");  break;
	case NullTemplate:	printf("(Null Template)");  break;
	default:		printf("(Unknown)");  break;
      }
  }

/*
 *  This routine will print a MenuOptions variable.
 */
 
PrintCommand( cmd,x,y,but )
	enum MenuOptions cmd;
	short x,y,but;
  {
    printf("Event = ");
    switch (cmd)
      {
	case CNull:		printf("CNull");		break;
	case COpenPolygon:	printf("COpenPolygon");		break;
	case COpenCurve:	printf("COpenCurve");		break;
	case CWideOpenArrow:	printf("CWideOpenArrow");	break;
	case CWideClosedArrow:	printf("CWideClosedArrow");	break;
	case CNarrowOpenArrow:	printf("CNarrowOpenArrow");	break;
	case CNarrowClosedArrow: printf("CNarrowClosedArrow");	break;
	case CClosedPolygon:	printf("CClosedPolygon");	break;
	case CClosedCurve:	printf("CClosedCurve");		break;
	case CClosedCircle:	printf("CClosedCircle");	break;
	case CClosedElipse:	printf("CClosedElipse");	break;
	case CClosedRectangle:	printf("CClosedRectangle");	break;
	case CPressEditSymbol:	printf("CPressEditSymbol");	break;
	case CFilledPolygon:	printf("CFilledPolygon");	break;
	case CFilledCurve:	printf("CFilledCurve");		break;
	case CFilledCircle:	printf("CFilledCircle");	break;
	case CFilledElipse:	printf("CFilledElipse");	break;
	case CFilledRectangle:	printf("CFilledRectangle");	break;
	case CText:		printf("CText");		break;
	case CMove:		printf("CMove");		break;
	case CCopy:		printf("CCopy");		break;
	case CErase:		printf("CErase");		break;
	case CAlter:		printf("CAlter");		break;
	case CRotate:		printf("CRotate");		break;
	case CScale:		printf("CScale");		break;
	case CStretch:		printf("CStretch");		break;
	case CGroup:		printf("CGroup");		break;
	case CRaise:		printf("CRaise");		break;
	case CLower:		printf("CLower");		break;
	case COpaque:		printf("COpaque");		break;
	case CDone:		printf("CDone");		break;
	case CUndo:		printf("CUndo");		break;
	case CAbort:		printf("CAbort");		break;
	case CLoad:		printf("CLoad");		break;
	case CSave:		printf("CSave");		break;
	case CPrint:		printf("CPrint");		break;
	case CRange:		printf("CRange");		break;
	case CClear:		printf("CClear");		break;
	case CQuit:		printf("CQuit");		break;
	case CFont1:		printf("CFont1");		break;
	case CFont2:		printf("CFont2");		break;
	case CHelp:		printf("CHelp");		break;
	case CNib:		printf("CNib %d",x);		break;
	case CPattern:		printf("CPattern %d",x);	break;
	case CDataPoint:	printf("CDataPoint (%d, %d) but = %d",x,y,but);break;
	case CCenter:		printf("CCenter %d",x);		break;
	case CAgain:		printf("CAgain");		break;
	case CToggleSelect:	printf("CToggleSelect");	break;
	case CDebug:		printf("CDebug");		break;

	default:		printf("**unknown (%d)",cmd);	break;
      }
  }

/*
 *  This routine will print a spline data structure.
 */
 
DebugSpline( sptr, indent )
	SPLINE *sptr;
	int indent;
  {
    register POINT *p;
    register short i;
    
    if (sptr == NULL)
      {
	printf("%*s    NULL SPLINE DATA STRUCTURE\n\r", indent, "");
	return;
      }
    printf("%*s", indent, "");
    printf("    order %d, border %d, closed %d, filled %d, opaque %d\n\r",
	sptr->order, sptr->border, sptr->closed, sptr->filled, sptr->opaque);
    printf("%*s", indent, "");
    printf("    numvert %d, nib %d, fill pattern %d",
	sptr->numvert, (int) sptr->nib, (int) sptr->pat);
    p = &(sptr->head);
    for (i = 0;  i < sptr->numvert;  i++)
      {
        if ((i % 5) == 0)
	    printf("\n\r%*s    ", indent, "");
	printf("(%d, %d)  ", p[i].x, p[i].y);
      }
    printf("\n\r");
  }

/*
 *  This routine will print an item descriptor.
 */
 
DebugItemDescriptor( id, indent )
	PART *id;
	int indent;
  {
    if (id == NULL)
      {
	printf("%*s    NULL PART\n\r", indent, "");
	return;
      }
    if (id->filnum==DupCheck)
	return;
    id->filnum = DupCheck;    
    printf("%*s", indent, "");
    printf("    number %d, symbol %d, refs %d, prev %x, next %x\n\r",
	id->number, id->symbol, id->refs, id->prev, id->next );
    printf("%*s", indent, "");
    printf("    xmin %d, xmax %d, ymin %d, ymax %d\n\r",
	id->xmin, id->xmax, id->ymin, id->ymax );
    printf("%*s", indent, "");
    printf("    type ");
    PrintObjType( id->type );
    switch (id->type) {
	case TextObj:
	    printf(", font number %d, centering %d, descent %d\n\r",
		id->subtype, id->typedata, id->base);
	    printf("%*s", indent, "");
	    if (id->data)
		printf("    data '%s'\n\r", id->data );
	    else
		printf("    data NULL\n\r");
	    break;
	
	case OpenSplineObj:
	case ClosedSplineObj:
	case OpenPolygonObj:
	case ClosedPolygonObj:
	case SILObj:
	    printf("\n\r");
	    DebugSpline( (SPLINE *) (id->data), indent );
	    break;
	
	case GroupObj:
	    printf("  Items are:\n\r");
	    DebugObjectList("",(OBJECT_HEADER *)id->data, indent + 4 );
	    break;
	
	case TemplateObj:
	    printf("  subtype ");
	    PrintTemplate( id->subtype );
	    printf("\n\r");
	    DebugSpline( (SPLINE *) (id->data), indent );
	    break;
	
	default:
	    printf("(Unknown)\n\r");
	    break;
    }
    printf("\n\r");
  }    

/*
 *  This routine will dump an object list (eg. activelist).
 */
 
DebugObjectList( name, list, indent )
	char *name;
	OBJECT_HEADER *list;
	int indent;
  {
    OBJECT *ob;
    if (list == NULL)
      {
	printf("DebugObjectList:  NULL list specified.\n\r");
	return;
      }
      
    /* Maybe parameters are okay. */
    printf("%*s", indent, "");
    printf("Object list %x:  first = %x, last = %x\n\r",
		list, list->first, list->last);
    ob = list->first;
    while (ob)
      {
	if (ob == CurrentObject) 
	    printf("%s%*s",FakeGroup?">v>":"**>",indent<3?0:indent-3,"");
	else
	    printf("%*s", indent, "");
	if (ob->parent != list)
	     printf("*** Invalid parent %x should be %x\n\r",ob->parent,list);
	printf("%x: prev %x, next %x, part %x,\n\r%*s       frame %d, call %d at (%d, %d)\n\r",
		ob, ob->prev, ob->next, ob->part,
		indent,"",ob->frame,ob->call, ob->dx, ob->dy );
	if (ob->part)
	    DebugItemDescriptor( ob->part, indent );
	if (ob == CurrentObject && FakeGroup)
	    printf("\n\r>^>\n\r");
	ob = ob->next;
      }
  }

/*
 *  This routine will print a short form of an ObjectList.
 */
 
DebugLinks( head )
    OBJECT_HEADER *head;
  {
    short i;
    OBJECT *ob;
    extern DUMP();
    PFI dump = DUMP;	/* Dummy to create reference for linker */
    
    /* Print the header information */
    if (head == NULL)
	printf("DebugLinks:  NULL object header.\n\r");
    printf("Object header:  first = %x, last = %x", head->first, head->last);
    
    /* Print the links. */
    i = 0;
    ob = head->first;
    while (ob)
      {
	if (i++ % 4 == 0)
	    printf("\n\r    ");
	if (ob->parent != head)
	     printf("*** Invalid parent %x should be %x\n\r",ob->parent,head);
	printf("(%x -- %x)   ", ob->prev, ob->next);
	ob = ob->next;
      }
    printf("\n\r");
  }


/* 
 * This procedure is never called but can be invoked by the debugger to
 * dump the main data structures.
 */
DUMP()
{
    long *crow;
    printf("CurrentObject = %x; FakeGroup = %s\n\r",CurrentObject,
	FakeGroup?"TRUE":"FALSE");
    DupCheck++;
    DebugObjectList("",activelist,4);
    printf("\n\r\n\r...Jumping to debugger...\n\r\n\r");
    crow = (long *)0xdead;
    *crow = 0;
}
    
