/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer, Eric Berglund,  5/83
 */

/*
 * This file, symbols.c, is part of the V debugger.  It contains the routines
 * needed to find and return the symbol table entries stored on the file
 * produced by ld.  The symbols themselves are cached for quicker access so
 * major portions of the code deal with handling of the cache.  This file
 * includes InitCache, QueryCache, FindSymByValue, FindSymByName, GetSymbolRec,
 * PrintSym, and FindRegister.
 */

#include <Venviron.h>
#include <b.out.h>
#include <Vio.h>
#include "Vdb.h"

extern ElementOf(), bufput(), numout(), getlong();
extern File *ReOpenFileIfNecessary();


/*
 * Declarations for a self-organizing symbol table cache.
 * The cache is maintained as a linked list of records which contain
 * symbol value ranges and the associated symbol table entry.
 * All references result in placing the cache entry at the beginning of the
 * list.
 */

typedef struct
  {
    unsigned long lowAddress;
    unsigned long highAddress;
    unsigned long offset;         /* offset in symbol table segment of debuggee */
    unsigned long next;
    SymRec symEntry;
  } CacheType;

CacheType Cache[CacheSize];

unsigned long HighAddress;
int NumberCacheHits = 0, NumberCacheMisses = 0;
int NumCallsByVal = 0, NumCallsByName = 0;

SymRec UsrSymbol;


InitCache()
  {
    int i;

    for (i = 0; i < CacheSize; i++)
      {
	Cache[i].lowAddress = 0;
	Cache[i].highAddress = 0;
	Cache[i].next = i + 1;
      }
    Cache[CacheSize-1].next = 0;
  }




QueryCache()
  {
    printf("Cache hits: %d,    Cache misses: %d,    Total: %d\n",
    	NumberCacheHits, NumberCacheMisses, 
	NumberCacheHits + NumberCacheMisses);
    if( NumberCacheHits > 0 )
	printf("Cache hit ratio: %d\n", 
	 (NumberCacheHits * 100) / (NumberCacheHits + NumberCacheMisses));
    printf("Number of calls to FindSymByValue: %d\n", NumCallsByVal);
    printf("Number of calls to FindSymByName: %d\n", NumCallsByName);
  }



/*  FindSymByValue  is passed a long value "v" and
 *  a pointer to a file containing the symbol table. It searches the
 *  file linearly for the symbol whose value is closest to, but not
 *  greater than, v.  It then returns a pointer to a sym structure which
 *  contains the type, length, and value of the newfound symbol and a pointer
 *  to a string containing the symbol's name.
 *  
 *  The routine also discovers the offset of the symbol table entry from the
 *  beginning of the file.  PrintSym can then use that entry to
 *  to find the symbol in the file and copy the information.
 *
 *  In the next generation this should be changed to read in a block
 *  of the symbol table at a time and search through that.  It would
 *  help if the symbols were kept in numerical order, however.
 */


SymRec *FindSymByValue ( v, symfileptr)
    register unsigned long v;
    File *symfileptr;
  {
    register int current, best;			/* Offsets in symbol file. */
    unsigned long bestvalue, currentvalue;
    register int i, j, k;

    v = (unsigned long)FixAddress(v);
    if ((v < (unsigned long) LinkOrigin) ||
	(v > (unsigned long) (DebugeeTeamSize-LoadOrigin+LinkOrigin)))
				/* Is this value even reasonable? */
      return(NULL);

    NumCallsByVal += 1;
    /*
     * Look for v in the cache.
     */
    for (k = -1, j = 0, i = Cache[0].next; i != 0; 
    		k = j, j = i, i = Cache[i].next)
      { int cacheLo = Cache[i].lowAddress, cacheHi = Cache[i].highAddress;
	DoFixAddress(&cacheLo); DoFixAddress(&cacheHi);
	if (cacheLo <= v && v < cacheHi)
	  {
	    bestvalue = (unsigned long)FixAddress(Cache[i].lowAddress);
	    best = Cache[i].offset;
	    NumberCacheHits += 1;
	    Cache[j].next = Cache[i].next;
				/* Place cache hit entry at front of cache. */
	    Cache[i].next = Cache[0].next;
	    Cache[0].next = i;
	    break;
	  }
      }
    if (i == 0)			/* Couldn't find v in the cache. */
      {
        NumberCacheMisses += 1;
	symfileptr = ReOpenFileIfNecessary( symfileptr );
	bestvalue = 0;
	fseek( symfileptr, N_SYMOFF(filhdr), 0 );
				/* Start at the beginning of the file */
	HighAddress = (unsigned long) (DebugeeTeamSize-LoadOrigin+LinkOrigin);
    
	for (current = 0; current < filhdr.ssize; 
			current += sizeof (struct nlist))
	  {						  
	    int type;
	    fseek( symfileptr, 4, 1 );          /* Skip n_un.n_strx */
	    type = getc( symfileptr ) & (N_TYPE | N_EXT);
	    fseek( symfileptr, 3, 1 );          /* Skip n_other & n_desc */
	    currentvalue = getlong( symfileptr );
	    if (! (type & N_EXT))   /* Skip file names, etc. */
		continue;
	    DoFixAddress(&currentvalue);
	    switch(type & N_TYPE)
	      {
		case N_TEXT:
		case N_BSS:
		case N_DATA:
		    break;

		default:
		    continue;       /* Skip anything else */
	      }
	    if (bestvalue < currentvalue && currentvalue <= v)
	      {
		best = current;
		bestvalue = currentvalue;
	      }
	    if ((v < currentvalue) && (currentvalue < HighAddress))
		HighAddress = currentvalue;
	  }
	/* j points to the last record on the cache list. */
	Cache[j].highAddress = HighAddress;
	Cache[j].offset = best;
	GetSymbolRec(&(Cache[j].symEntry), symfileptr, best);
	Cache[j].lowAddress = Cache[j].symEntry.symbol.n_value;
	Cache[k].next = 0;	/* Put cache entry at front of cache. */
	Cache[j].next = Cache[0].next;
	Cache[0].next = j;
      }

    /* The first entry in the cache now contains the right symbol. */
    if (bestvalue != 0 && v - bestvalue < LargestOffset)
      {
	offset = best;
	return(&(Cache[Cache[0].next].symEntry));
      }
    else
	return(NULL);
  }




/*
 * Buffering routines for the string table.  In an a.out file, the symbol names
 *   live in a string table which comes after the symbol table.  Dumb symbol
 *   lookup thus involves scanning through the symbol table, and following the
 *   pointer from each symbol-table entry to somewhere in the string table.
 *   If we use file seek()s all the time, this is hopelessly inefficient.
 *   Instead, we keep blocks of the string table in memory.  It's a pity to be
 *   sort-of duplicating the buffering in the file I/O routines, but it pays
 *   off.  Besides, here we can choose any number of parallel block buffers.
 * One could be a lot more clever than this: for instance, read in a number of
 *   symbol-table entries, sort them according to their indices into the string
 *   table, and then look up all their corresponding strings.  Or gradually 
 *   build a sort-of hash table which could say "the string at offset n in the
 *   string table does NOT match the one you want"
 */

#ifndef SBUF_NBUFS
#define SBUF_NBUFS 4
#endif
#ifndef SBUF_BUFLOG
#define SBUF_BUFLOG 8
#endif
#define SBUF_BUFSIZE (1<<SBUF_BUFLOG)

int  sBuf_valid = 0;
long sBufOffset[SBUF_NBUFS];
char sBufData  [SBUF_NBUFS][SBUF_BUFSIZE];

/*
 * sBuf_fetch: gets a bit of the string table and returns a pointer and length.
 *   Inputs: symfileptr	  = file descriptor for the a.out file
 *           stringoffset = offset in string table of the string we want
 *	     wantlength   = the number of bytes we'd like
 *   Outputs:sBuf_ptr	  = pointer to start of requested string
 *	     (return value) = number of bytes actually present.  May be less
 *			      than wantlength if the string crosses a buffer
 *			      boundary.
 * This is a fairly simple and efficient interface.  It is @i(lousy) if there
 * is any chance that multiple processes may want to use the string table.
 * Even with one process, it's a bit dubious: calling sBuf_fetch may destroy
 * the contents of any buffer.
 */

int sBuf_fetch(symfileptr, stringoffset, wantlength, sBuf_ptr)
    File *symfileptr;
    unsigned long stringoffset;
    int  wantlength;
    char **sBuf_ptr;
  {
    int i;
    unsigned long j;
    static int replaceNext;

    if (! sBuf_valid)
      {
	for (i=0; i<SBUF_NBUFS; i++)
	    sBufOffset[i] = -1;
	replaceNext = 0;
	sBuf_valid = 1;
      }
    j = stringoffset & ~(SBUF_BUFSIZE-1);
    for (i=0; i<SBUF_NBUFS; i++)
	if (sBufOffset[i] == j)
	    break;
    if (i == SBUF_NBUFS)
      {
	for (i=0; i<SBUF_NBUFS; i++)
	    if (sBufOffset[i] == -1)
		break;
	if (i == SBUF_NBUFS)
	    i = replaceNext++ % SBUF_NBUFS;  /* i.e. just FIFO replacement */
	sBufOffset[i] = j;
	fseek(symfileptr, N_STROFF(filhdr) + j, 0);
	fread(sBufData[i], SBUF_BUFSIZE, 1, symfileptr);
      }
    *sBuf_ptr = sBufData[i] + (stringoffset & (SBUF_BUFSIZE-1))  ;
    return   ( SBUF_BUFSIZE - (stringoffset & (SBUF_BUFSIZE-1)) );
  }
    
    

/* Given a symbol name, FindSymByName finds its type, length, and value.
 *  A ptr to the stored symbol entry is returned.
 *
 * At present this completely ignores the so-called cache used by 
 *   FindSymByValue; we neither search it nor write new values there.
 *   Should we?
 */


struct nlist *FindSymByName ( name, symfileptr )
    char *name;
    File *symfileptr;
  {
    register unsigned long current;          /* Current offset from SYMPOS. */
    int found_;                     /* True if near match was found */
    unsigned long current_;                  /* Offset to near-match entry */
    int found, i, found1;
    int length;
    unsigned long bestvalue, currentvalue;

    NumCallsByName += 1;
    
    found_ = 0;
    for(current = 0; current < filhdr.ssize;
	   current += sizeof(struct nlist))

      {						  
	int type;
	unsigned long stroff;                  /* Offset into string table */
	int sliceTotal, sliceCount;
	char c, *sliceChars;

	symfileptr = ReOpenFileIfNecessary( symfileptr );
	/* Go to current entry */
	fseek( symfileptr, N_SYMOFF(filhdr) + current, 0 );
	stroff = getlong( symfileptr );     /* Get n_un.n_strx */
	type = getc( symfileptr ) & (N_TYPE | N_EXT);
	if (! (type & N_EXT))   /* Skip file names, etc. */
	      continue;
	  /*
	   * Search for either foo or _foo so the user doesn't have to
	   * keep remembering to type in the stupid underscore.
	   * found will be true until a mismatch occurs.
	   * Similarly for found1 but it keeps track of the name with an
	   * underscore at the beginning.
	   * This routine should probably be rewritten to read in the whole
	   * symbol and then just do two comparisons.
	   */
	length = strlen(name);
	sliceTotal = 0;
	sliceCount = 0;
	found = 1;    /* Assume equal until mismatch found */
	found1 = 0;   /* Don't assume equal unless first char is _ */
	for (i = 0; i < length + 1 ; i++)	/* + 1 for the trailing \0 */
	  {
	    if (! sliceCount)	
	      {
		sliceCount = sBuf_fetch(symfileptr, stroff+sliceTotal, 
					length+2-sliceTotal, &sliceChars);
		/* "length+2" : we add 1 for trailing \0, 1 in case of "_" */
		sliceTotal += sliceCount;
	      }
	    c = *sliceChars++, sliceCount--;
	    if (name[i] != c)
		found = 0;  /* Not a direct match */
	    if (i == 0)
	      {
		if (c == '_')   /* Check for _ as first character */
		    found1 = 1;
	      }
	    else
	      {
		if (name[i-1] != c)
		    found1 = 0;   /* Not a near-match */
	      }
	    if (!found && !found1)
		break;      /* No point to looking any further */
	  }
	if( found )       /* If this was a direct match, all done */
          {
	    GetSymbolRec(&UsrSymbol, symfileptr, current);
	    return(&(UsrSymbol.symbol));
          }
	  /*
	   * If this was a near match (equal except for the leading
	   * underscore, keep searching in case there is a true match.
	   */
	if( found1 && *sliceChars == '\0' )
	  {
	    current_ = current; /* Keep track of its location */
	    found_ = 1;
	  }
      }

    if (found_)
      {
	GetSymbolRec(&UsrSymbol, symfileptr, current_);
	return(&(UsrSymbol.symbol));
      }

    return(NULL);                     /* Not found */
  }


/*
 * GetSymbolRec:
 * Stores the designated symbol table entry at symptr.
 * Adjusts the symbol type for the debugger's tastes.
 */

GetSymbolRec(symptr, symfileptr, offset)
    SymRec *symptr;
    File *symfileptr;
    int offset;
  {
    int i, j;
    unsigned long stroff;    /* Offset into string table for this symbol's name */
    int sliceCount, sliceTotal;
    char *sliceChars;

    fseek(symfileptr, N_SYMOFF(filhdr) + offset, 0);
    stroff = getlong( symfileptr ); /* Index into string table */
    symptr->symbol.n_type = getc(symfileptr);  /* Get n_type */
    fseek( symfileptr, 3, 1 );          /* Skip n_other & n_desc */
    symptr->symbol.n_value = getlong(symfileptr);    /* Get n_value */
    switch (symptr->symbol.n_type & N_TYPE)
      {
	case N_TEXT:  symptr->stype = INSTTYPE; break;
	case N_BSS:
	case N_DATA:  symptr->stype = LONGTYPE; break;
	default:      symptr->stype = NULLTYPE; break;
      }
    sliceTotal = 0;
    for (i = 0; i < MaxSymLength-1; )
      {
	sliceCount = sBuf_fetch(symfileptr, stroff+sliceTotal, 
				MaxSymLength-sliceTotal, &sliceChars);
	sliceTotal += sliceCount;
	for (j = 0; i < MaxSymLength-1 && j < sliceCount; i++, j++)
	    if ( (symptr->symName[i] = *sliceChars++) == '\0' )
		goto copy_done;
      }
copy_done:
    symptr->slength = i;
    symptr->symName[i] = '\0';
  }




/* Given a value "v", look for a symbol whose value is 'closest' to "v".
 * If one is found, output the name of that symbol, otherwise, output the
 * numeric value of "v" in the current radix.
 */

PrintSym ( v, fileptr )
    register unsigned long v;
    File *fileptr;
  {
    SymRec *symptr;
    int i; unsigned long svalue;

    symptr = FindSymByValue(v, fileptr);
    if (symptr)
      {
	for (i = 0; symptr->symName[i] != '\0'; i++)
				/* Print the symbol. */
	    bufput(symptr->symName[i]);

	svalue = symptr->symbol.n_value;
	v = (unsigned long)FixAddress(v); DoFixAddress(&svalue);
        if (svalue < v) 
          {
	    bufput('+');	/* And any offset from it. */
            numout(v - svalue);
          }
      }
    else
      numout(v);
  }


/*
 * Look up a register designator string in the Vdb symbol table. 
 * Return a pointer to the appropriate location storing the value of
 * the designated register.
 */
unsigned long *FindRegister(string)
    char *string;
{
    int reg_num;                /* Register number */
    static unsigned long psw;            /* Temporary storage for psw */

    if (string[0] != '%')
	return(NULL);
    string++;                   /* Skip over the % */
    if (sscanf(string, "r%d", &reg_num) == 1 &&
	  reg_num >= 0 && reg_num <= 15)
	return (unsigned long *) &SavedRegisters[reg_num];
    if (strcmp(string, "sp") == 0)
	return (unsigned long *) &SavedRegisters[SP];
    if (strcmp(string, "pc") == 0)
	return (unsigned long *) &SavedRegisters[PC];
    if (strcmp(string, "ap") == 0)
	return (unsigned long *) &SavedRegisters[AP];
    if (strcmp(string, "fp") == 0)
	return (unsigned long *) &SavedRegisters[FP];
    if (strcmp(string, "psl") == 0)
	return (unsigned long *) &SavedPSL;
    if (strcmp(string, "psw") == 0)
    {
	psw = SavedPSL & 0x0000ffff;    /* PSW is low half of PSL */
	return (unsigned long *) &psw;
    }
    return (unsigned long *) NULL; 	/* Not a register designator */
}

