/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer, Eric Berglund,  5/83
 */

/* This file, commands.c, is part of the V debugger, Vdb.  It contains the
 * loop for processing commands and several of the higher level routines
 * which implement commands.  This is the heart of the control structure
 * of Vdb.
 *
 * Routines included here:  CommandLoop, IncrememtPc, DecrementPc,
 * DisplayMemLoc, GetValToCompare, PrintHelp, and QueryDebugger.
 */


#include <b.out.h>
#include <Vio.h>
#include <Venviron.h>
#include "Vdb.h"

extern long *FindRegister();
extern enum DbCmds ParseToGetCommand();
memaddr IncrementDot(), DecrementDot();
extern ExitDebugger(), SetBreakpoint(), RemoveBreakpoint(), PrintBreakpoints(),
	PutMem(), TypeOut(), DisplayMemLoc(), ReadInputLine(),
	ParseToGetArg(), ParseForSymbol(), StackDump(),
	printnum(), ValidInstrPc(), ValidAddress();


/* Values used in the search command are "sticky"--remain the same
 * until changed by the user, and thus must be saved when control is
 * returned to the user program.
 */
long mask = ~0;			/* search mask */
long searchvalue;		/* search value */
memaddr lowlimit, hilimit;      /* search limits */

char InputBuffer[80];		/* Buffer used to hold user's last input
				   line. */

char *DebuggerSymbolNames[] =
  {
    "r0", "r1", "r2", "r3", "r4", "r5", "r6", "r7",
    "r8", "r9", "r10", "r11", "ap", "fp", "sp", "pc",
  };

#define NUM_DEBUG_SYMS (sizeof DebuggerSymbolNames / sizeof DebuggerSymbolNames[0])


/*
 * Process commands until we get a "go" or "continue".
 */
CommandLoop(bpno)
    short *bpno;		/* Used for index of breakpt. set at
				   current pc, if any.  0 if none set. */
  {
    char c, go = 0;
    short nbytes = 0;
    short inc;
    memaddr adrs;
    enum stype depType;
    enum stype watchmode;
    long test;
    enum stype searchmode;
    short bpnum;
    enum DbCmds cmd;
    int len, i;
    long arg1, arg2, arg3, arg4;
    int flag1, flag2, flag3, flag4;
    enum stype type1;
    char symbol[5];
    long *regadr;
    long savValue;
    SystemCode error;
    memaddr saveDot, startDot;
    long tmask;				/* temporary mask used in search */
    char disassembledCode[MAX_CODE_LINE_LENGTH];

    SsFlag = 0;				/* out of single step mode */
    SsSbrFlag = 0;
    strcpy( disassembledCode, OutputBuffer.buf );
    TempTypeOutMode = NULLTYPE;

    while (!go)
      {
	putchar('.');
	Flush(stdout);

	len = ReadInputLine(InputBuffer);
	if( len == -1 )
	  {
	    fprintf( stdout, "Debugger: Hit EOF on input too many times\n");
	    ExitDebugger();
	  }

  	cmd = ParseToGetCommand(InputBuffer, len);
	flag1 = ParseToGetArg(InputBuffer, 1, &arg1);
	type1 = SymType;
	flag2 = ParseToGetArg(InputBuffer, 2, &arg2);
	flag3 = ParseToGetArg(InputBuffer, 3, &arg3);
	flag4 = ParseToGetArg(InputBuffer, 4, &arg4);
	if ((flag1 == -1) || (flag2 == -1) || (flag3 == -1) || (flag4 == -1))
	  {
	    fprintf( stdout, "Debugger: Invalid symbol in arguments.\n");
	    Flush( stdout );
	    continue;
	  }

	switch(cmd)
	  {
	    case NoCmd:
		break;
	    case DisplayNextCmd:
	    case DisplayPrevCmd:
	    case DisplayPtrCmd:
		if (flag1)
		  {
		    Dot = (memaddr) arg1;
/*		    putchar('\n');		*/
		    if (cmd == DisplayPtrCmd)
		      {
			saveDot = Dot;
			Dot = (memaddr) GetMemLong(Dot, &error);
			if( error != OK )
			  {
			    fprintf( stdout, "Debugger: Value of Dot is illegal address.\n" );
			    Dot = saveDot;
			  }
		      }
		  }

		else
		  {
		    switch (cmd)
		      {
			case DisplayNextCmd:

			    Dot = IncrementDot(Dot, nbytes);
			    break;
			case DisplayPrevCmd:
			    Dot = DecrementDot(Dot, nbytes);
			    /* The default decrement of nbytes is a little
			     * weird, but as good as anything else. */
			    break;
			case DisplayPtrCmd:
			    saveDot = Dot;
			    Dot = (memaddr) GetMemLong(Dot, &error);
			    if( error != OK )
			     {
				fprintf( stdout, "Debugger: Content of Dot is illegal address.\n" );
			        Dot = saveDot;
			      }
			    break;
		      }
		  }

		putchar('\r');
		for (i = 0; i < 70; i++)
		    putchar(' ');
		putchar('\r');
		putchar('.');
		nbytes = DisplayMemLoc();
		if (nbytes == -2)
		  {
		    fprintf( stdout, "Debugger: Illegal address specified.\n");
		    nbytes = 0;
		  }
		break;

	    case DisplayNextPageCmd:
	    case DisplayPrevPageCmd:
		if (flag1)
		    Dot = (memaddr) arg1;
		if (flag2)
		    len = arg2;
		else
		    len = PAGE_LENGTH;

		if( cmd == DisplayPrevPageCmd )
		  {
		    startDot = Dot;
		    nbytes = 4;		/* Default typeout length.*/
		    for( i = 1; i < len; i++ )
		      {
			startDot = DecrementDot( startDot, nbytes );
		      }
		    Dot = startDot;
		  }

		nbytes = DisplayMemLoc();
		if (nbytes == -2)
		  {
		    fprintf( stdout, "Debugger: Illegal address specified.\n");
		    break;
		  }

		for (i = 1; i < len; i++)
		  {
		    Dot = IncrementDot(Dot, nbytes);
		    nbytes = DisplayMemLoc();
		    if (nbytes == -2)
		      {
			fprintf( stdout, "Debugger: Illegal address specified.\n");
			break;
		      }
		  }

		if( cmd == DisplayPrevPageCmd )
		  Dot = startDot;

		break;

	    case DisplayEqCmd:
	        if (flag1)
		    printnum(arg1);
		else
		    printnum(Dot);
		putchar('\n');
		break;

	    case DisplayRegCmd:
		for (i = 0; i < 8; i++)
		  {
		    fprintf( stdout,"%s ", DebuggerSymbolNames[i]);
		    inc = strlen(DebuggerSymbolNames[i]);
		    inc += printnum(SavedRegisters[i]);
		    while (inc++ < 25)
			putchar(' ');
		    fprintf( stdout,"%s ", DebuggerSymbolNames[i+8]);
		    printnum(SavedRegisters[i+8]);
		    putchar('\n');
		  }
		fprintf(stdout, "psl %08x\n", SavedPSL);
	        putchar('\n');
		break;


	    case ReplaceCmd:
		if (flag1)
		  {
		    if (flag3)
			depType = GiveTypeOutMode(arg3);
		    else
		      {
			SetTypeOutMode(arg1);
			depType = ActualTypeOutMode;
		      }
		    if( depType == STRTYPE )
		      {
			fprintf( stdout,"Cannot replace strings, just characters.\n");
		      }
		    else
		      {
			if (flag2)
		            nbytes = PutMem(depType, arg1, arg2);
			else
			    nbytes = PutMem(depType, arg1, 0);
			if (nbytes == -2)
			    fprintf( stdout, "Debugger: Illegal address specified for replacement.\n");
		      }
		  }
		else
		    fprintf( stdout, "Debugger: No memory location specified for replacement.\n");
		break;

	    case ReplaceRegCmd:
		ParseForSymbol(InputBuffer, symbol, 5);
		regadr = FindRegister(symbol);
		if (regadr)
		  {
		    if (flag2)
		      if( ( regadr == (long *)&ProgramCounter ) && !ValidInstrPc( arg2 ) )
			fprintf( stdout, "Debugger: Invalid address specified for pc.\n" );
		      else
			*regadr = arg2;
		    else
			*regadr = 0;
		  }
		else
		    fprintf( stdout, "Debugger: No register specified for replacement.\n");
		break;


	    case GoCmd:
		if (!IsRunnable()) break;
	        if (flag1)
		    if (ValidInstrPc(arg1))
			ProgramCounter = (long) arg1;
		    else
		      {
		        fprintf( stdout, "Debugger: Invalid starting address specified.\n");
			break;
		      }
		go = 1;
		break;

	    case GoBrkptCmd:
		if (!IsRunnable()) break;
	        if (cbptr)	/* Inside a break point? */
		    if (flag1)
		        cbptr->count = arg1;
		    else
			cbptr->count = 1;
		go = 1;
		break;


	    case SingleStepSbrCmd:
		if (!IsRunnable()) break;
	        SsSbrFlag = 1;
		/* NO break */

	    case SingleStepCmd:
		if (!IsRunnable()) break;
	        SsFlag = 1;
		if (flag1)
		    SsCount = arg1;
		else
		    SsCount = 1;
		go = 1;
		break;


	    case BpCmd:
		if (!IsRunnable()) break;
	        if (flag1)
				/* Remove or set a breakpoint. */
		  {
		    if (flag2)
		      {
			if ((arg2 <= 1) || (arg2 >= BPMAX))
			  {
			    fprintf( stdout, "Debugger: Invalid breakpoint number specified.\n");
			    break;
			  }
		      }
		    else
			arg2 = ANY;
		    if (arg1 != 0)
				/* Set a breakpoint. */
		      {
			if (!ValidInstrPc(arg1))
			  {
			    fprintf( stdout, "Debugger: Invalid instruction address given.\n");
			    break;
			  }
			bpnum = SetBreakpoint(arg2, arg1);
			if (arg1 == (long) ProgramCounter)
			    *bpno = bpnum;
				/* Signal that a breakpoint was set at the
				   current pc. */
		      }
		    else	/* Remove a breakpoint. */
		      {
		        RemoveBreakpoint(arg2);
			if ((arg2 == ANY) || (arg2 == *bpno))
			    *bpno = 0;
		      }
		  }
		else		/* Display breakpoints. */
		    PrintBreakpoints();
		break;


	    case SearchCmd:
	        if (flag1)
		  {
		    searchvalue = arg1;
		    if (flag4)
			searchmode = GiveTypeOutMode(arg4);
		    else
			searchmode = type1;
		  }

		if (flag2)
		  if( ValidAddress( arg2 ) )
		    lowlimit = (memaddr) arg2;
		  else
		    {
		      fprintf( stdout, "Debugger: Lower limit is invalid address.\n" );
		      break;
		    }

		if (flag3)
		  if( ValidAddress( arg3 ) )
		    hilimit = (memaddr) arg3;
		  else
		    {
		      fprintf( stdout, "Debugger: Upper limit is invalid address.\n" );
		      break;
		    }

		inc = ( searchmode == CHARTYPE || 
			searchmode == BYTETYPE ||
			searchmode == STRTYPE)?    1: 2;
		tmask = mask;

		if (searchmode == STRTYPE)
		  {
		    savValue = searchvalue;
		    searchvalue = (long) (*((char *) searchvalue));
				/* searchvalue points to user-specified
				   string - get the first byte. */
		    tmask = 0xff;
				/* Mask to look just at a byte. */
		  }

		for ( adrs = lowlimit; adrs < hilimit; adrs += inc)
		  {
		    test = GetValToCompare( searchmode, adrs);
		    if ( ( test & tmask ) == ( searchvalue & tmask ) )
		      {
			if (searchmode == NULLTYPE)
			  {
			    SetTypeOutMode(adrs);
			    TypeOut(ActualTypeOutMode, adrs);
			  }
			else if (searchmode == STRTYPE)
			  {
			    int len;
			    memaddr memAdr;
			    char *strPtr;

			    len = strlen(savValue);
			    memAdr = adrs + 1;
			    strPtr = (char *) (savValue + 1);
			    for (i = 1; i < len; i++)
				if (*strPtr++ != GetMemByte(memAdr++,&error)
					|| error != OK )
				    break;
			    if (i == len)
				TypeOut(searchmode, adrs);
			    else
				continue;
			  }
			else
			    TypeOut( searchmode, adrs );
			putchar('\n');
		      }
		  }
		if (searchmode == STRTYPE)
		    searchvalue = savValue;
		break;


	    case MaskCmd:
	        if (flag1)
		    mask = arg1;
		else
		    mask = 0;
		break;


	    case IRadixCmd:
	        if (flag1)
		    if( arg1 > 1 && arg1 < 26L )	/* possible radix ? */
			InputRadix = (short) arg1;
		    else
			InputRadix = DEFRADIX;
		else
		    InputRadix = DEFRADIX;
		break;

	    case ORadixCmd:
	        if (flag1)
		    if( arg1 > 1 && arg1 < 26L )	/* possible radix ? */
			OutputRadix = (short) arg1;
		    else
			OutputRadix = DEFRADIX;
		else
		    OutputRadix = DEFRADIX;
		break;


	    case TmpTypeCmd:
	        if (flag1)
		  {
	            TempTypeOutMode = GiveTypeOutMode(arg1);
		    if ((TempTypeOutMode == STRTYPE) && flag2)
			StringLength = arg2;
		  }
		else
		    TempTypeOutMode = GiveTypeOutMode(0);

		break;

	    case TypeCmd:
	        if (flag1)
		  {
	            TypeOutMode = GiveTypeOutMode(arg1);
		    if ((TypeOutMode == STRTYPE) && flag2)
			StringLength = arg2;
		  }
		else
		    TypeOutMode = GiveTypeOutMode(0);
		TempTypeOutMode = NULLTYPE;
		break;


	    case StackTraceCmd:
		if( flag1 )
		  StackDump( disassembledCode, arg1 );
		else
		  StackDump( disassembledCode, 0 );
		break;


	    case OffsetCmd:
		if( flag1 && arg1 > 0 )
		  LargestOffset = arg1;
		else
		  LargestOffset = DEFOFFSET;
		break;

	    case SymLengthCmd:
		if( flag1 && arg1>0 && arg1<=MAXSYMLEN )
		  MaxSymLength = arg1;
		else
		  MaxSymLength = DEF_MAXSYMLEN;
		InitCache();
		break;

	    case StopProcessCmd:
		if (!IsRunnable()) break;
		StopProcessFlag = !StopProcessFlag;
		if( StopProcessFlag )
		  fprintf( stdout,"Only the process with the exception will halt.\n");
		else
		  fprintf( stdout,"The whole team will halt on an exception. \n");
		break;


	    case HelpCmd:
		PrintHelp();
		break;


	    case WatchCmd:
	    case WatchBrkptCmd:
		if (!IsRunnable()) break;
	        if (flag1)	/* any args? */
		  {
		    if (arg1)	/* set or clear? */
		      {
		        if (flag2)
			    watchmode = GiveTypeOutMode(arg2);
			else
			    watchmode = type1;
			if (watchmode == STRTYPE || watchmode == INSTTYPE)
			  {
			    fprintf(stdout, "Can't watch strings or instruction\n");
			  }
			else
			  {
			    OldWatchValue = GetMem(watchmode, arg1, &error);
			    if (error != OK)
			        fprintf(stdout, "Invalid address given\n");
			    else
			      {
			        fprintf(stdout, "Current contents:\n");
				TypeOut(watchmode, arg1);
				fprintf(stdout, "\n");
				WatchFlag = 1;
				WatchBrkptFlag = (cmd == WatchBrkptCmd);
				WatchType = watchmode;
				WatchAddr = (memaddr)arg1;
			      }
			  }
		      }
		    else
		      WatchFlag = 0;
		  }
		else		/* query */
		  {
		    if (WatchFlag)
		      {
		        fprintf(stdout, "Currently watching%s:\n",
				WatchBrkptFlag ? " (breakpoints only)" : "");
			TypeOut(WatchType, WatchAddr);
			fprintf(stdout, "\n");
		      }
		    else
		        fprintf(stdout, "Not watching anything.\n");
		  }
		break;

	    case QuitCmd:
		ExitDebugger();
		break;

	    case StatsCmd:
		QueryDebugger();
		break;


	    case BadCmd:
	    default:
		fprintf( stdout, "Debugger: Command not understood.\n");
		break;
	  }
      }
  }

/*
 * Check if program is runnable and complain if not.
 */
int IsRunnable()
  {
    if (LinkOrigin != LoadOrigin)
      {
        fprintf(stdout,
	    "Debugger: Program not runnable; not loaded at link origin.\n");
	return (0);
      }
    else
        return (1);
  }


/*
 * Here to increment pc.
 */
memaddr IncrementDot( pc, inc )
	register memaddr pc;
	register int inc;
  {
    return( (memaddr) pc + inc );
  }


/*
 * Function to decrement pc. Returns pc - amount decremented.
 *	 Notice that decrementing over instructions is troublesome
 * because instructions can almost any length.  The routine
 * attempts to find the shortest possible instruction that could
 * have ended at pc, up to an arbitrary maximum of 20 bytes.
 */

memaddr DecrementDot( pc, dec )
	register memaddr pc;
	register int dec;
  {
    register int nbytes, i;
    int legal;
    SystemCode error;
    memaddr pos;

    if ((ActualTypeOutMode == INSTTYPE) && (dec > 0))
      {
	for( i = 1; i < 20; i++)
	  {
	    pos = pc - i;
	    if( ValidAddress( pos ) )
	      {
		nbytes = dasm(pos, Symfile, &legal);
						/* attempt to disassemble */
		if( nbytes == i && legal )
		    return( pos );
	      }
	  }
      }

    return( pc - dec );         /* If all else fails, do as instructed */
  }



/*
 * Here to get data from memory for search comparison
 * NOTE: for type STRTYPE this routine just returns the first byte stored at
 * pc so that an initial comparison can be made.
 */

long GetValToCompare( type, pc )
	memaddr pc;
	enum stype type;
  {
    SystemCode error;		/* pc has already been checked for validity */

    switch (type)
      {
	case NULLTYPE:          /* Default search mode */
	case LONGTYPE:
	    return((long)GetMemLong( pc, &error));
	case CHARTYPE:
	case BYTETYPE:
	    return((long)GetMemByte( pc, &error));
	case WORDTYPE:
	    return((long)GetMemWord( pc, &error));
	case INSTTYPE:
	    return((long) GetInstrWord( pc, &error ));
	case STRTYPE:
	    return((long)GetMemByte( pc, &error));
      }
  }


/*
 * Print command summary.
 */

PrintHelp()
  {
    char c;

    fprintf( stdout,"\nDebugger Commands:\n");
    fprintf( stdout," expression/\n");
    fprintf( stdout," expression\\    Set dot to expression and display its contents.\n");
    fprintf( stdout," expression@    Set dot to expression and display the contents\n");
    fprintf( stdout,"                of its contents.\n");
    fprintf( stdout," expression=    Display value of expression.\n");
    fprintf( stdout," /              Increment dot and display contents of that address.\n");
    fprintf( stdout," \\              Decrement dot and display contents of that address.\n");
    fprintf( stdout," @              Set dot to contents it points to and then print new contents.\n");
    fprintf( stdout," =              Print value of dot.\n");
    fprintf( stdout," expr,nlines,n  Display the nlines instructions which begin at expr.\n");
    fprintf( stdout,"                Default expr is dot; default nlines is 24.\n");
    fprintf( stdout," expr,nlines,p  Display the nlines instructions which end at expr.\n");
    fprintf( stdout,"                Default expr is dot; default nlines is 24.\n");
    fprintf( stdout," d              Display the contents of the registers.\n");
    fprintf( stdout," s              Print a trace of subroutine calls.\n");
    fprintf( stdout," expr,brknum,b  Set breakpoint brknum (2 <= brknum <= 15) at expr.\n");
    fprintf( stdout,"                If expr is 0, clear breakpoint brknum; if omitted,\n");
    fprintf( stdout,"                print all breakpoints.  If brknum omitted, first\n");
    fprintf( stdout,"                available breakpoint is used.\n");
    fprintf( stdout," expr,g         Execute starting at expr until exception, breakpoint,\n");
    fprintf( stdout,"                or termination. If expr omitted, start at current pc.\n");
    fprintf( stdout," expr,gb        Go past current breakpoint expr times.\n");
    fprintf( stdout,"\nType a character for more commands:");
    getchar(c);
    putchar('\n');

    fprintf( stdout," expr,x         Execute the next expr instructions.\n");
    fprintf( stdout," expr,y         Like x but subroutine calls are single instructions.\n");
    fprintf( stdout," type,t         Temporarily set typeout mode to type.  Type can be:\n");

    fprintf( stdout,"                's',strLength--string; or 'i'--instruction.  Apostrophes\n");
    fprintf( stdout,"                are considered part of the type specification.\n");
    fprintf( stdout," type,tt        Permanently set typeout mode to type.\n");
    fprintf( stdout," expr1,expr2,type,r\n");
    fprintf( stdout,"                Replace contents of expr1 with expr2.  Expr2 has type type.\n");
    fprintf( stdout," register,expr,rr\n");
    fprintf( stdout,"                Replace contents of register with expr.  Registers\n");
    fprintf( stdout,"                are referred to as %%r0-%%r15, %%ap, %%fp, %%sp, or %%pc.\n");
    fprintf( stdout," base,ir        Set input radix to base.\n");
    fprintf( stdout," base,or        Set output radix to base.\n");
    fprintf( stdout," displacement,of  Set maximum offset from a symbol to displacement.\n");
    fprintf( stdout," length,sl      Set maximum symbol length for display.\n");
    fprintf( stdout," sp             Switch from stopping the whole team at an exception to\n");
    fprintf( stdout,"                just stopping the process with the exception, or vice versa.\n");
    fprintf( stdout," pattern,lowlimit,highlimit,type,f\n");
    fprintf( stdout,"                Find pattern in the address range lowlimit to highlimit-1.\n");
    fprintf( stdout,"                Pattern's type is type, and a mask determines how much\n");
    fprintf( stdout,"                of it is significant:\n");
    fprintf( stdout," expr,m         Set find's mask to expr. -1,m forces a complete match;\n");
    fprintf( stdout,"                f,m matches only the last four bits; 0,m matches anything.\n");
    fprintf( stdout,"\nType a character for more commands:");
    getchar(c);
    putchar('\n');
    fprintf( stdout," expr,type,w    Watch a specified location to see when it changes\n");
    fprintf( stdout,"                If expr is omitted, show watched location.\n");
    fprintf( stdout,"		     If expr is 0, stop watching.  If type is omitted,\n");
    fprintf( stdout,"                use default.\n");
    fprintf( stdout," expr,type,wb   Same as \"w\" except the location is only\n");
    fprintf( stdout,"                examined at breakpoints; breakpoints do not\n");
    fprintf( stdout,"                case a stop unless the location has changed.\n");
    fprintf( stdout," h              This help command.\n");
    fprintf( stdout," #              Internal debugger statistics.\n");
    fprintf( stdout," q              Exit the debugger and the program. \n");
    fprintf( stdout,"\nFurther information can be found in the V manual.\n");
  }



/*
 * Print internal statistics.
 */

QueryDebugger()
  {
    QueryCache();
    fprintf( stdout,"CurExceptionPid %x  LinkOrigin %x  DebugeeTeamSize %x\n",
      CurExceptionPid, LinkOrigin, DebugeeTeamSize );
    fprintf( stdout,"ProgramCounter %x\n", ProgramCounter );
    QueryMemBuffer();
  }
