/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer, Eric Berglund,  5/83
 */


/* This file, stack.c, is part of the V debugger.  It contains the routines
 * necessary to implement the StackDump command.
 */


#include <b.out.h>
#include <Vio.h>
#include "Vdb.h"
#include <procdesc.h>




extern char *StackBottom;

extern PrintAddress(), GetMemLong();
extern short GetInstrWord();


char *kernelname[] =
  {
    "Forward",
    "GetReply",
    "_Receive",
    "Send",
    "ReplyWithSeg",
    "endofnames"
  };



/* StackDump prints the sequence of calls that led to the routine in which
 *  execution is halted, giving the address of each call and the current values
 *  of the first MAXNUMPARAMSTOPRINT parameters of each routine.
 *  Variables ending in "fp" hold various values of the frame pointer.  For
 *  full details of the C calling conventions for the SUN, contact your
 *  nearest SUN User's Guide, Chapter 2.
 */


StackDump( disassembledCode, maxLines )
	char *disassembledCode;
	int maxLines;
  {
    char *currentfp, *nextfp, *correctcurrentfp, *correctnextfp;
    long returnpc;
    unsigned short lastinstr, currentinstr;
    SystemCode error = OK;

    cbfp = &OutputBuffer;
    PrintMessage( "Halted at", ProgramCounter );

    currentfp = (char *) SavedRegisters[FP];
    if( currentfp == 0 )
      {
	fprintf( stdout, "\nStackdump: Frame pointer's value is 0. Either execution\n");
	fprintf( stdout, "hasn't begun or the frame pointer hasn't been set correctly.\n");
	Flush( stdout );
	return;
      }

/* In the following cases, the frame pointer, a6, has either not been updated
 * or been updated prematurely.  At the beginning of a routine, just before the
 * link statement has been executed; and at the end of a routine, when the unlk
 * has been executed but the return has not, the current frame pointer
 * should indicate the word above the top of the stack.
 *
 * In kernel routines, where the C calling convention for the Suns isn't
 * followed, we must special case the routines to read the appropriate
 * registers from the stack frame.
 */

    if( AtStartofRoutine( disassembledCode ) || AtEndofRoutine() ||
	InKernelRoutine( disassembledCode ) ||
	AtStartofKernelRoutine()  ||  AtEndofKernelRoutine() )
      {
	if( InKernelRoutine( disassembledCode ) &&
	    !( AtStartofKernelRoutine()  ||  AtEndofKernelRoutine() ) )
	  {
	    correctcurrentfp = (char *) SavedRegisters[SP]
				 + (NUM_REGS_SAVED - 1) * STACKENTRYSIZE;
	    correctnextfp = (char *) GetMemLong( correctcurrentfp, &error );
	    currentfp = correctnextfp;
	  }
	else
	  {
	    correctcurrentfp = (char *) SavedRegisters[SP] - STACKENTRYSIZE;
	    correctnextfp = currentfp;
	  }

	returnpc = PrintParams ( correctcurrentfp, correctnextfp, &error );
        if( error != OK )
	  {
	    fprintf( stdout,
	     "\nStackdump: Frame pointer holds invalid address: %x\n",
	     correctcurrentfp );
	    Flush( stdout );
	    return;
	  }
	
	if( --maxLines == 0 ) return;

	PrintMessage( "Called from", returnpc - SubCallLength( returnpc ) );
      }	    

    for ( ; ; )
      {
	nextfp = (char *) GetMemLong( currentfp, &error );
	returnpc = PrintParams (currentfp, nextfp, &error);
	if( error != OK )
	  {
	    fprintf( stdout,
	        "\nStackdump: Frame pointer holds invalid address: %x\n",
		currentfp );
	    Flush( stdout );
	    return;
	  }

	if(nextfp == 0 || --maxLines == 0 ) return;

	PrintMessage( "Called from", returnpc - SubCallLength( returnpc ) );  
	currentfp = nextfp;
      }
  }


/* Determine the symbolic representation of the address and print it.
 */

PrintMessage( message, address )
	char *message, *address;
  {
    fprintf( stdout, "%s ", message );
    PrintAddress( address );
    Flush( stdout );
  }



/* Print the first MAXNUMPARAMSTOPRINT parameters of the routine whose frame is
 * indicated by currentfp.  Note that the parameters are stored 2 words below
 * the frame pointer.  Refer to the Sun User's Guide for details of the C
 * calling conventions.
 */

PrintParams( currentfp, nextfp, error )
	char *currentfp, *nextfp;
	SystemCode *error;
  {
    int i, nparams;
    char *paramaddress, *stackaddress;
    long param;
    long funcStart;
    long returnpc;

    /* Check whether the attempt to read nextfp was good. */
    if( *error != OK ) return( 0 );

    returnpc = GetMemLong( currentfp + STACKENTRYSIZE, error );
    if( *error != OK ) return( 0 );

    if( nextfp != 0 )
      nparams = GetNumofParams( currentfp, error );
    else
      nparams = GetNumofProcessParams( currentfp );

    if( *error != OK ) return;

    putchar('(');
    stackaddress = currentfp + 2 * STACKENTRYSIZE;

    paramaddress = stackaddress;

    /* No parameters printed if nparams == 0. */
    for( i = 1; i <= nparams; i++ )
      {
	param = GetMemLong( paramaddress, error );
	if( *error != OK )
	  return( 0 );

	if (i != 1)
	  putchar( ',' );
	putchar(' ');
	printnum( param );

	paramaddress += STACKENTRYSIZE;
      }
    fprintf( stdout, " )    Stack Address: %x\n", stackaddress );
    Flush( stdout );
    return( returnpc );
  }



/* Get the number of parameters by finding the instruction at the return
 * address of the calling routine.  The compiler's convention is to pop
 * the stack at that point by adding the appropriate number of bytes to the
 * stack pointer: 4 if there was one parameter, 8 if there were two, etc. 
 * Unfortunately, there are two separate add instructions, and this whole
 * routine is quite compiler dependent.
 */


GetNumofParams( currentfp, error )
	char *currentfp;
	SystemCode *error;
  {
    char *returnpc;
    unsigned short instruction;
    int nameindex;
    int count;

    returnpc = (char *) GetMemLong( currentfp + STACKENTRYSIZE, error );
    if( *error != OK ) 
      {
	fprintf( stdout, "\nStackdump: Frame pointer holds invalid address: %x\n", currentfp );
	Flush( stdout );
	return( -2 );
      }

    instruction = GetInstrWord( returnpc, error );
    if( *error != OK ) return( -2 );

    switch( instruction )
      {
	case ADDQL4 :  return( 1 );
	case ADDQL8 :  return( 2 );
	case LEA    :
	case ADDW   :  count = GetMemWord( returnpc + 2, error ) / 4; break;
	case ADDL   :  count = GetMemLong( returnpc + 2, error ) / 4; break;
        default     :  return( 0 );
      }
    if( *error != OK )
      {
	fprintf( stdout, "\nStackdump: Return address holds invalid address: %x\n", returnpc );
	Flush( stdout );
	return( -2 );
      }
    if( count > MAXNUMPARAMSTOPRINT ) count = MAXNUMPARAMSTOPRINT;
    return( count );
  }




GetNumofProcessParams( currentfp )
	char *currentfp;
  {
    int nparams;
    nparams = (int) ( StackBottom - currentfp ) / STACKENTRYSIZE  - 2;
    return( nparams > MAXNUMPARAMSTOPRINT ?  MAXNUMPARAMSTOPRINT : nparams );
  }




#define SameString !strncmp

/* Given the string produced by the disassembler, which begins with the symbolic
 * address expression, isolate the name of the current routine and check it against
 * the list of kernel routines.  Note that because the disassembler truncates 
 * symbols to ten characters, the kernel name table above has truncated entries.
 * The index of the entry in the table is returned, but not currently used.
 */

InKernelRoutine( string )
	char *string;
  {
    char routinename[ MAXSYMLEN ];
    int i, j;

    i = 0;
    while( (*string != '\0') && (*string != '+') && (*string != ' ') &&
							(*string != '\t') )
	routinename[i++] = *string++;
    routinename[i] = '\0';

    j = 0;
    while( ( !SameString( routinename, kernelname[j], MaxSymLength ) )
	 && !SameString( kernelname[j], "endofnames", MaxSymLength ) )
	j++;
    return( SameString( kernelname[j], "endofnames", MaxSymLength ) ?  0 : j );
  }


/* Given the string returned by the disassembler, which begins with the symbolic
 * address representation, assume that we're at the start of a routine if no
 * offset has been calculated by the disassembler--i.e. the name is followed by
 * a blank, not a plus sign.  This works because the entries in the symbol table
 * are currently routine names only.  If this should change, it may be better
 * to get inaccurate stackdumps at the start of a routine rather than in the middle.
 */

AtStartofRoutine( string )
	char *string;
  {
    while( (*string != ' ') && (*string != '+') && (*string != '\t') )
	string++;
    return( *string == ' '  ||  *string == '\t' );
  }



/* The start of a kernel routine occurs when the current instruction is
 * a MOVEML #/3F3E, SP@-.
 */

AtStartofKernelRoutine()
  {
    SystemCode error;
    unsigned long currentinstr;

    currentinstr = GetInstrWord( ProgramCounter, &error )
				 << ( sizeof( short ) * 8 );
    currentinstr |= GetInstrWord( ProgramCounter + 1, &error );
    return( currentinstr == MOVEML3F3ESP );
  }



/* The end of a routine occurs when the current instruction is an RTS and
 * the previous instruction was an UNLK A6.
 */

AtEndofRoutine()
  {
    SystemCode error;
    unsigned short currentinstr, lastinstr;
    currentinstr = GetInstrWord( ProgramCounter, &error );
    lastinstr = GetInstrWord( ProgramCounter - 1, &error );
    return( ( currentinstr == RTS ) && ( lastinstr == UNLKA6 ) );
  }


/* The end of a kernel routine occurs when the current instruction is an RTS and
 * the previous instruction was a MOVEML SP@+, #/7CFC.
 */

AtEndofKernelRoutine()
  {
    SystemCode error;
    unsigned short currentinstr;
    unsigned long lastinstr;

    currentinstr = GetInstrWord( ProgramCounter, &error );
    lastinstr = GetInstrWord( ProgramCounter - 2, &error )
				 << ( sizeof( short ) * 8 );
    lastinstr |= GetInstrWord( ProgramCounter - 1, &error );
    return( ( currentinstr == RTS ) && ( lastinstr == MOVEMLSP7CFC ) );
  }


/* SubCallLength determines which possible subroutine call instruction
 * was used just before a given return address: JSR, BSR with a short
 * offset, or BSR with a long offset, and returns the number of bytes
 * in the instruction.
 */

SubCallLength( returnpc )
	char *returnpc;
  {
    SystemCode error;

    if( (GetInstrWord( returnpc - JSRLENGTH, &error ) & JSRMASK ) == JSRINST )
	return( JSRLENGTH );

    else if( (GetInstrWord( returnpc - BSRLONGLENGTH, &error ) & BSRLONGMASK )
								 == BSRINST )
	return( BSRLONGLENGTH );

    else if( (GetInstrWord( returnpc - BSRSHRTLENGTH, &error ) & BSRSHRTMASK )
								 == BSRINST )
	return( BSRSHRTLENGTH );

    else
	return( 0 );
  }
