/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer, Eric Berglund,  5/83
 *
 * If Addr24 is defined, mask off top 8 bits of addresses. Per Bothner. Mar 84
 */

/* This file, memoryaccess.c, is part of the V debugger.  It maintains a
 * buffer containing a copy of INSTRBLK bytes of program memory.  It includes:
 * GetFromMemBuffer, GetMemByte, GetMemWord, GetMemLong, GetInstrWord, PutMem,
 * PutMemData (the replacement for PutMemByte, PutMemWord, PutMemLong, and
 * PutMemString), InvalidMemBuffer, ValidInstrPc, ValidAddress, 
 * and QueryMemBuffer.
 */


#include <b.out.h>
#include <Vio.h> 
#include "Vdb.h"


unsigned char MemBuffer[INSTRBLK] = { 0 };
				/* Buffer for program memory. */
char *bfrbegpc, *bfrendpc;	/* First and last pc's in current MemBuffer. */
int BfrValid = 0;		/* Signals that mem. buffer may be invalid. */


/* GetFromMemBuffer ensures that the byte pointed to by pc is in
 * MemBuffer unless an error condition occurs. It reads the next
 * INSTRBLK bytes starting at that pc if *pc is not already in MemBuffer,
 * returning an error code if, for some reason, the contents of pc can
 * not be obtained.
 * Other routines which must access program memory do so through the
 * MemBuffer--GetFromMemBuffer is the only one with a MoveFrom instruction,
 */

SystemCode GetFromMemBuffer( pc )
	char *pc;
  {
    SystemCode error;
    int blocksize, bytestoend;

    DoFixAddress(&pc);
    if (BfrValid && (bfrbegpc <= pc) && (pc <= bfrendpc))
      return(OK);

    if( (pc < LinkOrigin) || (pc >= DebugeeTeamSize-LoadOrigin+LinkOrigin) )
      return(BAD_ADDRESS);

    bytestoend = (int)( DebugeeTeamSize-LoadOrigin+LinkOrigin - (int) pc );
    if( bytestoend < INSTRBLK )
      blocksize = bytestoend;
    else
      blocksize = INSTRBLK;

    error = MoveFrom( CurExceptionPid, MemBuffer, 
    		      pc-LinkOrigin+LoadOrigin, blocksize );
    if (error != OK)
      {
        BfrValid = 0;
        fprintf( stdout, "Debugger: Error while trying to access debuggee's memory\n %s\n", ErrorString( error ) );
      }
    else
      {
        bfrbegpc = pc;
        bfrendpc = pc + blocksize - 1;
        BfrValid = 1;			/* Valid mem. buffer */
      }
    return(error);
  }



/* Fetch the byte at loc. */

char GetMemByte( loc, error )
	char *loc;
	SystemCode *error;
  {
    DoFixAddress(&loc);
    if( (*error = GetFromMemBuffer(loc)) != OK)	return(-2);
    return( MemBuffer[ loc - bfrbegpc ] );
  }


/* Fetch the word at loc. */

short GetMemWord( loc, error )
	char *loc;
	SystemCode *error;
  {
    short word;

    DoFixAddress(&loc);
    if( (*error = GetFromMemBuffer(loc)) != OK ) return(-2);
    word = ( (short) MemBuffer[ loc - bfrbegpc] ) << 8;
    if( (*error = GetFromMemBuffer(loc+1)) != OK ) return(-2);
    word |= ( (short) MemBuffer[ loc+1 - bfrbegpc ] ) & 0xFF;
    return( word );
  }


/* Fetch a long at loc. */

long GetMemLong( loc, error )
	char *loc;
	SystemCode *error;
  {
    long lword;
    DoFixAddress(&loc);
    if( (*error = GetFromMemBuffer(loc) ) != OK ) return(-2);
    lword = ( (long) MemBuffer[ loc - bfrbegpc ] ) << 24;
    if( (*error = GetFromMemBuffer(loc+1) ) != OK ) return(-2);
    lword |= ( ( (long) MemBuffer[ loc+1 - bfrbegpc ] ) << 16 ) & 0xFF0000;
    if( (*error = GetFromMemBuffer(loc+2) ) != OK ) return(-2);
    lword |= ( ( (long) MemBuffer[ loc+2 - bfrbegpc ] ) << 8 ) & 0xFF00;
    if( (*error = GetFromMemBuffer(loc+3) ) != OK ) return(-2);
    lword |= ( (long) MemBuffer[ loc+3 - bfrbegpc ] ) & 0xFF;
    return( lword );
  }


/*
 * Determines if the instruction is a breakpoint instruction and returns
 * its true value.
 */

short GetInstrWord( pc, error )
    short *pc;
    SystemCode *error;
  {
    int bpno;
    short instrVal;

    instrVal = GetMemWord(pc, error);
    if ( instrVal == BPINST )
      {
	bpno = FindUserBreakpoint(pc);
	if (bpno != 0)
	  instrVal = bp[bpno].oldinst;
      }
    return( instrVal );
  }


long GetMem( type, addr, error )
short type;
char *addr;
SystemCode *error;
  {
    switch(type)
      {
    case 0:
    case LONGTYPE:
        return(GetMemLong(addr, error));
    case CHARTYPE:
    case BYTETYPE:
    	return(GetMemByte(addr, error));
    case WORDTYPE:
    	return(GetMemWord(addr, error));
    case INSTTYPE:
    	return(GetInstrWord(addr, error));
    default:
    	return(*error = BAD_ARGS);
      }
  }

/*
 * Here to PutMem data
 */
int PutMem( type, adrs, data )
	short type;
	char *adrs;
	long data;
{
    short bpno;
    char byte;
    short word;

    switch (type)
      {
	case 0:			/* Default PutMem */
	case LONGTYPE:
	    return(PutMemLong(adrs, &data));

	case CHARTYPE:
	case BYTETYPE:
	    byte = (char) data;
	    return(PutMemByte(adrs, &byte));

	case WORDTYPE:
	case INSTTYPE:
	    bpno = FindUserBreakpoint( adrs);
	    if ( bpno != 0 )
	      {
		bp[ bpno ].oldinst = (short) data;
		return(2);
	      }
	    else
	      {
		word = (short) data;
		return(PutMemWord(adrs, &word));
	      }

	default:
	    return( -2 );
      }
}

/*
 * Function to deposit data in team space.
 */

PutMemData( address, data, nbytes )
	char *address;
	char *data;
	int nbytes;
  {
    SystemCode error;
    int i, indx;

    DoFixAddress(&address);
    if( address < LinkOrigin || 
    	address + nbytes - 1 >= DebugeeTeamSize-LoadOrigin+LinkOrigin )
      return( -2 );
    error = MoveTo( CurExceptionPid, address-LinkOrigin+LoadOrigin,
    		    data, nbytes );
    if( error != OK ) return( -2 );

    /* Deposit data at loc in MemBuffer. */
    for (i = 0; i < nbytes; i++)
	if (BfrValid && (bfrbegpc <= address) && (address <= bfrendpc))
	  {
	    indx = address++ - bfrbegpc;
            MemBuffer[indx] = *data++;
	  }
     return( nbytes );
  }



/*
 * InvalidMemBuffer:
 * Invalidates the memory buffer cache.
 */

InvalidMemBuffer()
  {
    BfrValid = 0;
  }


/*
 * Checks if pc points to a legal instr.
 */

ValidInstrPc(pc)
    short *pc;
  {
    int legal;

    DoFixAddress(&pc);

    if ((((char *)pc) >= LinkOrigin) && 
        (((char *)pc) < DebugeeTeamSize-LoadOrigin+LinkOrigin) &&
	(!(((long)pc) & 1)))
      {
        dasm(pc, Symfile, &legal);
	if (legal)
	    return(1);
      }
    return(0);
  }


ValidAddress( address )
	char *address;
  {
    DoFixAddress(&address);
    return( address >= LinkOrigin &&
            address < DebugeeTeamSize-LoadOrigin+LinkOrigin );
  }



QueryMemBuffer()
  {
    printf("BfrValid %d  bfrbegpc %x  bfrendpc %x \n",
	BfrValid, bfrbegpc, bfrendpc );
  }
