/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/* 			68000 Disassembler
 *    			    V.R. Pratt
 *    			    Jan., 1981
 *			Modified by David Cheriton, 
 *			Eric Berglund, Marvin Theimer.
 */


/* January 13, 1983: Eric J. Berglund
 *   I have just taken a copy of Mr. Pratt's and Mr. Cheriton's disassembler
 *   and modified it to search for the symbolic representations of addresses
 *   on a file resident on a VAX, while the disassembler executes on a SUN
 *   workstation.  The fileid for this file must be passed to the dasm routine
 *   and is assumed to be in the usual format of .r files.
 */


/* This is a compact disassembler intended for use in the 68000 design
 * module.
 *
 * Principles of Operation
 * 
 * The disassembler has the following components.
 * 
 * 1.  A list of instruction patterns and their assembler forms,
 * each serving as the head of a function definition.
 * 
 * 2.  A list of effective addresses and their assembler forms, each serving
 * as the body of a function definition.
 * 
 * 3.  A function to locate the first entry in the appropriate list matching 
 * a given instruction or effective address, serving as the
 * function-identification component of an eval.
 * 
 * 4.  A function to bind the actual parameters in the instruction or
 * effective address to the formal parameters in the function head.
 * 
 * 5.  A function to substitute the formal parameters into the function body.
 * 
 * 
 * Lists
 * 
 * The two lists are machine readable forms of those given in /usr/pratt/ins68
 * (see /usr/pratt/ins.f for the format of this file).  In the machine
 * readable form each entry has the following structure.
 */

typedef struct Entry
       {char length;
	char modifier;
	unsigned short mask;
	unsigned short pattern;
       } *entry;

/* Following each entry is a character string, considered part of the entry.
 * The entries are concatenated to form a table.  The entry boundaries are
 * determined only by the length member, the tables being intended to be
 * searched forwards linearly.  Entries are aligned on word boundaries.
 * 
 * The length of an entry is the number of bytes in the entry, which may be
 * odd in which case the following entry is one byte further along than
 * indicated by length.
 * 
 * Fparams is an array of 6 formal parameters, each an unsigned nybble (4-bit
 * integer) encoding the type of that parameter.
 *
 * The mask and the pattern together are used to tell whether an item such
 * as an opcode matches entry e, by the test:
 */

#define match(item,ent) ((item ^ ent->pattern) & ent->mask) == 0

/* This file includes getlong, readfileheader, dasminit, dasm, instr,
 * locate, translate, bind, bindmod, subst, hex, whex, rev.
 */


#include <b.out.h>
#include <Vio.h>
#include <Vtermagent.h>
#include "Vdb.h"

#ifdef MC68000
#define MACHINE ".m68k"
#else  MC68000
#define MACHINE ".vax"
#endif MC68000


long opargs[NOARGS],		/* Where to put opcode args. */
            eargs [NEARGS];		/* Where to put ea args. */
int argno;			/* Global pointer into args. */
int nwords;			/* Words in instruction. */
unsigned short *mpc,		/* Main program counter. */
               *spc;		/* Start program counter. */

char longflag;			/* 1 -> long. */
char symbolicAddress;		/* 1 -> symbolic addresses. */
char sizmod;			/* 1 -> one bit size field. */

extern unsigned char optab[];	/* Table of opcode patterns */
extern unsigned char eatab[];	/* Table of eff. add. patterns */


extern InitCache(), PrintSym(), numout(), bufput();





/* Getlong reads a long from the .r file--most significant byte first,
 *   of course--and returns it.
 */

long getlong( fileptr )
  File *fileptr;
{
  long value;

  value = getc( fileptr ) << 24;
  value |= ( getc( fileptr ) << 16 ) & 0xFF0000;
  value |= ( getc( fileptr ) << 8 ) & 0xFF00;
  value |= getc( fileptr ) & 0xFF;
  return( value );
}






/* Readfileheader reads the 8 longs from the beginning of the provided
 *   file and stores them in the appropriate subfield of the filhdr structure.
 */

readfileheader( fileptr )
  File *fileptr;
{
  filhdr.fmagic = getlong( fileptr );
  filhdr.tsize = getlong( fileptr );
  filhdr.dsize = getlong( fileptr );
  filhdr.bsize = getlong( fileptr );
  filhdr.ssize = getlong( fileptr );
  filhdr.rtsize = getlong( fileptr );
  filhdr.rdsize = getlong( fileptr );
  filhdr.entry = getlong( fileptr );
}





/* Dasminit provides the fileid of the newly created opened .r file.
 * Exits program on error conditions.
 */

#define DebuggerPadMode		(CR_Input+LF_Output)

File *dasminit( fileServer, fileId, fileName )
	ProcessId fileServer;
	InstanceId fileId;
	char *fileName;
  {
    File *fileptr;
    SystemCode error;
    char c;

    fileptr = OpenFile(fileServer, fileId, FREAD, &error);

    if( fileptr == NULL  ||  error != OK )
      {
	fprintf( stderr,
	 "Debugger couldn't open file given fileServer and fileId.\n" );
	fprintf( stderr, "Trying fileName %s%s.\n", fileName, MACHINE );
	fflush( stderr );
	fileptr = Open( strcat( fileName, MACHINE ), FREAD, &error );
      }

    while ((fileptr == NULL) || (error != OK))
      {
	char filename[ 256 ];

        fprintf( stdout, "Debugger: couldn't open symbol table file ");
	fprintf( stdout, "for program to be debugged.\n");
        fprintf( stdout, "Enter name of file to use <CR to give up>: ");
	Flush( stdout );
	ModifyPad( stdin, CR_Input+Echo+LineBuffer+LF_Output ); 
	c = getchar();
	if( c == EOF && stdin->lastexception == BAD_BLOCK_NO )
	  {
	    ClearEof( stdin );
	    Resynch( stdin );
	  }
	else
	    ungetc( c, stdin );

	gets( filename );
	ModifyPad( stdin, DebuggerPadMode );
	if (filename[0] == '\0') ExitDebugger();
	fileptr = Open(filename, FREAD, &error);
      }
    readfileheader( fileptr );
    InitCache();
    return( fileptr );
  }


/* Finally, the main entry point to this collection of routines.
 * Given a word address, we interpret the word(s) at that address
 * as a 68000 instruction and disassemble them putting the disassembled
 * "source" text in one output buffer, and the hex ascii representation
 * in another. The number of words displayed are returned.  In the case
 * where no legal instruction could be found, 1 word is displayed as data.
 */
/* Symfile and symfile are treated very strangely in here!!!*/

int dasm ( address , symfile, legal)
    unsigned short *address;
    File *symfile;
    int *legal;

  {
     SystemCode error;
     short instrVal;

     Symfile = symfile;			/* Make into a local global to avoid
					   having to pass it around. */

     instrVal = GetInstrWord( address, &error );
				/* Get the true value of the first word of
				   the instruction, independent of whether a
				   breakpoint has been set at address. */
     if( error != OK )
       {
	 *legal = 0;
	 return( -1 );
       }

     mpc = address;			/* EJB: mpc will be changed by instr */
     OutputBuffer.pnt = 0;
     OutputBuffer.col = 1;
     cbfp = &OutputBuffer;

     PrintSym((long)mpc, Symfile);
     if (cbfp->col < 8) bufput('\t');
     bufput('\t');
     instr(instrVal);			/* dis-assemble the instruction */
     bufput('\0');

     HexBuffer.pnt = 0;
     HexBuffer.col = 1;
     cbfp = &HexBuffer;

     hex((long)spc);			/* address in hex */
     bufput('\t');

     whex(instrVal);
     spc++;
     while (spc < mpc)
       {
	 whex( GetInstrWord( spc++, &error ) );		/* EJB--output hex, word by word */
	 bufput(' ');
       }
     while (cbfp->col < 32) bufput('\t');
     bufput('\0');

     if (mpc == address)
       {			/* No legal instr. found.  Treat as data. */
         *legal = 0;
	 return(1);
       }
     else
       {
         *legal = 1;
         return((int) (mpc - address) );
       }
  }





/* The heart of this file is instr(), which returns the assembler form
 * of the instruction presently pointed to by mpc.  As a side effect it 
 * advances mpc to the word following the instruction unless the instr.
 * is not a legal one.
 */

instr( instrVal )				/* process next instruction */
	short instrVal;

  {
    char legal;
    char *ptr;

    sizmod = 0;					/* assume two-bit size field */
    longflag = 0;				/* assume short data */
    legal = 1;					/* innocent till proved g'ty */
    symbolicAddress = 0;			/* assume nonsymbolic data */
    spc = mpc;					/* remember mpc */
    if( instrVal )
						/* if word at pc isn't 0 */
      {
        mpc++;					/* point past opcode */
	translate(instrVal,optab,opargs);	/* translate opcode */
	ptr = OutputBuffer.buf;			/* Check if an illegal instr.
						   was disassembled. */
	while (*ptr)
	  {
	    if (*ptr++ == '*')
	      {
	        legal = 0;
		break;
	      }
	  }
      }
    else
	legal = 0;

    if (!legal) 
	mpc = spc;
  }





/* Match is used by the locate routine, which locates the first entry in the
 * given table matching the given code.
 */
/* EJB: recall that Match is defined near the beginning of the file. */


entry  locate ( code, tab )
  register code;
  register entry tab;
{
    while (1)				/* guaranteed to find something */
    {
	if (match(code,tab)) break;
	tab = (entry)((long)tab + tab->length + 1 & -2);/* go to next entry */
    }
    return(tab);
}



/* The function translate(item,tab,args) translates item, which is either
 * an opcode or an effective address, using the appropriate table tab, either
 * optab or eatab, using args as the base address of the stack of
 * arguments extracted from the item in the course of translation, by the
 * binding mechanisms.
 */

 translate ( item, tab, args )
    unsigned short item;
    char *tab;
    long args[];
  {
    entry ent = locate(item,tab);		/* locate entry for opcode */
    argno = 0;					/* where to store args */
    bind(item,ent->pattern,ent->mask,args);	/* bind item arguments */
    bindmod(ent->modifier,args);		/* bind modifier arguments */
    subst((long)ent+6,(long)ent+ent->length,args);	/* substitute args */
  }









/* The mask and pattern are also used to extract the arguments from an item 
 * such as an opcode or an effective address.  These are extracted 
 * right-to-left from the item by the function bind and stored in 
 * in successive locations of the area args supplied to bind.
 */

#define shft	{item >>= 1; pattern >>= 1; mask >>= 1;}


bind ( item, pattern, mask, args )
    register short unsigned item, pattern, mask;
    long args[];

  {
    while (1)					/* loop per argument */
    {
	while (mask&1) shft;			/* skip masked part */
	if (!pattern) break;			/* stopping cond. */
	args[argno++] = (long)(char)(item&(pattern ^ pattern-1)); /* bind */
	while (!(pattern&1)) shft;  shft;	/* skip field */
    }
  }




bindmod ( m, args ) 
    register char m;		                /* bind modifier */
    register long args[];	                /* bind arg */

  {
    SystemCode error;

    /*n*/  if (m&1 && (!(m&2) || !longflag))
		 args[argno++] = (short) GetMemWord((short*)mpc++,&error);
    /*i*/  if (m&2 && (!(m&1) || longflag))
		 {args[argno++] = GetMemLong((long*)mpc++,&error); mpc++;}
    /*g*/  if (m&8)
	      { int word = (unsigned short)GetMemWord((short*)mpc++,&error);
		args[argno++] = word >> 8;
		args[argno++] = word & 0xFF;
	      }
    /*l*/  if (m&16) longflag = 1;
    /*p*/  if (m&32) symbolicAddress = 1;
    /*t*/  if (m&64) sizmod = 1;
    /*u*/  if (m&128) args[argno++] = GetMemWord(mpc++,&error);
  }







/* The function subst takes a pair of pointers pointing to the two ends of
 * a string whose free variables are to be instantiated, and a pointer to
 * a stack of arguments to be bound to those variables.
 */

subst ( cod, lim, args )
  char *cod, *lim;
  long args[];

{
  while (cod<lim)				/* substitute up to lim */
    if (*cod >= 0) bufput(*cod++);		/* copy the char directly */
    else					/* execute the command *cod */
      {
	register long v = args[(*cod >> 4) & 7];/* get value v */
	char *siz = "bwl*";
	char *cc  = "rasrhilscccsneeqvcvsplmigeltgtle";/* condition codes */
	char r = 'a';
	switch (*cod++ & 15)
/*k*/	{case 0:  if (v<128) r += 3;
		  bufput(r);
		  bufput('0'+((v>>4)&7));
		  bufput(':');
		  bufput(v&8? 'l': 'w');
		  break;
/*d*/	 case 1:  r += 3;	/* that is: r = 'd' */
/*a*/	 case 2:  bufput(r);
		  bufput('0'+(v&7));
		  break;
/*E*/	 case 3:  v = ((v&7)<<3)|(v>>3);
/*e*/	 case 4:  translate((short)v,eatab,eargs); 
		  break;
/*s*/	 case 5:  v+=sizmod; bufput(siz[v]); 
		  if (v==2) longflag = 1;
		  break;
/*v*/	 case 6:  if (v == 0) v = 8;
/*n*/	 case 7:  if (v<0) {
			bufput('-');
			v = -v; 
		  }
		  numout(v);
		  break;
/*u*/	 case 8:  PrintSym((long)v, Symfile);  
		  break;
/*y*/	 case 9:  v = rev((long)v);
/*x*/	 case 10: hex((long)(v&0xffff)); break;
/*m*/	 case 11: if (v) cod += *cod; else cod++; break;
/*f*/	 case 12: while (v--) while (*cod++ != ',');
		  while (*cod != ',' && *cod != ')') bufput(*cod++); 
		  while (*cod++ != ')'); break;
/*c*/	 case 13: bufput(cc[v*2]); bufput(cc[v*2+1]); break;
/*r*/	 case 14: PrintSym((long)spc+2+v, Symfile); break;
	}
    }
}






/* Convert a long to its ascii hex representation regardless of the
 * "current radix", and send it to the output buffer.
 */

hex(n)
  register unsigned long n;
{
    if ( n > 15 ) hex( n >> 4 );
    n &= 15;
    bufput((char)( n + ( n>9 ? '7': '0')));
}





/* Convert an integer to its ascii hex representation in exactly four
 * digits, and send it to the output buffer. Insert leading zeroes if 
 * required.
 */

whex(n)
  register unsigned int n;
{
  register int i;
  register char c;

    for( i = 12; i >= 0; i -= 4)
    {
	c = n>>i & 15;
	bufput( c + ( c>9? '7': '0'));
    }
}

/* Given a long "y", return its bit reversal.
 */

long  rev( y )
  long y;
{
    register long z=0;
    register n=16;
    while (n--)
    {
	z = z<<1|(y&1);
	y>>=1;
    }
    return(z);
}
