/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/


/*
 * Print information about processes currently running.
 * If no arguments are provided then information on all processes running
 * is provided.
 * Specification of one or more process ids yields a complete dump of
 * the information returned by QueryProcessState for each pid specified.
 * The -l option specifies a long form of output for a complete dump which
 * includes the filenames from which the various teams were loaded.
 * The -t option specifies that information only for the specified team(s)
 * should be printed out.  The process id of any process on a team may be
 * used to designate a team.
 * Teams can be specified by a character string which is a suffix of the
 * team's filename.  Thus, for example, the team [bin]internetserver can
 * be specified by the character string internetserver or even by netserver.
 *
 * NOTE: Process ids can be abbreviated by leaving off the host and version
 * prefix.  For example, pid 0x9790047 can be specified simply as 0x47.
 * Also, process-ids must be specified as hex number preceded by  0x .
 * 
 */


#include <Vprocess.h>
#include <Vteams.h>
#include <Vdirectory.h>


#define FALSE 0
#define TRUE 1

#define INTRO_HDR	"Process Information for Host 0x%x:\n\n"
#define FORMAT_HDR1	"                  Team     Priority"
#define FORMAT_HDR2	"Process      Parent   Root      Net     Team    State"
#define FORMAT		"0x%08x   0x%04x  0x%04x    0x%04x  0x%04x   "
#define PID_MASK	0xffff


ProcessId TeamServer;
int Tindex;
ProcessId Teams[64];
static struct
  {
    ArbitraryDescriptor entry;
    char name[sizeof(ArbitraryDescriptor)];
  } TeamName;

extern ProcessId GetPid(), Creator();

ProcessId PidOfTeamFileName();
int Biopsy();



main( argc, argv)
  int argc;
  char *argv[];
  {
    ProcessId pid;
    Message msg;

    /*
     * Start up the actual biopsy routine in a separate process so that
     * it can have a larger stack than is currently available to the main
     * process.
     */
    pid = Create(4, Biopsy, 10000);
    if (pid== 0)
      {
        fprintf(stderr, "Error in creation of biopsy process!\n");
	fprintf(stderr, "Probable cause: out of memory.\n");
	exit(1);
      }
    if (!Ready(pid, 2, argc, argv))
      {
        fprintf(stderr, "Error in readying biopsy process!\n");
	exit(1);
      }
    Receive(msg);		/* Hang this process */
    exit(msg[0]);		/* until child returns status */
  }



/*
 * Biopsy:
 * Actual biopsy routine.  This has been separated out from the main 
 * program because it needs a larger stack than is currently available to
 * the main process.  Thus it is run as a separate process!
 */

Biopsy(argc, argv)
    int argc;
    char *argv[];
  {
    int longForm = FALSE;	/* Signals long or short form of output. */
    int teamsFlag = FALSE;	/* Signals whether pids should be interpreted
				   as designating teams or just processes. */

    int allFlag = FALSE;	/* Signals whether all processes should be
    				   printed. */
    register ProcessId headpid;
    ProcessId pid;
    Message msg;

    /*
     * Process options arguments.
     */
    argv++;			/* Skip over biopsy program name. */
    argc--;
    while ((argc > 0) && (argv[0][0] == '-'))
      {
        if (Equal(argv[0], "-l"))
	  {
	    longForm = TRUE;
	  }
	else if (Equal(argv[0], "-t"))
	  {
	    teamsFlag = TRUE;
	  }
	else if (Equal(argv[0], "-a"))
	  {
	    allFlag = TRUE;
	  }
	else
	  {
	    fprintf(stderr, "%s is not a valid option for this command\n", 
	    		argv[0]);
	    msg[0] = 1;
	    Send(msg, Creator(0));
	  }
	argv++;
	argc--;
      }

    /*
     * Print process info. either for all processes or specific ones only.
     */
    TeamServer = GetPid(TEAM_SERVER, LOCAL_PID);
    headpid = GetPid( ACTIVE_PROCESS, LOCAL_PID )&(~PID_MASK);
    printf("\n");
    if ( argc == 0)
	/* Do a general process dump */
      {
	/* first find the head of the show */
	headpid = 0;
	while( (pid = Creator(headpid)) != 0 ) headpid = pid;

	PrintAll(headpid, longForm);
      }
    else			/* Print out full process state of each
				   process specified. */
      {
        for (; argc-- > 0; argv++)
	  {
	    if ((argv[0][0] == '0') && (argv[0][1] == 'x'))
	      {
	        argv[0] += 2;
		if (sscanf( *argv, "%x", &pid ) != 1)
		  {
		    fprintf(stderr, "Invalid process id specified: %s\n\n",
		    		*argv);
		    continue;
		  }
	      }
	    else
	      {
		pid = PidOfTeamFileName(*argv);
		if (pid == 0)
		  {
		    fprintf(stderr,
		    		"Invalid team filename specified: %s\n\n", 
		    		*argv);
		    continue;
		  }
	      }
	    if ( pid <= PID_MASK)
	      {
	        pid |= headpid;	/* allow the user to leave the host part of 
				   the pid blank */
	      }
	    if (allFlag)
	      {
		/* first find the head of the show */
		headpid = pid;
		while( (pid = Creator(headpid)) != 0 ) headpid = pid;
		PrintAll(headpid, longForm);
	      }
	    else if (teamsFlag)
	      {
		PrintTeam(pid);
	      }
	    else
	      {
		PrintProcess(pid);
	      }
	  }
      }
    msg[0] = 0;
    Send(msg, Creator(0));	/* Exit the program. */
  }


/*
 * PrintAll:
 * Print information about all processes running.
 * If longForm is true then print each team's filename.
 */

PrintAll(pid, longForm)
    ProcessId pid;
    int longForm;
  {
    int i;

    /* If the short form is to be employed then print out column headers
       only once at the beginning. */
    if (!longForm)
      {
	puts( FORMAT_HDR1 );
	puts( FORMAT_HDR2 );
      }
    Tindex = 0;
    FindTeams(pid);
    for (i = 0; i < Tindex; i++)
      {
        if (longForm)		/* Print out team filename and repeat the
				   field headers. */
	  {
	    printf("\nTeam:  ");
	    if (i == 0)		/* The first team will always be here because
				   we start at the root of the process tree.
				   */
	      {
		printf("first team\n");
	      }
	    else if (GetTeamName(Teams[i]) != NULL)
	      {
		printf("%-40s\n", TeamName.entry.t.fileName);
	      }
	    else		/* Default case - we don't know anything about
				   the team. */
	      {
		printf("%x\n", Teams[i]);
	      }
	    puts( FORMAT_HDR1 );
	    puts( FORMAT_HDR2 );
	  }
	NextNode(Teams[i], Teams[i]);
	printf("\n");
      }
  }


/*
 * FindTeams:
 * Recursively descend the tree of processes and find the roots of all
 * teams.
 */

FindTeams(pid)
    ProcessId pid;		/* Root of tree to start from. */
  {
    ProcessBlock pb;

    if (pid == 0)
	return;

    if ( QueryProcessState( pid, &pb ) != pid)
      {
	fprintf(stderr, "Invalid Process Identifier 0x%x\n", pid );
	return;
      }

    if (!SameTeam(pid, pb.father))
      {
	Teams[Tindex++] = pid;
      }
    FindTeams(pb.son);
    FindTeams(pb.brother);
  }


/*
 * NextNode:
 * Recursively called routine which
 * prints out process information on the current team's processes.
 */

NextNode(pid, rootPid)
    register ProcessId pid;
    ProcessId rootPid;
  {
    ProcessBlock pb;

    if (pid == 0)
      {
	return;
      }

    if ( QueryProcessState( pid, &pb ) != pid)
      {
	fprintf(stderr, "Invalid Process Identifier 0x%x\n", pid );
	return;
      }

    if (SameTeam(pid, rootPid))
      {
	printf(FORMAT, pid, pb.father&PID_MASK, pb.team_root&PID_MASK
		, pb.priority, pb.team_priority );
	PrintState( &pb );
	NextNode(pb.son, rootPid);
      }

    NextNode(pb.brother, rootPid);
  }


/*
 * PidOfTeamFileName:
 * Returns the pid corresponding to the team filename specified by s.
 * Returns 0 if s does not correspond to an actual team.
 * s is matched against SUFFIXES of team filenames returned from the 
 * team server.
 */

ProcessId PidOfTeamFileName(s)
    char *s;
  {
    ProcessId headPid, pid;
    int i;

    if (Tindex == 0)		/* Haven't obtained a list of all team root
				   pids yet. */
      {
        /* Get the root pid of the whole system. */
        headPid = 0;
	while ((pid = Creator(headPid)) != 0)
	  {
	    headPid = pid;
	  }
	/* Get a list of all team root pids. */
	FindTeams(headPid);
      }

    /* Compare s to the suffix of the filename of each team in the system. */
    for (i = 0; i < Tindex; i++)
      {
	if (GetTeamName(Teams[i]) != NULL)
	  {
	    if (MatchesSuffix(TeamName.entry.t.fileName, s))
	      {
		break;
	      }
	  }
      }
    if (i == Tindex)
      {
	return(0);
      }
    else
      {
	return(Teams[i]);
      }
  }


/*
 * MatchesSuffix:
 * Returns TRUE if s1 is a suffix of s; else returns FALSE.
 */

int MatchesSuffix(s, s1)
    char *s, *s1;
  {
    char *p;
    char *p1;

    for (p = s + strlen(s) - 1, p1 = s1 + strlen(s1) -1;
    	 (p >= s) && (p1 >= s1);
	 p--, p1--)
      {
	if (*p != *p1)
	  {
	    break;
	  }
      }
    if (p1 < s1)
      {
	return(TRUE);
      }
    else
      {
	return(FALSE);
      }
  }


/*
 * GetTeamName:
 * Get the team filename for the specified pid.  TeamName is used to store
 * the team descriptor obtained.
 */

int GetTeamName(pid)
    ProcessId pid;
  {
    Message msg;
    DescriptorRequest *reqMsg = (DescriptorRequest *) msg;
    DescriptorReply *repMsg = (DescriptorReply *) msg;

    TeamName.name[0] = '\0';	/* Team server is queried by team root, 
				   not by name. */
    reqMsg->requestcode = NREAD_DESCRIPTOR;
    reqMsg->nameindex = ((int) &(TeamName.name[0])) - (int)(&TeamName);
    reqMsg->dataindex = 0;
    reqMsg->fileid = 0;
    reqMsg->namecontextid = (ContextId) pid;
    reqMsg->segmentptr = (char *) (&TeamName);
    reqMsg->segmentlen = sizeof(TeamName);
    TeamServer = Send(msg, TeamServer);
    if (repMsg->replycode != OK)
	return(0);
    return(1);
  }


/*
 * PrintState:
 * Print out the state of the process in form other than hex digits.
 * The cases are self explanatory.
 */

PrintState( pb)
    register ProcessBlock *pb;
  {
    switch (pb->state)
      {
	case READY:
	    printf("Ready");
	    break;

	case RECEIVE_BLKED:
	    printf("Receive blocked");
	    if (pb->blocked_on == 0)
		printf(", nonspecific");
	    else
	        printf(" on %x",pb->blocked_on);
	    break;

	case AWAITING_REPLY:
	    printf("Awaiting reply from %x",pb->blocked_on);
	    break;

	case SEND_BLKED:
	    printf("Send blocked on %x",pb->blocked_on);
	    break;

	case AWAITING_INT:
	    printf("Awaiting interrupt");
	    break;

	case DELAYING:
	    printf("Delaying");
	    break;

	case GETPID_BLKED:
	    printf("Remote GetPid, logical id %d",pb->blocked_on);
	    break;

	case MOVEFROM_BLKED:
	    printf("Remote MoveFrom, from %x",pb->blocked_on);
	    break;

	case MOVETO_BLKED:
	    printf("Remote MoveTo, to %x",pb->blocked_on);
	    break;

	default:
	    printf("Unknown state %d", pb->state );
	    break;

      }
    putchar('\n');
  }


/*
 * PrintTeam:
 * Same semantics as PrintAll except that info. is printed only for the
 * designated team.  The pid of any process on a team may be used to 
 * designate a team.
 */

PrintTeam(pid)
    ProcessId pid;
  {
    int firstTeamFlag = FALSE;
    ProcessBlock pb;

    /* Get the root pid of the team. */
    if (QueryProcessState(pid, &pb) != pid)
      {
	fprintf(stderr, "Invalid Process Identifier 0x%x\n", pid);
	return;
      }
    pid = pb.team_root;
    if (Creator(pid) == 0)	/* Is this the first team? */
      {
	firstTeamFlag = TRUE;
      }

    /* Print info. about this team. */
    printf("\nTeam:  ");
    if (firstTeamFlag)
      {
	printf("first team\n");
      }
    else if (GetTeamName(pid) != NULL)
      {
	printf("%-40s\n", TeamName.entry.t.fileName);
      }
    else
      {
	printf("%x\n", pid);	/* Don't know the team's name! */
      }
    puts( FORMAT_HDR1 );
    puts( FORMAT_HDR2 );
    NextNode(pid, pid);

    /* Print team's memory size. */
    printf("Team memory size: %d (0x%x)\n", pb.team_size, pb.team_size);
    printf("\n");
  }


/*
 * PrintProcess:
 * Print out everything that is in the process block.
 */

PrintProcess( pid )
    ProcessId pid;
  {
    ProcessBlock processblock;
    register ProcessBlock *pb = &processblock;

    if ( QueryProcessState( pid, pb ) != pid )
      {
	fprintf(stderr, "Invalid Process Identifier 0x%x\n", pid );
	return;
      }

    printf( "Information about Process 0x%x\n\n", pid );
    /* Convert Process specific info to a non-sexist format */
    printf(
"Process: Link 0x%x, Parent 0x%x,\n\t Sibling 0x%x, Child 0x%x, User 0x%x\n",
pb->link, pb->father, pb->brother, pb->son, User(pid) );

    /* team fields */
    printf( "Team: Root 0x%x, Priority 0x%04x, Size 0x%x\n",
	    pb->team_root, pb->team_priority, pb->team_size );

    /* Not sure what this stuff does, but regurgitate it anyway */
    printf( "Msg: Forwarded by 0x%x, Queue Head 0x%x, Queue Tail 0x%x\n"
		, pb->forwardedby, pb->msg_queue, pb->msg_queue_end );

    printf( "State: " );
    PrintState( pb );
    printf("pc: %x\n", pb->ps.proc_state.pc);
    putchar('\n' );
  }
