/*  amaze : a distributed multi-person Vkernel game.
 *  Copyright (c) 1983  Eric J. Berglund  &  David R. Cheriton
 *
 * This file, processes.c, is the 2nd of the 8 which make up the game amaze.
 * It contains the processes which make up the team that reads one player's
 * moves, determines the state of the other players in the game and displays
 * its perception of the game.
 *
 * It contains the routines:  main, KeyboardReader, AutoPilot, GameTimer,
 * StatusInquirer, and MissileMessenger.  All other files, except amaze.h,
 * contain routines called directly or indirectly by main.  None of these
 * other processes call other routines--they can quite properly be called
 * helper processes.
 */

#include "amaze.h"


/* main initializes the game and then serves as the game manager for this
 * player.  Its function is to receive one of the seven possible requests
 * ( given as requestcodes in amaze.h ), pass it to a routine which
 * handles that request, and then send back a reply.  The game manager
 * is the only process which Replies, and the only one which does not Send
 * messages.
 */

main( argc, argv )
	int argc;
	char *argv[];
  {
    extern	GameState *Initialize();
    Message	msg;
    GameRequest *req = (GameRequest *) msg;
    GameReply	*replymsg = (GameReply *) msg;
    ProcessId	pid;
    SystemCode	reply;
    GameState	*state;

    if( (state = Initialize()) == NULL ) return;

    while(1)
      {
	pid = Receive( req );

	switch( req->requestcode )
	  {
	    case GAME_INPUT_CHAR:
		reply = ProcessInputChar( state, req, pid );
		break;

	    case GAME_TIMER:
		/* Reply right away so timer can keep real time. */
		replymsg->replycode = OK;
		Reply( replymsg, pid );
		ProcessTimer( state, req, pid );
		replymsg->replycode = NO_REPLY;
		break;

	    case REQUEST_MONSTER_STATE:
		reply = ReturnLocalMonsterState( state, req, pid );
		break;

	    case REPORT_MONSTER_STATE:
		reply = UpdateRemoteMonsterState( state, req, pid );
		break;

	    case JOIN_GAME:
		reply = JoinGame( state, req, pid );
		break;

	    case AUTO_PILOT_CHAR:
		reply = ProcessAutoPilot( state, req, pid );
		break;

	    case MISSILE_REPORT:
		reply = NoteMissile( state, req, pid );
		break;

	    default:
		reply = NO_REPLY;
	  }

	if( reply != NO_REPLY )
	  {
	    replymsg->replycode = reply;
	    Reply( replymsg, pid );
	  }
      }
  }



/* The KeyboardReader process reads from the workstation keyboard and sends
 * the input character to the game manager.
 */

KeyboardReader( mymanagerpid )
	ProcessId mymanagerpid;
  {
    Message msg;
    register GameRequest *req = (GameRequest *) msg;

    while(1)
      {
	req->inputchar = getchar();
	req->requestcode = GAME_INPUT_CHAR;

	Send( req, mymanagerpid );
      }
  }





/* The autopilot process randomly generates direction characters for this
 * monster and sends them to the manager in the same form as the KeyboardReader.
 */

AutoPilot( mymanagerpid )
	ProcessId mymanagerpid;
  {
    extern rand();
    Message msg;
    register GameRequest *req = (GameRequest *) msg;

    while(1)
      {
	Delay( 1, 0 );  /* Delay of 1 Second */

	switch ( rand() % 4 )
	  {

	    /* Up */
	    case 0:  req->inputchar = 'i';
		     break;

	    /* Down */
	    case 1:  req->inputchar = ',';
		     break;

	    /* Left */
	    case 2:  req->inputchar = 'j';
		     break;

	    /* Right */
	    case 3:  req->inputchar = 'l';
		     break;

	  }

	req->requestcode = AUTO_PILOT_CHAR;

	Send( req, mymanagerpid );
      }
  }



/* The timer process for the game manager sends a message every delayclicks
 * clicks, which tells the manager to move each ALIVE monster in its
 * specified direction, and redraw the monsters in their new positions.
 */

GameTimer( mymanagerpid, delayclicks, state )
	ProcessId mymanagerpid;
	unsigned delayclicks;
	GameState *state;
  {
    Message msg;
    register GameRequest *req = (GameRequest *) msg;

    while( 1 )
     {
	Delay( 0, delayclicks);
	req->requestcode = GAME_TIMER;
	Send( req, mymanagerpid );
      }
  }




/* The status inquirer gets the state of the specified remote monster from
 * the specified remote manager and reports this its manager.
 * Each monster has one status inquirer process for each of the other
 * monsters.
 */

StatusInquirer( mymonsternumber, mymanagerpid, remotemonsternumber,
		remotemanager )
	unsigned mymonsternumber;
	ProcessId mymanagerpid;
	unsigned  remotemonsternumber;
	ProcessId remotemanager;

  {
    Message msg;
    register StatusInquirerRequest *req = (StatusInquirerRequest *) msg;
    StatusInquirerReply *replymsg = (StatusInquirerReply *) msg;
    ProcessId pid;

    while(1)
      {
	/* Put together the request for status information. */

	req->requestcode = REQUEST_MONSTER_STATE;
	req->monsternumber = mymonsternumber;
	req->manager = mymanagerpid;

	pid = Send( req, remotemanager );

	/* If the Send fails, we assume that the player has quit the game,
	 * setting his living_status to CREMATED so that the local manager
	 * can open his former spot to allow another player to join.
	 */

	if( pid == 0 || replymsg->replycode != OK )
	    req->living_status = CREMATED;

	/* Now report the monster's state to the local manager. */

	req->requestcode = REPORT_MONSTER_STATE;
	req->monsternumber = remotemonsternumber;
	req->manager = remotemanager;

	/* Monster state info is same as in reply from remote manager */

	Send( req, mymanagerpid );
      }
  }




/* MissileMessenger sends a MISSILE_REPORT to remotemanager, and then
 * destroys itself when the message is replied to.
 */

MissileMessenger( remotemanager, mymonsternumber, mymissile )
	ProcessId remotemanager;
	unsigned mymonsternumber;
	register MissileState *mymissile;
  {
    Message msg;
    register MissileRequest *req = ( MissileRequest * )  msg;

    req->requestcode = MISSILE_REPORT;
    req->x = mymissile->x;
    req->y = mymissile->y;
    req->direction = mymissile->direction;
    req->cornerindex = mymissile->cornerindex;
    req->monsternumber = mymonsternumber;
    Send( req, remotemanager );
    DestroyProcess( 0 );
  }
