/*  amaze : a distributed multi-person Vkernel game.
 *  Copyright (c) 1983  Eric J. Berglund  &&  David R. Cheriton
 *
 *  This file, inputhandler.c, is the 4th of the 8 files which make up the
 *   distributed game amaze.  It contains the two routines which the game
 *   manager ( main ) calls in response to a GAME_INPUT_CHAR or AUTO_PILOT_CHAR
 *   request:  ProcessInputChar and ProcessAutoPilot, respectively.  It
 *   also contains FireIfReady, which is called by ProcessInputChar to
 *   determine whether a missile can be fired in the current state.
 */

#include "amaze.h"

extern Node Corners[];
extern Edge Corridors[];


/* ProcessInputChar takes the action specified by the user's key strokes.
 *  No command is more than one character in length.  A quick summary of
 *  the commands:
 *
 *	i	Move monster up now or at next intersection.
 *	,	Move monster down now or at next intersection.
 *	j	Move monster left now or at next intersection.
 *	l	Move monster right now or at next intersection.
 *	k	Stop the monster.
 *	a	Let the monster's moves be chosen randomly by the autopilot.
 *
 *	e	Fire a missile up, if possible.
 *	c	Fire a missile down, if possible.
 *	s	Fire a missile left, if possible.
 *	f	Fire a missile right, if possible.
 *
 *	h	Hide the monster from other players--but no shooting allowed.
 *	v	Let the monster be seen again--can shoot again, too.
 *
 *	0	Set the monster's velocity to 0.
 *	1	Set the monster's velocity to VELOCITY1.
 *	2	Set the monster's velocity to VELOCITY2.
 *	3	Set the monster's velocity to VELOCITY3.
 *	4	Set the monster's velocity to VELOCITY4.
 *	5	Set the monster's velocity to VELOCITY5.
 *	6	Set the monster's velocity to VELOCITY6.
 *	7	Set the monster's velocity to VELOCITY7.
 *	8	Set the monster's velocity to VELOCITY8.
 *	9	Set the monster's velocity to VELOCITY9.
 *
 *	q	Quit the game, but continue to watch other players.
 *	^	Rejoin the game just above the door.
 *	r	Rejoin the game at a random corner in the maze.
 *	Ctrl-C	Exit the program.
 *
 *	R	Redraw the maze.
 */

ProcessInputChar( state, req, pid )
	GameState *state;
	GameRequest *req;
	ProcessId pid;
  {
    extern AutoPilot(), rand();
    register MonsterState *mymonster;
    Edge *mycorridor;
    Node *mycorner;
    MissileState *mymissile;
    int autopilotkill, i;

    if( pid != state->mykeyboardreader )
      {
	printf(" NO_REPLY in ProcessInputChar \n");
	return( NO_REPLY );
      }

    mymonster = &( state->monsterstates[ state->mymonsternumber ] );
    mycorridor = &( Corridors[ mymonster->corridorindex ] );
    mymissile = &( mymonster->missile );

    autopilotkill = FALSE;

    /* In responding to any input character, I update the state representation
     * of MY monster; there is no way for me to directly affect the state of
     * any other monster.  In this way, we avoid the synchronization problems
     * which arise when there are more than one possible writers to a single
     * data item.
     * In many cases, after updating my state, I notify any waiting status
     * inquirers of the change.
     */

    switch( req->inputchar )
      {

	/* Directions of movement. */

	    case 'i' :
		if( mymonster->next_dir != UP )
		  {
		    mymonster->next_dir = UP;
		    if( mycorridor->orientation == VERTICAL )
		      {
			mymonster->current_dir = UP;
			mymonster->cornerindex = mycorridor->endptindex 
								[ NEAR_CORNER ];
		      }
		    NotifyOthers( state );
		  }
		autopilotkill = TRUE;
		break;

	    case ',' :
		if( mymonster->next_dir != DOWN )
		  {
		    mymonster->next_dir = DOWN;
		    if( mycorridor->orientation == VERTICAL )
		      {
			mymonster->current_dir = DOWN;
			mymonster->cornerindex = mycorridor->endptindex 
								[ FAR_CORNER ];
		      }
		    NotifyOthers( state );
		  }
		autopilotkill = TRUE;
		break;

	    case 'j' :
		if( mymonster->next_dir != LEFT )
		  {
		    mymonster->next_dir = LEFT;
		    if( mycorridor->orientation == HORIZONTAL &&
/* Gross */		!( mymonster->current_dir == NONE && mymonster->
			   cornerindex == LEFT_WRAP_AROUND_CORNER ) )
		      {
			mymonster->current_dir = LEFT;
			mymonster->cornerindex = mycorridor->endptindex 
								[ NEAR_CORNER ];
		      }
		    NotifyOthers( state );
		  }
		autopilotkill = TRUE;
		break;

	    case 'l' :
		if( mymonster->next_dir != RIGHT )
		  {
		    mymonster->next_dir = RIGHT;
		    if( mycorridor->orientation == HORIZONTAL )
		      {
			mymonster->current_dir = RIGHT;
			mymonster->cornerindex = mycorridor->endptindex 
								[ FAR_CORNER ];
		      }
		    NotifyOthers( state );
		  }
		autopilotkill = TRUE;
		break;

	    case 'k' :
		if( mymonster->next_dir != NONE )
		  {
		    mymonster->next_dir = NONE;
		    mymonster->current_dir = NONE;
		    NotifyOthers( state );
		  }
		autopilotkill = TRUE;
		break;

	    case 'a' :
		if (state->myautopilot == NULL_PROCESS)
		  { 
		     state->myautopilot = Create(4,AutoPilot,256);
		     Ready(state->myautopilot,1,state->mymanager);
		  }
		break;


	/* Visibility. */

	    case 'h' :
		if( mymonster->living_status == ALIVE && mymonster->count == 0 )
		  {
		    mymonster->living_status = HIDDEN;
		    mymonster->count = HIDDEN_COUNT;
		    NotifyOthers( state );
		  }
		break;

	    case 'v' :
		if( mymonster->living_status == HIDDEN )
		  {
		    mymonster->living_status = ALIVE;
		    mymonster->count = VISIBLE_COUNT;
		    NotifyOthers( state );
		  }
		break;


	/* Firing missiles. */

	    case 'e' :
		if( mymissile->state==READY && mymonster->living_status==ALIVE )
		  FireIfReady( mymonster, mycorridor, UP, state );
		break;

	    case 'c' :
		if( mymissile->state==READY && mymonster->living_status==ALIVE )
		  FireIfReady( mymonster, mycorridor, DOWN, state );
		break;

	    case 's' :
		if( mymissile->state==READY && mymonster->living_status==ALIVE )
		  FireIfReady( mymonster, mycorridor, LEFT, state );
		break;

	    case 'f' :
		if( mymissile->state==READY && mymonster->living_status==ALIVE )
		  FireIfReady( mymonster, mycorridor, RIGHT, state );
		break;


	/* Velocity. */

	    case '0' :
		mymonster->velocity = 0;
		NotifyOthers( state );
		break;

	    case '1' :
		mymonster->velocity = VELOCITY1;
		NotifyOthers( state );
		break;

	    case '2' :
		mymonster->velocity = VELOCITY2;
		NotifyOthers( state );
		break;

	    case '3' :
		mymonster->velocity = VELOCITY3;
		NotifyOthers( state );
		break;

	    case '4' :
		mymonster->velocity = VELOCITY4;
		NotifyOthers( state );
		break;

	    case '5' :
		mymonster->velocity = VELOCITY5;
		NotifyOthers( state );
		break;

	    case '6' :
		mymonster->velocity = VELOCITY6;
		NotifyOthers( state );
		break;

	    case '7' :
		mymonster->velocity = VELOCITY7;
		NotifyOthers( state );
		break;

	    case '8' :
		mymonster->velocity = VELOCITY8;
		NotifyOthers( state );
		break;

	    case '9' :
		mymonster->velocity = VELOCITY9;
		NotifyOthers( state );
		break;


	/* Game participation. */

	    case 'q' :
		if( mymonster->living_status == ALIVE )
		  {
		    mymonster->living_status = DYING;
		    mymonster->count = DYING_COUNT;
		    NotifyOthers( state );
		  }
		autopilotkill = TRUE;
		break;

	    case '^' :
		if( mymonster->living_status == EMBRYO )
		  {
		    mymonster->living_status = ALIVE;
		    mymonster->count = 0;
		    mymonster->y = FIRST_ALIVE_Y;
		    mymonster->x = FIRST_ALIVE_X;
		    NotifyOthers( state );
		  }
		break;

	    case 'r' :
		if( mymonster->living_status == EMBRYO )
		  {
		    mymonster->living_status = ALIVE;
		    mymonster->count = 0;
		    mymonster->cornerindex = rand() % NUM_CORNERS;

		    if( mymonster->cornerindex == START_CORNER )
			mymonster->cornerindex = FIRST_CORNERINDEX;
/*		    if( mymonster->cornerindex == LEFT_WRAP_AROUND_CORNER )
 *			mymonster->cornerindex = RIGHT_WRAP_AROUND_CORNER;
 */
		    mycorner = &( Corners[ mymonster->cornerindex ] );
		    mymonster->x = mycorner->x;
		    mymonster->y = mycorner->y;

		    /* Instead of cascading else ifs, I have taken advantage
		     * of the values of UP, DOWN, LEFT, and RIGHT ( 0, 1, 2,
		     * and 3, respectively, to write this loop.  Note, though,
		     * that the key advantage of those values is the ability
		     * to use them as subscripts into the corridorindex array
		     * that is part of each corner structure.
		     */

		    i = UP;
		    while( mycorner->corridorindex[ i ] == WALLED )
			i++;
		    mymonster->corridorindex = mycorner->corridorindex[ i ];

		    mymonster->current_dir = NONE;
		    mymonster->next_dir = NONE;
		    NotifyOthers( state );
		  }
		break;

	    case '\03' :            		/* Control-C */
		ClearScreen();
/*		PrintTestResults( state );
 */
		exit();

	/* Redraw Command. */
	    case 'R' :
		DrawMaze();
		PrepareForRedraw( state );
		DrawPicture( state );
		break;


/* I used this debug routine as a convenient place to set a breakpoint
 * when using the debugger.
 *
 *	    case 'd' :
 *		Debug();
 *		break;
 */
      }


    if (state->myautopilot != NULL_PROCESS  &&  autopilotkill)
      {
	DestroyProcess( state->myautopilot ); 
	state->myautopilot = NULL_PROCESS;
      }

    return( OK );
  }


/* Process a character from the AutoPilot.  Currently, the autopilot only
 * provides direction characters, so this routine could easily be subsumed
 * in ProcessInputChar.  However, at some future time there will come an
 * ambitious soul who will add some intelligence to the autopilot that will,
 * perhaps, make it a worthy opponent.
 */

ProcessAutoPilot( state, req, pid )
	GameState *state;
	GameRequest *req;
	ProcessId pid;
  {
    register Edge *mycorridor;
    register MonsterState *mymonster;

    if( pid != state->myautopilot )
      {
	printf(" NO_REPLY in ProcessAutoPilot \n");
	return( NO_REPLY );
      }

    mymonster = &(state->monsterstates[state->mymonsternumber]);
    mycorridor = &( Corridors[ mymonster->corridorindex ] );


    switch( req->inputchar )
      {
	    case 'i' :
		if( mymonster->next_dir != UP )
		  {
		    mymonster->next_dir = UP;
		    if( mycorridor->orientation == VERTICAL )
		      {
			mymonster->current_dir = UP;
			mymonster->cornerindex = mycorridor->endptindex 
							[ NEAR_CORNER ];
		      }
		    NotifyOthers( state );
		  }
		break;

	    case ',' :
		if( mymonster->next_dir != DOWN )
		  {
		    mymonster->next_dir = DOWN;
		    if( mycorridor->orientation == VERTICAL &&
			mymonster->corridorindex != FIRST_CORRIDORINDEX )
		      {
			mymonster->current_dir = DOWN;
			mymonster->cornerindex = mycorridor->endptindex 
							[ FAR_CORNER ];
		      }
		    NotifyOthers( state );
		  }
		break;

	    case 'j' :
		if( mymonster->next_dir != LEFT )
		  {
		    mymonster->next_dir = LEFT;
		    if( mycorridor->orientation == HORIZONTAL &&
/* Gross */		!( mymonster->current_dir == NONE && mymonster->
			   cornerindex == LEFT_WRAP_AROUND_CORNER ) )
		      {
			mymonster->current_dir = LEFT;
			mymonster->cornerindex = mycorridor->endptindex 
							[ NEAR_CORNER ];
		      }
		    NotifyOthers( state );
		  }
		break;

	    case 'l' :
		if( mymonster->next_dir != RIGHT )
		  {
		    mymonster->next_dir = RIGHT;
		    if( mycorridor->orientation == HORIZONTAL )
		      {
			mymonster->current_dir = RIGHT;
			mymonster->cornerindex = mycorridor->endptindex 
							[ FAR_CORNER ];
		      }
		    NotifyOthers( state );
		  }
		break;

      }
    return( OK );
  }



/* FireIfReady does a series of tests to determine whether one of the
 * conditions exists that allows a shot to be fired and, if so, sets
 * variables appropriately to launch the missile.
 */

FireIfReady( mymonster, mycorridor, shotDirection, state )
	MonsterState *mymonster;
	Edge *mycorridor;
	short int shotDirection;
	GameState *state;
  {
    register MissileState *mymissile;
    short int shotOrientation, farOrNearCorner;
    Node *endptNear, *endptFar;
    int distFromFar, distFromNear;

    mymissile = &( mymonster->missile );

    if( shotDirection == UP || shotDirection == DOWN )
	shotOrientation = VERTICAL;
    else
	shotOrientation = HORIZONTAL;

    if( shotDirection == UP || shotDirection == LEFT )
	farOrNearCorner = NEAR_CORNER;
    else
	farOrNearCorner = FAR_CORNER;

    endptNear = &( Corners[ mycorridor->endptindex[ NEAR_CORNER ] ] );
    endptFar = &( Corners[ mycorridor->endptindex[ FAR_CORNER ] ] );

    /* If my monster is close enough to the Near corner of his corridor, and
     * if there is a corridor leaving from that corner in the direction I wish
     * to shoot, then shoot!  Note the special case:  If he's at the leftmost
     * edge of the screen and shooting left, the missile starts at the
     * rightmost edge.
     * Here again, I use the values of the directions to index the
     * corridorindex array.
     */

    distFromNear = Abs( mymonster->x - endptNear->x ) +
					Abs( mymonster->y - endptNear->y);
    distFromFar = Abs( mymonster->x - endptFar->x ) +
					Abs( mymonster->y - endptFar->y);

    if( distFromNear <= CLOSE_ENOUGH_TO_CORNER  )
      {
        if( distFromFar <= CLOSE_ENOUGH_TO_CORNER  &&
	  endptFar->corridorindex[ shotDirection ] != WALLED &&
	  !( endptNear->corridorindex[ shotDirection ] != WALLED  &&
	  distFromNear < distFromFar ) )

	  {
	    if( endptFar == &( Corners[ RIGHT_WRAP_AROUND_CORNER ] )
						 && shotDirection == RIGHT )
	      endptFar = &( Corners[ LEFT_WRAP_AROUND_CORNER ] );
	    mymissile->cornerindex = Corridors[ endptFar->
	     corridorindex[ shotDirection ] ].endptindex[ farOrNearCorner ];
	    mymissile->x = endptFar->x;
	    mymissile->y = endptFar->y;
	    mymissile->direction = shotDirection;
	    mymissile->state = TRIGGER_PULLED;
	    SendMissileMessage( state );
	  }

	else if( endptNear->corridorindex[ shotDirection ] != WALLED )
	  {
	    if( endptNear == &( Corners[ LEFT_WRAP_AROUND_CORNER ] )
						 && shotDirection == LEFT )
	      endptNear = &( Corners[ RIGHT_WRAP_AROUND_CORNER ] );
	    mymissile->cornerindex = Corridors[ endptNear->
	     corridorindex[ shotDirection ] ].endptindex[ farOrNearCorner ];
	    mymissile->x = endptNear->x;
	    mymissile->y = endptNear->y;
	    mymissile->direction = shotDirection;
	    mymissile->state = TRIGGER_PULLED;
	    SendMissileMessage( state );
	  }
     }

    /* If my monster is close enough to the Far corner of his corridor, and
     * if there is a corridor leaving from that corner in the direction I wish
     * to shoot, then shoot!  Note the special case:  If he's at the rightmost
     * edge of the screen and shooting right, the missile starts at the
     * leftmost edge.
     */

    else if( distFromFar <= CLOSE_ENOUGH_TO_CORNER )
     {
      if( endptFar->corridorindex[ shotDirection ] != WALLED )
	{
	  if( endptFar == &( Corners[ RIGHT_WRAP_AROUND_CORNER ] )
						 && shotDirection == RIGHT )
	    endptFar = &( Corners[ LEFT_WRAP_AROUND_CORNER ] );
	  mymissile->cornerindex = Corridors[ endptFar->
	    corridorindex[ shotDirection ] ].endptindex[ farOrNearCorner ];
	  mymissile->x = endptFar->x;
	  mymissile->y = endptFar->y;
	  mymissile->direction = shotDirection;
	  mymissile->state = TRIGGER_PULLED;
	  SendMissileMessage( state );
	}
     }

    /* Even if my monster's not near a corner, he can still shoot down the
     * corridor he's travelling.
     */

    else if( mycorridor->orientation == shotOrientation )
	{
	  mymissile->cornerindex = mycorridor->endptindex[ farOrNearCorner ];
	  mymissile->x = mymonster->x;
	  mymissile->y = mymonster->y;
	  mymissile->direction = shotDirection;
	  mymissile->state = TRIGGER_PULLED;
	  SendMissileMessage( state );
	}
  }
