/*
 * V Kernel - Copyright (c) 1982 by David Cheriton, Tim Mann
 *
 * Memory mapping routines--machine dependent
 *
 * $Revision: 1.43.1.11 $
 * $Locker:  $
 * $State: Exp $
 */

#include <Vquerykernel.h>
#include <b.out.h>
#include "config.h"
#include "externals.h"
#include "memory.h"
#include "uvaxmemory.h"
#include "process.h"
#include "uiocache.h"
#include "vm.h"
#ifdef VM
#include "meminstance.h"
#endif VM


#ifdef FIREFLY

#include "firefly.h"

#endif FIREFLY

/* Forward */
void InvalidateTLB _TAKES(( unsigned, int));
PageTableEntry  *GetTeamPTE _TAKES((Team *, char *));
BooleanInt TeamPageIsAccessible _TAKES ((Team	*, unsigned, unsigned *));
void MakePTEs _TAKES((PageTableEntry *, unsigned, unsigned));

/* Exports */
struct bhdr	KernelBheader = {0},
		TeamBheader = {0};	/* Force to data segment */
Process		*AddressableProcess;	/* the currently addressable process */
unsigned	AddressableTemp;	/* temp for SetAddressableProcess */
unsigned long KFlags = 0;		/* Kernel flags passed by Vload */

PageTableEntry	*SysPageTable;

struct vaxMemoryMap vaxMap	= {0};

unsigned num_sysPT_entries; 		

int SysMallocReady	= 0;

/*
 * Variables used to construct the System Page Table; these are initialized
 *   by Init_memory_mapping() and then manipulated by the BuildSysMap_*()
 *   routines which follow it.  The page table has to occupy contiguous
 *   physical memory; we place immediately after the first team and it grows
 *   toward high memory.  We also need to allocate memory (physically
 *   contiguous in some cases) for process/team descriptors, the virtual
 *   memory structures and so forth; for this we start at the top of memory
 *   and work down.
 *	building_map	is just kiddie-proofing; it makes sure the BuildSysMap
 *			routines are only called when appropriate.
 *	alloc_phys_addr	is the lowest address allocated so far for the above-
 *			mentioned data structures
 *	alloc_virt_addr is the starting virtual address of the next object we
 *			will create in system space
 *	alloc_sys_pte	is the physical address of the current end of the page
 *			table (i.e. the place to write the first PTE for the
 *			next object we create)
 */
static int building_map = 0;
static char *alloc_phys_addr, *alloc_virt_addr;
static PageTableEntry *alloc_sys_pte;

void
Init_memory_mapping()
  /*
   * Setup the memory mapping as required.  This routines starts off in
   *   Physical memory and ends up in Virtual memory.  It prints a message
   *   if it thinks it succeeds.
   *
   * We assume the loader program has placed the kernel at KERNEL_START and
   *   the first team at PHYS_TEAM_START, with the team's b.out header just
   *   below PHYS_TEAM_START, and the kernel's b.out header just below that.
   *
   * We assume that all the physical memory is allocated sequentially
   *   starting from 0.  If there are holes, things get more complicated.
   *
   * Physical memory-map:
   *	00000000: Kernel text, data, bss
   *		  ... free space (add to free-page list)
   *    00040000: On the firefly, cache initialization code goes here.
   *	00040000+VPAGESIZE: First team text, data, bss
   *		  System Page-Table
   *		  ... free space (add to free-page list)
   *		  Any other memory needed by things in the kernel
   *	End-of-physical-memory:
   *
   *    [Some of the "free space" is allocated later for objects in the
   *     kernel's virtual memory-map, such as P0 page-tables for teams.]
   *
   * Virtual memory-map (system space):
   *	80000000: Kernel text, data, bss
   *		  Space requested by other code (could be for I/O devices,
   *		    memory management for particular breeds of VAX, arrays of
   *		    process- and team-descriptors, etc.)
   *		  P0 page-tables for teams
   *		  Miscellaneous pages used by this memory manager
   *		  Whatever the machine-independent VM manager wants (probably
   *		    just the PageFrameArray)
   *		  System Page-Table
   */
  {
    register int		r11;

    extern char etext, edata, end;

    /* Copy the b.out headers into kernel data space */
    TeamBheader = *((struct bhdr *)(PHYS_TEAM_START - sizeof(struct bhdr)));
    KernelBheader = *((struct bhdr *)(PHYS_TEAM_START-2*sizeof(struct bhdr)));
	/* now the kernel flags */
	printx("PHYS_TEAM_START-(2*sizeof(struct bhdr)+sizeof(ulong)) = %x\n",
		PHYS_TEAM_START-(2*sizeof(struct bhdr)+sizeof(unsigned long)));
	KFlags = *((unsigned long *)(PHYS_TEAM_START-(2*sizeof(struct bhdr)+
			sizeof(unsigned long))));
	printx("KFlags = %x\n",KFlags);

    /* If the first team is NMAGIC (410), then the loader should have moved */
    /*   the data section to the first 1K boundary after the text.  We      */
    /*   round tsize up to the next 1K - a bit of a crock, but then we can  */
    /*   forget about the problem.					    */
    if ( TeamBheader.fmagic == NMAGIC )
	TeamBheader.tsize = (TeamBheader.tsize + SEGSIZE - 1) & ~(SEGSIZE-1);

    /* Zero the bss areas in case the loader didn't do it */
    bzero((char *)(PHYS_TEAM_START + TeamBheader.tsize + TeamBheader.dsize),
	  (int)TeamBheader.bsize);
    bzero(KERNEL_START + &edata, &end-&edata);

    /*
     * We choose to align everything on V page boundaries (1K) rather than just
     *   VAX page boundaries.  Then, if we ever do things like freeing space in
     *   the first team, everything should work OK.
     */

    vaxMap.kernel.phys  = (char *)KERNEL_START;
    vaxMap.kernel.bytes = VUpToPage(
		KernelBheader.tsize+KernelBheader.dsize+KernelBheader.bsize );

    vaxMap.team1.phys  = (char *)PHYS_TEAM_START;
    vaxMap.team1.bytes = VUpToPage(
		TeamBheader.tsize+TeamBheader.dsize+TeamBheader.bsize );

/*
 * Check to see if any of the data Vload put here overlaps.  We should allow
 * stuff to be on top of the kernel's bss, and move team1 up, but I don't
 * trust it yet.  The real solution is to have Vload put things in the right
 * place and pass us the address (ews).
 */

    if (vaxMap.kernel.phys + vaxMap.kernel.bytes > (char *)PHYS_TEAM_START)
	Kabort("Kernel too big; first team (and Vload?) must be moved");
#ifdef FIREFLY
/*
 * Force the cache initialization code to be between the kernel and the
 * first team.
 */
    if ((char *)FF_SECONDARY_INIT_PC < 
         vaxMap.kernel.phys+vaxMap.kernel.bytes
	|| vaxMap.team1.phys <= (char *)FF_SECONDARY_INIT_PC)
	Kabort("cache code overlaps first team or kernel.");
#endif FIREFLY
    /*
     * Find out how much physical memory (assumed to be contiguous from 0)
     *   we have.  We don't want GetMemorySize munging the code that we're
     *   executing right now, so we tell it to start looking after the last
     *   physical address in the kernel text/data/bss.
     */

    MachineConfig.fastMemory = GetMemorySize((unsigned)vaxMap.team1.phys);
    MachineConfig.slowMemory = 0;
    MachineConfig.memory = MachineConfig.fastMemory+MachineConfig.slowMemory;

    printx("Memory: 0x%x bytes.\r\n", MachineConfig.memory);
    if (vaxMap.team1.phys + vaxMap.team1.bytes  > (char *)MachineConfig.memory)
	Kabort("Not enough fast memory for kernel and first team (???)");

    FindMachineType();
    BasicInit_SpecificVAX();
    FindConfiguration();

    vaxMap.sysPT.phys = (char *)
			VUpToPage(vaxMap.team1.phys + vaxMap.team1.bytes);

    alloc_sys_pte   = (PageTableEntry *)vaxMap.sysPT.phys;
    alloc_virt_addr = (char *)SYS_SEG_START;
    alloc_phys_addr = (char *)MachineConfig.memory;
    building_map    = 1;

    /*
     * Map the kernel text/data/bss first so that we get up the mapping
     *   physical(x) = virtual(80000000+x) that everyone expects.
     */
    vaxMap.kernel.virt	= BuildSysMap_MapPhysPages
				(vaxMap.kernel.phys, MachineConfig.memory);
#ifdef FIREFLY
    vaxMap.ipcr_space.phys = (char *)FF_INTERPROCESSOR_COMM_REG_SPACE_PHYS_ADDR;
    vaxMap.ipcr_space.bytes  = 4*1024;
    vaxMap.ipcr_space.virt = BuildSysMap_MapPhysPages
				(vaxMap.ipcr_space.phys, vaxMap.ipcr_space.bytes);
    ipcrs_v = vaxMap.ipcr_space.virt; 
#endif FIREFLY
    vaxMap.space.bytes = 2 * VPAGESIZE * VPAGESIZE;
    vaxMap.space.virt = BuildSysMap_LeaveSpace(vaxMap.space.bytes);
    SysMallocReady |= 1;
    vaxMap.teamPTs.virt	= BuildSysMap_LeaveSpace
				(KernelConfig.maxTeams * TEAM_P0PT_BYTES);
    BuildSysMap_SpecificVAX();
    BuildSysMap_Config();
    BuildSysMap_VMmanager();
    BuildSysMap_TDs();

/*
 * We have to map the pte's into kernel virtual space, so we can get to
 * them.  Although we don't really need to access these new pte's after
 * we turn on the MMU, for historical reasons we make them map themselves.
 * Which leads to the following calculation.
 *
 * pps	= pte's to map in System Page Table
 *	= ceil( (bytes of pte's so far + pps * 8) / 1024)
 *
 * This leaves some core where we round up the pagesize; should give that
 * slop to vaxMap.space or something.
 */
#define QQQ (VPAGESIZE - sizeof (PageTableEntry) * PAGESIZERATIO)
    vaxMap.sysPT.bytes = VPAGESIZE
	* ( (( ((char *)alloc_sys_pte) - vaxMap.sysPT.phys) + QQQ - 1)
	   / QQQ);
#undef QQQ
    vaxMap.sysPT.virt = BuildSysMap_MapPhysPages(vaxMap.sysPT.phys,
						vaxMap.sysPT.bytes);
    num_sysPT_entries = alloc_sys_pte - (PageTableEntry *)vaxMap.sysPT.phys;

    building_map = 0;
    /* vaxMap.end.phys is just the end of the slice which contains the first */
    /*   team and the System Page Table.				     */
    vaxMap.end.phys = vaxMap.sysPT.phys + vaxMap.sysPT.bytes;
    vaxMap.end.virt = vaxMap.sysPT.virt + vaxMap.sysPT.bytes;
    vaxMap.end.bytes= 0;

    if (vaxMap.end.phys > alloc_phys_addr)
	Kabort("Ran out of room while building memory map");

    SysPageTable = (PageTableEntry *)vaxMap.sysPT.virt;

printx("SYSPAGETABLE = %x\n", SysPageTable);

#define CalcPTE(f, title)						      \
    vaxMap.f.pte = &SysPageTable[MD_PageNumber(vaxMap.f.virt-SYS_SEG_START)]; \
 /* printx("%13s: %08x %08x %08x %08x\n", title,			      \
		vaxMap.f.phys, vaxMap.f.virt, vaxMap.f.bytes, vaxMap.f.pte) */

    CalcPTE(kernel , "text/data/bss");
    vaxMap.team1.pte= (PageTableEntry *)BOGUS_ADDR;
    CalcPTE(teamPTs, "team P0 PTEs");
    CalcPTE(space  , "data structure space");
    CalcPTE(sysPT  , "system PTEs");
    CalcPTE(end    , "(end of map)");

    printx("Memory mapping complete; ");

    EnableMemMap();		/* Turn on memory management */

    InitMem_SpecificVAX();
    ChangeSysPagesProtection((unsigned)vaxMap.kernel.virt,
				MD_DownToPage(KernelBheader.tsize), VM_R___);

    /*
     * Now tell the machine-independent VM manager routines what they need to
     *   know.  First give them a chance to set everything up...
     */
    InitPageTables();
    /*
     *   .. and then tell them what parts of physical memory are actually free:
     */
    MarkPhysMemFree(0, vaxMap.kernel.phys);/* A no-op; vaxMap.kernel.phys = 0*/
#ifdef FIREFLY
    MarkPhysMemFree(vaxMap.kernel.phys+vaxMap.kernel.bytes,
		    (char *)FF_SECONDARY_INIT_PC); /* probably-op */
    MarkPhysMemFree((char *)FF_SECONDARY_INIT_PC, vaxMap.team1.phys); /* no-op */
#else !FIREFLY
    MarkPhysMemFree(vaxMap.kernel.phys+vaxMap.kernel.bytes, vaxMap.team1.phys);
#endif
    MarkPhysMemFree(vaxMap.end.phys, alloc_phys_addr);
    SysMallocReady |= 2;
  }

/* Initialize the internal registers which turn on memory management */

void
EnableMemMap()
  {
    register int r11;
    r11 = (unsigned)vaxMap.sysPT.phys;
    asm("	mtpr	r11, $sbr");
    r11 = num_sysPT_entries;
    asm("	mtpr	r11, $slr");
    asm("	mtpr	$0, $p0br");
    asm("	mtpr	$0, $p0lr");
    asm("	mtpr	$-1, $p1br");
    asm("	mtpr	$0x200000, $p1lr");

    /* turn on memory management */
    asm("	mtpr	$0, $tbia");
    asm("	mtpr	$1, $mapen");
  }


/*
 * BuildSysMap_MapPhysPages - build a list of Page Table Entries pointing to a
 *   slice of the physical address space.  Does not allocate pages (assumes
 *   that the caller has a good reason for mapping that range of physical
 *   addresses).  Can only be used early in the boot process when we're
 *   building the system-space memory map.
 * We round start address down, and limit up, to V page boundaries, not just
 *   VAX boundaries (it is assumed everywhere that VPAGESIZE > MD_PAGESIZE).
 *   It's not really clear whether this is necessary, but it may save some
 *   grief if we attempt to deallocate parts of the system address space.
 */
char *
BuildSysMap_MapPhysPages(start_phys_address, num_bytes)
    char		*start_phys_address;
    long		num_bytes;
  {
    register PageTableEntry *pte;
    register unsigned pfn;
    register long npages;
    char *start_virt_address;

    if (!building_map)
      {
	Kabort("BuildSysMap_MapPhysPages called at wrong time");
	return BOGUS_ADDR;
      }
    start_virt_address	= alloc_virt_addr + VByteInPage(start_phys_address);
    pfn			= VPageNumber(start_phys_address) * PAGESIZERATIO;
    npages		= NumVPages(start_phys_address + num_bytes) *
				PAGESIZERATIO - pfn;
    if (npages <= 0)
	return BOGUS_ADDR;
    alloc_virt_addr    += npages << MD_PAGEBITS;
    pte			= alloc_sys_pte;
    while (--npages >= 0)
	(pte++)->word = pfn++ | VALID_SYS_PAGE | VM_RW__;
    alloc_sys_pte	= pte;
    return start_virt_address;
  }

/*
 * Reserve a slice of the system virtual address space; fill the appropriate
 *   part of the system page table with "invalid PTE" entries.
 */
char *
BuildSysMap_LeaveSpace(num_bytes)
    register long	num_bytes;
  {
    register PageTableEntry *pte;
    register long npages;
    char *start_virt_address;

    if (!building_map)
      {
	Kabort("BuildSysMap_LeaveSpace called at wrong time");
	return BOGUS_ADDR;
      }
    start_virt_address	= alloc_virt_addr;
    num_bytes		= VUpToPage(num_bytes);
    if (num_bytes <= 0)
	return BOGUS_ADDR;
    npages		= num_bytes >> MD_PAGEBITS;
    alloc_virt_addr    += num_bytes;
    pte			= alloc_sys_pte;
    while (--npages >= 0)
	(pte++)->word = NOPAGE;
    alloc_sys_pte	= pte;
    return start_virt_address;
  }

char *
BuildSysMap_Contiguous(num_bytes)
    register long	num_bytes;
  {
    register PageTableEntry *pte;
    register unsigned pfn;
    register long npages;
    char *start_virt_address;

    if (!building_map)
      {
	Kabort("BuildSysMap_Contiguous called at wrong time");
	return BOGUS_ADDR;
      }
    start_virt_address	= alloc_virt_addr;
    num_bytes		= VUpToPage(num_bytes);
    if (num_bytes <= 0)
	return BOGUS_ADDR;
    npages		= num_bytes       >> MD_PAGEBITS;
    alloc_virt_addr    += num_bytes;
    alloc_phys_addr    -= num_bytes;
    pfn			= (unsigned)alloc_phys_addr >> MD_PAGEBITS;
    pte			= alloc_sys_pte;
    while (--npages >= 0)
	(pte++)->word = pfn++ | VALID_SYS_PAGE | VM_RW__;
    alloc_sys_pte	= pte;
    return start_virt_address;
  }

char *
BuildSysMap_AllocPages(num_bytes)
    unsigned long num_bytes;
  {
    /*
     * Caller said he didn't actually need physically contiguous pages, but
     *   we'll give them to him anyway.
     */
    return BuildSysMap_Contiguous(num_bytes);
  }


void
GetTeam1Location(team1_physaddr, team1_bytes)
    char     **team1_physaddr;
    unsigned *team1_bytes;
  {
    *team1_physaddr = (char *)vaxMap.team1.phys;
    *team1_bytes    =	      vaxMap.team1.bytes;
  }

void
ChangeSysPagesProtection(sys_virt_address, bytes, new_protection)
    register unsigned sys_virt_address;
    unsigned bytes;
    unsigned new_protection;
  {
    register PageTableEntry *pte;
    register int i;
    int MD_PagesToChange;

    new_protection &= VM_PROTECTION_MASK;
    pte = SysPageTable +
		MD_NumPages( MD_DownToPage(sys_virt_address) - SYS_SEG_START);
#ifdef REAL_OPTIMIZER
    MD_PagesToChange = MD_NumPages( MD_UpToPage(sys_virt_address+bytes) -
				   MD_DownToPage(sys_virt_address) );
#else
    MD_PagesToChange = MD_NumPages( sys_virt_address + bytes -
				   MD_DownToPage(sys_virt_address) );
#endif
    for (i = MD_PagesToChange; i > 0; i--)
      {
	if (pte->word != NOPAGE)
	    pte->word = pte->word & ~VM_PROTECTION_MASK | new_protection;
	++pte;
      }
    InvalidateTLB(sys_virt_address, MD_PagesToChange);
  }

void
InvalidateTLB(virt_addr, npages)
    register unsigned virt_addr;	/* r11 */
    register int npages;
  {
    for ( ; --npages >= 0; virt_addr += MD_PAGESIZE)
	{ ; asm("mtpr	r11,	$tbis"); }
  }


/*
 * Ms_td_init(): Machine-specific initialization for a team descriptor.
 *		 Done just once at boot time, not every time we recycle the
 *		 descriptor.
 */
void
Ms_td_init(td, i)
    register Team *td;
    register unsigned i;
  {
    td->team_space.p0len = 0;	/* the initial number of pages used */
    td->team_space.p0base = (unsigned)vaxMap.teamPTs.virt + i*TEAM_P0PT_BYTES;
  }


/*
 * KMapPage: Try to map physical Vpage (frameVfn) into P0 address space of
 *	     team (td) at (address), with user access rights = (permissions).
 * Returns 1 if successful, 0 if failed (because it couldn't allocate a page
 *	     needed for P0 page-tables; wait until the free-page list is non-
 *	     empty and try again).
 */
int
KMapPage(td, address, frameVfn, permissions)
    Team	*td;
    unsigned	address;
    unsigned	frameVfn;	/* Number of frame (physical address */
				/*   divided by VPAGESIZE)	     */
    unsigned	permissions;
  {
    register int		r11;
    register PageTableEntry	*pte,
				*sysPte;
    register unsigned		permMask;
    register int		pageNum,
				i;
	     unsigned		newPage;

    pte = & ((PageTableEntry *)td->team_space.p0base)
				[VPageNumber(address) * PAGESIZERATIO];
    sysPte = &SysPageTable[ MD_PageNumber((unsigned)pte - SYS_SEG_START) ];
    pageNum = frameVfn * PAGESIZERATIO;

    if (permissions & VM_WRITE)
	permMask = VM_RWRW;
    else if (permissions & VM_READ || permissions & VM_EXECUTE)
	permMask = VM_RWR_;
    else
	permMask = VM_RW__;	/* 0 means kernel-only access */
    permMask |= VALID_USER_PAGE;

    if (sysPte->word == NOPAGE)
      {
	if (AllocatePage_Vfn(&newPage) == 0)
	    return 0;
	newPage *= PAGESIZERATIO;

	/* We're adding PAGESIZERATIO pages to the page-table, not just one, */
	/*   so we have to align the pointer to an appropriate boundary.     */
	sysPte -= (sysPte - SysPageTable) % PAGESIZERATIO;

	for (i = PAGESIZERATIO; --i >= 0; ++newPage)
	    (sysPte++)->word = newPage | VALID_USER_PT_PAGE | VM_RW__;
	/* ----- should really use NOPAGE here */
	bzero((char *)VDownToPage(pte), VPAGESIZE);
      }
    if (pte->word & VM_MAPPED)
      {
	DecrTimesMapped(pte->fields.pfn / PAGESIZERATIO);
	for (r11 = address, i = PAGESIZERATIO; --i >= 0; r11 += MD_PAGESIZE)
	  {
	    ;asm("mtpr	r11,	$tbis");
	  }
      }
    for (i = PAGESIZERATIO; --i >= 0; ++pageNum)
      {
	(pte++)->word = permMask | pageNum;
      }
    if (td->team_space.p0len < MD_PageNumber(address + VPAGESIZE))
      {
	td->team_space.p0len = MD_PageNumber(address + VPAGESIZE);
      }
    return 1;
  }

void
KUnmapPages(td, start, limit)
	     Team	*td;
    register unsigned	start; /* r11 */
	     unsigned	limit;
  {
    register PageTableEntry *pte, *sysPte;
    register int vpages_left, vpages_left_in_sysptepg, i;
	/* "PTE block" here means "enough PTES to map one V (not VAX) page" */
	/*   ("enough" == PAGESIZERATIO).				    */

    pte = (PageTableEntry *)td->team_space.p0base + MD_PageNumber(start);
    sysPte = &SysPageTable[ MD_PageNumber((unsigned)pte - SYS_SEG_START) ];
    vpages_left_in_sysptepg = (MD_PAGESIZE - MD_PageOffs(pte)) /
				(PAGESIZERATIO * sizeof *pte);

    vpages_left = VPageNumber(VUpToPage(limit)) - VPageNumber(start);
    while (vpages_left > 0)
	/*
	 * Do one pass of this outer loop for each VAX page of PTEs
	 *   (we have to check whether the top-level PTE for the page
	 *    says it really isn't there).
	 */
      {
	if (vpages_left_in_sysptepg > vpages_left)
	    vpages_left_in_sysptepg = vpages_left;
	vpages_left -= vpages_left_in_sysptepg;
	if ( (sysPte++)->word & VM_VALID )
	  {
	    while (--vpages_left_in_sysptepg >= 0)
		/*
		 * Do one pass through this for each pte block
		 */
	      {
		if (pte->word & VM_MAPPED)
		    DecrTimesMapped(pte->fields.pfn / PAGESIZERATIO);
		for (i = PAGESIZERATIO; --i >= 0; )
		  {
		    pte->word = NOPAGE;
	/* Flush the TLB for this address.  If the currently AddressableProcess */
	/*   is not equal to 'td' then this is unnecessary, but we're likely */
	/*   to lose more than we gain if we check for it.		     */
		    asm("	mtpr	r11,	$tbis");
		    ++pte;
		    start += MD_PAGESIZE;
		  }
	      }
	  }
	else
	  {
	    pte   += vpages_left_in_sysptepg * PAGESIZERATIO;
	    start += vpages_left_in_sysptepg * VPAGESIZE;
	  }
	vpages_left_in_sysptepg = MD_PAGESIZE / (PAGESIZERATIO * sizeof *pte);
      }
/* ------- might also be nice if it checked the possibility of unmapping a 
	PTE page.
*/
  }


void
PhysicalInterTeamCopy(dest, destPd, src, srcPd, bytes)
   char * dest;
   Process * destPd;
   char * src;
   Process * srcPd;
   unsigned long bytes;
  /*
   * This routine is used by MoveFrom and MoveTo to copy bytes between
   *   different teams.  We do have to worry about the case where one
   *   of the `teams' is in fact the kernel process.
   */

  {
    register int r11;
    Process *oldPd;

    /* save current context */
    oldPd = GetAddressableProcess();

    /* map source team into P0 */
    if (((int)src & SYS_SEG_START) == 0) {
	r11 = srcPd->team->team_space.p0len;
	asm("	mtpr	r11, $p0lr");
	r11 = srcPd->team->team_space.p0base;
	asm("	mtpr	r11, $p0br");
    }
    
    /* do the copy */
    if (((int)dest & SYS_SEG_START) == 0) {
	/* map destination team into P1 */
	asm("	mtpr	$0, $p1lr");
	r11 = destPd->team->team_space.p0base;
	asm("	mtpr	r11, $p1br");
#ifndef PRODUCTION
if (GetTeamPTE(destPd->team, dest)->word == 0) {
    Process	*tpd;
    printx("pte=%x, p0br=%x, dest=%x ", *GetTeamPTE(destPd->team, dest),
	   destPd->team->team_space.p0base, dest);
#endif !PRODUCTION
    tpd = MapPid(destPd->team->team_root);
    printx("tpd=%x,segmentPtr=%x,segmentSize=%x,dataSegmentPtr=%x,dataSegmentSize=%x,state=%x ", tpd, tpd->srcSegment.ptr, tpd->srcSegment.size, tpd->dstSegment.ptr, tpd->dstSegment.size, tpd->state);
  }
/* --- should we invalidate the translation buffer ? */
	Copy((char *)(P1_SEG_START | MD_SegOffs(dest)), src, (unsigned)bytes);

	asm("	mtpr	$0x200000, $p1lr");
	asm("	mtpr	$-1, $p1br");
    }
    else
	Copy(dest, src, (unsigned)bytes);

    /* restore the old contexts */
    SetAddressableProcess(oldPd);
  }


char *
VirtToPhysAddr(sys_virt_addr)
    char *sys_virt_addr;
  {
    /* Gross: should be rewritten KH 10/90 */
    if (((int)sys_virt_addr) >= 0)
	Kabort("VirtToPhysAddr only works for System-space addresses");
    if (sys_virt_addr > vaxMap.end.virt)
	Kabort("VirtToPhysAddr: system address out of range");
    /* ---- Should really check whether it's mapped, too. */
    return MD_PageOffs(sys_virt_addr) + (char *)
	( SysPageTable[MD_PageNumber(sys_virt_addr-SYS_SEG_START)].fields.pfn
				<< MD_PAGEBITS );
  }

/*
 * More routines that tell you whether you can read a given location in
 *   memory.  These ones, however, are intended specifically for virtual
 *   addresses.  They don't actually try to read from the location, just
 *   look at the memory mapping registers and tables, which had better be set
 *   up properly.
 *
 * VIRTUAL MEMORY IMPLEMENTORS NOTE: If we ever decide to page the per-process
 *   (well, per-team) page tables, then the P0 and P1 cases will need some
 *   work.
 */

BooleanInt
KernelCanRead(address, length)
    unsigned	address, length;
  {
    unsigned	testadr;
    /* Test one byte in each page from address to address+length-1 */
    /*   If we can read one byte in the page, we can read anything */
    for (testadr = address; (unsigned)(testadr-address) < length;
			     /* ^ Does this work for all cases? */
	 testadr = (testadr+MD_PAGESIZE) & (-MD_PAGESIZE) /*gag*/)
	if (! KernelCanReadByte((char *)testadr))
	    return 0;
    return 1;
  }

BooleanInt
KernelCanReadByte(address)
    char	*address;
  {
    register unsigned r11 = MD_PageNumber(MD_SegOffs(address));
    register PageTableEntry *r10;

    switch((unsigned)address >> MD_SEGMENTBITS)
      {
	case 3:
	    ;asm("naughty:");
	    return 0;
	case 2:
	    asm("	mfpr	$slr,	r0");
	    asm("	cmpl	r11,	r0");
	    asm("	bgequ	naughty	  ");
	    r10 = &(SysPageTable[r11]);
	    break;
	case 1:	   /* We don't use P1 at present, but let's get this right */
	    asm("	mfpr	$p1lr,	r0");
	    asm("	cmpl	r11,	r0");	/* Bounds-check */
	    asm("	blssu	naughty	  ");
	    asm("	ashl	$2, r11,r10");	/* Form virtual address of */
	    asm("	mfpr	$p1br,	r0");   /*   P1 page table entry   */
	    asm("	addl2	r0,	r10");
		/* Check that this gives an address in System Space.  Should */
		/*   be, unless our software is really screwy or the hardware*/
		/*   is less clever than we thought.  Still, check it anyway;*/
		/*   if things went wrong we might otherwise recurse.	     */
	    asm("	bgeq	naughty   ");
	    if (! KernelCanReadByte((char *)r10))     /* We'll actually want to read*/
		return 0;		      /* a PTE (4 bytes) but this   */
	    break;			      /* gives the right answer     */
	case 0:
	    asm("	mfpr	$p0lr,	r0");
	    asm("	cmpl	r11,	r0");
	    asm("	bgequ	naughty   ");
	    asm("	ashl	$2, r11,r10");
	    asm("	mfpr	$p0br,	r0");
	    asm("	bgeq	naughty   "); /* Checks P0BR->System space  */
	    asm("	addl2	r0,	r10");
	    if (! KernelCanReadByte((char *)r10))
		return 0;
	    break;
      }
    if (!r10->fields.valid || r10->fields.protection < 2)
	return 0;
    return 1;
  }
  

ResponseCode
ClearModifiedPages( req, segPtr, segSize )
  KernelRequest *req;
  register long **segPtr;
  unsigned long segSize;
  {
    return MODE_NOT_SUPPORTED;  /* Some day... */
  }


/*
 * Get memory statistics for QueryKernel operation
 */
void
GetMemoryStats(reply)
    register MemoryStatisticsReply *reply;
  {
/* We go and ask the machine-independent VM routines.  Can we move this
   procesure to somewhere more sensible? */

    reply->unusedFastMemory = NumFreePages * VPAGESIZE;
    reply->unusedSlowMemory = 0;
  }

void
DumpMap( td )
	Team	*td;
{
	unsigned	i;
	int		acc;
	PageTableEntry	*pte;

	printx("Team %08x pt: ", td->team_root);
	
	pte = &SysPageTable
		[ MD_PageNumber(td->team_space.p0base - SYS_SEG_START) ];
printx("teamPT.virt = %08x, pte = %08x;  td->p0base = %08x, pte = %08x\n",
	vaxMap.teamPTs.virt, vaxMap.teamPTs.pte, td->team_space.p0base, pte);

	acc = (((pte->word) & VM_MAPPED) != 0);
	printx(acc ? "0.." : "[0..");

	for (i = 1, pte++;
	     i < MD_NumPages(TEAM_P0PT_BYTES);
	     i++, pte++) {
		if ( (((pte->word) & VM_MAPPED) != 0) != acc ) {
			printx(acc ? "%d " : "%d] ", i-1 );
			acc = !acc;
			printx(acc ? "%d.." : "[%d..", i);
		};
	};
	printx(acc ? "%d\n" : "%d]\n", i-1 );

	printx("Team %08x va: ", td->team_root);

	i = 0;
	acc = TeamPageIsAccessible(td, i, &i);
	printx(acc ? "0.." : "[0..");
	for( i++; i < MD_NumPages(TEAM_LIMIT); i++) {
		if (TeamPageIsAccessible(td, i, &i) != acc) {
			printx(acc ? "%d " : "%d] ", i-1 );
			acc = !acc;
			printx(acc ? "%d.." : "[%d..", i );
		}
	}
	printx(acc ? "%d\n" : "%d]\n", i-1 );
}

BooleanInt
TeamPageIsAccessible(td, team_page, new_page)
	Team	*td;
	unsigned team_page;
	unsigned *new_page;
{
	PageTableEntry *pte;

	pte = (PageTableEntry *)td->team_space.p0base + team_page;
	if ( (SysPageTable[MD_PageNumber(MD_SegOffs(pte))].word & VM_MAPPED)
		 != 0) 
          {
	    return( (pte->word & VM_MAPPED) != 0 );
	   } 
         else 
           {
		if (new_page) 
                  *new_page += (PTES_PER_MD_PAGE - 1);
/* -------------- why -1? */
		return( 0 );
	   }
}

/*
 * SysMalloc: return the kernel virtual address of space to hold the given
 * number of bytes.  The space will be in the vaxMap.space area, and is backed
 * from the freelist.
 *
 * On a multiprocessor, this need to be synchronized.  It's OK for allocating
 * PDs, since the FreePdq is locked, but I don't think TDs are
 * locked the same way.
 */
char	*
SysMalloc(bytes, contiguousp)
unsigned	bytes;
{
    static unsigned		 bytes_used	= 0;
    register char		*ptr;
    register PageTableEntry	*tptep;
		unsigned	 ClusterNumber;

#ifndef PRODUCTION
    if (SysMallocReady != 3)
	Kabort("SysMalloc: not initialized yet");
#endif
    if (bytes_used + bytes > vaxMap.space.bytes)
      {
	Kabort("SysMalloc: out of space");
	return NULL;
      }
/*
 * This will never push us past the space allowed.
 */
    if (contiguousp)
      {
	if (VByteInPage(bytes_used) + bytes > VPAGESIZE)
	    if (bytes > VPAGESIZE)
		Kabort("SysMalloc: Requested contiguous space larger than V page size");
	    else
		bytes_used = VUpToPage(bytes_used);
      }
    for (tptep = vaxMap.space.pte + VPageNumber(bytes_used) * PAGESIZERATIO;;)
      {
	if (tptep->word == NOPAGE)
	  {
	    if (AllocatePage_Vfn(&ClusterNumber) == 0)
		return NULL;
	    MakePTEs(tptep, ClusterNumber, VALID_SYS_PAGE | VM_RW__);
	  }
	tptep += PAGESIZERATIO;
	if (bytes + bytes_used <= MD_PAGESIZE * (tptep - vaxMap.space.pte))
	    break;
      }
    ptr = vaxMap.space.virt + bytes_used;
    bytes_used += bytes;
    return ptr;
}

/*
 * Make the pte's point to memory.
 */
void
MakePTEs(ptep, ClusterNumber, prot)
  register PageTableEntry	*ptep;
  unsigned	ClusterNumber;
  unsigned	prot;
{
    register int	count;
    register unsigned	pfn;

    pfn = ClusterNumber * PAGESIZERATIO;
    for (count = PAGESIZERATIO; count-- > 0; ptep++)
	{
	ptep->word = prot | pfn++;
	}
}

PageTableEntry *
GetTeamPTE(td, address)
   Team	*td;
   char	*address;
  {
    register PageTableEntry	*pte,
				*sysPte;

#ifndef PRODUCTION
    if (address > (char *)(td->team_size))
      {
	return NULL;
      }
#endif
    pte = & ((PageTableEntry *)td->team_space.p0base)
				[VPageNumber(address) * PAGESIZERATIO];
    sysPte = &SysPageTable[ MD_PageNumber((unsigned)pte - SYS_SEG_START) ];
    if (sysPte->word == NOPAGE)
	return NULL;
    return pte;
  }

PageFrame *
GetTeamFrameFast(pd)
   Process	*pd;
  {
    PageTableEntry	*ptep	= GetTeamPTE(pd->team, (char *)pd->vm.vaddr);

    if (ptep == NULL)
	return NULL;
    if (!ptep->fields.valid)
	return NULL;
    if ( (pd->vm.flags & VM_PAGEFRAME_WRITE)
	&& (ptep->word & VM_PROTECTION_MASK) != VM_RWRW)
	return NULL;
    return &PageFrames[ptep->fields.pfn / PAGESIZERATIO];
  }

unsigned
KernelVirtualToVPageNumber(vaddr)
   unsigned	vaddr;
  {
    PageTableEntry	*ptep;

    ptep = & SysPageTable[ MD_PageNumber(vaddr-SYS_SEG_START)];
    if (ptep->fields.valid == 0)
	Kabort("Kernel address not valid");
    return ptep->fields.pfn;
  }

#if PAGESIZERATIO != 2
	UnmapTeamFrameIfMapped won't work.
#endif

void
UnmapTeamFrameIfMapped(td, fp, vaddr)
   Team		*td;
   PageFrame	*fp;
   unsigned	 vaddr;
  {
    register PageTableEntry	*ptep;

/* WRONG: if (vaddr < pd->team.p0len/512) */
    ptep = GetTeamPTE(td, (char *)vaddr);
    if (ptep->word == NOPAGE)
      return;
    if (ptep->fields.pfn != (fp - PageFrames) * PAGESIZERATIO)
      {
	printx("pfn was %x, index was %x", ptep->fields.pfn, fp - PageFrames);
	Kabort("see above");
	return;
      }
    if (fp->timesMapped < 1)
      printx("Unmap: mapped %d times\n", fp->timesMapped);
    if (ptep[0].fields.modified || ptep[1].fields.modified)
      {
	fp->pfdFlags |= PFD_MODIFIED | PFD_DIRTY;
      }
    fp->timesMapped--;
    ptep[0].word = ptep[1].word = 0;    
  }

#ifdef VM
/*
 * Migrate the state of a page out of the various `caches', into core.
 * On the vax, this means gathering the modified bits from the pte's.
 *
 * As a side-effect, VM_REDUCE_ACCESS (to VM_READONLY_ACCESS or no access)
 * is applied to the pte's.
 */
void
FlushPageState(pf, flags)
   PageFrame	*pf;
   unsigned	 flags;
  {
    Team	*td;
    unsigned    vaddr;
    int		 ctr;
    BoundRegion	*binding	= NULL;
    PageTableEntry	*ptep;

    for (ctr = pf->timesMapped; ctr > 0; ctr--)
      {
	while ( (td = GetNextAddressMapping(pf, &vaddr, &binding)) != NULL)
	  {
    	    ptep = GetTeamPTE(td, (char *)vaddr);
	    /* The page could be deferred-copy, or not */
	    if (ptep->fields.pfn / PAGESIZERATIO != pf - PageFrames)
		continue;
	    if (ptep->fields.valid)
		break;
	  }
	if (td == NULL)
	    { printx("FPP: not mapped"); return; }
        if (ptep[0].fields.modified || ptep[1].fields.modified)
          {
	    pf->pfdFlags |= PFD_DIRTY | PFD_MODIFIED;
	    ptep[0].fields.modified = ptep[1].fields.modified = 0;
          }
	if (flags & VM_REDUCE_ACCESS)
	  {
	    if (flags & VM_READONLY_ACCESS)
		ptep[0].fields.protection = ptep[1].fields.protection
		     = (VM_R_R_ >> 27);
	    else
	      {
		ptep[0].word = ptep[1].word = NOPAGE;
		DecrTimesMapped(pf - PageFrames);
	      }
	  }
      }
  }
#endif VM

#ifdef DEBUG
/*
 * Print out data from a given team's space, in hex.
 */
void
TeamDataPrint(td, addr, bytes)
   Team	*td;
   char	*addr;
   unsigned long	bytes;
  {
    register int r11;
    Process *oldPd;

    /* save current context */
    oldPd = GetAddressableProcess();

    /* map source team into P0 */
    if (((int)addr & SYS_SEG_START) == 0) {
	r11 = td->team_space.p0len;
	asm("	mtpr	r11, $p0lr");
	r11 = td->team_space.p0base;
	asm("	mtpr	r11, $p0br");
    }

    for (r11 = 0; r11 < bytes; r11++)
	printx("%2x", addr[r11]);
  }
#endif DEBUG
