/*
 * contour.c - program to trace a contour (zero) of a polynomial in x,y
 *
 * Compile with -lgraphics
 *
 * Vaughan Pratt	July 17, 1981
 *
 * Bill Nowicki July 1982
 *	- Changed to use framebuf instead of graphmacs.h
 *	- clears the screen first
 */

#include "framebuf.h" 

/*
 * POINT(x,y) sets the given point to whatever the prevailing
 * function is.
 */
#define	POINT(x,y) { register short r;\
    GXsetX(x);\
    *(short*)(GXBase|GXupdate|GXsource|GXselectY|(y<<1))= r; \
    }

int init = 1000;
int GXBase;

typedef struct {int x; int y;} vector;

vector *vplus();

/* CGOL-style parser for parsing a polynomial
   Typical polynomial:	2x3+3x2y-13xy2+2y3-240
   Alternate notation:	2*x^3 + 3*x^2*y - 13*x*y^2 + 2*y^3 - 240
*/

char expr[200], *tok;
int xx,yy;

char lbp[128] = {
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,	/* ^@-^O */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,	/* ^P-^_ */
0,0,0,0,0,0,0,0,0,0,8,7,0,7,8,0,	/*   -/  */
9,9,9,9,9,9,9,9,9,9,0,0,0,0,0,0,	/*  0-?  */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,	/*  @-O  */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,0,	/*  P-_  */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,	/*  `-o  */
0,0,0,0,0,0,0,0,8,8,0,0,0,0,0,0		/*  p-^? */
};

parse(rbp) {		/* parse expression starting at tok */

	int t;
	for (t = nud(); rbp < lbp[*tok]; t = led(t));
	return t;
}

nud() {			/* parse and evaluate expression with no left arg */

	int tem, tk;
	switch (tk = *tok++) {
		case '+':	return parse(7);
		case '-':	return -parse(7);
		case 'x':	return xx;
		case 'y':	return yy;
		case ' ':	return nud();
		case '(':	tem = parse(0); tok++; return tem;
		default:	if ('0'<=tk && tk<='9')
					return nval(tk-'0');
				else return 100000;
	}
}

led(l) {		/* parse and evaluate expression with left arg l */

	int tk;
	switch (tk = *tok++) {
		case '+':	return l + parse(7);
		case '-':	return l - parse(7);
		case ' ':	return led(l);
		case '*':
		case '.':	return l * parse(8);
		case '^':	return expt(l,parse(8));
		case 'x':
		case 'y':	tok--; return l * parse(8);
		default:	if ('0'<=tk & tk<='9') {
					tok--;
					return expt(l,parse(9));
				}
				else return 100000;
	}
}

nval(n) {		/* parse and evaluate a number */

	char tk = *tok;
	if ('0' <= tk && tk <= '9') {
		tok++;
		return nval(n*10 + tk-'0');
	}
	else return n;
}

eval(v) vector *v; {	/* parse expression, evaluating x,y as v */

	xx = v->x; yy = v->y;
	tok = expr;
	return parse(0);
}

main() {

	register int x1, x2, y1, y2;
	int x, y;
	vector v;

	GXBase = GXProbe();
	if (!GXBase) {
		printf("No frame buffer\n");
		exit(1);
	}
	ScreenClear();
	v.x = v.y = 2;
	menu();
	while (1) {
		register short r;
		scanf("%s",expr);
		if (expr[0] == 'q') return;
		if (expr[0] == 'm') {
			menu();
			continue;
		}
		printf("");
		GXwidth = 1;
		GXfunction = GXset;
		y=GXbottom/2;			/* draw X axis */
		GXsetY(y);
		for (x=0; x<GXright; x++)
			*(short*)(GXBase|GXupdate|GXselectX|(x<<1) ) = r;
		x=GXright/2;			/* draw Y axis */
		GXsetX(x);
		for (y=0; y<GXbottom; y++) 
			*(short*)(GXBase|GXupdate|GXselectY|(y<<1) ) = r;
		contour(eval);
	}
}

menu()
{

      printf("Contour tracer.  Try some of the following expressions.\n");
      printf("\nDiagonal	x-y");
      printf("\nCircle		x2+y2-90000");
      printf("\nEllipse		x2+y2-xy-100000");
      printf("\nHypercircle	x4+y4-10^10");
      printf("\nTriangle	x2y-y2x-250xy-10000");
      printf("\nStar		x4+y4+200x2y2-10^10");
      printf("\nCusp		x3y2-2x2y-1000000");
      printf("\nFalls		1000xy-x3-100000y");
      printf("\nDeer (fake)	x6-10000y3");
      printf("\nJellyfish (fake)(100x+y2)3+(y+x3)2-100000");
      printf("\nQuit		q");
      printf("\nMenu		m");
      printf("\nRestart		c 1000");
      printf("\n\n");
}

onscreen(v) vector *v; {

	return -GXright/2 <= v->x && v->x < GXright/2 &&
		-GXbottom/2 <= v->y && v->y < GXbottom/2;
}

contour(f) int (*f)(); {
/* locate and draw one contour of f.  Assumes f not too wild.  */

	vector pos, dir, midpos, middir;
	int i, new, old;
	pos.x = pos.y = 0;		/*start at origin */
	dir.x = 1; dir.y = 0;		/*heading right */
	for (i = 0; i<10; i++) 		/* get direction right*/
		old = step(f,&pos,&dir);
	while ((new=step(f,&pos,&dir)) < old)	/* locate contour */
		old = new;
	for (i=0; i<2000 && !onscreen(&pos); i++)  /* get back on screen */
		step(f,&pos,&dir);
	for (i= 0; i<10; i++) step(f,&pos,&dir);	/* get into swing */
	midpos = pos, middir = dir;
	POINT(pos.x+400,512-pos.y)		/* draw this point */
	step(f,&pos,&dir);			/* leave midpos */
	POINT(pos.x+400,512-pos.y)		/* draw next point */
	for (i=0; i<2000 && onscreen(&pos) &&
		(pos.x != midpos.x || pos.y != midpos.y); i++) {
		step(f,&pos,&dir);
		POINT(pos.x+400,512-pos.y)
	}
	if (pos.x != midpos.x || pos.y != midpos.y) {
		pos = midpos;			/* back to old pos */
		dir.x = -middir.x; dir.y = -middir.y;/*but reverse direction */
		for (i=0; i<2000 && onscreen(&pos); i++) {  /* draw contour */
			step(f,&pos,&dir);
			POINT(pos.x+400,512-pos.y)
		}
	}
}

vector ll, rr;				/* storage for turnleft, turnright */

norm(x) {return x>0? 1: x==0? 0: -1;}	/* to get a unit lattice point */

vector *
turnleft(d) vector *d; {
/* return the vector rotated 45 degrees from d */

	ll.x = norm(d->x - d->y);
	ll.y = norm(d->x + d->y);
	return &ll;
}

vector *
turnright(d) vector *d; {
/* return the vector rotated -45 degrees from d */

	rr.x = norm( d->x + d->y);
	rr.y = norm(-d->x + d->y);
	return &rr;
}

step(f,pos,dir) int (*f)(); vector *pos, *dir; {
/* step one lattice point from pos in direction d within 45 degrees of dir,
with d chosen to minimize f(pos+d) */

	int left = abs((*f)(vplus(pos,turnleft (dir)))), /* try all three */
	    mid  = abs((*f)(vplus(pos,          dir ))), /* directions near */
	    right= abs((*f)(vplus(pos,turnright(dir)))); /* direction dir */
	
	if (left <= mid || right <= mid)	/* direction change needed? */
		if (left < right) { 		/* if so, which way? */
			*dir = *turnleft(dir);
			mid = left;
		}
		else {
			*dir = *turnright(dir);
			mid = right;
		}

	vadd(pos,dir);				/* take one step */
	return mid;
}

vector temp;	/* Yuck.  Replace with malloc when available. */

vector *
vplus(u,v) vector *u, *v; {

	temp.x = u->x + v->x;
	temp.y = u->y + v->y;
	return &temp;
}

vadd(u,v) vector *u, *v; {u->x += v->x; u->y += v->y;}


/* Exponentiation */

expt(a,b) {

	if (b <= 0) return 1;
	if (b & 1) return a * expt(a,b-1);
	return expt(a*a,b>>1);
}
