/*
 * postmortem.c
 *
 * DESCRIPTION
 * A simple program that looks through the process descriptors in
 *  the V kernel and prints out the current state of each living
 *  process.  This is intended for postmortems on user programs,
 *  not for debugging the kernel, so it assumes the kernel data
 *  structures are consistent.
 *
 * AUTHOR
 * Tim Mann
 *
 * HISTORY
 * 03/11/82 - TPM - Created.
 * 03/29/82 - TPM - Changed to use page 0 pointer to find Pd_bundle
 *	instead of asking user.  Added more information to printout.
 * 04/05/82 - TPM - Uses relative pathname
 * 04/29/82 - TPM - More information
 * 07/27/82 - WEZ - Pid printed out in hexadecimal.
 * 09/06/82 - TPM - Print team root pid, improved format.
 * 11/17/82 - TPM - Now knows about remote operations
 * 09/09/83 - TPM - Knows about new process states
 */

#include "../kernel/mi/process.h"

/* main:
 *
 * Gets the address of the Pd_bundle from a pointer provided by the
 *   kernel, then loops through the bundle, using info() to print
 *   information about each live process.
 * Kernel memory must be read-enabled for this to work.
 */
main()
  {
    Process **bundle, *queue;
    int i;

    bundle = P_Pd_bundle;
    for (i=0; i < MAX_PROCESSES; i++)
      {
	info(*bundle);
	bundle++;
      }

    /* Print pids of processes in ready queue */
    printf("Ready queue: ");
    queue = *P_Readyq_head;
    while (queue != NULL)
      {
	printf("%x ",queue->pid);
	queue = queue->link;
      }
    printf("\n");

    printf("Process %x is active\n",(*P_Active)->pid);

  }


/* info:
 *
 * Prints information about a process descriptor, if it represents a
 *  live process.  The information printed includes
 *	- The pid.
 *	- The current state (ready/blocked/etc.)
 *	- If blocked sending or receiving, the process blocked on.
 *	- The parent's pid.
 *	- The team root's pid.
 */

info(proc)
Process *proc;
  {

    if (proc->pid == 0)
	return;
    printf("Process %x (parent %x, team %x): ",proc->pid,
			proc->father->pid, proc->team->team_root);

    switch (proc->state)
      {
	case READY:
	    printf("Ready");
	    break;

	case RECEIVE_BLKED:
	    printf("Receive blocked");
	    if (proc->blocked_on == 0)
		printf(", nonspecific");
	    else
	        printf(" on %x",proc->blocked_on);
	    break;

	case AWAITING_REPLY:
	    printf("Awaiting reply from %x",proc->blocked_on);
	    break;

	case SEND_BLKED:
	    printf("Send blocked on %x",proc->blocked_on);
	    break;

	case AWAITING_INT:
	    printf("Awaiting interrupt");
	    break;

	case DELAYING:
	    printf("Delaying, %d clicks left",proc->blocked_on);
	    break;

	case GETPID_BLKED:
	    printf("Remote GetPid, logical id %d",proc->blocked_on);
	    break;

	case MOVEFROM_BLKED:
	    printf("Remote MoveFrom, from %x",proc->blocked_on);
	    break;

	case MOVETO_BLKED:
	    printf("Remote MoveTo, to %x",proc->blocked_on);
	    break;

	case REPLIED_TO:
	    printf("Alien, replied to by %x", proc->blocked_on);
	    break;

	case FORWARDED_TO:
	    printf("Alien, forwarded to %x", proc->blocked_on);
	    break;

	default:
	    printf("Unknown state %d", proc->state);
	    break;

      }

    printf("\n");

  }
